#! /usr/bin/perl -w
#
# ops2c.pl
#
# Generate a C header and source file from the operation definitions in
# an .ops file.
#

use strict;
use Parrot::OpsFile;

sub Usage {
    print STDERR <<_EOF_;
usage: $0 input.ops [input2.ops ...]\n";
_EOF_
    exit 1;
}

#
# Process command-line argument:
#

Usage() unless @ARGV;

my $file = 'core.ops';

my $base = $file;
$base =~ s/\.ops$//;

my $incdir  = "include/parrot/oplib";
my $include = "parrot/oplib/${base}_ops.h";
my $header  = "include/$include";
my $source  = "${base}_ops.c";


#
# Read the input file:
#
$file = shift @ARGV;
die "$0: Could not read ops file '$file'!\n" unless -e $file;

my $ops = new Parrot::OpsFile $file;

for $file (@ARGV) {
    die "$0: Could not read ops file '$file'!\n" unless -e $file;
    my $temp_ops = new Parrot::OpsFile $file;
    for(@{$temp_ops->{OPS}}) {
       push @{$ops->{OPS}},$_;
    }
}
my $cur_code = 0;
for(@{$ops->{OPS}}) {
   $_->{CODE}=$cur_code++;
}

my $num_ops     = scalar $ops->ops;
my $num_entries = $num_ops + 1; # For trailing NULL


#
# Open the output files:
#

if (! -d $incdir) {
    mkdir($incdir, 0755) or die "ops2c.pl: Could not mkdir $incdir $!!\n";
}

open HEADER, ">$header"
  or die "ops2c.pl: Could not open header file '$header' for writing: $!!\n";

open SOURCE, ">$source"
  or die "ops2c.pl: Could not open source file '$source' for writing: $!!\n";


#
# Print the preamble for the HEADER and SOURCE files:
#

my $preamble = <<END_C;
/*
** !!!!!!!   DO NOT EDIT THIS FILE   !!!!!!!
**
** This file is generated automatically from '$file'.
** Any changes made here will be lost!
*/

END_C

print HEADER $preamble;
print HEADER <<END_C;
#include "parrot/parrot.h"

extern INTVAL    ${base}_numops;

extern op_func_t ${base}_opfunc[$num_entries];
extern op_info_t ${base}_opinfo[$num_entries];

END_C

print SOURCE $preamble;
print SOURCE <<END_C;
#include "$include"

END_C

print SOURCE $ops->preamble;

print SOURCE <<END_C;

INTVAL ${base}_numops = $num_ops;

/*
** Op Function Table:
*/

op_func_t ${base}_opfunc[$num_entries] = {
END_C


#
# Iterate over the ops, appending HEADER and SOURCE fragments:
#

my @op_funcs;
my $index = 0;

foreach my $op ($ops->ops) {
    my $func_name  = $op->func_name;
    my $arg_types  = "opcode_t *, struct Parrot_Interp *";
    my $prototype  = "opcode_t * $func_name ($arg_types)";
    my $args       = "opcode_t cur_opcode[], struct Parrot_Interp * interpreter";
    my $definition = "opcode_t *\n$func_name ($args)";
    my $source     = $op->source(\&map_ret_abs, \&map_ret_rel, \&map_arg, \&map_res_abs, \&map_res_rel);

    print HEADER "$prototype;\n";
    print SOURCE sprintf("  %-22s /* %6ld */\n", "$func_name,", $index++);

    push @op_funcs, "$definition {\n$source}\n\n";
}

#
# Finish the SOURCE file's array initializer:
#

print SOURCE <<END_C;
  NULL
};


/*
** Op Function Definitions:
*/

END_C

print SOURCE @op_funcs;


#
# Op Info Table:
#

print SOURCE <<END_C;

/*
** Op Info Table:
*/

op_info_t ${base}_opinfo[$num_entries] = {
END_C

$index = 0;

foreach my $op ($ops->ops) {
    my $type       = sprintf("PARROT_%s_OP", uc $op->type);
    my $name       = $op->name;
    my $full_name  = $op->full_name;
    my $func_name  = $op->func_name;
    my $body       = $op->body;
    my $arg_count  = $op->size;
    my $arg_types  = "{ " . join(", ", map { sprintf("PARROT_ARG_%s", uc $_) } $op->arg_types) . " }";

    print SOURCE <<END_C;
  { /* $index */
    $type,
    "$name",
    "$full_name",
    "$func_name",
    "", /* TODO: Put the body here */
    $arg_count,
    $arg_types
  },
END_C

  $index++;
}

print SOURCE <<END_C;
};

END_C

exit 0;


#
# map_ret_abs()
#

sub map_ret_abs
{
  my ($addr) = @_;
  return "return $addr";
}


#
# map_ret_rel()
#

sub map_ret_rel
{
  my ($offset) = @_;
  return "return cur_opcode + $offset";
}


#
# map_arg()
#

sub map_arg
{
  my ($type, $num, $self) = @_;

  my %arg_maps = (
    'op' => "cur_opcode[%ld]",

    'i'  => "interpreter->int_reg->registers[cur_opcode[%ld]]",
    'n'  => "interpreter->num_reg->registers[cur_opcode[%ld]]",
    'p'  => "interpreter->pmc_reg->registers[cur_opcode[%ld]]",
    's'  => "interpreter->string_reg->registers[cur_opcode[%ld]]",
  
    'ic' => "cur_opcode[%ld]",
    'nc' => "interpreter->code->const_table->constants[cur_opcode[%ld]]->number",
    'pc' => "%ld /* ERROR: Don't know how to handle PMC constants yet! */",
    'sc' => "interpreter->code->const_table->constants[cur_opcode[%ld]]->string",
  );

  die "Unrecognized type '$type' for num '$num' in opcode @{[$self->full_name]}" unless exists $arg_maps{$type};

  return sprintf($arg_maps{$type}, $num);
}


#
# map_res_rel()
#

sub map_res_rel
{
  my ($offset) = @_;
  return "interpreter->resume_addr = cur_opcode + $offset";
}


#
# map_res_abs()
#

sub map_res_abs
{
  my ($addr) = @_;
  return "interpreter->resume_addr = $addr";
}


