package Sub::Spec::Gen::ForModule;
{
  $Sub::Spec::Gen::ForModule::VERSION = '0.02';
}

use 5.010;
use strict;
use warnings;
use Exporter::Lite;
use Log::Any '$log';
use Module::Info;
use SHARYANTO::Array::Util qw(match_array_or_regex);

our @ISA       = qw(Exporter);
our @EXPORT_OK = qw(gen_spec_for_module);

# VERSION

our %SPEC;

$SPEC{gen_spec_for_module} = {
    summary => 'Generate spec for subroutines in a module',
    description_fmt => 'org',
    description => <<'_',

This function can be used to automatically generate spec for "traditional"
functions, which means regular Perl subroutines in a Perl module.

The resulting spec will be put in %<PACKAGE>::SPEC. Functions that already have
spec in the %SPEC will be skipped. The spec will have C<result_naked> clause set
to true, C<args_as> set to C<array>, and C<args> set to C<{args => ["any" =>
{arg_pos=>0, arg_greedy=>1}]}>. In the future, function's arguments will be
parsed from POD or other indicators.

_
    args => {
        module => ['str*' => {
            summary => 'The module name',
        }],
        load => ['bool*' => {
            summary => 'Whether to load the module using require()',
            default => 1,
        }],
        include_subs => ['any' => { # or regex
            summary => 'If specified, only include these subs',
            of => [['array*'=>{of=>'str*'}], 'str*'], # 2nd should be regex*
        }],
        exclude_subs => ['any' => { # or regex
            summary => 'If specified, exclude these subs',
            of => [['array*'=>{of=>'str*'}], 'str*'], # 2nd should be regex*
        }],
    },
};
sub gen_spec_for_module {
    my %args = @_;

    my $inc = $args{include_subs};
    my $exc = $args{exclude_subs};

    # XXX schema
    my $module = $args{module}
        or return [400, "Please specify module"];
    my $load = $args{load} // 1;

    eval {
        my $modulep = $module; $modulep =~ s!::!/!g;
        require "$modulep.pm";
    };
    my $eval_err = $@;
    return [500, "Can't load module $module: $eval_err"] if $eval_err;

    my $mod = Module::Info->new_from_loaded($module);

    for my $sub ($mod->subroutines) {
        next if ref($sub);
        $sub =~ s/.+:://;
        $log->tracef("Adding spec for subroutine %s ...", $sub);
        if (defined($inc) && !match_array_or_regex($sub, $inc)) {
            $log->info("Subroutine $sub skipped: doesn't match include_subs");
            next;
        }
        if (defined($exc) &&  match_array_or_regex($sub, $exc)) {
            $log->info("Subroutine $sub skipped: doesn't match include_subs");
            next;
        }
        no strict 'refs';
        my $pspec = \%{"$module\::SPEC"};
        if ($pspec->{$sub}) {
            $log->debugf("SPEC keys: %s", [keys %$pspec]);
            $log->info("Subroutine $sub skipped: already has spec");
            next;
        }

        no strict 'vars'; # for $VERSION
        my $spec = {
            summary => $sub,
            description => "This spec is automatically generated by ".
                __PACKAGE__." version ".($VERSION//"?")." on ".localtime(),
            result_naked => 1,
            args_as => 'array',
            args => {
                args => ['array*' => {
                    of => 'any',
                    summary => 'Arguments',
                    arg_pos => 0,
                    arg_greedy => 1,
                }],
            },
        };

        $pspec->{$sub} = $spec;
    }

    [200, "OK"];
}

1;
#ABSTRACT: Generate spec for subroutines in a module


=pod

=head1 NAME

Sub::Spec::Gen::ForModule - Generate spec for subroutines in a module

=head1 VERSION

version 0.02

=head1 SYNOPSIS

In Foo.pm:

 package Foo;
 sub sub1 { ... }
 sub sub2 { ... }
 1;

In another script:

 use Sub::Spec::Gen::FromModule qw(gen_spec_for_module);
 gen_spec_for_module(module=>'Foo');

Now Foo's functions have sub specs (in %Foo::SPEC).

=head1 DESCRIPTION

B<NOTICE>: This module and the L<Sub::Spec> standard is deprecated as of Jan
2012. L<Rinci> is the new specification to replace Sub::Spec, it is about 95%
compatible with Sub::Spec, but corrects a few issues and is more generic.
C<Perinci::*> is the Perl implementation for Rinci and many of its modules can
handle existing Sub::Spec sub specs.

This module provides gen_spec_for_module().

This module uses L<Log::Any> for logging framework.

This module's functions has L<Sub::Spec> specs.

=head1 FUNCTIONS

None are exported by default, but they are exportable.

=head2 gen_spec_for_module(%args) -> [STATUS_CODE, ERR_MSG, RESULT]


Generate spec for subroutines in a module.

This function can be used to automatically generate spec for "traditional"
functions, which means regular Perl subroutines in a Perl module.

The resulting spec will be put in %<PACKAGE>::SPEC. Functions that already have
spec in the %SPEC will be skipped. The spec will have C<result_naked> clause set
to true, C<args_as> set to C<array>, and C<args> set to C<{args => ["any" =>
{arg_pos=>0, arg_greedy=>1}]}>. In the future, function's arguments will be
parsed from POD or other indicators.

Returns a 3-element arrayref. STATUS_CODE is 200 on success, or an error code
between 3xx-5xx (just like in HTTP). ERR_MSG is a string containing error
message, RESULT is the actual result.

Arguments (C<*> denotes required arguments):

=over 4

=item * B<exclude_subs> => I<array|str>

If specified, exclude these subs.

=item * B<include_subs> => I<array|str>

If specified, only include these subs.

=item * B<load>* => I<bool> (default C<1>)

Whether to load the module using require().

=item * B<module>* => I<str>

The module name.

=back

=head1 FAQ

=head1 SEE ALSO

L<Sub::Spec>

=head1 AUTHOR

Steven Haryanto <stevenharyanto@gmail.com>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2012 by Steven Haryanto.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut


__END__

