#
# $Id: Factor.pm,v 0.03 2004/01/06 17:26:07 sts Exp $

package Math::Factor;

our $VERSION = '0.03';

use integer;
use strict;
use warnings;

our ($format_factors, $format_match_number,
     $format_match_matches);

use Exporter;
use base qw(Exporter);

our (@EXPORT_OK, %EXPORT_TAGS, @subs_factor,
     @subs_match, @subs);

@subs_factor = qw(factor show_factor);
@subs_match = qw(match show_match);
@subs = (@subs_factor, @subs_match);

@EXPORT_OK = @subs;
%EXPORT_TAGS = (  all     =>    [ @subs ],
                  factor  =>    [ @subs_factor ],
                  match   =>    [ @subs_match ],
);

sub croak {
    require Carp;
    &Carp::croak;
}

=head1 NAME

Math::Factor - factorise integers and calculate matching multiplications.

=head1 SYNOPSIS

 use Math::Factor q/:all/;

 @numbers = (9);

 # data manipulation
 %factors = factor (\@numbers);
 %matches = match (\%factors);

 # output
 show_factor (\%factors);
 show_match (\%matches);

=head1 DESCRIPTION

see above.

=head1 FUNCTIONS

=head2 factor

Factorises numbers.

 %factors = factor (\@numbers);

Each number within @numbers will be entirely factorised and its factors will be
saved within %factors, accessible by the number itself e.g  the factors of 9 may
be accessed by @{$factors{9}}.

=cut

sub factor {
    my $number_ref = $_[0];
    croak q~usage: factor (\@numbers)~
      unless ref $number_ref eq 'ARRAY' && @$number_ref;

    my %factor;
    foreach (@$number_ref) {
        for (my $i = 1; $i <= $_; $i++) {
            if ($_ % $i == 0)  {
                push @{$factor{$_}}, $i;
            }
        }
    }

    return %factor;
}

=head2 match

Evaluates matching multiplications.

 %matches = match (\%factors);

The factors of each number within %factors will be multplicated against
each other and results the equal the number itself, will be saved to %matches.
The matches are accessible through the according numbers e.g. the first two numbers
that matched 9, may be accessed by $matches{9}[0][0] and $matches{9}[0][1], the second
ones by $matches{9}[1][0] and $matches{9}[1][1], and so on.

=cut

sub match {
    my $number_ref = $_[0];
    croak q~usage: match (\%factors)~
      unless ref $number_ref eq 'HASH' && %$number_ref;

    my ($i, %match);
    foreach (keys %$number_ref) {
        $i = 0;
        my @cmp = my @base = @{$$number_ref{$_}};
        foreach my $base (@base) {
            foreach my $cmp (@cmp) {
                if ($cmp >= $base) {
                    if ($base * $cmp == $_) {
                        $match{$_}[$i][0] = $base;
                        $match{$_}[$i][1] = $cmp;
                        $i++;
                    }
                }
            }
        }
    }

    return %match;
}

=head2 show_factor

Outputs all numbers and their according factors to STDOUT.

 show_factor (\%factors);

 9
 -
 1 3 9

=cut

sub show_factor {
    my $number_ref = $_[0];
    croak q~usage: show_factors (\%factors)~
      unless ref $number_ref eq 'HASH' && %$number_ref;

    print <<'HEADER';
-------
FACTORS
-------

HEADER

    my $ul;
    eval $format_factors;
    croak $@ if $@;

    foreach (sort {$a <=> $b} keys %$number_ref) {
        $ul = '-' x length;
        write;
    }
}

=head2 show_match

Outputs all numbers and their according matching multiplications to STDOUT.

 show_match (\%matches);

 9
 -
         1        * 9
         3        * 3

=cut

sub show_match {
    my $number_ref = $_[0];
    croak q~usage: show_match (\%match)~
      unless ref $number_ref eq 'HASH' && %$number_ref;

    print <<'HEADER';
-------
MATCHES
-------

HEADER

    no warnings;
    foreach (sort {$a <=> $b} keys %$number_ref) {
        my $ul = '-' x length;

        eval $format_match_number;
        croak $@ if $@;
        write;

        my $i;
        eval $format_match_matches;
        croak $@ if $@;

        for ($i = 0; $$number_ref{$_}[$i]; $i++) { write }

        print <<'WHITESPACE';


WHITESPACE
    }
}

$format_factors = '
    format =
@<<<<<<<<<<<<<<<<<<<<<<<<<
$_
@<<<<<<<<<<<<<<<<<<<<<<<<<
$ul
@<<<<<<<<<<<<<<<<<<<<<<<<<
"@{$$number_ref{$_}}"


.
';

$format_match_number = '
    format =
@<<<<<<<<<<<<<<<<<<<<<<<<<
$_
@<<<<<<<<<<<<<<<<<<<<<<<<<
$ul
.
';

$format_match_matches = '
    format =
        @<<<<<<<<* @<<<<<<<<<<<<<<<<<<<<<
        $$number_ref{$_}[$i][0] $$number_ref{$_}[$i][1]
.
';

1;
__END__

=head2 LIMITATIONS

Evaluating ranges of factors or multiplication matches is unimplemented aswell as iterating factors or multiplication matches.

=head2 EXPORT

C<factor(), match(), show_factor(), show_match()> upon request.

B<TAGS>

C<:all - *()>

C<:factor - factor(), show_factor()>

C<:match - match(), show_match()>

=head1 SEE ALSO

perl(1)

=head1 LICENSE

This program is free software; you may redistribute it and/or modify it under the same terms as Perl itself.

=head1 AUTHOR

Steven Schubiger

=cut
