# Crypt::Argon2
[![Build Status](https://travis-ci.org/skinkade/p6-crypt-argon2.svg?branch=master)](https://travis-ci.org/skinkade/p6-crypt-argon2)

[Argon2](https://github.com/P-H-C/phc-winner-argon2) is the winner of the
Password Hashing Competition. It is both memory- and compute-hard. This module
is a NativeCall binding using the Argon2**i** variant, which is resistant to
side-channel attacks.



## Synopsis
Password hashing and verification are one function each, and utilze a
`crypt()`-style output string:
```
> use Crypt::Argon2

> my $hash = argon2-hash("password")
$argon2i$v=19$m=65536,t=2,p=2$q5gCW1J92SNJPlPtoOsP1Q$BCGwLfZsfL7vyF69PhHsQA

> argon2-verify($hash, "password")
True

> argon2-verify($hash, "wrong")
False

> argon2-hash("password", :t_cost(4), :m_cost(2**18), :parallelism(4), :hashlen(24))
$argon2i$v=19$m=262144,t=4,p=4$Ou7t7DzIXXJnEIok0kr10A$0VC9/L+aXKI34i1FQHla4LxQz30/3G0H
```


Key generation returns `(Buf $key, Argon2-meta $meta)`, where `$key` can be
regenerated by running the function again with `$meta` as its second parameter:
```
> use Crypt::Argon2::DeriveKey;

> my ($key, $meta) = argon2-derive-key("password", :hashlen(16))
(Buf:0x<02 78 d7 dc 29 4d 8b 9a fb 89 0d 91 be 09 64 d0> ... )

> argon2-derive-key("password", $meta)
Buf:0x<02 78 d7 dc 29 4d 8b 9a fb 89 0d 91 be 09 64 d0>
```



## Cost Parameters
Hashing takes three cost parameters: time cost as an iteration count, memory cost
in KiB, and parallelism in thread count. By default, password hashing uses two
iterations and 64 MiB of memory, and key generation uses three iterations and
128 MiB of memory. Both defaults use two threads.

Parameters that may better fit your environment can be tested with the included
`bin/argon2-benchmark.p6`. As an example, doubling either memory use or
iteration count will roughly double hashing time, meaning you can somewhat
'trade' these costs if need be:
```
$ perl6 -Ilib bin/argon2-benchmark.p6
Running 10 iterations of argon2-verify() with the following settings:
    Iterations: 2
    Memory cost: 65536 KiB
    Parallelism: 2 threads
    Hash length: 16 bytes
Time per verification: 122.20 ms

$ perl6 -Ilib bin/argon2-benchmark.p6 --m_cost=32768 --t_cost=4
Running 10 iterations of argon2-verify() with the following settings:
    Iterations: 4
    Memory cost: 32768 KiB
    Parallelism: 2 threads
    Hash length: 16 bytes
Time per verification: 110.47 ms
```
