package Sah::Schema::simple_uint_range;

our $AUTHORITY = 'cpan:PERLANCAR'; # AUTHORITY
our $DATE = '2021-07-16'; # DATE
our $DIST = 'Sah-Schemas-IntRange'; # DIST
our $VERSION = '0.002'; # VERSION

our $schema = [str => {
    summary => 'Simple unsigned integer range, e.g. 1-10',
    match => qr/\A(?:[0-9]+)\s*-\s*(?:[0-9]+)\z/,
    prefilters => ['IntRange::check_simple_uint_range'],
    examples => [
        {data=>'', valid=>0},
        {data=>'1', valid=>0},
        {data=>'-2', valid=>0},

        {data=>'1-1', valid=>1},
        {data=>'1-2', valid=>1},
        {data=>'1 - 2', valid=>1},
        {data=>'0-100', valid=>1},
        {data=>'-1-2', valid=>0,
         summary=>'Negative'},
        {data=>'-10--1', valid=>0,
         summary=>'Negative'},

        {data=>'1-', valid=>0},
        {data=>'1-1.5', valid=>0, sumary=>'Float'},
        {data=>'9-2', valid=>0, summary=>'start value cannot be larger than end value'},
        {data=>'1-2-3', valid=>0},
        {data=>' 1-2 ', valid=>0, summary=>'Leading and trailing whitespace is currently not allowed'},
    ],
}, {}];

1;
# ABSTRACT: Simple unsigned integer range, e.g. 1-10

__END__

=pod

=encoding UTF-8

=head1 NAME

Sah::Schema::simple_uint_range - Simple unsigned integer range, e.g. 1-10

=head1 VERSION

This document describes version 0.002 of Sah::Schema::simple_uint_range (from Perl distribution Sah-Schemas-IntRange), released on 2021-07-16.

=head1 SYNOPSIS

To check data against this schema (requires L<Data::Sah>):

 use Data::Sah qw(gen_validator);
 my $validator = gen_validator("simple_uint_range*");
 say $validator->($data) ? "valid" : "INVALID!";

 # Data::Sah can also create validator that returns nice error message string
 # and/or coerced value. Data::Sah can even create validator that targets other
 # language, like JavaScript. All from the same schema. See its documentation
 # for more details.

To validate function parameters against this schema (requires L<Params::Sah>):

 use Params::Sah qw(gen_validator);

 sub myfunc {
     my @args = @_;
     state $validator = gen_validator("simple_uint_range*");
     $validator->(\@args);
     ...
 }

To specify schema in L<Rinci> function metadata and use the metadata with
L<Perinci::CmdLine> to create a CLI:

 # in lib/MyApp.pm
 package MyApp;
 our %SPEC;
 $SPEC{myfunc} = {
     v => 1.1,
     summary => 'Routine to do blah ...',
     args => {
         arg1 => {
             summary => 'The blah blah argument',
             schema => ['simple_uint_range*'],
         },
         ...
     },
 };
 sub myfunc {
     my %args = @_;
     ...
 }
 1;

 # in myapp.pl
 package main;
 use Perinci::CmdLine::Any;
 Perinci::CmdLine::Any->new(url=>'MyApp::myfunc')->run;

 # in command-line
 % ./myapp.pl --help
 myapp - Routine to do blah ...
 ...

 % ./myapp.pl --version

 % ./myapp.pl --arg1 ...

Sample data:

 ""  # INVALID

 1  # INVALID

 -2  # INVALID

 "1-1"  # valid

 "1-2"  # valid

 "1 - 2"  # valid

 "0-100"  # valid

 "-1-2"  # INVALID (Negative)

 "-10--1"  # INVALID (Negative)

 "1-"  # INVALID

 "1-1.5"  # INVALID

 "9-2"  # INVALID (start value cannot be larger than end value)

 "1-2-3"  # INVALID

 " 1-2 "  # INVALID (Leading and trailing whitespace is currently not allowed)

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Sah-Schemas-IntRange>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Sah-Schemas-IntRange>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Sah-Schemas-IntRange>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2021 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
