package Dist::Zilla::Plugin::Rinci::InstallCompletion;

our $DATE = '2014-11-09'; # DATE
our $VERSION = '0.02'; # VERSION

use 5.008001;
use strict;
use warnings;
use utf8;

use Moose;
use namespace::autoclean;

use List::Util qw(first);

with (
    'Dist::Zilla::Role::InstallTool',
    'Dist::Zilla::Role::FileFinderUser' => {
        default_finders => [':ExecFiles'],
    },
);

sub setup_installer {
  my ($self) = @_;

  unless (@{ $self->found_files }) {
      $self->log_debug('No scripts in this distribution, skipped');
      return;
  }

  # first, try MakeMaker
  my $build_script = first { $_->name eq 'Makefile.PL' }
      @{ $self->zilla->files };
  $self->log_fatal('No Makefile.PL found. Using [MakeMaker] is required')
      unless $build_script;

  my $content = $build_script->content;

  no strict 'refs';
  my $header = "
# modify generated Makefile to install bash completion for scripts. this piece\n".
"# is generated by " . __PACKAGE__ . " version " .
    (${__PACKAGE__ ."::VERSION"} // 'dev').".\n";

  my $body = <<'_';
INSTALL_COMPLETION:
{
    print "Modifying Makefile to install bash completion on install\n";
    open my($fh), "<", "Makefile" or die "Can't open generated Makefile: $!";
    my $content = do { local $/; ~~<$fh> };

    $content =~ s/^(install :: pure_install doc_install)/$1 comp_install/m
        or die "Can't find pattern in Makefile (1)";

    $content =~ s/^(uninstall :: .+)/$1 comp_uninstall/m
        or die "Can't find pattern in Makefile (2)";

    $content .= qq|\ncomp_install :\n\t| .
        q|$(PERLRUN) -E'if(eval { require App::BashCompletionProg; 1 }) { system "bash-completion-prog", "--verbose", @ARGV }' -- $(EXE_FILES)| .
        qq|\n\n|;

    $content .= qq|\ncomp_uninstall :\n\t| .
        q|$(PERLRUN) -E'if(eval { require App::BashCompletionProg; 1 }) { system "bash-completion-prog", "--remove", "--verbose", @ARGV }' -- $(EXE_FILES)| .
        qq|\n\n|;

    open $fh, ">", "Makefile" or die "Can't write modified Makefile: $!";
    print $fh $content;
}
_

  $content .= $header . $body;

  return $build_script->content($content);
}

no Moose;
1;
# ABSTRACT: Install bash completion for scripts when distribution is installed

__END__

=pod

=encoding UTF-8

=head1 NAME

Dist::Zilla::Plugin::Rinci::InstallCompletion - Install bash completion for scripts when distribution is installed

=head1 VERSION

This document describes version 0.02 of Dist::Zilla::Plugin::Rinci::InstallCompletion (from Perl distribution Dist-Zilla-Plugin-Rinci-InstallCompletion), released on 2014-11-09.

=head1 SYNOPSIS

In your dist.ini:

 [Rinci::InstallCompletion]

=head1 DESCRIPTION

This plugin modifies C<Makefile.PL> so that when a user installs your
distribution with C<make install>, L<bash-completion-prog> is invoked to add
C<complete> bash command for your scripts. This is convenient because at the
next shell the user starts, bash tab completion is already activated for your
scripts.

L<bash-completion-prog> recognizes several ways/hints to add completion to your
scripts. Please see its documentation for more details.

Some notes:

First, user must already install and setup L<bash-completion-progs> (included in
L<App::BashCompletionProg> distribution). But if App::BashCompletionProg is
installed after your distribution is installed, she can simply run
C<bash-completion-prog> to scan PATH and add completion for all recognized
scripts, including yours.

Second, this plugin's implementation strategy is currently as follow (probably
hackish): insert some code in the generated C<Makefile.PL> after
C<WriteMakefile()> to insert some targets in the C<Makefile> generated by
C<Makefile.PL>.

Third, currently only MakeMaker is supported, L<Module::Build> is not.

=for Pod::Coverage setup_installer

=head1 SEE ALSO

L<App::BashCompletionProg>

CLI scripts using the L<Perinci::CmdLine> framework will automatically have bash
completion capability. L<bash-completion-prog> detects this.

You can also use L<Getopt::Long::Complete>.

Or, if your CLI scripts use L<Getopt::Long>, you can use
L<Dist::Zilla::Plugin::GenBashCompletion::GetoptLong> to produce a completion
script based on the spec passed in Getopt::Long::GetOptions.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Dist-Zilla-Plugin-Rinci-InstallCompletion>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Dist-Zilla-Plugin-Rinci-InstallCompletion>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Dist-Zilla-Plugin-Rinci-InstallCompletion>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2014 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
