package App::ruler;

our $DATE = '2015-08-09'; # DATE
our $VERSION = '0.02'; # VERSION

use feature 'say';
use strict 'subs', 'vars';
use warnings;

use Exporter;
our @ISA = qw(Exporter);
our @EXPORT_OK = qw(
                       ruler
               );

our %SPEC;

my $term_width;
if (eval { require Term::Size; 1 }) {
    ($term_width, undef) = Term::Size::chars();
} else {
    $term_width = 80;
}

$SPEC{ruler} = {
    v => 1.1,
    summary => 'Print horizontal ruler on the terminal',
    args_rels => {
        'choose_one&' => [
            #[qw/color random_color/],
        ],
    },
    args => {
        length => {
            schema => ['int*', min=>0],
            cmdline_aliases => {l=>{}},
        },
        background_pattern => {
            schema => ['str*', min_len=>1],
            default => '-',
            cmdline_aliases => {bg=>{}},
        },

        major_tick_every => {
            schema => ['int*', min=>1],
            default => 10,
            cmdline_aliases => {N=>{}},
        },
        major_tick_character => {
            schema => ['str', max_len=>1],
            default => '|',
            cmdline_aliases => {M=>{}},
        },

        minor_tick_every => {
            schema => ['int*', min=>1],
            default => 1,
            cmdline_aliases => {n=>{}},
        },
        minor_tick_character => {
            schema => ['str', max_len=>1],
            default => '.',
            cmdline_aliases => {m=>{}},
        },

        number_every => {
            schema => ['int*', min=>1],
            default => 10,
        },
        number_start => {
            schema => ['int*', min=>0],
            default => 10,
        },
        number_format => {
            schema => ['str*'],
            default => '%d',
            cmdline_aliases => {f=>{}},
        },
    },
};
sub ruler {
    my %args = @_;

    my $len = $args{length} // $term_width;

    my $bgpat = $args{background_pattern} // '-';
    my $mintickchar = $args{minor_tick_character} // '.';
    my $majtickchar = $args{major_tick_character} // '|';

    my $ruler = $bgpat x (int($len / length($bgpat)) + 1);

    if (length $mintickchar) {
        my $every = $args{minor_tick_every} // 1;
        for (1..$len) {
            if ($_ % $every == 0) {
                substr($ruler, $_-1, 1) = $mintickchar;
            }
        }
    }
    if (length $majtickchar) {
        my $every = $args{major_tick_every} // 10;
        for (1..$len) {
            if ($_ % $every == 0) {
                substr($ruler, $_-1, 1) = $majtickchar;
            }
        }
    }

    # draw numbers
    {
        my $numevery = $args{number_every} // 10;
        my $numstart = $args{number_start} // 10;
        my $fmt = $args{number_format} // '%d';
        for ($numstart..$len) {
            if ($_ % $numevery == 0) {
                my $num = sprintf($fmt, $_);
                substr($ruler, $_, length($num)) = $num;
            }
        }
    }

    # final clip
    $ruler = substr($ruler, 0, $len);
    $ruler .= "\n"
        unless $len == ($^O =~ /Win32/ ? $term_width-1 : $term_width);

    [200, "OK", $ruler];
}

1;
# ABSTRACT: Print horizontal ruler on the terminal

__END__

=pod

=encoding UTF-8

=head1 NAME

App::ruler - Print horizontal ruler on the terminal

=head1 VERSION

This document describes version 0.02 of App::ruler (from Perl distribution App-ruler), released on 2015-08-09.

=head1 SEE ALSO

L<hr> (L<App::hr>)

=head1 FUNCTIONS


=head2 ruler(%args) -> [status, msg, result, meta]

Print horizontal ruler on the terminal.

Arguments ('*' denotes required arguments):

=over 4

=item * B<background_pattern> => I<str> (default: "-")

=item * B<length> => I<int>

=item * B<major_tick_character> => I<str> (default: "|")

=item * B<major_tick_every> => I<int> (default: 10)

=item * B<minor_tick_character> => I<str> (default: ".")

=item * B<minor_tick_every> => I<int> (default: 1)

=item * B<number_every> => I<int> (default: 10)

=item * B<number_format> => I<str> (default: "%d")

=item * B<number_start> => I<int> (default: 10)

=back

Returns an enveloped result (an array).

First element (status) is an integer containing HTTP status code
(200 means OK, 4xx caller error, 5xx function error). Second element
(msg) is a string containing error message, or 'OK' if status is
200. Third element (result) is optional, the actual result. Fourth
element (meta) is called result metadata and is optional, a hash
that contains extra information.

Return value:  (any)

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/App-ruler>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-App-ruler>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=App-ruler>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
