package App::CSVUtils::csv_gen;

use 5.010001;
use strict;
use warnings;

our $AUTHORITY = 'cpan:PERLANCAR'; # AUTHORITY
our $DATE = '2022-12-23'; # DATE
our $DIST = 'App-CSVUtils'; # DIST
our $VERSION = '0.999_2'; # TRIAL VERSION

use App::CSVUtils qw(
                        gen_csv_util
                        compile_eval_code
                );

gen_csv_util(
    name => 'csv_gen',
    summary => 'Generate CSV data using Perl code',
    description => <<'_',


_
    add_args => {
        fields => $App::CSVUtils::argspecopt_fields{fields},
        eval_fields => {
            summary => 'Code to generate list of fields',
            schema => 'str*',
            description => <<'_',

This is an alternative to supplying a static list of fields via `fields` option.

Code is expected to return the list of fields as an arrayref.

_
        },
        eval => {
            summary => 'Code to generate row',
            schema => 'str*',
            description => <<'_',

Code will be compiled in the `main` package.

Code is expected to return the row data, which can be a hashref or an arrayref.
When code returns false, this signals `csv-gen` to stop the output. Note that
you can also limit the number of rows generated by setting the `num_rows`
option.

_
            cmdline_aliases => {e=>{}},
        },
        num_rows => {
            summary => 'Limit the number of rows to generate',
            schema => 'uint*',
            cmdline_aliases => {n=>{}},
        },
    },

    add_args_rels => {
        req_one => ['fields', 'eval_fields'],
    },

    reads_csv => 0,

    examples => [
        {
            summary => 'Generate 10 numbers from 1 to 10',
            src => q{[[prog]] -f num -n 10 -e '[++$i]'},
            src_plang => 'bash',
            test => 0,
        },
        {
            summary => 'Generate 10 numbers from 1 to 10 along with their squares',
            src => q{[[prog]] -f num -f square -n 10 -e '[++$i, $i*$i]'},
            src_plang => 'bash',
            test => 0,
        },
    ],

    after_read_input => sub {
        my $r = shift;

        # set output fields
        if ($r->{util_args}{eval_fields}) {
            my $code = compile_eval_code($r->{util_args}{eval_fields}, 'eval_fields');
            local $main::r = $r;
            my $fields = $code->();
            die [500, "Code in eval_fields did not return list of fields as arranref"]
                unless ref $fields eq 'ArRAY';
            $r->{output_fields} = $fields;
        } else {
            $r->{output_fields} = $r->{util_args}{fields};
        }

        # print rows
        {
            my $code = compile_eval_code($r->{util_args}{eval}, 'eval');
            local $main::r = $r;
            while (1) {
                my $row = $code->();
                last unless $row;
                last if defined $r->{util_args}{num_rows} &&
                    ($r->{output_data_rownum}//0) >= $r->{util_args}{num_rows};
                $r->{code_print_row}->($row);
            }
        }
    },
);

1;
# ABSTRACT: Generate CSV data using Perl code

__END__

=pod

=encoding UTF-8

=head1 NAME

App::CSVUtils::csv_gen - Generate CSV data using Perl code

=head1 VERSION

This document describes version 0.999_2 of App::CSVUtils::csv_gen (from Perl distribution App-CSVUtils), released on 2022-12-23.

=head1 FUNCTIONS


=head2 csv_gen

Usage:

 csv_gen(%args) -> [$status_code, $reason, $payload, \%result_meta]

Generate CSV data using Perl code.

This function is not exported.

Arguments ('*' denotes required arguments):

=over 4

=item * B<eval> => I<str>

Code to generate row.

Code will be compiled in the C<main> package.

Code is expected to return the row data, which can be a hashref or an arrayref.
When code returns false, this signals C<csv-gen> to stop the output. Note that
you can also limit the number of rows generated by setting the C<num_rows>
option.

=item * B<eval_fields> => I<str>

Code to generate list of fields.

This is an alternative to supplying a static list of fields via C<fields> option.

Code is expected to return the list of fields as an arrayref.

=item * B<fields> => I<array[str]>

Field names.

=item * B<num_rows> => I<uint>

Limit the number of rows to generate.

=item * B<output_escape_char> => I<str>

Specify character to escape value in field in output CSV, will be passed to Text::CSV_XS.

This is like C<--input-escape-char> option but for output instead of input.

Defaults to C<\\> (backslash). Overrides C<--output-tsv> option.

=item * B<output_filename> => I<filename>

Output filename.

Use C<-> to output to stdout (the default if you don't specify this option).

Encoding of output file is assumed to be UTF-8.

=item * B<output_header> => I<bool>

Whether output CSV should have a header row.

By default, a header row will be output I<if> input CSV has header row. Under
C<--output-header>, a header row will be output even if input CSV does not have
header row (value will be something like "col0,col1,..."). Under
C<--no-output-header>, header row will I<not> be printed even if input CSV has
header row. So this option can be used to unconditionally add or remove header
row.

=item * B<output_quote_char> => I<str>

Specify field quote character in output CSV, will be passed to Text::CSV_XS.

This is like C<--input-quote-char> option but for output instead of input.

Defaults to C<"> (double quote). Overrides C<--output-tsv> option.

=item * B<output_sep_char> => I<str>

Specify field separator character in output CSV, will be passed to Text::CSV_XS.

This is like C<--input-sep-char> option but for output instead of input.

Defaults to C<,> (comma). Overrides C<--output-tsv> option.

=item * B<output_tsv> => I<bool>

Inform that output file is TSV (tab-separated) format instead of CSV.

This is like C<--input-tsv> option but for output instead of input.

Overriden by C<--output-sep-char>, C<--output-quote-char>, C<--output-escape-char>
options. If one of those options is specified, then C<--output-tsv> will be
ignored.

=item * B<overwrite> => I<bool>

Whether to override existing output file.


=back

Returns an enveloped result (an array).

First element ($status_code) is an integer containing HTTP-like status code
(200 means OK, 4xx caller error, 5xx function error). Second element
($reason) is a string containing error message, or something like "OK" if status is
200. Third element ($payload) is the actual result, but usually not present when enveloped result is an error response ($status_code is not 2xx). Fourth
element (%result_meta) is called result metadata and is optional, a hash
that contains extra information, much like how HTTP response headers provide additional metadata.

Return value:  (any)

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/App-CSVUtils>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-App-CSVUtils>.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 CONTRIBUTING


To contribute, you can send patches by email/via RT, or send pull requests on
GitHub.

Most of the time, you don't need to build the distribution yourself. You can
simply modify the code, then test via:

 % prove -l

If you want to build the distribution (e.g. to try to install it locally on your
system), you can install L<Dist::Zilla>,
L<Dist::Zilla::PluginBundle::Author::PERLANCAR>,
L<Pod::Weaver::PluginBundle::Author::PERLANCAR>, and sometimes one or two other
Dist::Zilla- and/or Pod::Weaver plugins. Any additional steps required beyond
that are considered a bug and can be reported to me.

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2022, 2021, 2020, 2019, 2018, 2017, 2016 by perlancar <perlancar@cpan.org>.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=App-CSVUtils>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=cut
