##-*- Mode: CPerl -*-

## File: DDC::Query.pm
## Author: Bryan Jurish <moocow@cpan.org>
##======================================================================

package DDC::Query;
use DDC::Utils qw(:escape);
use strict;

##======================================================================
## Globals etc.
our @ISA = qw();

##======================================================================
## $q = $CLASS_OR_OBJ->new(%args)
##  + %args, %$q:
##    (
##     class   => $class,     ##-- subclass
##     dtrs    => \@args,     ##-- array of sub-queries (e.g. DDC::Query objects)
##     #...
##    )
sub new {
  my ($that,%args) = @_;
  my $class = $args{class};
  my ($new);
  if ($class && defined($new=UNIVERSAL::can((__PACKAGE__ . "::$class"), 'new'))) {
    delete($args{class});
    return $new->((__PACKAGE__ . "::$class"),%args) if ($new ne __PACKAGE__->can('new'));
  }
  return bless(\%args,
	       ($class ? (__PACKAGE__ . "::$class") : (ref($that)||$that))
	      );
}

## $q = $q->negate()
sub negate {
  $_[0]{negated} = !$_[0]{negated};
  return $_[0];
}

## $bool = $q->negatable()
sub negatable { return 0; }

## $bool = $q->negated()
sub negated { return $_[0]->negatable ? $_[0]{negated} : 0; }

## @items = $q->expandItems()
##  + returns list of source items to be expanded (e.g. by CAB); empty list for none
sub expandItems { return qw(); }

## @dtrs = $q->dtrs()
sub dtrs {
  return $_[0]{dtrs} ? @{$_[0]{dtrs}} : qw();
}

## $str = $q->toString()
##  + stringification operator
sub toString {
  my $q = shift;
  confess(ref($q)."::toString(): abstract method called");
}

## \@nodes = $q->dfs()
##  + returns query nodes in depth-first search order
sub dfs {
  my $q = shift;
  my @nodes = ($q);
  foreach ($q->dtrs) {
    push(@nodes,@{$_->dfs});
  }
  return \@nodes;
}

##======================================================================
## Root: root queries (with filters)
package DDC::Query::Root;
our @ISA = qw(DDC::Query);

## $q = $CLASS_OR_OBJ->new(%args)
##  + %args, %$q:
##    (
##     root => $q,            ##-- query root
##     filters => \@filters,  ##-- array of DDC::Query::Filter objects
##    )
sub new {
  my $that = shift;
  return $that->SUPER::new(
			   root=>$that->SUPER::new(),
			   filters=>[],
			   @_
			  );
}

sub dtrs {
  return $_[0]{root};
}

## $str = $q->toString()
##  + stringification operator
sub toString {
  my $q = shift;
  return join(' ', map {$_->toString} $q->{root}, @{$q->{filters}});
}

##======================================================================
## Negatable: negatable queries
package DDC::Query::Negatable;
our @ISA = qw(DDC::Query);

## $bool = $q->negatable()
sub negatable { return 1; }

## $str = $q->negString($str)
##  + stringification operator
sub negString {
  my ($q,$str) = @_;
  return $q->negated ? ('!'.$str) : $str;
}

##======================================================================
## BinOp
package DDC::Query::BinOp;
our @ISA = qw(DDC::Query::Negatable);
sub new {
  my $that = shift;
  return $that->SUPER::new(op=>undef,dtrs=>undef,@_);
}

sub toString {
  my $q = shift;
  return $q->negString('('.join(" $q->{op} ", map {$_->toString} @{$q->{dtrs}}).')');
}

##--------------------------------------------------------------
## Q1 && Q2
package DDC::Query::And;
our @ISA = qw(DDC::Query::BinOp);
sub new {
  my $that = shift;
  return $that->SUPER::new(op=>'&&',@_);
}

##--------------------------------------------------------------
## Q1 || Q2
package DDC::Query::Or;
our @ISA = qw(DDC::Query::BinOp);
sub new {
  my $that = shift;
  return $that->SUPER::new(op=>'||',@_);
}

##--------------------------------------------------------------
## Q1 with Q2
package DDC::Query::With;
our @ISA = qw(DDC::Query::BinOp);
sub new {
  my $that = shift;
  return $that->SUPER::new(op=>'WITH',@_);
}
## $bool = $q->negatable()
sub negatable { return 0; }

##======================================================================
## Atomic: sequenceable queries
package DDC::Query::Atomic;
our @ISA = qw(DDC::Query::Negatable);

##======================================================================
## Token: single-token queries

##--------------------------------------------------------------
package DDC::Query::Token;
use DDC::Utils qw(:escape);
our @ISA = qw(DDC::Query::Atomic);
sub new {
  my $that = shift;
  return $that->SUPER::new(
			   index=>undef,
			   #value=>undef,
			   #expand=>[]
			   @_);
}

sub toString {
  my $q = shift;
  return $q->negString(
		       $q->indexString . $q->valueString . $q->expandString
		      );
}

## $str = $q->indexString()       ##-- uses $q->{index}
## $str = $q->indexString($index)
sub indexString {
  my $q     = shift;
  my $index = @_ ? shift : $q->{index};
  return '' if (!defined($index) || $index eq '');
  return '$'.escapeq($index).'=';
}

## $str = $q->valueString() ##-- uses $q->{value}
## $str = $q->valueString($val)
sub valueString {
  my $q = shift;
  my $value = @_ ? shift : $q->{value};
  $value = '' if (!defined($value));
  return escapeq($value);
}

## $str = $q->expandString() ##-- uses $q->{expand}
## $str = $q->expandString(\@expand)
sub expandString {
  my $q      = shift;
  my $expand = @_ ? shift : $q->{expand};
  return '' if (!$expand || !@$expand);
  return join('', map {'|'.escapeq($_)} @$expand);
}

##--------------------------------------------------------------
## $INDEX=@WORD
package DDC::Query::TokExact;
our @ISA = qw(DDC::Query::Token);
sub valueString {
  my $q = shift;
  return '@'.$q->SUPER::valueString(@_);
}

##--------------------------------------------------------------
## $INDEX=WORD
package DDC::Query::TokExpand;
use DDC::Utils qw(:escape);
our @ISA = qw(DDC::Query::Token);

##--------------------------------------------------------------
## $INDEX=WORD*
package DDC::Query::TokPrefix;
our @ISA = qw(DDC::Query::Token);
sub valueString {
  my $q = shift;
  return $q->SUPER::valueString(@_).'*';
}

##--------------------------------------------------------------
## $INDEX=*WORD
package DDC::Query::TokSuffix;
our @ISA = qw(DDC::Query::Token);
sub valueString {
  my $q = shift;
  return '*'.$q->SUPER::valueString(@_);
}

##--------------------------------------------------------------
## $INDEX=*WORD*
package DDC::Query::TokInfix;
our @ISA = qw(DDC::Query::Token);
sub valueString {
  my $q = shift;
  return '*'.$q->SUPER::valueString(@_).'*';
}

##--------------------------------------------------------------
## $INDEX=/REGEX/
##  + 'value' argument should contain surrounding slashes and all relevant escapes!
package DDC::Query::TokRegex;
#use DDC::Utils qw(:escape);
our @ISA = qw(DDC::Query::Token);
sub valueString {
  my $q = shift;
  return @_ ? shift : $q->{value};
}

##--------------------------------------------------------------
## $INDEX=*
package DDC::Query::Any;
our @ISA = qw(DDC::Query::Token);
sub valueString {
  return '*';
}

##--------------------------------------------------------------
## TokSet: $INDEX={W1,W2,...} : abstract
package DDC::Query::TokSet;
use DDC::Utils qw(:escape);
our @ISA = qw(DDC::Query::Token);
sub new {
  my $that = shift;
  return $that->SUPER::new(values=>[],@_);
}

## $str = $q->valueString() ##-- uses $q->{values}
## $str = $q->valueString(\@values)
##   + returned string includes brackets
sub valueString {
  my $q      = shift;
  my $values = @_ ? shift : $q->{values};
  return '{'.join(',', map {escapeq($_)} @{$values||[]}).'}';
}

##--------------------------------------------------------------
## TokSetExpand: $INDEX={W1,W2,...}
package DDC::Query::TokSetExpand;
our @ISA = qw(DDC::Query::TokSet);
sub new {
  my $that = shift;
  return $that->SUPER::new(expand=>[],@_);
}

## @items = $q->expandItems()
sub expandItems { return @{$_[0]{expand}||[]}; }

##--------------------------------------------------------------
## TokSetExact: $INDEX=@{W1,W2,...}
package DDC::Query::TokSetExact;
our @ISA = qw(DDC::Query::TokSet);
sub valueString {
  my $q = shift;
  return '@'.$q->SUPER::valueString(@_);
}

##--------------------------------------------------------------
## TokSetPrefix: $INDEX={W1,W2,...}*
package DDC::Query::TokSetPrefix;
our @ISA = qw(DDC::Query::TokSet);
sub valueString {
  my $q = shift;
  return $q->SUPER::valueString(@_).'*';
}

##--------------------------------------------------------------
## TokSetSuffix: $INDEX=*{W1,W2,...}
package DDC::Query::TokSetSuffix;
our @ISA = qw(DDC::Query::TokSet);
sub valueString {
  my $q = shift;
  return '*'.$q->SUPER::valueString(@_);
}

##--------------------------------------------------------------
## TokSetInfix: $INDEX=*{W1,W2,...}*
package DDC::Query::TokSetInfix;
our @ISA = qw(DDC::Query::TokSet);
sub valueString {
  my $q = shift;
  return '*'.$q->SUPER::valueString(@_).'*';
}

##--------------------------------------------------------------
## TokThes: $INDEX={THES1:THES2:...}
package DDC::Query::TokThes;
our @ISA = qw(DDC::Query::Token);
sub valueString {
  my $q = shift;
  return ':{'.$q->SUPER::valueString(@_).'}';
}

##--------------------------------------------------------------
## TokMorph: $INDEX=[M1,M2,..]
package DDC::Query::TokMorph;
use DDC::Utils qw(:escape);
our @ISA = qw(DDC::Query::Token);
sub valueString {
  my $q     = shift;
  my $items = @_ ? shift : $q->{items};
  return '['.join(',', map {escapeq($_)} @{$items||[]}).']';
}

##--------------------------------------------------------------
## TokLemma: $INDEX=%LEMMA
package DDC::Query::TokLemma;
our @ISA = qw(DDC::Query::Token);
sub valueString {
  my $q = shift;
  return '%'.$q->SUPER::valueString(@_);
}

##--------------------------------------------------------------
## TokChunk: $INDEX=^CHUNK
package DDC::Query::TokChunk;
our @ISA = qw(DDC::Query::Token);
sub valueString {
  my $q = shift;
  return '^'.$q->SUPER::valueString(@_);
}

##--------------------------------------------------------------
## TokAncor: $.BREAK=POSITION
package DDC::Query::TokAnchor;
our @ISA = qw(DDC::Query::Token);
sub indexString {
  my $q     = shift;
  my $index = @_ ? shift : $q->{index};
  return '$.'.(defined($index) && $index ne '' ? escapeq($index) : '').'=';
}

##--------------------------------------------------------------
## <FILE
package DDC::Query::TokFile;
our @ISA = qw(DDC::Query::Token);
sub valueString {
  my $q = shift;
  return '<'.$q->SUPER::valueString(@_);
}

##======================================================================
## Multi-token queries

##--------------------------------------------------------------
## Near :
##     NEAR(W1,W2,   N)
##   | NEAR(W1,W2,W3,N)
package DDC::Query::Near;
our @ISA = qw(DDC::Query::Negatable);
sub new {
  my $that = shift;
  return $that->SUPER::new(dtrs=>[],dist=>1,@_);
}
sub toString {
  my $q = shift;
  return $q->negString('NEAR('
		       .join(',',
			     (map {$_->toString} @{$q->{dtrs}}),
			     $q->{dist}
			    )
		       .')'
		      );
}


##--------------------------------------------------------------
## Phrase: "W1 #D1 W2 ... WN"
package DDC::Query::Phrase;
our @ISA = qw(DDC::Query::Atomic);
sub new {
  my $that = shift;
  return $that->SUPER::new(args=>[],@_); ##-- $args=[[$w1,?$d1,?$op1],...,[$wN]]
}
sub dtrs {
  my $q = shift;
  return map {$_->[0]} @{$q->{args}};
}
sub toString {
  my $q = shift;
  return $q->negString('"'
		       .join(' ',
			     map {
			       ($_->[0]->toString . ($_->[1] ? " #$_->[1]" : ''))
			     } @{$q->{args}})
		       .'"'
		      );
}

package DDC::Query::Seq;
our @ISA = qw(DDC::Query::Phrase);

1; ##-- be happy

