use 5.008001;
use strict;
use warnings;

package BSON::Bool;
# ABSTRACT: Boolean data for BSON

our $VERSION = '0.13'; # TRIAL

use overload
  bool => \&value,
  '==' => \&op_eq,
  'eq' => \&op_eq;

sub new {
    my ( $class, $bool ) = @_;
    bless { value => $bool ? 1 : 0 }, $class;
}

sub value {
    $_[0]->{value} ? 1 : 0;
}

sub true {
    return $_[0]->new(1);
}

sub false {
    return $_[0]->new(0);
}

sub op_eq {
    return ref( $_[0] ) eq ref( $_[1] ) && $_[0]->value == $_[1]->value;
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

BSON::Bool - Boolean data for BSON

=head1 VERSION

version 0.13

=head1 SYNOPSIS

    use BSON;

    my $true  = BSON::Bool->true;
    my $false = BSON::Bool->false;
    my $odd   = BSON::Bool->new( time % 2 )

    print "Odd times!" if $odd;

=head1 DESCRIPTION

This module is needed for L<BSON> and it manages BSON's boolean element.

=for Pod::Coverage op_eq

=head1 METHODS

=head2 new

Main constructor which takes a single parameter. Zero or C<undef> create
a C<false> instance, and everything else creates a C<true> instance.

    my $true  = BSON::Bool->new(255);
    my $false = BSON::Bool->new;

=head2 true

As a secondary constructor it returns a C<true> instance.

=head2 false

As a secondary constructor it returns a C<false> instance.

=head2 value

Returns C<0> or C<1> for C<false> and C<true>.

=head1 OVERLOAD

All boolean operations are overloaded, so the class instance can
be used as a boolean variable itself.

    if ( BSON::Bool->true ) {
        print "You kick ass!";
    }

=head1 SEE ALSO

L<BSON>

=head1 AUTHORS

=over 4

=item *

minimalist <minimalist@lavabit.com>

=item *

David Golden <david@mongodb.com>

=back

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2015 by minimalist and MongoDB, Inc..

This is free software, licensed under:

  The Apache License, Version 2.0, January 2004

=cut
