package Forks::Queue;
use strict;
use warnings;
use base 'Exporter';
use Carp;

our $VERSION = '0.01';
our $DEBUG = 0;

sub new {
    my ($pkg,%opts) = @_;
    if ($opts{impl}) {
        $pkg = delete $opts{impl};
        if ($pkg =~ /[^\w:]/) {
            croak "Forks::Queue cannot be instantiated. Invalid 'impl' $pkg";
        }
        if (eval "require Forks::Queue::$pkg; 1") {
            $pkg = "Forks::Queue::" . $pkg;
            return $pkg->new(%opts);
        } elsif (eval "require $pkg; 1") {
            return $pkg->new(%opts);
        } else {
            croak "Forks::Queue cannot be instantiated. ",
                "Did not recognize 'impl' option '$opts{impl}'";
        }
    }
    croak "Forks::Queue cannot be instantiated. ",
        "Use an implementation or pass the 'impl' option to the constructor";
}


sub put {
    my $self = CORE::shift;
    return $self->push(@_);
}

sub get {
    my $self = CORE::shift;
    if ($self->{style} eq 'fifo') {
        return @_ ? $self->shift(@_) : $self->shift;
    } else {
        return @_ ? $self->pop(@_) : $self->pop;
    }
}

sub peek {
    my $self = CORE::shift;
    if ($self->{style} eq 'lifo') {
        return $self->peek_back;
    } else {
        return $self->peek_front;
    }
}

sub pending {
    my $self = CORE::shift;
    return $self->status->{avail};
}

sub _croak {
    my $func = shift;
    croak "Forks::Queue: $func not implemented in abstract base class";
}

sub push       { _croak("push/put") }
sub peek_front { _croak("peek") }
sub peek_back  { _croak("peek") }
sub shift      { _croak("shift/get") }
sub pop        { _croak("pop/get") }
sub status     { _croak("pending/status") }

1;

=head1 NAME

Forks::Queue - queue that can be shared across processes

=head1 VERSION

0.01

=head1 SYNOPSIS

  use Forks::Queue;
  $q = Forks::Queue->new( impl => ..., style => 'lifo' );

  # put items on queue
  $q->put("a scalar item");
  $q->put(["an","arrayref","item"]);
  $q->put({"a"=>"hash","reference"=>"item"});
  $q->put("list","of","multiple",["items"]);
  $q->end;        # no more jobs will be added to queue

  # retrieve items from queue, possibly after a fork
  $item = $q->get;
  $item = $q->peek;      # get item without removing it
  @up_to_10_items = $q->get(10);
  $remaining_items = $q->pending;

=head1 DESCRIPTION

Interface for a queue object that can be shared across processes.
Available implementations are L<Forks::Queue::File|Forks::Queue::File>,
L<Forks::Queue::Shmem|Forks::Queue::Shmem>,
L<Forks::Queue::SQLite|Forks::Queue::SQLite>.

=head1 METHODS

Many of these methods pass or return "tasks". For this distribution,
a "task" is any scalar or reference that can be serialized and
shared across processes.

This will include scalars and most unblessed references

  "42"
  [1,2,3,"forty-two"]
  { name=>"a job", timestamp=>time, input=>{foo=>[19],bar=>\%bardata} }

but will generally prohibit data with blessed references and code references

  { name => "bad job", callback => \&my_callback_routine }
  [ 46, $url13, File::Temp->new ]

=head2 new

=head2 $queue = Forks::Queue->new( %opts )

Instantiates a new queue object with the given configuration.

If one of the options is C<impl>, the constructor from that
C<Forks::Queue> subclass will be invoked.

If the C<impl> option is not provided, this call will fail.

Options that should be supported on all implementations include

=over 4

=item * C<< style => 'fifo' | 'lifo' >>

Indicates whether the L<"get"> method will return items in
first-in-first-out order or last-in-first-out order (and which
end of the queue the L<"peek"> method will examine)

=item * C<< limit => int >>

A maximum size for the queue. Set to a non-positive value to
specify an unlimited size queue.

=item * C<< on_limit => 'block' | 'fail' >>

Dictates what the queue should do when an attempt is made to
add tasks beyond the queue's limit. If C<block>, the queue
will block and wait until items are removed from the queue.
If C<fail>, the queue will warn and return immediately without
changing the queue.

=item * C<< join => bool >>

If true, expects that the queue referred to by this constructor
has already been created in another process, and that the current
process should access the existing queue. This allows a queue to
be shared across unrelated processes (i.e., processes that do not
have a parent-child relationship).

  # my_daemon.pl - may run "all the time" in the background
  $q = Forks::Queue::File->new(file=>'/var/spool/foo/q17');
  # creates new queue object
  ... 

  # worker.pl - may run periodically for a short time, launched from
  #             cron or from command line, but not from the daemon
  $q = Forks::Queue->new( impl => 'File', join => 1,
                          file => '/var/spool/foo/q17',
  # the new queue attaches to existing file at /var/spool/foo/q17
  ...

C<join> is not necessary for child processes forked from a process with
an existing queue

  $q = Forks::Queue->new(...)
  ...
  if (fork() == 0) {
      # $q already exists and the child process can begin using it,
      # no need for a  Forks::Queue  constructor with  join
      ...
  }

=item * C<< persist => bool >>

Active C<Forks::Queue> objects affect your system, writing to disk or
writing to memory, and in general they clean themselves up when they
detect that no more processes are using the queue. The C<persist> option,
if set to true, instructs the queue object to leave its state intact
after destruction.

An obvious use case for this option is debugging, to examine the
state of the queue after abnormal termination of your program.

A second use case is to create persistent queues -- queues that are
shared not only among different processes, but among different 
processes that are running at different times. The persistent queue
can be used by supplying both the C<persist> and the C<join> options
to the C<Forks::Queue> constructor.

    $queue_file = "/tmp/persistent.job.queue";
    $join = -f $queue_file;
    $q = Forks::Queue->new( impl => 'File', file => $queue_file,
                            join => $join, persist => 1 );
    ... work with the queue ...
    # the queue remains intact if this program exits or aborts


=back

=head2 put

=head2 $count = $queue->put(@tasks)

Place one or more "tasks" on the queue, and returns the number of
tasks successfully added to the queue.

Adding tasks to the queue will fail if the L<"end"> method of
the queue had previously been called from any process.

=head2 push

=head2 $count = $queue->push(@tasks)

Equivalent to L<"put">, adding tasks to the end of the queue and
returning the number of tasks successfully added. The most recent
tasks appended to the queue by C<push> or C<put> will be the first
tasks taken from the queue by L<"pop"> or by L<"get"> with LIFO
style queues, and the last tasks removed by L<"shift"> or L<"get">
with FIFO style queues.

For now there is no C<unshift> method to add tasks to the
I<front> of the queue (and to be retrieved first in FIFO queues),
but maybe someday there will be.

=head2 end

=head2 $queue->end;

Indicates that no more tasks are to be put on the queue,
so that when a process tries to retrieve a task from an empty queue,
it will not block and wait until a new task is added.
This method may be called from any process that has access to the queue.


=head2 get

=head2 $task = $queue->get

=head2 @tasks = $queue->get($count)

Attempt to retrieve one or more "tasks" on the queue. If the
queue is empty, and if L<"end"> has not been called on the queue,
this call blocks until a task is available or until the L<"end">
method has been called from some other process. If the queue is
empty and L<"end"> has been called, this method returns an
empty list in list context or C<undef> in scalar context.

If a C<$count> argument is supplied, returns up to C<$count> tasks or however
many tasks are currently available on the queue, whichever is fewer.

=head2 pop

=head2 $task = $queue->pop

=head2 @tasks = $queue->pop($count)

Retrieves one or more tasks from the "back" of the queue.
For LIFO style queues, the L<"get"> method is equivalent to this method.
Like C<"get">, this method blocks while the queue is empty and the
L<"end"> method has not been called on the queue.

If a C<$count> argument is supplied, returns up to C<$count> tasks or however
many tasks are currently available on the queue, whichever is fewer.

=head2 shift

=head2 $task = $queue->shift

=head2 @tasks = $queue->shift($count)

Retrieves one or more tasks from the "front" of the queue.
For FIFO style queues, the L<"get"> method is equivalent to this method.
Like C<"get">, this method blocks while the queue is empty and the
L<"end"> method has not been called on the queue.

If a C<$count> argument is supplied, returns up to C<$count> tasks or however
many tasks are currently available on the queue, whichever is fewer.

=head2 peek

=head2 $task = $queue->peek

=head2 $task = $queue->peek_front

=head2 $task = $queue->peek_back

Returns a task from the queue without removing it. The C<peek_front>
and C<peek_back> methods inspect the task at the front and the back of
the queue, respectively. The generic C<peek> method is equivalent to
C<peek_front> for FIFO style queues and C<peek_back> for LIFO style
queues.

If the queue is empty, these methods will return C<undef>.

Note that unlike the 
L<<"peek" method in C<Thread::Queue>|Thread::Queue/"peek">>,
C<Forks::Queue::peek> returns a copy of the item on the queue,
so manipulating a reference returned from C<peek> while B<not>
affect the item on the queue.


=head2 pending

=head2 $num_tasks_avail = $queue->pending

Returns the total number of tasks available on the queue. There is no
guarentee that the number of available tasks will not change between a
call to C<pending> and a subsequent call to L<"get">

=head2 status

=head2 $status = $queue->status

Returns a hash reference with meta information about the queue.
The information should at least include the number of tasks remaining in
the queue. Other implementations may provide additional information
in this return value.

=head1 DEPENDENCIES

The C<Forks::Queue> module and all its current implementations require
the L<JSON|JSON> module.

=head1 SEE ALSO

L<Thread::Queue|Thread::Queue>, L<File::Queue|File::Queue>,
L<Queue::Q|Queue::Q>, L<MCE::Queue|MCE::Queue>,
L<Queue::DBI|Queue::DBI>, L<Directory::Queue|Directory::Queue>.

=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Forks::Queue


You can also look for information at:

=over 4

=item * RT: CPAN's request tracker (report bugs here)

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=Forks-Queue>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/Forks-Queue>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/Forks-Queue>

=item * Search CPAN

L<http://search.cpan.org/dist/Forks-Queue/>

=back


=head1 LICENSE AND COPYRIGHT

Copyright (c) 2017, Marty O'Brien.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.10.1 or,
at your option, any later version of Perl 5 you may have available.

See http://dev.perl.org/licenses/ for more information.

=cut

# TODO:
#
#     non-blocking get, pop, shift
#     unshift (inefficient for FQ::File)
#     enqueue/dequeue for Threads::Queue compatibility
#     insert, extract for Threads::Queue compatibility, random access
#         (inefficient for FQ::File)
#     peek(INDEX) for random access
#     new( list => [ ] ) to populate queue at initialization
#     priorities
#     Directory implementation (see Queue::Dir)
#     use SIGIO to wait for tasks/capacity, not just sleep
#
