package WebService::Strike;

use 5.014000;
use strict;
use warnings;
use parent qw/Exporter/;

our @EXPORT    = qw/strike strike_search/; ## no critic (ProhibitAutomaticExportation)
our @EXPORT_OK = qw/strike_query strike strike_search/;
our $VERSION = '0.002';
our $BASE_URL = 'http://getstrike.net/api/torrents/';

use JSON::MaybeXS qw/decode_json/;
use HTTP::Tiny;
use Sort::ByExample qw/sbe/;
use WebService::Strike::Torrent;

sub _ht { HTTP::Tiny->new(agent => "WebService-Strike/$VERSION") }

sub _query {
	my ($url) = @_;

	my $ht = _ht;
	my $response = $ht->get($url);
	die $response unless $response->{success}; ## no critic (RequireCarping)
	$response = decode_json $response->{content};

	die $response unless ref $response eq 'ARRAY'; ## no critic (RequireCarping)
	map { WebService::Strike::Torrent->new($_) } @{$response->[1]};
}

sub strike_query {
	my (@hashes) = @_;
	my $url = "$BASE_URL/info/?hashes=" . join ',', map { uc } @hashes;

	my $sorter = sbe(\@hashes, {xform => sub { $_[0]->hash }});
	my @torrents = $sorter->(_query $url);
	wantarray ? @torrents : $torrents[0]
}

sub strike_search {
	my ($query, $full, %args) = @_;
	$args{q} = $query;
	my $url = "$BASE_URL/search/?" . HTTP::Tiny->www_form_urlencode(\%args);

	my @torrents = _query $url;
	@torrents = $torrents[0] unless wantarray;
	@torrents = strike_query map { $_->hash } @torrents if $full;
	wantarray ? @torrents : $torrents[0]
}

BEGIN { *strike = \&strike_query }

1;
__END__

=encoding utf-8

=head1 NAME

WebService::Strike - Get torrent info from getstrike.net API

=head1 SYNOPSIS

  use WebService::Strike;
  my $t = strike 'B425907E5755031BDA4A8D1B6DCCACA97DA14C04';
  say $t->title;               # Arch Linux 2015.01.01 (x86\/x64)
  say $t->magnet;              # Returns a magnet link
  my $torrent = $t->torrent;   # Returns the torrent file
  $t->torrent('file.torrent'); # Downloads the torrent file to 'file.torrent'

  my @debian = strike_search 'Debian';
  say 'Found ' . @debian . ' torrents matching "Debian"';
  say 'First torrent has info hash ' . $debian[0]->hash;

  my $mp = strike_search 'Modern perl', 1;
  say 'Torrent has ' . $mp->count . ' files. They are:';
  say join ' ', @{$mp->file_names};

=head1 DESCRIPTION

Strike API is a service for getting information about a torrent given
its info hash. WebService::Strike is a wrapper for this service.

=over

=item B<strike>(I<@info_hashes>)

Returns a list of L<WebService::Strike::Torrent> objects in list
context or the first such object in scalar context. Dies in case of
error.

=item B<strike_query>

Alias for B<strike>. Not exported by default.

=item B<strike_search>(I<$search_term>, [I<$full>])

Searches for torrents given a search term. Returns a list of
L<WebService::Strike::Torrent> objects in list context or the first
such object in scalar context.

If I<$full> is false (default), the returned objects will be
incomplete: their B<file_names> and B<file_lengths> accessors will
return undef.

If I<$full> is true, B<strike> will be called with the info hashes of
the found torrents, thus obtaining complete objects.

=back

=head1 SEE ALSO

L<WebService::Strike::Torrent>, L<http://getstrike.net/api/>, L<WWW::Search::Torrentz>

=head1 AUTHOR

Marius Gavrilescu, E<lt>marius@ieval.roE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2015 by Marius Gavrilescu

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.20.2 or,
at your option, any later version of Perl 5 you may have available.


=cut
