package App::VOJournal::VOTL;
#
# vim: set sw=4 ts=4 tw=76 et ai si:
#
# Author Mathias Weidner <mamawe@cpan.org>
# Version 0.1
# Copyright (C) 2015 user <user@work>
# Modified On 2015-05-15 22:18
# Created  2015-05-15 22:18
#
use strict;
use warnings;

=head1 NAME

App::VOJournal::VOTL - deal with vimoutliner files

=head1 VERSION

Version 0.3.0

=cut

use version; our $VERSION = qv('0.3.0');

=head1 SYNOPSIS

    use App::VOJournal::VOTL;

    my $votl = App::VOJournal::VOTL->new();

    $votl->read_file($infilename);

    $votl->write_file($outfilename);
    $votl->write_file($outfilename, \&filter);

    $votl->write_file_no_checked_boxes($outfilename);
    $votl->write_file_unchecked_boxes($outfilename);

=head1 SUBROUTINES/METHODS

=head2 new

=cut

sub new
{
    my $class = shift;
    my $arg = shift;
    my $self = {};

    $self->{objects} = [];

    bless($self, $class);
    return $self;
}

=head2 read_file

Reads a vimoutliner file.

    $votl->read_file( $filename );

=cut

sub read_file {
    my ($self,$filename) = @_;

    if (open my $input, '<', $filename) {
        $self->{objects} = [];
        while (<$input>) {
            if (/^(\t*)([:;|<>])(.*)$/) {
                $self->_add_something($1, {
                    children => [],
                    type => $2,
                    value => $3
                });
            }
            elsif (/^(\t*)(\[[_X]\])(.*)$/) {
                $self->_add_something($1, {
                    children => [],
                    type => $2,
                    value => $3
                });
            }
            elsif (/^(\t*)([^\t].*)$/) {
                $self->_add_something($1, {
                    children => [],
                    type => '',
                    value => $2
                });
            }
            else {
                die "unknown line: $_";
            }
        }
        close $input;
        return 1 + $#{$self->{objects}};
    }
    return;
} # read_file()

=head2 write_file

Writes a vimoutliner file.

    $votl->write_file( $filename );

    sub filter { ... }

    $votl->write_file( $filename, \&filter);

=cut

sub write_file {
    my ($self,$filename,$filter) = @_;

    if (open my $output, '>', $filename) {
        foreach my $object (@{$self->{objects}}) {
            _write_object($object,0,$output,$filter);
        }
        close $output;
    }
} # write_file()

=head2 write_file_no_checked_boxes

Writes a vimoutliner file that contains no checked boxes.

    $votl->write_file_no_checked_boxes( $filename );

This is a convenience function using C<write_file()> and a predifined
filter.

=cut

sub write_file_no_checked_boxes {
    my ($self,$filename) = @_;
    my $filter = sub {
        my ($object) = @_;
        return ! _checked_box($object);
    };
    $self->write_file( $filename, $filter );
} # write_file_no_checked_boxes()

=head2 write_file_unchecked_boxes

Writes a vimoutliner file that only consists of unchecked boxes at level
zero and their descendants.

    $votl->write_file_unchecked_boxes( $filename );

This is a convenience function using C<write_file()> and a predifined
filter.

=cut

sub write_file_unchecked_boxes {
    my ($self,$filename) = @_;
    my $filter = sub {
        my ($object,$indent) = @_;
        return $indent ? 1 : _unchecked_box($object);
    };
    $self->write_file( $filename, $filter );
} # write_file_unchecked_boxes()

sub _add_something {
    my ($self,$tabs,$newobject) = @_;
    my $indent = length $tabs;
    my $objects = $self->_descend_objects($indent);
    push @$objects, $newobject;
} # _add_something()

sub _checked_box {
    my ($object) = @_;

    return ($object->{type} =~ /^\[X\]$/);
} # _checked_box()

sub _descend_objects {
    my ($self,$indent) = @_;
    my $objects = $self->{objects};

    while (0 < $indent) {
        if (0 > $#$objects) {
            my $newobject = {
                children => [],
            };
            push @$objects, $newobject;
            $objects = $newobject->{children};
        }
        else {
            $objects = $objects->[$#$objects]->{children};
        }
        $indent--;
    }
    return $objects;
} # _descend_objects()

sub _unchecked_box {
    my ($object) = @_;

    return ($object->{type} =~ /^\[_\]$/);
} # _unchecked_box()

sub _write_object {
    my ($object,$indent,$outfh, $filter) = @_;

    if (defined $filter) {
        return unless ($filter->($object,$indent));
    }
    print $outfh "\t" x $indent, $object->{type}, $object->{value}, "\n";

    foreach my $co (@{$object->{children}}) {
        _write_object($co,$indent + 1,$outfh,$filter);
    }
} # _write_object()

1;
# __END__

=head1 FORMAT OF VIMOUTLINER FILES

Vimoutliner files are text files with a hierarchical structure.

The hierarchical structure is characterized by the number of tabulator
signs (0x09) at the beginning of the line.

A line can be a simple-heading or an object, depending on the first
nontabulator sign of the line.

A simple heading starts with any non-whitespace character except
C<< : ; | < > >>.
A checkbox is a special form of a heading that starts with either
C<< [_] >> or C<< [X] >> after the leading tabulator signs.
A checkbox may contain a percent sign (C<%>) as a placeholder for
the percentage completed.
This percent sign must follow the initial C<< [_] >> after a separating
whitespace.

The following text objects are defined for vimoutliner files:

=over 4

=item C<:> - body text

The text following the C<:> will be wrapped automatically.

=item C<;> - preformatted body text

This text won't be wrapped automatically.

=item C<|> - table

The table headings can be marked with C<||>.

=item C<< > >> - user defined text.

This text will also be wrapped automatically.

=item C<< < >> - user defined preformatted text.

This text won't be wrapped automatically.

=back

=head1 AUTHOR

Mathias Weidner, C<< <mamawe at cpan.org> >>

=head1 BUGS

Please report any bugs or feature requests to C<bug-app-vojournal at rt.cpan.org>, or through
the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=App-VOJournal>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.

=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc App::VOJournal::VOTL


You can also look for information at:

=over 4

=item * RT: CPAN's request tracker (report bugs here)

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=App-VOJournal>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/App-VOJournal>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/App-VOJournal>

=item * Search CPAN

L<http://search.cpan.org/dist/App-VOJournal/>

=back


=head1 ACKNOWLEDGEMENTS


=head1 LICENSE AND COPYRIGHT

Copyright 2015 Mathias Weidner.

This program is free software; you can redistribute it and/or modify it
under the terms of the the Artistic License (2.0). You may obtain a
copy of the full license at:

L<http://www.perlfoundation.org/artistic_license_2_0>

Any use, modification, and distribution of the Standard or Modified
Versions is governed by this Artistic License. By using, modifying or
distributing the Package, you accept this license. Do not use, modify,
or distribute the Package, if you do not accept this license.

If your Modified Version has been derived from a Modified Version made
by someone other than you, you are nevertheless required to ensure that
your Modified Version complies with the requirements of this license.

This license does not grant you the right to use any trademark, service
mark, tradename, or logo of the Copyright Holder.

This license includes the non-exclusive, worldwide, free-of-charge
patent license to make, have made, use, offer to sell, sell, import and
otherwise transfer the Package with respect to any patent claims
licensable by the Copyright Holder that are necessarily infringed by the
Package. If you institute patent litigation (including a cross-claim or
counterclaim) against any party alleging that the Package constitutes
direct or contributory patent infringement, then this Artistic License
to you shall terminate on the date that such litigation is filed.

Disclaimer of Warranty: THE PACKAGE IS PROVIDED BY THE COPYRIGHT HOLDER
AND CONTRIBUTORS "AS IS' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.
THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
PURPOSE, OR NON-INFRINGEMENT ARE DISCLAIMED TO THE EXTENT PERMITTED BY
YOUR LOCAL LAW. UNLESS REQUIRED BY LAW, NO COPYRIGHT HOLDER OR
CONTRIBUTOR WILL BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, OR
CONSEQUENTIAL DAMAGES ARISING IN ANY WAY OUT OF THE USE OF THE PACKAGE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

