package Cisco::UCS::Common::PSU;

use warnings;
use strict;

use Carp qw(croak);
use Scalar::Util qw(weaken);

our $VERSION	= '0.1';

our @ATTRIBUTES	= qw(dn id model operability power presence revision serial thermal vendor voltage);

our %ATTRIBUTES	= (
		operational	=> 'operState',
		performance	=> 'perf'
		);

=head1 NAME

Cisco::UCS::Common::PSU - Class for operations with a Cisco UCS PSU.

=head1 SYNOPSIS

Quick summary of what the module does.

Perhaps a little code snippet.

    foreach my $psu (sort $ucs->chassis(1)->get_psus) {
      print 'PSU ' . $psu->id . ' voltage: ' . $psu->voltage . "\n" 
    }

    # PSU 1 voltage: ok
    # PSU 2 voltage: ok
    # PSU 3 voltage: ok
    # PSU 4 voltage: ok

=head1 METHODS

=head3 id

Returns the ID of the PSU.

=head3 dn

Returns the distinguished name of the PSU.

=head3 serial

Returns the serial number of the PSU.

=head3 model

Returns the model number of the PSU.

=head3 revision

Returns the hardware revision number of the PSU.

=head3 vendor

Returns the vendor name of the PSU.

=head3 presence

Returns the presence status of the PSU.

=head3 operability

Returns the operability status of the PSU.

=head3 voltage

Returns the voltage status of the PSU.

=head3 power

Returns the power status of the PSU.

=head3 thermal

Returns the thermal status of the PSU.

=head3 operational

Returns the operational status of the PSU.

=head3 performance

Returns the performance status of the PSU.

=cut

sub new {
        my ($class, %args) = @_;
        my $self = {};
        bless $self, $class;
        defined $args{dn}       ? $self->{dn}   = $args{dn}             : croak 'dn not defined';
        defined $args{ucs}      ? weaken($self->{ucs} = $args{ucs})     : croak 'ucs not defined';
        my %attr = %{$self->{ucs}->resolve_dn(dn => $self->{dn})->{outConfig}->{equipmentPsu}};
        
        while (my ($k, $v) = each %attr) { $self->{$k} = $v }
                
        return $self;
}


=head1 AUTHOR

Luke Poskitt, C<< <ltp at cpan.org> >>

=head1 BUGS

Some methods may return undefined, empty or not yet implemented values.  This is dependent on the
software and firmware revision level of UCSM and components of the UCS cluster.  This is not a
bug but is a limitation of UCSM.

Please report any bugs or feature requests to C<bug-cisco-ucs-common-psu at rt.cpan.org>, or through
the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=Cisco-UCS-Common-PSU>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.


=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Cisco::UCS::Common::PSU


You can also look for information at:

=over 4

=item * RT: CPAN's request tracker

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=Cisco-UCS-Common-PSU>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/Cisco-UCS-Common-PSU>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/Cisco-UCS-Common-PSU>

=item * Search CPAN

L<http://search.cpan.org/dist/Cisco-UCS-Common-PSU/>

=back


=head1 ACKNOWLEDGEMENTS


=head1 LICENSE AND COPYRIGHT

Copyright 2012 Luke Poskitt.

This program is free software; you can redistribute it and/or modify it
under the terms of either: the GNU General Public License as published
by the Free Software Foundation; or the Artistic License.

See http://dev.perl.org/licenses/ for more information.


=cut

{
        no strict 'refs';

        while ( my ($pseudo, $attribute) = each %ATTRIBUTES ) {
                *{ __PACKAGE__ . '::' . $pseudo } = sub {
                        my $self = shift;
                        return $self->{$attribute}
                }
        }

        foreach my $attribute (@ATTRIBUTES) {
                *{ __PACKAGE__ . '::' . $attribute } = sub {
                        my $self = shift;
                        return $self->{$attribute}
                }
        }

}

1;
