# $Id: Tabs.pm,v 1.17 2002/09/07 04:33:17 koos Exp $
package CGI::Widget::Tabs;

use 5.006;
use strict;
use warnings;

use URI::Escape();
use HTML::Entities();

our $VERSION = '1.2.1';



# ----------------------------------------------
sub new {
# ----------------------------------------------
    my $proto = shift;
    my $class = ref($proto) || $proto;
    my $self = {};
    bless ($self, $class);
    return $self;
}



# ----------------------------------------------
sub headings {
# ----------------------------------------------
    #
    #  ( "Software", "Hardware, ...) or ( -sw=>"Software", -hw=>"Hardware", ... );
    #
    my $self = shift;
    if ( @_ ) {
        $self->{headings} = [ @_ ];
    };
    return @{ $self->{headings} || [] };
}



# ----------------------------------------------
sub default {
# ----------------------------------------------
    #
    # The default active tab
    #
    my $self = shift;
    if ( @_ ) {
        $self->{default} = shift;
    }
    return $self->{default}
}



# ----------------------------------------------
sub cgi_object {
# ----------------------------------------------
    #
    # The cgi object to retrieve the parameters from.
    # Could be a CGI object or a CGI::Minimal object.
    #
    my $self = shift;
    if ( @_ ) {
        $self->{cgi_object} = shift;
    }
    return $self->{cgi_object};
}



# ----------------------------------------------
sub cgi_param {
# ----------------------------------------------
    #
    # CGI parameter specifing the tab. Defaults to "tab".
    #
    my $self = shift;
    if ( @_ ) {
        $self->{cgi_param} = shift;
    }
    return $self->{cgi_param}||'tab';
}



# ----------------------------------------------
sub active {
# ----------------------------------------------
    #
    # The active pab page
    # In order of precendence:
    # 1. The tab clicked by the user
    # 2. The default tab
    # 3. The first tab in the list
    #
    my $self = shift;
    my $really_active = $self->cgi_object->param($self->cgi_param);

    if ( $really_active ) {  # tab has been clicked
        return $really_active;
    }
    return $self->default || ($self->headings)[0]; # the default tab or the first
}



# ----------------------------------------------
sub class {
# ----------------------------------------------
    #
    # The CSS class for display of the tabs
    # Defaults to 'tab'.
    #
    my $self = shift;
    if ( @_ ) {
        $self->{class} = shift;
    }
    return $self->{class} || 'tab';
}



# ----------------------------------------------
sub display {
# ----------------------------------------------
    #
    # save a few keystrokes
    #
    my $self = shift;
    print $self->render;
}



# ----------------------------------------------
sub render {
# ----------------------------------------------
    #
    # Process the lot and display it.
    #
    my $self        = shift;
    my $cgi         = $self->cgi_object;
    my @headings    = $self->headings;  # we either get ( "Tab1", "Tab2", ... ) or ( -t1=>"Tab1", -t2=>"Tab2", ... );
    my $class       = $self->class;
    my $cgi_param   = $self->cgi_param;
    my $active      = $self->active;
    my $spacer      = '<td class="'.$class.'_spc"></td>';
    my @html;
    my $params;


    # reproduce the URL incl all CGI params, _except_ the varying tab
    $params = "";
    foreach ( $cgi->param() ) {
        next if $_ eq $cgi_param;
        $params .= $_.'='.URI::Escape::uri_escape($cgi->param($_)||"").'&';
    }  # now we only have to add the tab cgi_param + value

    @html = ();
    push @html, "<!-- Generated by CGI::Widget::Tabs v$VERSION -->\n";
    push @html, '<table class="',$class,'">',"\n<tr>\n";
    push @html, $spacer,"\n";

    if ( @headings ) {
        # --- Did we get the -t=>"Tab" version?
        if ( substr($headings[0],0,1) eq '-' ) {
            my $heading_key;
            my $heading_text;
            while ( @headings ) {
                $heading_key  = shift @headings;
                $heading_text = shift @headings;
                push @html, '<td class="',$class;
                push @html, '_actv' if $heading_key eq $active;
                push @html, '">';
                push @html, &_link($heading_text,'?'.$params.$cgi_param.'='.URI::Escape::uri_escape($heading_key));
                push @html, "</td>";
                push @html, $spacer,"\n";
            }
        } else {
            # --- No, we got the ("Tab1", "Tab2", ...)  version.
            foreach my $heading ( @headings ) {
                push @html, '<td class="',$class;
                push @html, '_actv' if $heading eq $active;
                push @html, '">';
                push @html, &_link($heading,'?'.$params.$cgi_param.'='.URI::Escape::uri_escape($heading));
                push @html, "</td>";
                push @html, $spacer,"\n";
            }
        }
    }
    push @html, "</tr>\n</table>\n";
    return join("",@html);
    push @html, "<!-- End CGI::Widget::Tabs v$VERSION -->";
}



# ----------------------------------------------
sub _link {
# ----------------------------------------------
    #
    # Internal. Create a link for some text to a URI
    # Expects = (<text>,<url>) pair.
    #
    return '<a href="'.$_[1].'">'.HTML::Entities::encode_entities($_[0]).'</a>';
}

1;
__END__

=head1 NAME

CGI::Widget::Tabs - Create tab widgets in HTML

=head1 SYNOPSIS

    use CGI::Widget::Tabs;
    my $tab = CGI::Widget::Tabs->new;

    use CGI;
    my $cgi = CGI->new;          # interface to the query params
    $tab->headings(@titles);     # e.g. qw/Drivers Cars Courses/
    $tab->default("Courses");    # the default active tab
    $tab->active;                # the currently active tab
    $tab->class("my_tab");       # the CSS class to use for markup
    $tab->cgi_object($cgi);      # the object holding the query params
    $tab->cgi_param("t");        # the CGI query parameter to use
    $tab->render;                # the resulting HTML code
    $tab->display;               # same as `print $tab->render'

    # See the EXAMPLE section for a complete example

=head1 DESCRIPTION

=head2 Introduction

CGI::Widget::Tabs lets you simulate tab widgets in HTML. You could benefit
from tabs if you want to serve only one page. Depending on the tab selected
you fetch and display the underlying data. There are two main reasons for
taking this approach:

1. For the end user not to be directed to YAL or YAP (yet another link / yet
another page), but keep it all together: The single point of entry paradigm.

2. For the Perl hacker to generate and display multiple data sources within
the same script environment.

The nice thing about CGI::Widget::Tabs is that the tabs know their internal
state. So you can ask a tab for instance which tab page is highlighted. This
way you get direct feedback on what the user clicked.

=head2 "Hey Gorgeous!"

Of course tabs are useless if you can't "see" them. Without proper make up
they print as ordinary text. So you really need to fancy them up with some
eye candy. The designed way is that you provide a CSS style sheet and have
CGI::Widget::Tabs use that. See the class() method for how to do this.



=head1 METHODS OVERVIEW

=over 4

=item B<new()>

Returns a new CGI::Widget::Tabs object. Example:

    use CGI::Widget::Tabs;
    my $tab = CGI::Widget::Tabs->new;



=item B<headings(LIST)>

Sets/retrieves the headings to be displayed on the tabs. If the optional
argument LIST is given the headings are set. Otherwise they are retrieved.
You can specify LIST in two ways:

=over 4

=item * a plain list

=item * a keyword/value list

=back

The keyword/value list comes in handy if you don't want to check the value
returned by active() against very long words. Moreover, if you change the tab
headings but use the same keys you don need to change your code. So it is
less  error prone. As a pleasant side effect, the URL's get significantly
shorter. Do notice that the keys want to be unique.

Example:

    $tab->headings( qw/Planes Trains Automobiles/ );

    $tab->headings( -p=>"Planes", -t=>"Trains", -a=>"Automobiles" );

    my @h = $tab->headings;




=item B<default(STRING)>

Sets/retrieves the default active tab. If the optional argument STRING is
given the default tab is set. Otherwise it is retrieved. Example:

    $tab->default("Trains");  # if you use a normal list

    $tab->default("-t");      # take this for a key/value list

    my $default = $tab->default;



=item B<active()>

Returns the current active tab. This is (in order of precedence) the tab
being clicked on, the default tab, or the first in the list. Example:

    if ( $tab->active eq "Trains" ) {  # display the train tables
         ....

    if ( $tab->active eq "-t" ) {      # the same for a k/v list
         ....


=item B<class(STRING)>

Sets/retrieves the name of the CSS class used for the tabs markup. If the
optional argument STRING is given the class is set, otherwise it is
retrieved. In the accompanying style sheet, there are four class elements you
need to provide:

=over 4

=item 1. A table element for containment of the entire tab widget

=item 2. A td element for a normal tab

=item 3. A td element for the active tab

=item 4. A td element for the spacers

=back

The class names of these elements are gotten from the class() method. The
element for an active tab gets "_actv" added to the class name. The spacer
element gets "_spc" added. For instance, if you'd run

    $tab->class("my_tab");

then the four elements look like:

    <table class="my_tab">    # the entire table
    <td class="my_tab">       # normal tab
    <td class="my_tab_actv">  # highlighted tab
    <td class="my_tab_spc">   # spacer


Look at the example in the EXAMPLE section to see how this works out.



=item B<cgi_object(OBJECT)>

Sets/retrieves the CGI or CGI::Minimal object. If the optional argument
OBJECT is given, the CGI object is set, otherwise it is retrieved.
CGI::Widget::Tabs uses this object to pass and retrieve information on the
CGI query parameters. If you want you can use some other CGI object handler.
However such an object handler must provide a param() method with
corresponding behaviour as do CGI or CGI::Minimal. Note that currently only
CGI and CGI minimal have been tested. Example:

    my $cgi = CGI::Minimal->new;
    $tab->cgi_object($cgi);

    my $cgi_obj = $tab->cgi_object;



=item B<cgi_param(STRING)>

Sets/retrieves the CGI query parameter. This parameter identifies the tab in
the CGI query string. If the optional argument STRING is given, the query
parameter is set. Otherwise it is retrieved. Usually you can leave this
untouched. In that case the default parameter "tab" is used. You will need to
set this if you have more CGI query parameters on the URL with "tab" already
being taken. Another situation is if you use multiple tabs widgets on one
page. They both would use "tab" by default causing conflicts. Example:

   # Lets paint a fruit tab and a vegetable tab
   my $fruits_tab = CGI::Widget::Tabs->new;
   my $vegies_tab = CGI::Widget::Tabs->new;

   # this is our link with the outside world
   my $cgi = CGI::Minimal->new;
   $fruits_tab->cgi_object($cgi);
   $vegies_tab->cgi_object($cgi);

   # In the CGI params collection the first is identified by
   # 'ft' and the second by 'vt'
   $fruits_tab->cgi_param("ft");
   $vegies_tab->cgi_param("vt");




=item B<display()>

Prints the tab widget to the default filehandle (usually STDOUT). Example:


    $tab->display;       # this is the same as...

    print $tab->render;  # ... but saves you a few keystrokes




=item B<render()>

Renders the tab widget and returns the resulting HTML code. This is useful if
you need to print the tab to a different file handle. Another use is if you
want to manipulate the HTML. For instance to insert session id's or the like.
See the class() method and the EXAMPLE section somewhere else in this
document to see how you can influence the markup of the tab widget. Example:

    my $html = $tab->render;
    print HTML $html;  # there's a session id filter behind HTML


=back




=head1 EXAMPLE

As an example probably is the most explanatory, here is something to work
with. The following code is a simple but complete example. Copy it and run it
through the webservers CGI engine. (For a even more complete and useful demo
with multiple tabs, see the file tabs-demo.pl in the CGI::Widget::Tabs
installation directory.)

    #!/usr/bin/perl -w

    use CGI::Widget::Tabs;
    use CGI;

    print <<EOT;
    Content-Type: text/html;

    <head>
    <style type="text/css">
    table.my_tab   { border-bottom: solid thin black }
    td.my_tab      { padding: 2 12 2 12; background-color: #FAFAD2 }
    td.my_tab_actv { padding: 2 12 2 12; background-color: #C0D4E6 }
    td.my_tab_spc  { width: 15 }
    </style></head>
    <body>
    EOT

    my $cgi = CGI->new;
    my $tab = CGI::Widget::Tabs->new;
    $tab->cgi_object($cgi);
    $tab->class("my_tab");
    $tab->headings( "Hardware", "Lease Cars", "Xerox", "Mobiles");
    $tab->default("Lease Cars");
    $tab->display;
    print "<br>We now should run some intelligent code ";
    print "to process <strong>", $tab->active, "</strong><br>";
    print "</body></html>";




=head1 BUGS

As a side effect, the CGI query parameter to identify the tab (see the
cgi_param() method) is always moved to the end of the query string.



=head1 CONTRIBUTIONS

I would appreciate receiving your CSS style sheets used for the tabs markup.
Especially if you happened to be professionaly concerned with markup and
layout. For techies like us it is not always easy to see what goes and what
doesn't. If you send in a nice one, I will gladly bundle it with the next
release.



=head1 CREDITS

Bodo Eing E<lt>eingb@uni-muenster.deE<gt>


=head1 AUTHOR

Koos Pol E<lt>koos_pol@raketnet.nlE<gt>


=head1 DOWNLOAD

The latest version of CGI::Widget::Tabs is available from
CPAN (http://cpan.perl.org) or the CGI::Widget::Tabs homepage
(http://users.raketnet.nl/koos_pol/en/Tabs/index.html)



=head1 SEE ALSO

the manpages for CGI or CGI::Minimal, the CSS1 specs from the World Wide Web
consortium (http://www.w3.org/TR/REC-CSS1)

=cut
