use strict;
package ObjStore::AVHV::Fields;
use Carp;
use ObjStore;
use base 'ObjStore::HV';
use vars qw($VERSION %LAYOUT_VERSION);
$VERSION = '0.02';

'ObjStore::Database'->
    _register_private_root_key('layouts', sub { 'ObjStore::HV'->new(shift, 30) });
push(@ObjStore::Database::OPEN1, \&verify_class_fields);

sub verify_class_fields {
    return if $] < 5.00450;
    my ($db) = @_;
    my $layouts = $db->_private_root_data('layouts');
    map { get_certified_layout($layouts, $_) } keys %$layouts;
}

sub is_compat {
    my ($l, $tlay) = @_;
    for my $k (keys %$tlay) {
	next if $k =~ m'^_';
	return if ($l->{$k} || -1) != $tlay->{$k};
    }
    1;
}

sub get_transient_layout {
    my ($class) = @_;
    no strict 'refs';
    croak '\%{'.$class.'\::FIELDS} not found'
	if !defined %{"$class\::FIELDS"};
    \%{"$class\::FIELDS"};
}

sub get_certified_layout {
    my ($layouts, $of) = @_;
    my $l = $layouts->{$of};
    return if !$l || !$l->{__VERSION__};

    return $l if $l->{__VERSION__} == ($LAYOUT_VERSION{$of} or 0);

    my $tlay = get_transient_layout($of);
    return if !is_compat($l, $tlay);

    $LAYOUT_VERSION{$of} = $l->{__VERSION__};
    $l;
}

sub new { #XS? XXX
    my ($class, $db, $of) = @_;
    my $layouts = $db->_private_root_data('layouts');

    my $l = get_certified_layout($layouts, $of);
    return $l if $l;

    my $old = $layouts->{$of};
    if ($old and $ObjStore::RUN_TIME == ($old->{__VERSION__} or 0)) {
	confess "ObjStore::AVHV must be notified of run-time manipulation of field layouts by changing \$ObjStore::RUN_TIME to be != \$layout->{__VERSION__}";

	# We only check the previous layout.  Potentially, an older layout
	# could have the same version.  This will be caught by the
	# UNIVERSAL is_evolved check (given a version mismatch!).
    }

    $l = $layouts->{$of} = get_transient_layout($of);
    $l->{__VERSION__} = $ObjStore::RUN_TIME;
    $l->{__CLASS__} = $of;
    bless $l, $class;
    $l->const;
    $LAYOUT_VERSION{$of} = $l->{__VERSION__};
    $l;
}

sub is_evolved {1} #const anyway

package ObjStore::AVHV;
use Carp;
use base 'ObjStore::AV';
use vars qw($VERSION);
$VERSION = '0.04';

sub new {
    require 5.00450;
    my ($class, $near, $init) = @_;
    croak "$class->new(near, init)" if @_ < 2;
    my $fmap = 'ObjStore::AVHV::Fields'->new($near->database_of, $class);
    my $o = $class->SUPER::new($near, $fmap->{__MAX__}+1);
    $o->[0] = $fmap;
    if ($init) {
	while (my ($k,$v) = each %$init) {
	    croak "Bad key '$k' for $fmap" if !exists $fmap->{$k};
	    $o->{$k} = $v;
	}
    }
    $o;
}

sub is_evolved {
    my ($o) = @_;
    ($o->SUPER::is_evolved() and
     $o->[0]{__VERSION__} == $ObjStore::AVHV::Fields::LAYOUT_VERSION{ ref($o) });
}

sub _avhv_relayout {
    require 5.00450;
    my ($o, $to) = @_;
    my $new = 'ObjStore::AVHV::Fields'->new($o->database_of, $to);
    return if $new == $o->[0];
    
    my $old = $o->[0];
    ObjStore::peek($old), croak "Found $old where ObjStore::AVHV::Fields expected"
	if ($old && !$old->isa('ObjStore::AVHV::Fields') &&
	    !$old->isa('ObjStore::HV'));

    #copy interesting fields to @tmp
    my @save;
    while (my ($k,$v) = each %$old) {
	next if $k =~ m'^_';
	push(@save, [$k,$o->[$v]]) if exists $new->{$k};
    }
    
    #clear $o & copy @save back using new schema
    for (my $x=0; $x < $o->_count; $x++) { $o->[$x] = undef }
    for my $z (@save) { $o->[ $new->{$z->[0]} ] = $z->[1]; }
    $o->[0] = $new;
    ();
}

sub BLESS {
    return shift->SUPER::BLESS(@_) if ref $_[0];
    my ($class, $o) = @_;
    _avhv_relayout($o, $class);
    $class->SUPER::BLESS($o);
}

sub evolve { bless $_[0], ref($_[0]); }

#sub POSH_CD { my ($a, $f) = @_; $a->{$f}; }  # now XS!

# Hash style, but in square brackets
sub POSH_PEEK {
    require 5.00450;
    my ($val, $o, $name) = @_;
    my $fm = $val->[0];
    my @F = sort grep(!m'^_', keys %$fm);
    $o->{coverage} += scalar @F;
    my $big = @F > $o->{width};
    my $limit = $big ? $o->{summary_width}-1 : $#F;
    
    $o->o($name . " [");
    $o->nl;
    $o->indent(sub {
	for my $x (0..$limit) {
	    my $k = $F[$x];
	    my $v = $val->[$fm->{$k}];
	    
	    $o->o("$k => ");
	    $o->peek_any($v);
	    $o->nl;
	}
	if ($big) { $o->o("..."); $o->nl; }
    });
    $o->o("],");
    $o->nl;
}

1;

=head1 NAME

  ObjStore::AVHV - Hash interface, array performance

=head1 SYNOPSIS

  package MatchMaker::Person;
  use base 'ObjStore::AVHV';
  use fields qw/ height hairstyle haircolor shoetype favorites /;

=head1 DESCRIPTION

Support for extremely efficient records.

=head1 TODO

=over 4

=item * More documentation

=item *

This could be implemented with zero memory overhead if we stored the
layout in the per-class globals.  Will wait for performance numbers
and overload % before considering such techniques.

=back

=cut
