# -*- Perl -*-
#
# logic expression related routines. see Logic::Expr::Parser for how the
# expressions are built

package Logic::Expr;
our $VERSION = '0.01';
use parent qw(Exporter);

our ( @EXPORT_OK, %EXPORT_TAGS );

BEGIN {
    @EXPORT_OK = qw(TRUE FALSE LE_NOT LE_AND LE_OR LE_COND LE_BICOND);
    %EXPORT_TAGS =
      ( all => [qw(TRUE FALSE LE_NOT LE_AND LE_OR LE_COND LE_BICOND)] );
}

sub TRUE ()  { 1 }
sub FALSE () { 0 }
# -1 is reserved for testing (FAKE_OP); 0 or 1 for ops would conflate
# with the prior should a truth value be assigned to an op slot
sub LE_NOT ()    { 2 }    # ! ~
sub LE_AND ()    { 3 }    # &
sub LE_OR ()     { 4 }    # | v
sub LE_COND ()   { 5 }    # ->
sub LE_BICOND () { 6 }    # ==

# 'atoms' contains a name to a scalar reference in 'bools'; 'bools' is
# an array of unique atoms in an expression (for easy iteration by the
# "solutions" method); 'expr' is the parse tree of the expression as
# probably generated by Logic::Expr::Parser. 'bools' must be modified
# in-place to not break the scalar references from the other two
# structures. for example, given X&Y,
#  * atoms - { X => \$bools[0], Y => \$bools[1] }
#  * bools - [ 1, 1 ]
#  * expr  - [ LE_AND, ... ]

sub atoms { $_[0]->{atoms} }
sub bools { $_[0]->{bools} }
sub expr  { $_[0]->{expr} }

sub new
{
    my ( $class, %param ) = @_;
    my $self = { map { $_ => $param{$_} } qw(atoms bools expr) };
    bless $self, $class;
    return $self;
}

# brute force all possible boolean states for an expression
sub solutions
{
    my ($self) = @_;
    my @orig;
    my $bools = $self->{bools};
    for my $x (@$bools) {
        push @orig, $x;
        $x = TRUE;
    }
    my @solutions = [ [@$bools], _solve( $self->{expr} ) ? TRUE : FALSE ];
    # the reverse index ordering is to match that of the logic book,
    # backwards binary counting
    my $i = $#$bools;
    while ( $i >= 0 ) {
        if ( $bools->[$i] ) {
            $bools->[$i] = FALSE;
            push @solutions, [ [ $bools->@* ], _solve( $self->{expr} ) ? TRUE : FALSE ];
            $i = $#$bools;
        } else {
            $bools->[ $i-- ] = TRUE;
        }
    }
    for my $x (@$bools) { $x = shift @orig }
    return \@solutions;
}

# solve the expression using the current state in bools
sub solve { _solve( $_[0]->{expr} ) ? TRUE : FALSE }

sub _solve
{
    my ($ptr) = @_;
    my $rt = ref $ptr;
    return $$ptr if $rt eq 'SCALAR';    # lookup from bools
    if ( $rt eq 'ARRAY' ) {
        if ( $ptr->[0] == LE_NOT ) {
            return !_solve( $ptr->[1] );
        } elsif ( $ptr->[0] == LE_AND ) {
            return _solve( $ptr->[1] ) && _solve( $ptr->[2] );
        } elsif ( $ptr->[0] == LE_OR ) {
            return _solve( $ptr->[1] ) || _solve( $ptr->[2] );
        } elsif ( $ptr->[0] == LE_COND ) {
            return !_solve( $ptr->[1] ) || _solve( $ptr->[2] );
        } elsif ( $ptr->[0] == LE_BICOND ) {
            return !( _solve( $ptr->[1] ) ^ _solve( $ptr->[2] ) );
        }
        die "unknown op $ptr->[0]";
    }
    die "unexpected reference type '$rt'";
}

1;
__END__

=head1 NAME

Logic::Expr - logical expression parsing and related routines

=head1 SYNOPSIS

  use Logic::Expr::Parser;

  my $expr = Logic::Expr::Parser->new->from_string('Xv~Y');

  use Data::Dumper; 
  print Dumper $expr->solutions;

  my $bools = $expr->bools;
  $bools->[1] = 0;    # fiddle with Y
  print $expr->solve;

=head1 DESCRIPTION

L<Logic::Expr::Parser> parses logic expressions of a particular syntax
and returns a C<Logic::Expr> object. This object in turn has various
methods for solving all possible solutions, etc.

=head1 METHODS

=over 4

=item B<atoms>

Hash reference. Atom name (such as C<X>) to scalar reference in
B<bools> mapping.

=item B<bools>

Array reference. A list of atoms in the logic expression. Set to true by
default. B<atoms> and B<expr> contain scalar references to these slots.
These values must be updated in-place; a new array will break the
references in the other attributes.

=item B<expr>

The parse tree, which could be a scalar reference or something more
complicated involving array references and more.

=item B<new>

Constructor. Must supply the I<atoms>, I<bools>, and I<expr> parameters,
as shown in the code for L<Logic::Expr::Parser>.

=item B<solutions>

Solves all possible B<bools> states returning an array reference of
the results.

=item B<solve>

Solves the B<expr> using the current state of B<bools>. B<bools> atoms
are all set to true by default though could be changed manually.

=back

=head1 FUNCTIONS

These are available for export, mostly to the parser module, probably
via the C<:all> tag.

=over 4

=item B<TRUE>
=item B<FALSE>
=item B<LE_NOT>
=item B<LE_AND>
=item B<LE_OR>
=item B<LE_COND>
=item B<LE_BICOND>

=back

=head1 BUGS

None known.

=head1 SEE ALSO

There are various other logic modules on CPAN. In particular
L<Math::Logic::Predicate> appears to contain section 2 material while
this module only has section 1 material from

"A Modern Formal Logic Primer". Paul Teller. 1989.

https://tellerprimer.ucdavis.edu/

=head1 COPYRIGHT AND LICENSE

Copyright 2022 Jeremy Mates

This program is distributed under the (Revised) BSD License:
L<https://opensource.org/licenses/BSD-3-Clause>

=cut
