#include <arch/neuron.h>
#include <arch/layer.h>
#include <utils/transfer.h>
#include <string.h>

layer_node::layer_node(layer_node *p, char name[20], int number) {
    last  = p;
    next  = 0;
    above = 0;
    below = 0;

    char tname[255];
    if(number == -'b') {
        sprintf(tname, "%s-bias", name);
    } else {
        sprintf(tname, "%s-%i", name, number);
    }
    my_neuron = new neuron(tname);
    my_neuron->set_is_in_layer_node(this);
}

layer::layer(char n[20], int s) {
    head = new layer_node(0, n, 1);
    layer_node *temp = head;
    for(int i=1; i<s; i++) {
        temp->next = new layer_node(temp, n, i+1);
        temp = temp->next;
    }
    size = s;
    input                      = new real[s];
    output                     = new real[s];
    delta                      = new real[s];
    weighted_delta_from_above  = new real[s];
    cols = 0;
    rows = 0;
    strcpy(name, n);
}

layer::layer(char n[20], int s, real bias_v) {
    head = new layer_node(0, n, 1);
    layer_node *temp = head;
    for(int i=1; i<s; i++) {
        temp->next = new layer_node(temp, n, i+1);
        temp = temp->next;
    }
    temp->next = new layer_node(temp, n, -'b');
    s++;
    size = s;
    input                      = new real[s];
    output                     = new real[s];
    delta                      = new real[s];
    weighted_delta_from_above  = new real[s];
    strcpy(name, n);
    temp->next->my_neuron->set_input_as_bias(bias_v);
    temp->next->my_neuron->set_transfer_function(SUM);
}

void layer::set_matrix_size(int r, int c) {
    rows = r; cols = c;

    if(cols) {
        layer_node *t1 = head;
        layer_node *t2 = head;
        for(int i=0; i<cols; i++) {
            t2 = t2->next;
        }
        // t1 is the head of the first row.
        // t2 is the head of the second row?
        for(int i=1; i<rows; i++) {   // one fewer loops than the #of rows
            for(int j=0; j<cols; j++) {
                t1->below = t2;
                t2->above = t1;
#if DEBUG || ORIENTATION_DEBUG
                fprintf(stderr, "set %s->below = %s and %s->above = %s.\n",
                    t1->my_neuron->query_name(),
                    t2->my_neuron->query_name(),
                    t2->my_neuron->query_name(),
                    t1->my_neuron->query_name()
                );
#endif
                t1 = t1->next;
                t2 = t2->next;
            }
        }
    }
}

void layer::set_transfer_function(int type) {
    layer_node *temp = head;
    while(temp) {
        temp->my_neuron->set_transfer_function(type);
        temp = temp->next;
    }
}

char *layer::query_name() {
    return name;
}

void layer::dendrites_touch(layer *l) {
    layer_node *temp = head;
#if DEBUG || CONNECTION_DEBUG
    fprintf(stderr, "Connecting the dendrites in the %s layer to the %s layer.\n",
        name,
        l->query_name()
    );
#endif
    while(temp) {
        temp->my_neuron->dendrites_touch(l);
        temp = temp->next;
    }
}

void layer::start_foreach() {
    current_node = head;
}

int  layer::has_a_neuron() {
    return (current_node) ? 1:0;
}

void layer::next_neuron() {
    current_node = current_node->next;
}

neuron *layer::query_current_neuron() {
    return current_node->my_neuron;
}

void layer::set_input(real *inputs) {
    layer_node *temp = head;
    int counter = 0;
    while(temp) {
        if(!temp->my_neuron->query_biased())
            temp->my_neuron->set_input(inputs[counter++]);
        temp = temp->next;
    }
}

real *layer::query_output() {
    layer_node *temp = head;
    int counter = 0;
    while(temp) {
        output[counter++] = temp->my_neuron->query_output();
        temp = temp->next;
    }
    return output;
}

real *layer::query_input() {
    layer_node *temp = head;
    int counter = 0;
    while(temp) {
        input[counter++] = temp->my_neuron->query_input();
        temp = temp->next;
    }
    return input;
}

real *layer::query_delta() {
    layer_node *temp = head;
    int counter = 0;
    while(temp) {
        delta[counter++] = temp->my_neuron->query_delta();
        temp = temp->next;
    }
    return delta;
}

void layer::set_delta(real *d) {
    layer_node *temp = head;
    int counter = 0;
    while(temp) {
        temp->my_neuron->set_delta(d[counter++]);
        temp = temp->next;
    }
}

real *layer::weighted_sums_of_deltas_from_above() {
    layer_node *temp = head;
    int counter = 0;
    while(temp) {
#if DEBUG || ERROR_DEBUG
        fprintf(stderr, "Getting weighted sum from above layer for [%s].\n",
            temp->my_neuron->query_name()
        );
#endif
        weighted_delta_from_above[counter++] = 
            temp->my_neuron->axon_fires_at->weighted_sum_of_delta_from_above();
        temp = temp->next;
    }
    return weighted_delta_from_above;
}

void layer::reinitialize_weights_with(real max, real min) {
    layer_node *temp = head;
    while(temp) {
        temp->my_neuron->dendrites->reinitialize_weights_with(max, min);
        temp = temp->next;
    }
}

void layer::save_weights(FILE *F) {
    layer_node *temp = head;
    while(temp) {
        temp->my_neuron->save_weights(F);
        temp = temp->next;
    }
}

void layer::restore_weights(FILE *F) {
    layer_node *temp = head;
    while(temp) {
        temp->my_neuron->restore_weights(F);
        temp = temp->next;
    }
}
