#!/usr/bin/perl

use Test::More tests =>9991;
BEGIN { use_ok ('Geo::Coordinates::UTM'); }

use constant maxerror => 1e-2;

use warnings;
use strict;

sub fleq ($$;$) {
    if (abs($_[0] - $_[1]) < maxerror) {
        pass($_[2]);
    }
    else {
        fail($_[2]);
        diag("floating point value $_[0] too different to reference $_[1]");
    }
}

my $latlon = "CCDEFGHJKLMNPQRSTUVWXX";

while(<DATA>) {
    chomp;
    next if /^\s*(?:#.*)?$/;

    my ($ellipsoid, $latitude, $longitude, $zone, $easting, $northing) = split /\|/;
    my ($z, $e, $n) = latlon_to_utm($ellipsoid, $latitude, $longitude);
    is($z, $zone, "zone $.");
    fleq($e, $easting, "easting $.");
    fleq($n, $northing, "northing $.");

    my ($lat, $lon) = utm_to_latlon($ellipsoid, $z, $easting, $northing);
    fleq($lon, $longitude, "longitude $.");
    fleq($lat, $latitude, "latitude $.");

    my ($zone_number, $zone_letter) = $zone =~ /^(\d+)(\w)/;
    ($z, $e, $n) = latlon_to_utm_force_zone($ellipsoid, $zone_number, $latitude, $longitude);
    is($z, $zone, "fz zone $.");
    fleq($e, $easting, "fz easting $.");
    fleq($n, $northing, "fz northing $.");



    my $z1 = $zone_number + int(-2 + rand 5);
    $z1 -= 60 if $z1 > 60;
    $z1 += 60 if $z1 < 1;

    my $l1 = ($latlon =~ /(.)($zone_letter)(.)/, '')[rand(4)];
    ($z, $e, $n) = latlon_to_utm_force_zone($ellipsoid, "$z1$l1", $latitude, $longitude);
    ($lat, $lon) = utm_to_latlon($ellipsoid, $z, $e, $n);
    fleq($lon, $longitude, "fz longitude (zone $zone) $.");
    fleq($lat, $latitude, "fz latitude (zone $zone) $.");
}


__DATA__
# ellipsoid|latitude|longitude|zone|easting|northing
Bessel 1841 Nambia |-25.9668774017417|176.847283481794|60J|484713.786930711|7128217.21692427
Airy|-62.6643472980663|-18.1318011641218|27E|646897.012158895|3049077.01849759
Fischer 1968|35.6355896701843|-128.779407270838|9S|519972.615374218|3943564.81408124
Clarke 1880|-23.7249091815666|-58.725260492694|21K|324128.165780259|7375382.16974803
Clarke 1880|32.1214082107167|39.4139996279968|37S|539054.300206758|3553697.31179058
Everest 1830 India|-77.8630147119741|-167.673126297052|3C|437290.67927147|1356475.5866066
Airy|18.3773457007429|166.226582243424|58Q|629566.39702383|2032235.17088274
Australian National|56.0384628238617|-162.332085699794|3V|666204.74513802|6213592.60573265
GRS 1967|73.8433062387819|84.4955735592175|45X|422233.129750653|8196218.48575924
Bessel 1841|45.3359588450991|40.0400031298722|37T|581477.812337138|5020289.06897684
Helmert 1906|59.9201815839611|-99.2827685844408|14V|484189.728077767|6642625.01984108
Bessel 1841 Nambia |-14.7469801908782|85.8375051526079|45L|374882.74629932|8369470.82995574
WGS 66|76.7872465101191|65.0798587586657|41X|553057.061015891|8523979.51532624
Clarke 1880|-77.955182771371|144.231341269374|55C|435521.950639508|1345300.73476931
Fischer 1968|-46.0648299617229|79.4920266271394|44G|383370.850926947|4897630.12315441
Airy|-29.748187686171|-135.405807319046|8J|460765.866906721|6709275.23812834
Airy|7.02664660839798|25.1759536336006|35N|298528.58347161|777035.572913441
GRS 1980|-23.328081037079|-157.911520393938|4K|611282.26616725|7419742.38608919
Krassovsky|11.2850029025532|160.123416303008|57P|622616.284394897|1247750.19696123
Modified Fischer 1960|63.3429435572545|-65.5753780930164|20V|371108.78887873|7026412.7414732
Hough|18.8829467124758|-72.6426446845981|18Q|748345.681250186|2089517.14788254
Bessel 1841 Nambia |-33.5599610482547|149.259628455931|55H|709740.807412104|6284663.16960231
Fischer 1968|-42.9864757600699|73.2901330922691|43G|360599.261179762|5239255.45090894
GRS 1967|19.5586150524789|78.6857081619374|44Q|257192.91521481|2164288.44689143
WGS-84|-15.038207087342|-169.633239244409|2L|646923.005840822|8336993.38390601
International|52.870630356142|-150.706593044311|5U|654367.360576001|5860466.52005626
South American 1969|43.8281075242737|-20.83767446837|27T|513051.867864041|4852811.12725856
Clarke 1866|-28.5721039983461|-146.696242503652|6J|529707.632536775|6839556.76681453
Fischer 1968|-66.5351466743693|-131.100699832755|9D|406672.67060956|2618850.77469032
Hough|-10.2204937285656|-44.0815836680508|23L|600591.91301086|8870073.83572614
Modified Airy|0.898212672075346|-58.3646301216307|21N|348173.89644761|99297.8933194118
Clarke 1880|12.8121044499734|-117.115209168146|11P|487496.968220537|1416233.08300573
WGS-72|-28.3158449838178|-21.1430172657167|27J|485979.391986859|6867801.56555617
Bessel 1841 Nambia |-79.6912736020321|112.830861585599|49C|536566.508624811|1153141.93576399
WGS 60|-78.8665004247374|39.967100365045|37C|520846.260662007|1244711.5272432
Clarke 1866|74.9153423507582|135.063206941098|53X|501836.223148577|8314007.12896517
WGS-84|41.5891103734802|-81.6341167095867|17T|447147.34666104|4604351.4001664
Modified Fischer 1960|25.4822842637392|-22.2489427610583|27R|374459.310920946|2818951.29091641
WGS 66|-66.5839792744398|-5.53083678652641|30D|387791.471253292|2612712.21918927
WGS-72|-44.8707187405112|-41.0925340273647|24G|334705.685616504|5029282.28503198
WGS 60|-24.7353470675559|-52.8387111449346|22J|314039.036011629|7263093.13430589
Bessel 1841|-26.8339414009232|-141.993673622831|7J|401276.788144163|7031861.41595582
GRS 1980|17.9170061216219|-46.1051315533236|23Q|382947.346574745|1981350.77504379
Australian National|-69.910030919324|80.6622601498745|44D|487052.492719956|2244096.69079924
Airy|64.2107184266502|-150.308156994563|5W|630628.557760497|7122705.87567602
Bessel 1841 Nambia |81.0005265211464|156.109807830971|57X|449553.59199732|8993659.42672691
Everest 1830 India|-44.0402069211306|-164.867320105091|3G|510628.725995342|5124106.23691839
WGS 66|-54.5230601118632|27.5600989664397|35F|536252.302400058|3958128.61163491
Bessel 1841|-24.4602191417681|91.2138615618262|46J|318983.753466054|7293914.86908363
Fischer 1960 Mercury |33.0259050093003|-9.93203850642828|29S|412956.344341581|3654564.255788
Australian National|-5.70111443119308|25.5610353055634|35M|340648.901716452|9369633.73439646
Bessel 1841 Nambia |-3.50452292344862|167.529810174952|58M|781044.108772687|9612292.48902955
Modified Everest|-57.9061206776167|47.6086324677171|38E|654550.967210381|3579384.87397302
Fischer 1968|48.5641594911566|46.9250439431983|38U|642023.973557101|5380810.85322907
GRS 1980|11.8857960495429|-69.6237873633386|19P|432065.387622261|1314001.40617086
Modified Airy|32.0382237301017|-17.8614173779133|28S|229821.22416974|3547882.97401606
Helmert 1906|45.4416429793514|-59.3491564433593|21T|316277.376858519|5034750.87757134
WGS-72|27.7536629637847|119.926453251009|50R|788459.379020248|3073346.45219109
GRS 1967|-55.787428837664|87.425142939811|45F|526660.597095824|3817475.35726579
WGS-72|60.1716718625519|-102.07370998013|13V|662339.584484367|6674125.86173384
Bessel 1841 Nambia |55.417832675197|-71.3975083853801|19U|348265.063742001|6143350.46036408
WGS 60|15.3056746944921|-51.6344494052987|22P|431888.308485876|1692243.97784434
Bessel 1841 Nambia |74.5035749101927|-132.760897779056|8X|566754.922433039|8268697.04301107
International|8.34811284230717|116.449373561545|50P|439370.695202269|922835.199162216
WGS-84|-71.0460379758696|-59.4540977299395|21D|411056.160432079|2115666.73559241
Hough|-30.0147970685507|101.506294569531|47J|741735.132957834|6676942.3119651
GRS 1980|3.01754275388058|93.3767818058026|46N|541868.939529967|333539.481681074
WGS 60|32.6786136157925|-112.005036610064|12S|405772.331172202|3616124.00750729
WGS 66|-52.1477679701243|90.6690202722451|46F|340515.02674961|4219957.55407511
WGS 66|-8.50731004424314|-131.092290608796|9L|269675.711564331|9058994.17064866
Modified Everest|-52.5100287918179|55.776026532083|40F|416945.796319106|4182074.45877259
Krassovsky|2.35421603292527|-166.672378192892|3N|314032.55165339|260329.243641693
Airy|6.12240529128917|120.268101155136|51N|197644.936494717|677460.011121505
WGS-84|-36.6197210264493|142.101788019074|54H|598520.568544018|5946746.77361204
WGS 66|-0.432086253200822|166.137230041298|58M|626550.17977436|9952231.95321937
International|48.0700584095067|110.389532276226|49U|454521.485817479|5324371.30721701
WGS-84|-33.4090477416402|-39.3454643750168|24H|467877.817501539|6303311.17152226
WGS 66|70.9826595293591|-131.895156906671|9W|394742.944428888|7877986.93165589
Australian National|15.5550588780301|-11.4023259491394|29P|242344.279022172|1721174.52964556
International|58.8522081830389|10.1135752098733|32V|564252.730916659|6524279.99360641
Bessel 1841|68.5749166287382|105.185954758293|48W|507579.568049395|7606164.50430552
Bessel 1841 Nambia |10.7456100701673|173.009499976332|59P|719731.21301151|1188472.20156892
Modified Fischer 1960|-73.2030861148231|80.9096812505457|44C|497086.729780324|1876836.77661911
South American 1969|14.3338908807793|-57.924038236296|21P|400353.97399461|1584857.89768469
WGS 60|0.173063552093197|174.654184750992|60N|238896.084654498|19144.9772340554
WGS-72|61.5258317535112|-148.628585384002|6V|413382.786255001|6822439.4668495
Hough|58.8088516856014|-18.5639931125748|27V|640713.868051983|6521356.64366606
GRS 1967|-72.8367776282282|154.38437786016|56C|545594.402894699|1917173.2722071
Krassovsky|-32.3399134101337|-5.75376797980923|30H|240808.716533218|6418489.38898975
Modified Fischer 1960|-33.7935346775831|-90.1426680918991|15H|764547.93250672|6257051.72192377
Krassovsky|30.3110423004335|16.5781096795273|33R|651740.167029391|3354366.59423781
Airy|47.2977172681515|113.86735995409|49T|716750.76632314|5241853.62752025
Everest 1830 India|37.8845511113926|-135.133529573281|8S|488259.813017341|4192639.24977099
International|38.1792716151171|145.370215224618|55S|357247.309392793|4227031.33905273
Fischer 1968|-25.4762717068519|-126.563494069095|9J|744958.173012669|7180064.22592105
South American 1969|-79.5365248748027|166.324319120481|58C|526847.415913854|1169807.88427186
International|-78.4110632561182|-14.989240240315|28C|500241.311608922|1295500.987858
GRS 1967|-68.1951952117815|-149.551187703291|6D|394272.096152166|2433160.52905247
GRS 1967|-51.4537410204381|55.8720091094686|40F|421624.660166934|4299092.29734687
Hough|63.6032565144119|169.461688394098|59V|423695.90923465|7053765.46831524
Australian National|48.1242033834005|115.483535903583|50U|387151.395738734|5331235.78001154
Helmert 1906|76.8564843583623|-77.8858858063325|18X|426772.850216472|8532641.63221763
Modified Airy|-73.1413850157385|21.6219781938462|34C|520130.588120968|1884577.3296713
Airy|10.8443718593394|-106.547472578294|13P|330856.145329966|1199121.84577201
Modified Fischer 1960|53.8678378667685|37.5391657940209|37U|403942.856525831|5969826.99179774
Bessel 1841|-72.7737535405884|-7.41034977684023|29C|552533.249632339|1924931.2973827
Krassovsky|72.7221765690611|127.281429296735|52X|443034.475691343|8070443.9635078
WGS 66|59.0836358798848|-60.7869841003324|20V|626819.698008205|6551474.17215566
Krassovsky|50.3319354501851|50.6262418599614|39U|473398.31678153|5575702.66835709
Hough|-67.0761049801843|17.6673796933028|33D|615915.210570116|2557596.82151162
Hough|44.6070598377999|-79.0353060704053|17T|655907.465248995|4941178.68965883
WGS 66|56.9743083375776|-0.986959536978418|30V|622362.141454348|6316335.89134893
WGS-84|54.8376254894191|-30.4398113516179|25U|664413.814250686|6079726.64178547
Everest 1830 India|7.80734863804564|-94.7379708763827|15N|308385.662043342|863327.07384326
WGS-84|42.1388081786232|94.1239212872368|46T|592878.528530307|4665799.35497352
Modified Fischer 1960|-50.2668337384696|137.577966166824|53F|683719.888476084|4428502.00655906
Modified Fischer 1960|-16.0823420814511|-30.9872437684145|25K|715297.878249588|8220901.80950808
Airy|-66.2328424771405|-54.6449818303734|21D|605880.044371087|2652719.21929313
Everest 1830 India|-6.84757411014701|39.7458086007605|37M|582392.885672852|9243096.70318807
Bessel 1841 Nambia |41.1513527127688|-32.831879043666|25T|514105.333591258|4555176.5780732
Modified Fischer 1960|55.2863200295101|-152.321131692079|5U|543116.177973657|6126884.4743664
Modified Airy|-29.451084280348|73.6605567348573|43J|370119.219850812|6741628.56584502
WGS-84|46.2940205986409|150.032190433477|56T|271420.393487589|5130997.45336144
Australian National|-35.5603588368332|143.139947059774|54H|693948.558643153|6062691.79771528
Helmert 1906|-53.9063545868914|82.9790651234818|44F|630011.144548621|4025019.9690289
Modified Airy|66.3869857142563|13.6620169842286|33W|440206.910329835|7362851.54781893
Clarke 1880|3.55952457280554|139.373212276875|54N|319298.674952978|393562.559924183
Everest 1830 India|15.5789614743989|56.2216245557286|40P|416557.455087749|1722377.53283664
Fischer 1960 Mercury |-76.1197975163221|22.9216199653918|34C|551450.063399398|1550567.96761959
Bessel 1841 Nambia |62.5602541413105|-117.997775255694|11V|448708.571885748|6936356.0146201
Bessel 1841|25.8584360955937|-104.751556557646|13R|524890.477666797|2859750.40000131
South American 1969|-60.2453279878719|-123.675313033119|10E|462612.136941668|3321051.91420429
Everest 1830 India|-13.3967974757903|-126.350499493418|9L|786904.874593123|8517563.4085654
International|-78.6540825243318|-54.4721874213712|21C|555502.455472315|1267176.87975221
WGS-84|33.1031044705639|-31.4288634606255|25S|646605.556167168|3663815.17274866
GRS 1980|39.864851371075|125.435164112611|51S|708285.405378073|4415595.279023
Modified Fischer 1960|24.5603404065612|13.7258802175184|33R|370967.839649069|2716872.74675363
WGS-84|31.1835240023235|-162.328830839704|3R|754559.405865657|3453015.12274526
GRS 1980|-34.2718803006205|164.478603281853|58H|452003.843228432|6207574.82668561
Helmert 1906|-42.7919587323294|92.3815411151324|46G|449420.132525342|5262051.72898115
WGS-72|-4.94673118776331|-80.5149320978738|17M|553776.827294459|9453204.45974023
WGS-84|-73.2258582070606|138.51474392925|54C|419963.573951727|1872662.51467536
WGS-84|41.5297041790066|63.8470364720335|41T|570664.013037394|4597908.28877691
WGS 60|77.0361181286736|-98.4142481028422|14X|514668.694868441|8550917.34758758
WGS 66|20.0790026532708|-14.464297336787|28Q|556009.987965339|2220316.12071077
Krassovsky|-6.00344938468989|36.1574599181569|37M|185287.196520777|9335585.25880476
Modified Everest|-68.8160173533496|14.676753274757|33D|486966.088944928|2366915.10268845
Australian National|3.02752351643922|-124.084031766269|10N|379534.010557107|334696.786137724
GRS 1980|31.8984482198827|52.117044756982|39R|605629.951837584|3529723.67951414
WGS 66|74.6542374159547|-135.647311441461|8X|480878.565589977|8285143.40165614
South American 1969|83.9704718107223|-108.616547185029|12X|527943.727104414|9325408.64703107
GRS 1980|79.7816615344539|-61.7504610732985|20X|524744.601763056|8857481.62384563
Helmert 1906|-12.4972548881506|173.469367356779|59L|768388.781170199|8617192.85884249
Modified Everest|-0.0333210044673962|-61.6024142622206|20M|655511.489525822|9996316.19606776
International|8.62042639421796|139.997359207479|54P|389673.488184949|953043.367171278
International|-61.3939064996927|138.219515457428|54E|351505.090233679|3190009.82327212
GRS 1967|65.0887919939408|41.303007713009|37W|608223.128297521|7220348.55919927
International|46.2413904918871|121.635515867872|51T|394800.729968883|5121870.24260744
GRS 1980|-47.3313525669031|-110.005253139642|12G|575157.082866381|4757533.06930344
GRS 1967|30.1759452074515|53.8822080416733|39R|777549.951224282|3341804.03364465
International|72.7862934022847|-36.9202439307848|24X|568688.332726323|8078053.33070449
Clarke 1880|72.2274730462678|-159.477262251739|4X|483739.21203988|8014138.17609897
Airy|3.36434536658854|-78.7299942178312|17N|752205.649781205|372134.292407954
Fischer 1968|-72.3360912280983|39.8545868602318|37C|528941.948106945|1973353.39675481
International|72.9835079681788|-49.9778655264105|22X|533387.693130936|8099147.85615482
GRS 1980|12.2373773046462|-73.0315699421465|18P|714130.783723963|1353582.46324605
International|17.5564984788136|0.530788167946383|31Q|237879.643619451|1942847.50130151
Clarke 1880|-29.1939118745363|-25.8974851302705|26J|607190.607908043|6770281.83031495
Hough|18.4236922686371|44.0329328107573|38Q|397864.828502237|2037322.04830449
Helmert 1906|-55.7257200519642|-39.3305671047284|24F|479237.323070903|3824332.21868817
Clarke 1880|62.7892003516927|138.718571240205|54V|383618.125864685|6963881.60847544
South American 1969|0.78292342312632|128.412208510108|52N|434598.198776632|86541.4981972444
WGS-84|-27.652387191139|-48.8436865295465|22J|712719.422786103|6939445.2050421
WGS-72|-3.77668299665972|-122.361287598351|10M|570920.827908218|9582531.05510427
Helmert 1906|12.2125919178269|34.1018626785483|36P|619860.907566081|1350320.2490256
Modified Fischer 1960|-64.9827462710613|49.1714083186869|39D|413723.304575586|2792196.73372064
Clarke 1880|-53.2894648591547|162.585056567767|58F|339023.328177969|4093117.5127795
WGS 66|-75.176100990501|8.40545564675551|32C|483020.611439362|1656649.69283395
Everest 1830 India|-62.76253179113|-120.736048514809|10E|615569.748606089|3039567.73729965
Bessel 1841|77.8474580819954|75.0719404353057|43X|501690.413180512|8640407.64058068
Airy|60.0883875744786|78.8832820339202|44V|382274.569898478|6662630.04796129
GRS 1980|6.86791795256509|82.9058101369685|44N|710595.301322104|759566.64047955
Helmert 1906|-25.1486192701264|-60.2587436964082|20J|776356.231647104|7215754.8911117
Fischer 1960 Mercury |-31.0907288279179|-130.504900261109|9J|356460.906976224|6559350.88909112
Hough|83.3034310927728|41.0424407071055|37X|526585.922826274|9250914.07811744
GRS 1967|36.157812987843|-80.5274659873982|17S|542503.656142608|4001569.55286877
Bessel 1841 Nambia |-23.8016326968279|113.616632533595|49K|766582.05136411|7365498.27580319
Modified Fischer 1960|-14.0954798769091|-66.4459766070721|19L|775778.537895163|8440210.54429321
Bessel 1841|58.8533406912518|-179.162298730316|1V|375272.233064267|6525054.78126247
Helmert 1906|45.5646600873306|41.2801567423253|37T|677938.186800496|5048262.27080925
International|-4.25682403341571|-158.385843431127|4M|568156.966540089|9529452.09253501
Australian National|-10.2396079585079|80.0663873956458|44L|397751.464505241|8867943.42015668
WGS 60|-6.45978982694385|165.944261001114|58M|604415.027458815|9285868.20291786
Everest 1830 India|2.46765863645264|131.945603393902|52N|827570.523962609|273093.585400534
Fischer 1968|26.7463518300473|-116.517402378847|11R|547988.406857761|2958441.42149618
Krassovsky|20.0631367869546|111.26478583037|49Q|527687.465143011|2218529.5750089
WGS 66|73.7012261959657|-38.7242366547331|24X|508638.445492653|8178735.94805407
South American 1969|61.4130744054392|137.6835458632|53V|643225.377741332|6811768.81428633
Airy|66.8336694422557|-173.376297609368|2W|395711.297936072|7414245.67215566
WGS 66|12.7220546126373|-20.9055196076801|27P|510256.881255311|1406402.51722625
Modified Everest|-2.61328816468108|171.512101726323|59M|556918.556365826|9711160.83417942
Clarke 1866|-57.4541680123041|-130.443050722577|9E|413409.006486901|3631349.13135726
Fischer 1960 Mercury |51.4294619444677|-134.426757206853|8U|539851.769289503|5697769.8334818
Modified Everest|2.28243265372846|14.8972739302596|33N|488579.654468245|252260.530583355
Modified Airy|-44.2274911522972|-99.0913856107339|14G|492702.247291288|5103381.03462047
Modified Fischer 1960|34.845999480128|-172.842681346269|2S|331527.238996913|3857527.40427864
WGS 66|-27.6717341576511|35.2571995620831|36J|722635.72880466|6937119.75584766
Clarke 1880|19.9214964644881|-103.340496315229|13Q|673701.583465184|2203462.29147583
Everest 1830 India|35.9033449443135|60.965216789012|41S|316400.725293952|3974789.25293768
Fischer 1968|-70.8866573713732|143.831957290921|54D|603460.988321752|2132808.35578507
Everest 1830 India|-22.3947896414373|-53.2728376932893|22K|266039.36945633|7521909.49338389
Clarke 1880|77.253651666356|-108.539994870488|12X|560576.698466607|8576149.92634876
Clarke 1866|-14.7744885399655|-114.250436293483|11L|796009.593505733|8364904.53788469
Modified Fischer 1960|-47.2835863742135|-59.191938200446|21G|334243.165697886|4760972.76377597
Bessel 1841|0.627576788773666|121.75770702851|51N|361777.26926594|69375.8831925983
Helmert 1906|35.5270250226151|-113.936196672304|12S|233755.711730637|3935498.85758821
WGS 60|-33.223048415447|115.605070330761|50H|370014.623536236|6323099.25976853
Bessel 1841|15.1512499774216|28.8316366349176|35P|696781.690904831|1675715.56353851
WGS-72|-26.3447201781794|-6.68114732673885|29J|731422.371811788|7084061.48345215
Fischer 1960 Mercury |-33.938207351344|-29.3323169968633|26H|284436.199696227|6242225.01690586
GRS 1980|9.66932355523414|135.2497375123|53P|527397.411103909|1068862.12638087
Modified Airy|-32.7058399004368|174.953950987426|60H|308242.440173207|6379850.95900793
Bessel 1841 Nambia |66.2465231510228|84.198386529071|45W|374120.870039801|7349526.4870879
Modified Airy|50.8737243510672|83.6381085935457|44U|685579.204020539|5638483.70887218
Hough|56.7408506479957|0.127408767098842|31V|324308.934224721|6292247.29781798
Bessel 1841|71.8012333183597|40.6955204065713|37W|559088.516001731|7966736.92275303
WGS 66|-52.4311003520785|-171.753428330806|2F|448775.367361333|4190738.18382811
WGS-84|4.91799601358203|-174.161351096588|1N|814843.179721062|544268.756726857
Airy|-14.0856058079262|49.1765228887165|39L|303142.760003801|8442147.39744786
International|49.3846014716743|-14.0416879127554|28U|569554.902599737|5470762.28973179
Airy|69.220585642219|-59.2208043165802|21W|412096.661436904|7679944.77773849
Fischer 1960 Mercury |-76.5846547366596|-77.9034548953845|18C|424832.175027394|1497680.61326941
Clarke 1880|10.608280927345|82.4248037123807|44P|655871.033867479|1172918.19418541
Modified Fischer 1960|16.7250661031614|128.826983831256|52Q|481556.883101656|1849155.82565983
WGS 66|60.0683632022629|-99.8412238570303|14V|453176.10555296|6659330.32703425
Bessel 1841 Nambia |-32.0157118930196|107.904783245058|48H|774348.490806195|6454435.61418285
Clarke 1866|-31.9234244312324|173.118835166358|59J|700325.993533926|6466280.42986774
WGS-84|51.6679580472244|-108.989590187903|12U|639027.955062374|5726022.28078556
Hough|-51.0011625108647|-54.3423078987196|21F|686476.157269598|4346672.48753887
GRS 1967|-59.9389054272473|-61.8795693810463|20E|562608.458178879|3354839.77949208
Australian National|50.1718071915274|67.1972927440295|42U|371269.55871383|5559308.20281266
WGS-84|-3.73801622350375|136.461458917486|53M|662297.69471267|9586696.04586191
Fischer 1960 Mercury |-26.2412579857586|162.195749946704|58J|219863.081341649|7094549.46466467
International|-69.8322416178419|-89.1494931729758|16D|417302.461573099|2251172.62552692
Australian National|3.91022775262462|-65.8170705656094|20N|187128.549038698|432730.70440583
Krassovsky|44.3698039723548|169.136911121965|59T|351556.307977092|4914722.17444927
WGS 60|-18.0555132847462|155.867903593094|56K|803613.095125796|8001305.57822568
Modified Everest|-54.6259919474662|-123.313277389303|10F|479777.186704073|3947357.70675638
Airy|-36.6264035300906|134.99343245925|53H|499412.854917175|5946856.3211378
Fischer 1968|-64.7028468665263|-144.493538449197|6D|619483.744676044|2822280.38211136
Bessel 1841|-14.2825540716464|-8.60977917959545|29L|542083.632113683|8421141.25811252
WGS-72|-56.258286649426|-163.141529918464|3E|615126.742636836|3763622.07333328
Helmert 1906|-24.2891613620279|-118.627835617817|11J|334786.717298404|7312764.19690656
South American 1969|23.5099134075299|127.000908136699|52Q|295878.466143084|2601398.73519904
Krassovsky|-69.690250674077|60.3734070598664|41D|398278.469251553|2266347.30333324
Clarke 1866|-59.8494395740546|122.742924285467|51E|485595.275245875|3365535.55903271
Modified Fischer 1960|56.9288155218193|81.8587657773521|44V|552267.101541496|6309811.72492785
Australian National|61.4536699669284|63.9407845422517|41V|550154.777488763|6813706.56138963
South American 1969|-26.502127541824|14.9879525569598|33J|498799.487439884|7068696.57391453
Modified Everest|-38.2431799960341|-14.3495443590032|28H|556911.406675945|5767363.93694015
GRS 1980|-75.7825430160861|-40.6761858243941|24C|454051.783478387|1588427.4011035
WGS-72|82.2383679007745|120.568822823906|51X|463354.426972196|9132205.31349973
Fischer 1968|7.92296843467804|-96.9570538596618|14N|725219.741514597|876338.172525959
Fischer 1968|-19.9308655432569|145.16619317145|55K|308064.905482826|7795115.29552426
International|75.9867442179318|-90.4266070591966|15X|569542.17905459|8435461.6998636
Fischer 1960 Mercury |17.0447305123131|-6.29135235292608|29Q|788339.595659693|1886512.51086928
Modified Fischer 1960|-60.2950177427096|-13.6835137239781|28E|572772.710825702|3314983.92842035
Helmert 1906|37.4357201532249|-85.0270428866|16S|674549.989625671|4145081.37924612
Hough|35.6600801334844|9.50571157112122|32S|545774.74132739|3946355.15904645
Clarke 1880|-9.84544515632457|46.4989532273512|38L|664376.501285566|8911406.55149287
Krassovsky|63.821988119785|-126.670894806394|9V|614628.393342825|7079392.69750763
South American 1969|-76.6120600397079|178.530333926786|60C|539550.428422506|1495973.25229497
WGS-84|54.4415433100825|-37.4849768318624|24U|598251.491171143|6033706.65760297
Bessel 1841|65.1718409480116|52.9430330692395|39W|591015.307945058|7228241.42821334
Everest 1830 India|4.1092313712135|-149.552607865103|6N|216628.857513332|454618.36664913
Hough|-77.8999341206689|27.887834456852|35C|520775.6631623|1352549.45938882
International|25.4369817657623|149.655192970789|55R|767051.285817717|2816032.7288958
Australian National|28.0164056429588|116.128858501526|50R|414357.978012302|3099336.27248927
GRS 1967|7.68888455007252|-138.327255304684|7N|794858.155765854|850827.329682249
Fischer 1960 Mercury |-76.9090576943497|35.0402945038869|36C|551577.026470986|1462438.80204871
Bessel 1841|61.156958816071|-155.511428588157|5V|364877.891602252|6782153.07329031
WGS-84|59.4291479588837|-161.499127532557|4V|358231.72009861|6590502.18341198
GRS 1967|-59.3011153156042|-61.2354196954256|20E|600485.434981191|3425064.30035265
GRS 1980|54.6593950416621|-2.51061615615328|30U|531569.667367315|6057000.29220798
Fischer 1968|-47.1281912174952|50.1725036213203|39G|437239.78569376|4780243.9569017
WGS-72|83.8839409904563|-79.7877824820193|17X|514418.23487005|9315286.3006078
Airy|38.187250669113|101.148090625558|47S|688108.758947143|4228472.33846966
Australian National|40.4233287289302|-27.9202776886765|26T|421931.790966255|4475166.41927601
Modified Fischer 1960|-13.3534110039822|97.4180521743581|47L|328677.349843285|8523229.45486707
GRS 1967|-36.1268078791445|3.1911889781901|31H|517203.851940813|6001955.95026858
Hough|-65.870671273649|71.6684075390161|42D|621697.545785028|2691868.70068821
Clarke 1880|-44.7032654610135|-179.11990259409|1G|332051.149235128|5048137.03030904
Modified Fischer 1960|22.0901201808674|-154.639491339707|5Q|330860.319022677|2443721.35295042
International|-36.4970906357279|44.0792333090701|38H|417532.704160334|5960455.15419492
Modified Fischer 1960|-76.8256755231839|22.6055579265839|34C|540843.989525844|1472095.56654229
Modified Fischer 1960|9.42600952825298|100.153074513884|47P|626595.757046135|1042163.73404676
Fischer 1960 Mercury |-36.346316353164|-96.398790164706|14H|733436.635836264|5974475.67284948
Modified Everest|-66.7583953856327|26.7257352715995|35D|487924.72166237|2596289.51724235
Clarke 1866|51.7115957509787|30.0125031658341|36U|293606.036207907|5732968.3699442
Clarke 1866|-31.2837122315436|141.698325749239|54J|566470.367076028|6538929.79048126
Airy|-67.730115486232|-32.1677567873324|25D|535192.196539537|2487579.24867699
Fischer 1960 Mercury |-61.3789763560355|80.6680931147184|44E|482262.78505763|3194919.7360347
Airy|-40.7436553025038|-162.438424391526|3G|716255.16608294|5486865.01835559
Clarke 1880|-30.1283800393934|21.589714111449|34J|556805.491811363|6667102.23169263
Helmert 1906|13.6017891913806|4.28201244301661|31P|638692.314932118|1504068.09475283
Bessel 1841 Nambia |-59.1465134755096|60.3201061555438|41E|346731.813282999|3441151.33899344
Bessel 1841|-66.609095246857|-21.5125741735211|27D|477295.231767644|2612887.47073067
Clarke 1880|-63.8259685220904|114.201993631965|50E|362317.8229666|2919638.28366636
Australian National|-19.6397037105378|-143.912477759287|7K|194548.806212761|7825769.88730422
Australian National|35.9410194725523|-67.1015687021588|19S|671238.866603644|3979085.97648599
Bessel 1841|7.27661137347829|91.3268685808775|46N|315307.755939251|804591.857856239
Hough|72.922457410921|162.213360937298|58X|408686.132152079|8094025.85023433
Bessel 1841 Nambia |-36.9601032111618|-30.3991765762935|25H|731529.568159891|5906744.79429311
Modified Fischer 1960|-18.8015712644725|52.0217417437556|39K|607669.886101478|7920810.81720111
Fischer 1960 Mercury |-70.90038116343|148.032449794065|55D|537703.385474828|2133353.54080759
Krassovsky|-41.5074840432045|61.8734566936584|41G|405984.079354732|5404211.13854388
Bessel 1841|-5.1718845881116|138.189518350482|54M|188440.915420551|9427701.41297688
Modified Fischer 1960|-51.4485222518388|-82.8688201059572|17F|370138.780906146|4298619.59692328
WGS-84|-43.9281773266984|18.3568387889561|34G|287828.504571372|5132707.86372208
Fischer 1968|-58.126242846328|-123.733795783528|10E|456779.119764497|3556982.90295966
WGS-84|35.9830944429197|-51.4632468389748|22S|458239.612495156|3982172.57360767
Modified Fischer 1960|7.92034917790765|-166.760325296477|3N|305946.002271006|875906.522534943
Everest 1830 India|32.4276120971253|89.4739671969567|45S|732586.053085813|3590217.85740724
Bessel 1841|-7.63336873896428|109.72152462771|49M|358995.071764828|9156105.50349597
Modified Everest|-61.919422486652|98.6519032579521|47E|481722.357968947|3135433.33817663
GRS 1967|38.9380723781206|55.982629061217|40S|411826.018814325|4310411.23681255
Everest 1830 India|17.2033529535872|142.549875731645|54Q|664788.878725138|1902557.20798029
Everest 1830 India|3.51367877732811|69.2129208248267|42N|523645.375960571|388344.287441782
South American 1969|-52.5479321655823|-127.738620100488|9F|585531.707088946|4177248.50957216
WGS-72|63.726542935188|156.599312598602|57V|381452.108915804|7068768.13511563
Bessel 1841 Nambia |35.736632935535|176.956179976245|60S|496037.934794331|3954399.21594676
Bessel 1841|55.4975709171341|-78.0823863127947|17U|684267.613962615|6153392.93846948
Clarke 1866|-58.0712332189186|-128.923163882043|9E|504532.86963763|3563567.52626988
Helmert 1906|-5.93537682167586|-78.4660428053257|17M|780557.345033466|9343290.05479785
WGS-72|-53.6337691363712|-162.788386059629|3F|646228.923837574|4054952.68557501
Fischer 1960 Mercury |-46.07565590805|113.635780889383|49G|703814.533338941|4894142.06338873
Bessel 1841|-31.4578616453078|20.3777169248345|34J|440885.310576684|6519829.75182186
WGS-84|-59.2984405310237|141.143983842965|54E|508200.504291571|3426706.82953906
Bessel 1841|76.6538444636136|-112.653057094772|12X|457415.004519802|8507820.47501947
Modified Everest|-17.5523066991544|55.1139289149168|40K|299834.68520058|8058498.63411851
Helmert 1906|-61.204813370697|65.7637694190397|41E|648487.901174713|3211189.30203972
Everest 1830 India|48.5250525506666|-20.9990720265785|27U|500068.506987135|5374145.91042368
Bessel 1841|-1.20032996822832|132.645342496335|53M|237998.887096428|9867227.1733205
Bessel 1841 Nambia |13.9662360902587|70.2679598706246|42P|636943.782961323|1544230.06417956
Clarke 1880|-15.9951294333541|166.822990474024|58L|695081.683535882|8230903.82122531
WGS-84|81.5797399695856|-76.9465867189825|18X|468183.172195887|9058451.61243734
Modified Fischer 1960|83.3133164180588|30.6688064990395|35X|547662.306737|9252980.55274197
Australian National|-71.4105503642801|-117.24748601195|11D|491194.457804536|2076767.59049489
Clarke 1880|81.3341073182585|-160.561269352652|4X|473738.843813149|9030684.76077098
Modified Everest|-28.5959366786306|-13.7981108967801|28J|617504.867185874|6836444.19024189
WGS 66|36.2690986226404|-26.1790408354939|26S|573739.766584399|4014113.1321735
Clarke 1880|-47.7687627175188|109.298113767341|49G|372476.375474718|4708310.94529955
Australian National|63.6026250697529|80.7684803288119|44V|488515.305616927|7052780.42616271
Airy|-21.2705334097663|-71.5522690900974|19K|235185.885568155|7645929.54108685
Clarke 1880|47.772836745878|-123.277503857473|10T|479207.766144661|5290776.58154676
Modified Fischer 1960|-3.87298369317118|-3.11505593164526|30M|487226.154449426|9571909.90612281
Fischer 1960 Mercury |16.7231133718651|23.4334254226964|34Q|759465.719555568|1850521.13299857
Helmert 1906|-47.7539261269925|145.685645022044|55G|401489.559088296|4710156.9702429
WGS 60|0.0607278042791393|26.317049222428|35N|424002.574512162|6712.76305566759
Bessel 1841 Nambia |-57.0824455247107|116.415211584664|50E|464558.072268503|3673856.74376325
Modified Airy|-17.6012235042979|7.65436254519685|32K|357237.73498344|8053624.25823094
Helmert 1906|7.92639407259973|-160.505945240478|4N|333996.27900145|876471.003757981
GRS 1980|67.7170047755081|174.437251264525|60W|391583.409854323|7513557.46303015
WGS-84|-50.1002620650065|-148.516625451767|6F|391536.330325534|4449120.2003253
Clarke 1866|-64.9008826045014|-32.92680469008|25D|503464.573780328|2802783.62794413
WGS-72|25.4186654530492|24.1542133335679|35R|213738.473452555|2814359.65729429
Everest 1830 India|19.9257929142667|125.722856561983|51Q|785001.778022374|2205399.33361409
WGS 66|-61.1272541848579|-156.35144329409|4E|642649.05585218|3220142.50795681
South American 1969|52.6736601112187|-63.4478187276288|20U|469720.579788245|5836083.43784335
Helmert 1906|56.4421530215516|-24.0688735541033|26V|680686.454220116|6259210.01067666
Bessel 1841|27.8661616376909|-166.900364113362|3R|312923.718893124|3083524.49453697
Airy|-34.4680435931789|54.9345960834725|40H|310319.804456314|6184277.9834853
Australian National|-43.5480493545099|24.7125510086506|35G|315216.198236052|5175763.51540781
WGS-84|-67.8314847082767|147.838912234307|55D|535324.55987669|2475683.96687909
Clarke 1866|-52.967305136132|43.9291021422245|38F|428076.026893311|4131047.67230072
WGS 60|59.9690346663454|-173.10613369115|2V|382427.71663974|6649866.58539052
WGS 66|-76.6913457453743|-107.647735885876|13C|431983.441702402|1486130.00104086
WGS-84|45.549232836855|141.275915413489|54T|521537.592181178|5044003.1728565
Clarke 1880|36.4443073501673|48.479619374658|39S|274095.156718866|4035893.15040978
Clarke 1866|52.4742698058643|69.3696838032176|42U|525110.734980984|5813636.07518162
Fischer 1968|-19.246515548107|77.7866891246323|43K|792955.71811642|7869539.81576068
Modified Fischer 1960|-16.1954920879762|-107.238336299968|13K|260698.704453383|8208129.25229961
WGS 66|11.7313630692862|-154.771207628585|5P|306970.496353046|1297456.77052189
South American 1969|72.01806364414|-119.89778453295|11X|400185.158483407|7993377.0438093
GRS 1980|-66.7418504172919|94.8280829858511|46D|580543.829646494|2596216.58524228
Modified Everest|46.2419710596774|106.374545382119|48T|605953.487321928|5121390.32953657
Bessel 1841|24.6177477302516|151.989913720747|56R|397767.959482338|2722732.09027553
Modified Airy|43.5867339402175|-69.3072339121787|19T|475200.698464384|4825502.82962144
WGS 60|-77.8099196539147|-142.572911774295|7C|462927.68548746|1362304.96294244
Fischer 1968|-14.2955642353335|-24.6631653029101|26L|752093.02478479|8418310.74002738
Australian National|54.3052549490755|85.8913474963792|45U|427862.204089916|6018073.26369083
Modified Fischer 1960|57.6002108731327|50.4445900768829|39V|466805.206184824|6384359.29478799
Modified Everest|14.1314998544428|41.4342499202958|37P|762761.719073292|1563515.21618802
Modified Airy|7.79779464170012|142.608334014096|54N|677323.720151688|862194.373491165
Fischer 1968|-66.0001919506498|-2.6182588154974|30D|517325.734423953|2679997.43575552
WGS 66|65.14498791186|-154.366624194813|5W|435907.926150273|7225315.43514335
South American 1969|-43.9189895052331|78.6189673855849|44G|308840.418830436|5134351.85220029
Fischer 1968|74.228419070809|-168.457623815752|2X|577111.848023473|8239188.33705899
Fischer 1968|-61.1174992232197|72.8655675773182|43E|384998.099287568|3222231.59986867
Fischer 1968|-34.523440708619|39.3507302508559|37H|532189.29356177|6179736.7863213
Modified Fischer 1960|-61.0656072718444|116.749524775144|50E|486480.777748199|3229856.42779464
WGS 60|9.75207739464967|-107.035825682817|13P|276670.022144441|1078679.52937295
South American 1969|-0.87445590651518|157.097393621038|57M|288272.114782561|9903292.27479183
Bessel 1841 Nambia |-79.4942460769396|-49.1422156037689|22C|537805.173536543|1175102.38195371
Bessel 1841 Nambia |58.4053337919174|73.5580779185547|43V|415747.639515939|6474154.46516948
Modified Everest|-28.8661064946533|11.9124571706936|32J|784064.732710511|6803618.23930587
International|-30.70120357214|-179.490858873814|1J|261422.801043857|6600812.70749438
WGS 60|-74.3304214295556|21.1568833178708|34C|504729.711113954|1751052.70212271
Bessel 1841 Nambia |-70.3285772013314|-151.219100704398|5D|566892.029829997|2197234.16309721
Modified Everest|-67.0658033719383|101.326923799037|47D|601150.21649125|2560160.51220915
Everest 1830 India|10.2339750438857|-20.7328676758006|27P|529251.441045121|1131204.54307861
Modified Fischer 1960|-78.8513697348469|-65.8643432999023|20C|438196.238549743|1245070.5378123
Bessel 1841|8.82455671027429|-40.2060134917927|24P|367385.021311208|975576.127024787
Krassovsky|44.7237331035981|14.8772285958125|33T|490277.186403817|4952356.07444208
WGS-84|8.95074030996383|148.734588986774|55P|690709.223812861|989855.415746449
Modified Everest|-39.1584520282846|84.1830230710414|45H|256632.491082458|5662231.72460031
Modified Fischer 1960|-22.2110071489991|140.899509485076|54K|489642.66697156|7543804.6457914
Fischer 1960 Mercury |68.4115907330898|-171.250095111272|2W|489730.795726341|7588812.90209535
Modified Fischer 1960|68.7539563417144|96.4551647130405|47W|397109.331716695|7629082.60528133
Airy|-35.5834348149979|98.7693019097364|47H|479100.880390609|6062505.29880333
Helmert 1906|74.6782517638957|-162.073304566651|3X|586291.920247614|8289919.81487471
International|-16.2792054758096|132.498789296363|53K|232683.451393997|8198522.13670446
GRS 1967|56.5494045156185|111.703723534613|49V|543264.769534525|6267472.99590159
Modified Airy|-32.7322827906578|-141.603970596305|7H|443416.637263558|6378609.17527228
WGS-84|26.3400130571838|147.983815737086|55R|598175.7695275|2913713.03045754
Australian National|-35.4871617798279|10.1080576409794|32H|600511.595586113|6072352.4807616
Fischer 1968|-60.42659554902|118.62662168969|50E|589552.559052192|3299955.34833029
GRS 1967|64.0078680744605|-80.0462664515183|17W|546634.905340866|7098264.58273509
Australian National|-56.3553498166306|164.053133094201|58E|441489.791224601|3753945.38903853
Airy|-1.25111244904043|-109.574136173314|12M|658627.565057275|9861680.28638384
International|-37.8199760136105|113.194890774834|49H|693196.141631393|5811819.79932521
WGS 66|52.4941711400558|-120.598740025818|10U|663014.355163077|5818720.87437903
Fischer 1960 Mercury |-68.720788407246|-103.599822318467|13D|556704.715914434|2376086.96797372
Hough|-44.4500025316326|-174.563276761252|1G|693886.088233491|5075256.59986455
Clarke 1866|-3.51982081628771|145.424646996725|55M|325007.172697524|9610826.23688903
WGS 66|-20.871916821857|-124.553311141748|10K|338413.824094576|7691244.52423999
South American 1969|-49.4019308447694|-156.160264457986|4F|706016.893027757|4523964.83154775
Helmert 1906|7.31189546667598|-32.086860602443|25N|600793.883876541|808339.044927794
Everest 1830 India|-12.8052187724202|-106.990549583911|13L|283965.001031009|8583684.77137217
GRS 1980|-38.3116671497867|70.3408830401204|42H|617228.467275085|5758753.46118053
Fischer 1968|-2.54978610051285|-130.244361801552|9M|361658.672175697|9718102.47138579
Bessel 1841 Nambia |-50.5653348058264|57.4074743097657|40F|528855.451133649|4398928.61084085
Krassovsky|6.09652300145197|156.713897402188|57N|246972.820499015|674422.825978094
Bessel 1841|37.1905197648192|-107.111304510006|13S|312625.069220189|4117684.40815484
Helmert 1906|-22.3401222728482|-91.9729727847697|15K|605761.056353208|7529138.56066856
Everest 1830 India|-16.7005123185909|44.024646699329|38K|396026.815177164|8153469.51159811
Fischer 1960 Mercury |-38.7882421374715|18.2761364986861|34H|263414.05598518|5703174.77728816
Modified Fischer 1960|-23.2669915396283|1.36327810269933|31K|332581.717431735|7425969.54625788
WGS 66|32.602891570934|-124.087581447656|10S|397947.271582908|3607790.81008776
Bessel 1841|75.0814303976418|99.7908455726753|47X|522723.248107582|8331942.96369686
GRS 1967|-10.8124495428775|85.4639400403561|45L|332070.28642228|8804331.57882047
Everest 1830 India|23.0597014577442|-57.6757891130015|21Q|430783.120417681|2550078.17086006
Modified Fischer 1960|67.5295492304662|93.3517965853129|46W|515004.847957829|7490481.88278995
Modified Airy|41.3095857483432|152.664006874965|56T|471878.735853166|4572692.06418473
International|-74.1357819344287|144.254128655037|55C|416239.182030482|1770652.95885674
WGS-84|63.282845717702|70.6637485016623|42V|583449.73277667|7018186.04640878
Helmert 1906|72.181825587322|175.558314100301|60X|450764.977983535|8009885.95038014
Modified Fischer 1960|-15.6833807064034|-148.595744171624|6L|328983.074956022|8265436.37959629
Australian National|-39.5921253086549|163.413220736405|58H|363746.764077851|5616293.42341647
Everest 1830 India|-72.9375338307636|114.935797251714|50C|432420.38077794|1906225.79373092
Fischer 1968|-21.1028081784124|-51.6978500846356|22K|427521.569056343|7666308.44785589
Helmert 1906|83.4644626498568|-118.152195591397|11X|485359.52016526|9268549.22310173
Hough|25.9404826627254|121.22785158056|51R|322536.475346436|2870279.58566473
Modified Fischer 1960|-0.704952803441174|81.6827276016036|44M|575966.800564356|9922075.63081988
Everest 1830 India|10.6540723763841|-111.318132473095|12P|465210.655540227|1177655.23306891
Krassovsky|-53.0405545142223|55.9418498122142|40F|429053.639509113|4122591.45094977
WGS-84|30.3590516472597|-2.4285353569168|30R|554917.399354236|3358710.69471739
Everest 1830 India|-15.6530831651842|-72.285767513321|18L|790949.51326443|8267716.33719248
Hough|17.0787924938892|-110.199586165332|12Q|585167.145376267|1888434.1339691
Hough|57.9090739866782|-38.3382168047217|24V|539217.427348228|6418805.99124219
Bessel 1841 Nambia |54.1757464391621|-102.671292206841|13U|651972.339184778|6005041.48968616
South American 1969|79.9641476775162|147.883597710196|55X|517189.37983916|8877746.19146799
WGS-72|-20.2032350733216|-97.3941159408291|14K|667784.011982125|7765216.82824449
Everest 1830 India|-16.183101608287|-109.768574077366|12K|631622.098609694|8210560.19600829
International|-30.7358835820862|104.485558017519|48J|450750.975320071|6599505.14511099
Everest 1830 India|-60.3501081077845|-141.2516049998|7E|486116.798235404|3310261.71152634
Fischer 1968|24.9802536616046|-5.16145506773213|30R|281819.398265724|2764507.81955774
Everest 1830 India|-52.1830615470703|170.613000557412|59F|473544.874784138|4219096.95347133
International|-35.7967150574717|-18.3888637406092|27H|735974.434624929|6035388.74266603
Australian National|-73.4663659748172|-0.381821631021637|30C|583140.612694473|1845644.27229113
WGS-84|72.6424249299604|-156.212459385437|4X|592791.692895918|8062746.92416458
Everest 1830 India|46.5329216761366|41.8620533419204|37T|719440.484082758|5156754.05239321
Clarke 1866|15.6240798902831|59.311147466294|40P|747791.520925822|1728594.63392776
Airy|-21.8609067994208|9.02969776740377|32K|503068.12784467|7582731.90321979
Fischer 1968|51.0861451673117|100.270392072722|47U|588976.355432494|5660187.3536803
Everest 1830 India|-76.509396270127|-28.3854885262883|26C|463928.868320505|1508513.22571268
Airy|33.0435908211805|67.4525769654728|42S|355523.651590389|3656928.54777551
Modified Fischer 1960|-33.8748453648909|76.9238781573236|43H|677939.35343285|6250041.66357151
Modified Airy|43.68220567917|-94.6019716647452|15T|370895.858628072|4837305.5956775
WGS 60|75.556652200765|31.77441956466|35X|632795.325481602|8391117.52715257
Fischer 1960 Mercury |1.90484759998438|88.0993804869|45N|622274.46751187|210583.828897115
Fischer 1968|74.7264778451559|104.126784537284|48X|474324.267098746|8293298.11492066
Airy|-5.61606168543625|-35.6051553628571|25M|211421.444925238|9378635.18713333
Bessel 1841|72.0304526811685|-53.3874072139042|22X|417821.451189947|7993099.29846485
South American 1969|68.0060734863935|153.339109011226|56W|514172.661289953|7543606.95137232
WGS-72|56.5190342700986|-52.3175583497797|22V|418934.27195175|6264624.88530562
WGS 60|33.454629198597|-36.2506445382603|24S|755547.902554731|3705089.87062815
Helmert 1906|47.9437087482423|-31.6956131504656|25T|597407.280435401|5310923.05688906
Bessel 1841|-33.3883835392547|-35.6746013662907|25H|251244.003063768|6302824.90710211
GRS 1967|-16.8950109771943|112.797138576367|49K|691426.393250037|8131180.29981954
Krassovsky|76.9743322772294|95.6435873926593|46X|566492.697417878|8545551.37720499
Hough|33.0573234824604|-67.1100453497309|19S|676454.915920869|3659217.9351208
Fischer 1968|-35.0009503623938|-143.659282694902|7H|257308.360026432|6123608.69762627
Fischer 1960 Mercury |59.737778856236|72.4061606704836|43V|354202.157180503|6625093.73459735
Clarke 1866|36.7997248327861|-51.059375001078|22S|494703.094148861|4072454.30487608
WGS-72|-22.7212614719316|66.3186736100846|42K|224581.668929568|7484846.18798233
WGS-84|24.0413987826917|144.552382136581|55R|251078.931333155|2660976.92020615
Krassovsky|15.7411699536286|-70.4368081603732|19P|346060.84697715|1740860.18554268
Fischer 1960 Mercury |-38.416873012896|136.61687019342|53H|641154.459567608|5746670.04730366
Airy|75.1111360681007|-105.648176004422|13X|481411.665457517|8335436.59376783
WGS-72|15.9898459397665|-6.92766956161293|29P|721774.310715024|1768916.89978102
Modified Everest|37.3272235589447|-14.9966435239728|28S|500297.319459075|4130823.27232731
Modified Fischer 1960|74.8358202707451|58.650789127815|40X|548195.984225444|8305985.58452905
GRS 1967|52.7066482223222|-151.519985739937|5U|599993.616572481|5840686.1018475
Modified Fischer 1960|49.5629167435162|-166.757867699769|3U|372883.133676315|5491539.12253069
Modified Fischer 1960|45.1616735140287|-155.508540040729|5T|302845.240466088|5003989.56463272
Australian National|81.2799403513549|45.890388216268|38X|515069.735951684|9024599.80349185
WGS 60|-78.390563972176|-52.3747777402343|22C|469118.006237355|1297638.52066937
Airy|-51.2720711081613|-96.321040183213|14F|686847.098199668|4316931.45551411
Clarke 1866|44.3979462093858|-26.7891229771826|26T|516793.95405029|4915878.09328516
GRS 1967|81.1185797320184|158.809569815969|57X|496717.709977577|9006477.65726973
South American 1969|62.8841650789915|-154.672411471|5V|414959.351842125|6973812.70255123
Krassovsky|49.4567042265903|103.090365512484|48U|361607.67394771|5480076.66429183
Everest 1830 India|-30.8978509634734|105.874165942018|48J|583531.645798337|6581684.89345526
Helmert 1906|10.4239114013636|-134.553944065799|8P|548821.981466514|1152329.09228408
Fischer 1968|-14.3635141638131|85.5751971642576|45L|346365.083769423|8411591.54939332
International|-58.1599293244641|136.375457627382|53E|580939.972640099|3552511.92897569
Australian National|-77.2174630984881|-37.0327090558632|24C|548580.6554521|1428115.61053291
Airy|16.4692277911876|-79.067077420247|17Q|706331.189838366|1821704.9046601
Modified Airy|27.6125923107236|26.7569487636121|35R|476020.686265944|3053995.57092482
Hough|-79.8431533099182|74.8820801895407|43C|497678.485962582|1135820.04684957
Fischer 1968|39.4424564716321|10.2237529655039|32S|605305.53725867|4366605.04054707
WGS 66|83.196374724526|78.2591455307592|44X|463762.546697809|9239252.84946338
Modified Fischer 1960|-52.6547479524176|-0.134794204403761|30F|693794.55145452|4162261.10257489
Clarke 1880|33.0276535233388|-6.68640128078508|29S|716090.632772291|3656456.1425668
Krassovsky|4.08228333008744|17.8363973867809|33N|814958.13166492|451786.026936226
Modified Airy|-39.3887589326368|-136.785892148378|8H|346221.332000731|5639022.44181471
Everest 1830 India|58.986841806808|-70.3678887693292|19V|421397.402506779|6538721.37908011
Modified Everest|42.1693834811411|39.7295454863853|37T|560250.249304305|4668432.10035574
GRS 1967|29.3326268570903|-51.0483758355738|22R|495303.34363792|3244850.76433517
WGS-84|7.04533528057556|168.503947958448|59N|224250.001023189|779497.597073143
South American 1969|18.3913558396439|39.4622837586553|37Q|548830.084241193|2033553.68138444
WGS-72|43.3423088948644|103.04137897556|48T|341243.319910815|4800690.27818285
WGS 66|-33.080199810012|-102.323391655186|13H|749848.473798014|6336631.23215716
Clarke 1866|8.0321337552283|-68.9838780945176|19P|501776.502019669|887794.182782833
Modified Fischer 1960|15.0950459162041|-145.339085063032|6P|678503.897514149|1669518.71967872
Hough|-8.9990092407821|86.6726481004456|45L|464018.399447268|9005248.52354217
South American 1969|72.5194567898129|22.3133847970652|35X|343013.78079102|8053031.58326701
Modified Everest|-25.0435835460112|-43.9801976556679|23J|602861.180134226|7230057.30112824
GRS 1980|-51.4202269621504|-6.48892733274465|29F|674592.176844861|4300450.89603998
WGS 66|40.3871815006471|-89.4383897439884|16T|293029.705638927|4473591.64197495
Everest 1830 India|9.38209970709205|0.0876976315080924|31P|180152.310509047|1038340.72890792
Krassovsky|64.3893067241981|-107.784073206706|13W|365753.613740627|7143463.22528339
Modified Fischer 1960|-35.3727311905739|30.3131889819878|36H|255910.622933052|6082292.5244988
Modified Everest|-50.884843057428|-120.011479003353|10F|710171.556924558|4359249.49555364
Everest 1830 India|33.6140933458078|38.6637996827992|37S|468817.347530674|3719095.12929728
Fischer 1960 Mercury |-3.32020205596149|-52.8909730540748|22M|289893.779744772|9632810.91100335
Fischer 1968|20.7917885041413|-25.225784262193|26Q|684669.230263311|2300126.51785401
Helmert 1906|-14.2127980376064|122.989079486528|51L|498821.766902596|8428722.06226813
WGS 66|-78.5684308789236|53.7523481300903|39C|560876.062633349|1276748.05984679
International|-65.6937263377432|153.905606246024|56D|541595.67536817|2713743.97777915
Modified Airy|30.201056677098|-37.5402930410718|24R|640490.27936823|3341616.96646036
WGS 60|-38.758343657961|-86.8254346180221|16H|515167.274252896|5710003.14908597
Modified Everest|-58.1035653565808|150.77561630124|56E|368932.611318396|3558226.66336061
Modified Airy|56.8825038440556|-34.5820276395089|25V|403610.298336|6304724.05765501
Clarke 1880|51.2714285223691|-156.139931248393|4U|699508.891305899|5683583.41611182
Airy|1.39284564877605|77.0477269755479|43N|727822.140131258|154040.427495729
Australian National|44.9802896904291|-15.3741335360755|28T|470502.338265644|4980846.28478141
Krassovsky|83.5579214910344|-90.7397380079151|15X|528306.205823406|9279456.67222026
Everest 1830 India|37.7857847729914|-14.4416555803224|28S|549156.436304083|4181820.78751488
Fischer 1960 Mercury |-61.701279603885|-109.563835242315|12E|575953.808330079|3158223.97850769
WGS-84|24.2326399951906|-79.062658556384|17R|696718.260342495|2681348.82207355
Everest 1830 India|43.6602450114622|-20.4178948091065|27T|546928.328066937|4833854.38550678
Everest 1830 India|55.4873135633743|-125.267888848591|10U|356725.51946693|6150742.57105599
Krassovsky|-31.5720510241762|-15.8585134656944|28J|418531.680150931|6506616.00826256
WGS 66|35.3661005399907|-85.9866282732489|16S|592059.808944601|3914119.07205552
Clarke 1866|32.4208060286875|76.937152446372|43S|682152.58038389|3588542.34630273
WGS 66|-6.69302194064262|-135.387707080612|8M|457149.606302018|9260168.57095612
Airy|74.7458996714221|-108.372422622758|12X|577135.875330461|8296295.25184684
GRS 1967|-35.2124876563367|-147.244003192974|6H|477791.91384306|6103351.85888688
South American 1969|81.9328635801389|8.49571787111145|31X|585975.385698496|9101452.80734558
Modified Fischer 1960|71.0475925625243|158.517879537902|57W|482523.797200695|7882799.87192288
Clarke 1866|-67.8367517890222|-60.5253825205291|20D|604160.609910702|2473436.6769716
South American 1969|79.1367794213384|-88.0902975863896|16X|477061.913091487|8785485.31297382
Hough|72.3668276271435|-99.8247367769618|14X|472115.045220251|8030112.11404364
WGS-84|-68.6639990376337|6.7736216562588|32D|409616.879681467|2381466.78974735
Clarke 1866|56.369203542617|102.961680938191|48V|374094.712380892|6248823.41196407
WGS-84|-79.5560155735765|-19.9481835778127|27C|521284.378331306|1167776.81989535
Bessel 1841 Nambia |-67.0377287833116|-59.5034521454521|21D|391048.791434465|2562911.18620036
WGS 66|-63.4676614074895|95.4086833317115|46E|620029.038655935|2960037.63001368
South American 1969|33.6933995698695|-146.310453903762|6S|563906.784394215|3728387.84802602
South American 1969|11.3416812063234|159.563223193075|57P|561457.856902116|1253823.43878579
Fischer 1960 Mercury |-68.6940247397257|-31.1808135748804|25D|573758.390281356|2378625.78677763
Clarke 1880|6.71973049546777|-172.636084777834|2N|319159.706097632|743000.401368137
Australian National|36.3086982158058|63.4214308360104|41S|537834.219163851|4018284.82898117
Fischer 1968|35.1974088657091|-118.057213694691|11S|403757.739191764|3895458.18488702
Hough|-4.42838352784139|-70.5647231317435|19M|326376.719731627|9510341.39153632
WGS-84|-19.3804954357771|-6.53520013344746|29K|758887.269588734|7855222.03720286
Krassovsky|-57.9836668551134|-81.2675783600112|17E|484176.216337418|3572964.48348991
Modified Everest|34.9481436840763|12.0606506614141|33S|231607.163413898|3870915.28917129
Australian National|79.236171289559|94.7958731641434|46X|537437.136489546|8796940.5357783
Modified Everest|-27.9209542415175|77.2321534797487|43J|719632.274481275|6909797.31797105
International|-45.4276624904735|-134.326919191893|8G|552655.379844583|4969225.34546059
GRS 1967|76.2728211278903|18.1778642247056|33X|584141.222101057|8467925.19538858
Fischer 1968|-69.3907396692136|-111.470431563789|12D|481519.990281627|2301977.55246257
Bessel 1841 Nambia |-76.4827436614873|-41.5857163208352|24C|432561.680440323|1510272.14799416
Airy|61.8598869545891|14.1487794546553|33V|455216.126579519|6858336.49169917
GRS 1980|3.54293898179039|174.81595471932|60N|257373.118686125|391892.089515591
WGS-72|6.43559288380004|70.1275112628407|42N|624686.264141689|711493.689115344
Everest 1830 India|-69.6958144297604|-3.23156291840803|30D|491033.701805086|2268876.61472753
Modified Fischer 1960|-13.6616054426364|-39.6015977039248|24L|434937.807979566|8489611.57889917
South American 1969|-40.6811869518647|-50.0113450272116|22G|583547.683754327|5496147.66654361
Hough|-60.2090171926739|52.0819893747434|39E|559969.851170418|3324786.20727016
WGS-72|-63.5227806183433|-39.9430377383544|24E|453090.224112772|2955818.35072272
Modified Everest|-22.4282388469635|35.7538912859063|36K|783437.336357769|7517364.78402059
Krassovsky|38.1470176795979|-102.817670288185|13S|691233.84319542|4224452.05459327
Helmert 1906|-75.764810180546|-13.6315324059096|28C|537560.662041156|1590536.63650773
Clarke 1866|-8.50075525742962|96.5405996471208|47L|229233.860609575|9059542.10157434
Krassovsky|-52.9675436537454|64.3303263543463|41F|589344.919631673|4130408.88209887
Airy|-68.2378766344258|-50.8097459659533|22D|507871.009831584|2431198.39338533
WGS 66|-12.7313128054761|-64.4291515367262|20L|344841.014492025|8592149.01628657
Bessel 1841|-18.9501061526646|38.6586280418123|37K|464063.98193321|7904861.55624693
Everest 1830 India|64.8849261880904|-34.385429632717|25W|434401.321303812|7195583.52951348
Krassovsky|-6.23523856015014|61.9921669719659|41M|388504.895476856|9310672.69476967
Airy|45.9364621381409|-144.591077990782|6T|686722.108556106|5089439.31202525
Bessel 1841 Nambia |-41.7227017465263|135.419185112835|53G|534862.538353316|5381328.22392285
Hough|21.4735367545533|4.89361943334174|31Q|696205.000197323|2375730.39458351
GRS 1980|-21.7374795357266|-178.027089960346|1K|393784.733598419|7595877.3059553
Clarke 1866|1.11371231230186|96.3026729934599|47N|199796.526934701|123228.381607099
Helmert 1906|-42.2447882820635|148.743873857217|55G|643872.210666963|5321522.79835123
GRS 1967|-55.5082875522462|120.521386357409|51F|343474.271686372|3845831.94926407
WGS 66|-36.7930712788868|-128.334102090963|9H|559409.604455623|5927871.16135374
Bessel 1841|2.59190488789272|-96.1074926912265|14N|821636.26224841|286825.234396734
WGS-72|-68.1181386727381|114.204706271511|50D|383772.706672737|2441334.7457218
WGS-84|32.3174575039114|28.2193918652693|35S|614783.206731468|3576277.39499587
Krassovsky|-39.811978734667|3.78709389932027|31H|567371.522662312|5592736.7214318
Modified Everest|-23.6322931397874|-0.255268600112544|30K|779992.39535468|7383997.73657273
International|53.9779283879979|87.9694566884359|45U|563583.21400295|5981629.19073352
Australian National|24.1059026836102|135.622432706402|53R|563257.447787468|2666101.07383442
International|72.7694468971967|-140.383485718303|7X|520384.364843945|8075087.79302461
Modified Everest|23.1428709247058|-122.95004279108|10Q|505113.597136107|2559136.50570467
Everest 1830 India|-77.6903878204971|-13.8304619840113|28C|527827.974026918|1376890.38742851
Fischer 1960 Mercury |-44.4157598109597|-87.5805746193215|16G|453779.000155021|5081758.20598543
GRS 1980|18.1359589654202|117.935500001011|50Q|598961.879223097|2005479.11379272
WGS-72|-22.596136275271|-49.1643023245693|22K|688702.961431238|7500025.5597641
Airy|-34.8407401773983|-114.716488082732|11H|708776.91538343|6142510.70717211
GRS 1980|-76.9367577495753|-59.3680234703989|21C|440266.847516055|1459081.53659704
WGS 66|46.518172851519|-9.98749020308736|29T|424253.355877641|5152102.18418369
Helmert 1906|16.8119465009574|-137.444548243455|8Q|239467.43242697|1860380.09760617
Bessel 1841 Nambia |14.4335229345694|-94.4247636968231|15P|346433.334360955|1596016.4170649
Modified Everest|28.2398462994175|124.373493483903|51R|634733.877371954|3124285.38394313
Modified Airy|-63.2441189329501|-87.0535576308296|16E|497310.209540149|2987998.34547748
WGS 66|73.0561558429291|15.0022407744042|33X|500072.890529556|8106753.78560861
WGS-84|-4.25710482646879|-74.65062212455|18M|538770.702344551|9529444.54690344
Modified Fischer 1960|-48.4931240490302|-11.9804914332424|29F|279807.238697117|4624579.23684606
GRS 1967|13.2227128671312|89.5479246751464|45P|776135.015365332|1463174.94252545
South American 1969|21.674278117251|-28.9329628537659|26Q|299997.022294727|2398029.43960629
Clarke 1880|-74.9252370263923|-168.875488646101|2C|561668.808359551|1683844.62106312
Airy|-9.38880410300303|-164.099963984486|3L|598813.602750072|8962104.47258725
Bessel 1841|-61.5142721040737|82.5140461720557|44E|580546.150265335|3179709.66104064
WGS-84|53.3140589741412|-69.7109902933663|19U|452632.479758524|5907443.6127846
WGS-84|26.8452334860052|125.236227949537|51R|722204.347303138|2971252.44794547
Australian National|-11.1746859748619|15.826898304786|33L|590283.185310593|8764574.8230252
Clarke 1880|30.4003613827056|-138.295887677506|7R|759807.740180679|3365992.3797569
South American 1969|42.9846328642507|47.41718439222|38T|697074.732058576|4761960.06481922
Hough|19.0459577207393|-66.544478895799|19Q|758440.767196144|2107707.71526985
Everest 1830 India|-7.15301063600984|74.0477367299137|43M|394865.695046315|9209289.57392232
Clarke 1866|-38.7067894493418|46.7515724688266|38H|652304.815741446|5714512.01631406
Fischer 1968|11.34509048111|159.939172525569|57P|602482.093246013|1254305.66752389
WGS 60|42.2564118788|94.6521450075466|46T|636278.453390094|4679591.36911273
WGS-84|-6.74337642759511|-87.7464531577773|16M|417506.977405363|9254556.72665592
GRS 1980|-76.5258130651961|107.073784750378|48C|553929.348972323|1505192.20233762
WGS-84|-73.8856817689306|114.974238808441|50C|437251.455667048|1799649.77175727
WGS-72|-52.6123953746928|4.1845884237506|31F|580206.754740603|4170188.64069806
WGS 60|-26.3045496152241|-179.637259783815|1J|236695.457923214|7087886.60076451
Helmert 1906|57.1197759801584|-65.2848484129996|20V|361660.253625722|6333102.09387657
Australian National|-46.8026630277537|-146.057645086433|6G|571906.491394573|4816315.66138727
Fischer 1968|-38.8595886694269|-157.481289586982|4H|631770.881255304|5697696.92248195
Modified Airy|-71.5563645532973|-165.108382426494|3D|496173.455514958|2061456.81896975
Fischer 1960 Mercury |0.717262356738061|-47.3461691165798|23N|238875.797684349|79346.4748434853
Bessel 1841 Nambia |-68.1455688932281|-4.49980025017155|30D|437707.315023181|2440851.78266126
Modified Everest|-27.65633315202|-111.348116883049|12J|465668.86769379|6941062.70761179
Modified Airy|63.5537020760048|-144.646568009667|6V|616908.348370855|7048641.03204597
Modified Fischer 1960|-33.5449843659575|-164.383861979095|3H|557201.177364444|6288109.94487799
Airy|15.7489874065237|-141.503488432538|7P|446068.548421825|1741117.94656241
International|58.6892449552004|80.4596173920891|44V|468673.150489128|6505725.64168696
GRS 1980|25.2210504127705|144.485612271205|55R|246677.552850693|2791794.24275699
Helmert 1906|-41.551994945666|-1.40066060386943|30G|633381.876860952|5398679.52378975
South American 1969|-57.6307505484448|57.8849575679472|40E|552845.809747856|3612031.19090921
WGS 60|-32.9284160914746|-174.576952777604|1H|726562.382772007|6354024.72841715
Hough|-39.9479180710408|-113.008106912967|12H|328447.622528534|5576098.66825838
WGS-84|40.3248428813374|-160.148762758248|4T|402407.11765976|4464445.77433068
GRS 1967|3.31173201258406|154.009951087563|56N|612202.061032156|366108.35119472
WGS 66|28.9979523989394|-2.20074398076918|30R|577850.463790909|3208025.5654761
GRS 1980|-79.9635778969601|-178.667252355676|1C|467566.90724735|1122014.74233502
GRS 1980|-28.1404380710867|2.76719521902493|31J|477139.756036296|6887218.5214191
WGS-72|56.2359238641698|-68.5581977658789|19V|527386.035487193|6232423.83814674
Modified Airy|71.6561573030662|-87.2518203737133|16W|491155.675656447|7949687.29695553
WGS 66|-46.6297050217688|-114.041672020975|11G|726452.401634049|4831726.6137232
WGS 66|-38.0427668571867|-129.059507690042|9H|494778.412474625|5789433.4874595
Modified Airy|-69.2402688763852|-24.800577426813|26D|586975.517349849|2318160.9593455
Hough|9.56384274223181|62.9406545872575|41P|493487.383880019|1057182.94014935
South American 1969|-46.8469119358787|-38.0672250491928|24G|571117.112868006|4811407.33578285
Everest 1830 India|-25.0908577321962|161.463556354802|57J|748428.190892001|7222957.07705386
Modified Airy|18.7839700221431|-126.996294416839|9Q|711168.156257141|2077901.49788667
WGS-84|-45.3040121513681|-64.4219664945598|20G|388521.640548901|4982293.35744939
Modified Airy|4.04089836078211|95.4519521522546|46N|772215.929927842|447013.638618119
Airy|-53.5052105677487|55.674155680334|40F|412075.085358846|4071151.88986659
Modified Airy|-49.8542982204404|-148.98014037214|6F|357684.245192289|4476289.22427171
Airy|-19.924852961044|142.515599627239|54K|658614.54620333|7796267.89371607
Krassovsky|-73.8457860680485|-15.7619272576376|28C|476338.263037295|1804873.35678869
Modified Airy|-12.1388293273085|169.145455145205|59L|298212.817838979|8657544.10835646
Helmert 1906|-20.6495932786313|61.8645666308221|41K|381715.530248675|7716194.06811822
Hough|-72.4446397371833|156.385717615514|57C|412009.207290674|1959484.91983283
Modified Fischer 1960|-57.8207958205553|-30.2409868350846|25E|663870.035817209|3587878.78510991
WGS 60|-55.6461352480418|-47.7695909092247|23F|325715.63162533|3829794.08846462
Clarke 1866|82.4805136982259|-62.4089350226835|20X|508635.699332399|9158394.05091295
Krassovsky|-38.4590396240553|-137.526805656057|8H|279523.168075079|5740151.42420126
GRS 1980|-78.7367904753334|-58.1529696802598|21C|474862.053659018|1259154.01023277
Modified Airy|-22.6288991538721|170.41270562919|59K|439656.803531558|7497698.20028734
Modified Everest|-29.9062720083772|-107.335703081878|13J|274514.855468831|6689576.069179
International|54.3236892363582|38.940629940069|37U|496138.384948591|6019667.71714888
International|11.8872813349318|-156.614193481505|4P|759903.723761244|1315219.67337801
Modified Everest|-45.8145584621569|-125.263168074934|10G|324199.919405038|4924520.43142026
Airy|32.6925299668516|-176.033693596588|1S|590573.447777679|3617362.2870565
International|71.3260720855773|-15.7034472675775|28W|474860.978374177|7914124.19850538
Fischer 1960 Mercury |-65.3831306077382|-19.2740730234159|27D|580213.80535437|2747710.25787988
Fischer 1960 Mercury |7.41469980650278|-25.5377638669697|26N|661376.478929836|819863.19996279
Modified Everest|-58.3377584854288|23.6213103356336|34E|653434.666354812|3531328.04907606
Clarke 1866|-45.4011399935517|-3.67021028390411|30G|447545.516270955|4972485.85287706
Clarke 1880|76.0133307404128|1.43173516895885|31X|457690.116022384|8437028.29682883
Bessel 1841 Nambia |0.873437099366427|17.709455940437|33N|801542.058920662|96641.9249119846
Everest 1830 India|-14.2522284443073|148.517482197081|55L|663688.323378082|8423970.19874018
Clarke 1866|19.2910014942622|154.810764141391|56Q|690276.747249775|2133892.54323373
Modified Airy|83.8736841121004|-93.4633003900201|15X|494480.663598899|9312918.61273545
WGS 66|27.9105808517974|-114.737096267022|11R|722710.841913365|3089359.80810113
Hough|-66.7103506791713|-163.929330619116|3D|547238.257467145|2600450.96143766
Fischer 1960 Mercury |68.4100151885887|172.920171351635|59W|578839.425828237|7589844.93822321
Modified Airy|-43.4936979172786|-108.005350993801|12G|742099.892053403|5180517.60120696
Modified Everest|20.611572749674|95.0141817504877|46Q|709872.881284022|2280283.20559547
Modified Fischer 1960|73.5018285051955|-87.1087981665402|16X|496551.299954334|8156490.97058023
Fischer 1960 Mercury |-72.0007246603803|149.568700548992|55C|588568.507013918|2009058.48464906
Fischer 1968|25.4108623693357|-173.60982122417|2R|237467.958442498|2813018.27764348
Modified Fischer 1960|-52.4482676575713|130.044784731109|52F|571005.582000375|4188568.97393183
Clarke 1866|18.7930904926345|-128.929172698011|9Q|507463.842398304|2077810.33628841
Fischer 1968|-9.08044162519566|-55.3959187488937|21L|676294.232341948|8995861.85096771
Clarke 1866|-13.7107859557436|-67.232111870136|19L|691181.534772492|8483653.45773915
Fischer 1960 Mercury |-53.16958692877|5.45626995411084|31F|664181.141770938|4106016.06272776
GRS 1967|-10.9307156116851|160.919815072309|57L|709814.244687714|8791010.53661072
Clarke 1866|61.840701739178|147.574676316599|55V|530257.9494396|6856365.59613915
Modified Fischer 1960|45.8596976113066|-154.163439232821|5T|409684.725783863|5079135.07345781
Krassovsky|56.1547581234987|-95.7171272150228|15V|331237.70537991|6226737.48747185
Hough|-76.5237906512958|89.8965930806374|45C|575326.440650159|1504429.16058046
Krassovsky|-51.0066362439736|-147.431943936014|6F|469694.546337919|4349249.05722189
Helmert 1906|57.8832336496386|76.6056412497072|43V|595212.429675596|6416907.36827539
Bessel 1841 Nambia |19.6016505122604|-19.1854426989667|27Q|690290.206284916|2168231.3242731
Fischer 1960 Mercury |20.3362643995417|-98.4488898131541|14Q|557526.537221322|2248801.44877574
Modified Fischer 1960|-68.0688856566031|-110.870592313735|12D|505393.785062839|2449425.15691561
GRS 1967|20.850006814971|40.1470707936055|37Q|619338.823180836|2305980.67422259
GRS 1980|55.7535141567594|-26.8089461682993|26U|511991.327475882|6178663.46001652
GRS 1980|-37.9078436621771|-62.4593662924421|20H|547525.707536079|5804272.07380389
Fischer 1960 Mercury |26.302399776458|-167.025801627561|3R|297757.701793122|2910773.85143444
Bessel 1841|59.9481641511859|14.7004180385657|33V|483265.985001577|6644980.3441916
WGS 66|-72.8728207406813|2.09798578460521|31C|470351.395794861|1913474.69472682
GRS 1980|-79.2934388277983|-82.2099104758762|17C|474908.557057085|1197015.15964602
Bessel 1841 Nambia |-74.2142209441718|-141.146843212543|7C|495541.510841435|1764833.48889397
Bessel 1841 Nambia |4.5804626564387|-141.567040160174|7N|437108.230046971|506272.749446891
Hough|-35.2932053173919|177.82816444711|60H|575303.620019015|6094136.6705393
Clarke 1866|4.79877261774084|91.9005157035924|46N|378072.152176982|530485.083953134
Australian National|4.31819357773433|148.341303333551|55N|648846.682889955|477432.045712188
GRS 1967|-0.536415382730496|-28.8663782338951|26M|292290.145222193|9940678.0846416
Everest 1830 India|47.0259710812879|106.01297005246|48T|576962.469038422|5208054.37174447
Fischer 1960 Mercury |81.2616361755885|-62.1740818999193|20X|514007.776139712|9022552.50769724
Bessel 1841 Nambia |-58.9746255467707|-177.53261758132|1E|469380.946951368|3463245.38415885
Krassovsky|75.202734121666|-80.1098347183888|17X|525377.184671297|8346561.89221642
GRS 1980|-64.7423100175756|88.2538777805772|45D|559694.661688893|2819673.17954986
Modified Airy|68.4405530459342|56.1329753394985|40W|464450.239753445|7591369.54710399
Helmert 1906|5.38878174069093|86.0615672917646|45N|396028.757479999|595726.323556076
Airy|58.458478137613|78.9273960414481|44V|379083.71630144|6481126.81167578
International|-72.6377654098696|146.091066025669|55C|469725.167520368|1939477.50278594
Clarke 1880|74.2296382934929|109.946828432378|49X|468049.521013516|8237721.52220915
WGS 66|-15.2398242115612|-71.3113426303562|19L|251733.780530189|8313829.44180753
Modified Fischer 1960|9.00344434737539|103.450152837158|48P|329631.358738607|995597.395347121
GRS 1967|56.2752065615606|-173.241901293847|2V|361185.803166911|6238990.89334462
WGS 60|-73.4806739412066|-162.288205492393|3C|586039.188422178|1843906.86860808
Clarke 1880|-18.9689884174488|-14.6128931228321|28K|540751.695772183|7902740.48388794
Helmert 1906|-17.9942206904023|-34.6568930071818|25K|324569.607047934|8009648.62460335
Fischer 1968|48.7138081136984|61.6373411170215|41U|399762.933119943|5396552.3084814
International|30.9984249356909|179.525452314159|60R|741146.27662824|3432216.25502221
Bessel 1841 Nambia |-57.1902087911709|86.6120495495682|45E|476555.84094318|3661947.14284555
Bessel 1841 Nambia |74.9890482312621|115.359644932033|50X|452586.530991172|8322254.3734065
Hough|83.3267093233001|151.023487330639|56X|474360.930530461|9253481.41288155
South American 1969|-59.1320673341979|-24.9696533015371|26E|616190.025647276|3443451.68581982
Helmert 1906|59.6492760173468|-153.631045394764|5V|464429.876979204|6612590.86082979
WGS-84|-65.3681345803636|133.039009712316|53D|408812.438401013|2749097.8630899
Fischer 1960 Mercury |35.0602051259169|22.6207839390437|34S|647798.21464408|3880941.37831861
Hough|-9.12758521604582|134.28673773185|53L|421626.56508008|8990972.25596531
Everest 1830 India|65.5525175731477|84.2346016872739|45W|372345.580388078|7272063.40937412
Everest 1830 India|-57.2751738994497|-136.295871944183|8E|421875.353558092|3651882.11813481
Everest 1830 India|-36.2272859459059|40.1666674992205|37H|604833.432037742|5990565.88090623
Modified Fischer 1960|-14.6814694647313|0.944565035566285|31L|278659.397646236|8375891.77073548
Clarke 1880|77.0899837373827|-177.37036550633|1X|490762.294306622|8556644.94411781
Clarke 1866|-20.6503905163223|123.816835293962|51K|585092.321577382|7716465.20011188
Helmert 1906|16.8751999296532|79.839028892396|44Q|376332.678774068|1866133.00937845
Krassovsky|13.3629450241432|83.2502966301276|44P|743726.224974388|1478406.00569355
Fischer 1960 Mercury |-12.0398980004405|5.29298186867322|31L|749637.534243359|8667984.82186195
GRS 1980|-55.8497165220386|94.6322191885729|46F|602187.511827394|3809441.66072786
Clarke 1880|-78.7564727096092|-54.0603855607228|21C|563963.799631354|1255782.3744227
Clarke 1880|-3.44050710182705|-167.54322817865|3M|217411.244072476|9619374.46570054
Airy|20.7652852235598|24.4988587119468|35Q|239616.643787471|2298032.26719669
Hough|11.2072070609038|133.05422229312|53P|287543.056146242|1239583.14014647
WGS 60|-29.4606899624163|77.1506085252619|43J|708559.918226045|6739029.403319
Fischer 1960 Mercury |-32.9833503340519|-34.8312832728173|25H|328884.309500239|6349050.2595105
Modified Airy|12.2604140277024|169.362814677288|59P|321950.552013747|1355752.70968001
Krassovsky|8.58934073415716|132.030158875263|53P|173066.753535838|950734.625926016
Bessel 1841|-36.3498006126314|-43.041346816992|23H|675735.138534041|5975872.32774493
Hough|77.9022157466906|26.7954965475094|35X|495215.264878104|8647556.1339883
South American 1969|11.5287938423674|-140.059676220787|7P|602541.773309467|1274622.02109957
Clarke 1880|79.5535825046842|-134.462896064561|8X|510872.437451809|8831628.81761795
WGS-72|-7.95699955362879|56.2788840543598|40M|420523.297787146|9120386.89401319
WGS 66|39.6262193018376|-176.036941916601|1S|582653.657742031|4386720.5495526
GRS 1980|-11.3177479724301|-95.8093597965797|15L|193313.931187608|8747410.21254279
Modified Airy|-44.7052098342806|90.6378860262389|46G|312900.99665284|5047614.04138694
Australian National|29.8002253598859|-104.335563090513|13R|564212.334133172|3296845.4997507
Bessel 1841|41.8314905539578|-173.548903177189|2T|288367.559315677|4633741.47913723
Helmert 1906|-20.9397399626419|177.846022326138|60K|587964.598045775|7684264.41023451
Fischer 1968|-59.8452259351522|-52.7266500965733|22E|403249.052228122|3364547.63080443
GRS 1967|-26.0680962505341|-75.3472580617998|18J|465267.026288494|7116718.83458407
WGS-72|6.13274640677975|-78.4135573734237|17N|786265.710109449|678569.139023455
WGS 60|-79.1267737619683|-0.497114312399844|30C|552691.944836942|1214704.98371191
South American 1969|-24.3874082122701|2.09033702193366|31J|407753.379704524|7302569.18988596
Helmert 1906|-16.2686031840395|-174.197788153224|1K|799513.539257512|8199280.42366317
Helmert 1906|-65.009556508361|-50.718315828489|22D|513278.630717144|2790376.06412403
Australian National|-55.7846246270823|-8.32707606363726|29F|542201.691681254|3817664.07014439
WGS-84|-2.14570899466443|-27.0138509712139|26M|498459.807119314|9762833.56587461
Modified Airy|50.5508187838413|162.960081893214|58U|355507.078365311|5601252.79891444
Bessel 1841 Nambia |-66.381295969703|122.926332336821|51D|496706.83833171|2638268.02666923
Bessel 1841|-74.1566204321942|-151.940327936064|5C|532286.383291318|1771089.53239425
International|-56.7773271199879|104.109799199876|48E|445597.814984988|3706906.46675089
South American 1969|27.0451770530908|-127.452387125271|9R|653496.876977914|2992392.13391933
Modified Everest|16.6533858718592|-75.3914558444994|18Q|458261.614045128|1841112.3463134
Helmert 1906|1.6211019381423|-8.95468391684346|29N|505040.591083726|179182.951008382
Hough|-79.3459703487028|75.3528067186413|43C|507281.837003628|1191294.06142229
Fischer 1968|9.87371925597843|-138.448940756621|7P|779780.076676466|1092522.33805817
Modified Airy|77.5054494824172|142.785779294229|54X|543116.422773128|8602835.2796839
Hough|-6.32607395830634|-27.5052848310088|26M|444112.653566865|9300728.35043521
Bessel 1841|-0.0910099239023623|95.0673363021535|46M|730066.05358549|9989935.03823886
Hough|-60.1860598622687|87.9891714690919|45E|554863.943886888|3327423.55886512
Clarke 1880|42.7672642900646|72.2749277733085|43T|277031.146413509|4738264.31582415
Airy|-54.2538586313642|-5.15546632307803|30F|359595.430977426|3986540.06130879
Clarke 1880|70.9009290296928|69.6718184437712|42W|524535.172788418|7866243.29064019
Clarke 1866|-75.4375274575894|-46.4610180060382|23C|458996.170076484|1627222.40308494
Fischer 1968|-42.5910116349162|-88.3363817075098|16G|390353.013513949|5283722.53197248
WGS 60|-16.2577944751071|65.5060188818163|41K|767850.088394029|8200898.27701474
Modified Airy|47.9327065339794|-30.6265090385473|25T|677253.434952812|5310972.49609313
Krassovsky|48.0543835382343|11.5146551275475|32U|687381.516822389|5325497.80279335
Helmert 1906|-65.7052573046267|8.94677858908338|32D|497556.594152793|2712865.67965816
Modified Airy|35.7503764940392|-158.543756760842|4S|541244.132229659|3955941.63182338
Modified Everest|68.6981405174636|-3.49173188301748|30W|480067.434221323|7619988.82853236
Clarke 1866|-49.6225623898587|-82.5922994799865|17F|384992.288174722|4502334.93622299
Clarke 1880|-72.9274219494125|54.9832602215294|40C|433920.668093239|1906731.94887622
Modified Everest|-78.1493278918028|-43.7645050392462|23C|528317.525952313|1325621.50497254
Modified Airy|50.3284304777149|-116.349343196205|11U|546305.934088509|5574743.56401665
Clarke 1880|74.3023926315722|57.7342949183142|40X|522177.010068379|8245693.15699376
Modified Everest|-71.1307076350216|124.488647544682|51D|553720.782653881|2108200.22570884
Australian National|-75.3836626308948|150.243885218516|56C|422391.239184291|1631749.55441896
WGS-84|-67.2819203386667|94.9847876897432|46D|585526.36217775|2535825.03253516
WGS 66|-7.33142024775275|-26.8547347397694|26M|516033.147116551|9189610.03464806
Bessel 1841|14.3845934585824|169.599431922321|59P|349010.82547103|1590566.17248307
Hough|-73.5905477131022|177.199081842649|60C|506277.734217249|1833555.22256906
Fischer 1968|-2.68544485105829|79.3817002460973|44M|320095.84750932|9703055.5230926
Clarke 1866|-30.5073958458502|-46.0901047080719|23J|395395.675865987|6624665.94069665
Modified Fischer 1960|-63.3249904908332|41.0527349079867|37E|602805.818489815|2976531.25053698
Clarke 1866|-1.61003780386984|-117.034684535432|11M|496141.951029954|9822053.3213996
WGS-72|22.2783220833026|-69.2474982047965|19Q|474503.127829063|2463654.08162659
Everest 1830 India|75.4062085839801|53.3845653068144|39X|567039.69438561|8369335.58218084
Fischer 1960 Mercury |-75.2950782667526|-178.158104283004|1C|467186.901308802|1643107.86498181
Modified Fischer 1960|-40.2912721838004|3.90237217008689|31G|576698.669833549|5539507.07929589
Everest 1830 India|12.5048339573667|164.721599566687|58P|469755.161628261|1382284.32549215
International|-70.861115582977|144.444757796835|55D|406518.067588682|2135911.86041503
Australian National|-39.0087760164844|128.573686310582|52H|463089.104976199|5682148.0829442
South American 1969|73.807142396556|-115.251875625078|11X|554408.057902347|8191348.5456761
Fischer 1960 Mercury |51.8498991358998|175.293568765728|60U|382464.763738667|5745750.1017286
International|-73.9340855053218|74.7425982819513|43C|492048.415878299|1795070.05808418
GRS 1967|41.5629961721581|-136.965130682639|8T|336139.286774015|4603138.53579646
Bessel 1841|-59.6710831240381|13.3001963122808|33E|404268.047419541|3384685.0169031
International|-33.2705816307066|116.401056326882|50H|444217.744224665|6318499.28092214
Modified Everest|-7.27009314416794|61.8252878394142|41M|370336.013272432|9196284.46072599
Australian National|32.9943063348287|102.666457327905|48S|281970.654886131|3653087.01761561
WGS 66|11.1567598395545|-128.612450268711|9P|542315.247328112|1233341.57263956
Bessel 1841|5.10114272125|-172.925883146935|2N|286525.515725696|564109.463758122
Helmert 1906|41.6627475008016|-15.4890406084613|28T|459285.217250389|4612497.02870654
South American 1969|-36.7445764915712|-78.6394456850302|17H|710751.157606417|5930849.76011075
Fischer 1968|72.8733459275791|-122.46795335476|10X|517487.891683684|8086448.58318298
Fischer 1968|44.0196798232483|-117.856778278222|11T|431331.056764627|4874428.84101119
GRS 1967|-43.7030177642338|-48.5089088709353|22G|700718.230928293|5158079.09345144
Bessel 1841|-26.8135758426241|94.4792485550086|46J|646997.56360082|7033647.16480529
Hough|53.9301429854063|-146.870663259846|6U|508492.405700046|5975774.51019271
Everest 1830 India|-47.8611775619757|40.1835619792767|37G|588511.051331584|4698956.34103352
Modified Fischer 1960|-73.6648488045976|74.0610750467826|43C|470524.793740196|1825093.94174954
South American 1969|61.9610023976897|63.1148689558553|41V|506024.232855557|6869865.28922263
International|43.4698233958116|35.8886648568375|36T|733664.6411131|4817131.52833695
Australian National|-56.3177692882303|-115.243132061643|11E|608665.108609965|3757144.23835995
WGS 66|-73.7207364181313|132.44630394897|53C|420119.155356025|1817398.31465082
Krassovsky|32.2233424859976|76.8582006310517|43S|675106.474070044|3566769.66690917
Hough|43.6310874514881|64.7483362936526|41T|641041.885719619|4832383.47063849
Modified Fischer 1960|-61.2080880296666|102.463480846558|48E|363732.126432108|3211367.70037112
Modified Fischer 1960|5.10963323934364|-107.211356055709|13N|254841.77882497|565206.654431613
South American 1969|-56.2942252885404|-58.1668315969431|21E|427782.753677175|3760539.49115581
Modified Everest|61.8742450583816|30.7385520424077|36V|381098.876467544|6861555.44452573
Hough|56.3836532184208|171.061522099043|59V|503799.017756851|6248805.32104046
WGS-84|52.1044166549965|-23.9891611068262|27U|295292.996206518|5776867.36485757
Modified Everest|-34.0592932973628|-122.764773834357|10H|521704.51684377|6231557.59718691
Clarke 1880|9.06980525867839|-172.367930033708|2P|349657.78031744|1002761.43802815
GRS 1967|18.5919153826264|122.746400734063|51Q|473244.145736401|2055700.62775077
GRS 1967|-62.1673774104218|5.13268599528686|31E|611076.958202436|3105321.16786896
South American 1969|-33.1494050115219|-169.556576809933|2H|634617.121837616|6331209.72983344
Fischer 1968|39.5210797761432|12.1665988564939|33S|256441.044309833|4378450.24836053
Bessel 1841 Nambia |60.7885385490899|132.042267868838|53V|339021.421716668|6742246.91430738
Clarke 1880|43.3208155437471|51.946165210592|39T|576721.455931461|4796567.54046122
Helmert 1906|-24.9694098171367|84.9550430201676|45J|293562.493836199|7236854.16780551
WGS 66|-9.46161423400288|-158.273072236867|4L|579797.801248827|8954028.0301497
Modified Fischer 1960|40.5372777346581|-74.3079451901248|18T|558608.379282159|4487638.64005147
International|-2.26356759019897|-31.8344630054278|25M|629608.63761131|9749751.66495352
Fischer 1960 Mercury |-72.2614528544406|119.534014068318|50C|586149.146495398|1980049.67234961
Bessel 1841|77.5023029466433|131.350470486375|52X|556759.076725438|8603029.63594708
Clarke 1880|-53.4604671533592|-34.8758926958529|25F|375451.474968435|4075175.39221818
Bessel 1841|-79.3397083073356|144.276180396578|55C|443776.855244779|1191757.83070873
Bessel 1841 Nambia |49.0974326589454|89.3121608972213|45U|668762.430803155|5440379.52173242
South American 1969|74.4645700512893|153.603102135067|56X|518030.289104969|8263989.00926495
Airy|-27.2120018194124|-87.1741607137689|16J|482754.713316899|6990277.65594674
Krassovsky|48.7117163587868|28.461064419856|35U|607481.632125278|5396534.23448614
International|40.7293812995294|124.00298895184|51T|584701.468512033|4509278.51047989
Modified Everest|-12.7819361861178|-105.776181287007|13L|415765.742233648|8586958.79870971
Bessel 1841|55.2391798279743|75.19796302144|43U|512586.408586539|6120791.13320645
Modified Airy|-46.921566562964|-101.540332827325|14G|306616.86551945|4800980.60197698
Modified Fischer 1960|17.724839026044|-175.62673930873|1Q|645612.618441054|1960281.89866835
Krassovsky|19.4648592457662|-48.4572007231992|22Q|766950.542989052|2154277.65137338
Australian National|70.1201574662691|-90.2486037538632|15W|604388.919603768|7781657.98304426
Modified Airy|14.0892747017256|-135.590682747916|8P|436243.422626902|1557521.32159313
WGS 60|-33.9370126981898|50.7853151622091|39H|480159.597679272|6244787.62457039
South American 1969|83.329091517088|114.74883082491|50X|470811.083481606|9253799.11086044
Krassovsky|27.5894439325656|178.022588263852|60R|600926.724111736|3052195.76719672
WGS-72|62.6124018645957|-152.843232504822|5V|508045.760930399|6942413.18907581
GRS 1967|35.7120520680791|-172.284906446331|2S|383771.361250814|3952785.90710229
Bessel 1841 Nambia |78.1671960802171|39.4547062231896|37X|510407.489927674|8676243.87720712
Bessel 1841 Nambia |17.2786654888002|-157.873566530507|4Q|619716.648038466|1910573.5761343
WGS 66|5.58953543986964|-46.1432397926129|23N|373378.432581889|617954.984063968
GRS 1980|-79.1662346627871|-137.965652323083|8C|437797.174989476|1209891.22189056
Fischer 1960 Mercury |-34.3474364886817|-68.8559373490841|19H|513249.528092841|6199290.12746289
WGS 60|71.9658915215373|137.868839943863|53W|599095.601148268|7987525.9318378
Clarke 1880|83.8141975744418|153.667842733124|56X|508034.599356588|9307250.04701179
WGS-72|-8.20874367016177|-97.4872858748408|14L|666632.616787222|9092311.1956724
WGS 66|0.7476776507023|178.566129759528|60N|674278.35336832|82672.0696026921
Modified Everest|-74.6998716676132|146.85808008686|55C|495820.469816437|1710769.86532762
Modified Fischer 1960|-57.9214955961981|146.912180271276|55E|494797.677468314|3580004.80867212
Bessel 1841|61.8225976885022|81.0634215842767|44V|503340.73527673|6853698.51960782
Krassovsky|59.7798787426991|28.6869404769118|35V|594712.817108638|6628217.95518085
GRS 1967|83.4496014163334|-99.2350427598224|14X|497006.475020804|9266688.50916806
International|12.5602816534855|90.2090266563485|46P|196697.59184345|1390133.17309262
Krassovsky|37.9142141118883|-154.032450111026|5S|409245.41146596|4196873.7426866
Clarke 1866|-50.1750238384579|-100.757339338565|14F|374513.843059306|4440650.11543203
South American 1969|-52.5636720554095|-69.9087356110228|19F|438401.839553792|4175857.28953485
South American 1969|-10.1255529943583|49.9692909257087|39L|387076.387724463|8880523.43704295
International|38.0229695182947|-147.466835551994|6S|459023.929864415|4208536.18312033
Bessel 1841 Nambia |7.45649671111561|-170.766245887299|2N|525789.901661608|824152.397620362
South American 1969|66.7109698160545|-44.5563241730743|23W|519574.698307027|7399256.13255862
Bessel 1841 Nambia |2.20642294074061|72.5247153742171|43N|224707.851830917|244086.229727224
South American 1969|-30.4009216851484|-108.130503902328|12J|775694.664052742|6633280.8332553
International|70.1803631534128|-45.06634572005|23W|497489.268574147|7786195.93234554
Clarke 1880|-20.5075986329816|-90.0692123798184|15K|805689.724207915|7729799.39980131
Modified Everest|71.5800330954972|159.45799695965|57W|516149.540364133|7941308.14396297
Airy|31.4126302698542|-16.2001882154048|28R|385925.886231874|3475714.86048509
Krassovsky|79.1507223031669|-119.6265991257|11X|444824.848380571|8788190.01866969
WGS-72|23.1356683825182|6.08419531331421|32Q|201393.7381288|2561524.80997276
Everest 1830 India|83.3496123997676|2.92317586567407|31X|499006.856909385|9254401.38364661
Bessel 1841|-69.1444293903902|-88.8580867731769|16D|426193.944376015|2329234.05388112
WGS-72|-3.27954852563762|116.140255987473|50M|404484.090911754|9637466.46985707
Hough|16.2024001845186|-150.545044916175|5Q|762467.491056604|1792881.97694997
South American 1969|6.77014511747912|42.5472202537557|38N|228875.517455995|749026.351383918
Krassovsky|32.8620310383455|-18.9805036434028|27S|688965.393412196|3637863.8897583
Airy|46.7500350281965|-73.8454222525877|18T|588177.121231382|5177656.40631951
Krassovsky|-55.7958919678941|-25.8157950281471|26F|574244.336770508|3815893.78703706
GRS 1967|52.8768541695847|90.6573922412181|46U|342351.093794555|5861162.30316593
GRS 1980|9.85905715999888|-18.4769085913153|27P|776722.586203672|1090873.18344845
Krassovsky|70.1739880015593|4.45113750016787|31W|554926.104441408|7786065.56512699
GRS 1967|6.69850460694468|-145.554186612166|6N|659809.060345206|740657.540771707
Bessel 1841|-78.6539814894128|-33.0168755701951|25C|499629.420012918|1269594.5506556
GRS 1980|-31.9643829858903|-108.46683051504|12J|739398.381204866|6460709.42166475
Fischer 1960 Mercury |-4.03971995978874|-113.400443502335|12M|233471.018584199|9553086.18402409
WGS 66|-14.6181452765399|-58.6106396667203|21L|326520.457923874|8383290.86384354
Bessel 1841 Nambia |42.5236116143582|21.6592063774399|34T|554138.368752377|4707714.91001905
WGS-72|-62.1786021254442|-123.907566201028|10E|452743.780035811|3105594.05220819
Clarke 1880|2.21324887158521|25.0895475924523|35N|287528.730961752|244745.997395303
Hough|80.188760347551|-59.6217698414627|21X|450139.445261854|8903878.26223262
WGS-72|-47.0281479502178|-46.5424862445039|23G|382794.217541861|4790554.80309277
South American 1969|79.731897849524|153.236500083527|56X|504706.321675731|8851702.74539223
WGS 66|15.7447570785022|54.7411139932626|40P|257959.670828063|1741999.23736902
GRS 1980|65.649537280093|120.849790726309|51W|401087.322895615|7282538.54147401
