# NAME

Test2 - Framework for writing test tools that all work together.

# EXPERIMENTAL RELEASE

This is an experimental release. Using this right now is not recommended.

# DESCRIPTION

This package exports all the functions necessary to write and/or verify testing
tools. Using these building blocks you can begin writing test tools very
quickly. You are also provided with tools that help you to test the tools you
write.

# SYNOPSYS

## WRITING A TOOL

The `context()` method is your primary interface into the Test2 framework.

    package My::Ok;
    use Test2 qw/context/;

    our @EXPORT = qw/my_ok/;
    use base 'Exporter';

    # Just like ok() from Test::More
    sub my_ok($;$) {
        my ($bool, $name) = @_;
        my $ctx = context(); # Get a context
        $ctx->ok($bool, $name);
        $ctx->release; # Release the context
        return $bool;
    }

See [Test2::Context](https://metacpan.org/pod/Test2::Context) for a list of methods avabilable on the context object.

## TESTING YOUR TOOLS

The `intercept { ... }` tool lets you temporarily intercept all events
generated by the test system:

    use Test2 qw/intercept/;

    use My::Ok qw/my_ok/;

    my $events = intercept {
        # These events are not displayed
        my_ok(1, "pass");
        my_ok(0, "fail");
    };

    my_ok(@$events == 2, "got 2 events, the pass and the fail");
    my_ok($events->[0]->pass, "first event passed");
    my_ok(!$events->[1]->pass, "second event failed");

# EXPORTS

All exports are optional, you must specify subs to import.

    use Test2 qw/context intercept run_subtest/;

## context(...)

Usage:

- $ctx = context()
- $ctx = context(%params)

The `context()` function will always return the current context to you. If
there is already a context active it will be returned. If there is not an
active context one will be generated. When a context is generated it will
default to using the file and line number where the currently running sub was
called from.

Please see ["CRITICAL DETAILS" in Test2::Context](https://metacpan.org/pod/Test2::Context#CRITICAL-DETAILS) for important rules about what
you can and acannot do with a context once it is obtained.

**Note** This function will throw an exception if you ignore the context object
it returns.

### OPTIONAL PARAMETERS

All parameters to `context` are optional.

- level => $int

    If you must obtain a context in a sub deper than your entry point you can use
    this to tell it how many EXTRA stack frames to look back. If this option is not
    provided the default of `0` is used.

        sub third_party_tool {
            my $sub = shift;
            ... # Does not obtain a context
            $sub->();
            ...
        }

        third_party_tool(sub {
            my $ctx = context(level => 1);
            ...
            $ctx->release;
        });

- wrapped => $int

    Use this if you need to write your own tool that wraps a call to `context()`
    with the intent that it should return a context object.

        sub my_context {
            my %params = ( wrapped => 0, @_ );
            $params{wrapped}++;
            my $ctx = context(%params);
            ...
            return $ctx;
        }

        sub my_tool {
            my $ctx = my_context();
            ...
            $ctx->release;
        }

    If you do not do this than tools you call that also check for a context will
    notice that the context they grabbed was created at the same stack depth, which
    will trigger protective measures that warn you and destroy the existing
    context.

- stack => $stack

    Normally `context()` looks at the global hub stack initialized in
    [Test2::Global](https://metacpan.org/pod/Test2::Global). If you are maintaining your own [Test2::Context::Stack](https://metacpan.org/pod/Test2::Context::Stack)
    instance you may pass it in to be used instead of the global one.

- hub => $hub

    Use this parameter if you want to onbtain the context for a specific hub
    instead of whatever one happens to be at the top of the stack.

- on\_init => sub { ... }

    This lets you provide a callback sub that will be called **ONLY** if your call
    to c&lt;context()> generated a new context. The callback **WILL NOT** be called if
    `context()` is returning an existing context. The only argument passed into
    the callback will be the context object itself.

        sub foo {
            my $ctx = context(on_init => sub { 'will run' });

            my $inner = sub {
                # This callback is not run since we are getting the existing
                # context from our parent sub.
                my $ctx = context(on_init => sub { 'will NOT run' });
                $ctx->release;
            }
            $inner->();

            $ctx->release;
        }

- on\_release => sub { ... }

    This lets you provide a callback sub that will be called when the context
    instance is released. This callback will be added to the returned context even
    if an existing context is returned. If multiple calls to context add callbacks
    then all will be called in reverse order when the context is finally released.

        sub foo {
            my $ctx = context(on_release => sub { 'will run second' });

            my $inner = sub {
                my $ctx = context(on_release => sub { 'will run first' });

                # Neither callback runs on this release
                $ctx->release;
            }
            $inner->();

            # Both callbacks run here.
            $ctx->release;
        }

## release($;$)

Usage:

- release $ctx;
- release $ctx, ...;

This is intended as a shortcut that lets you release your context and return a
value in one statement. This function will get your context, and an optional
return value. It will release your context, then return your value. Scalar
context is always assumed.

    sub tool {
        my $ctx = context();
        ...

        return release $ctx, 1;
    }

This tool is most useful when you want to return the value you get from calling
a function that needs to see the current context:

    my $ctx = context();
    my $out = some_tool(...);
    $ctx->release;
    return $out;

We can combine the last 3 lines of the above like so:

    my $ctx = context();
    release $ctx, some_tool(...);

## intercept(&)

Usage:

    my $events = intercept {
        ok(1, "pass");
        ok(0, "fail");
        ...
    };

This function takes a codeblock as its only argument, and it has a prototype.
It will execute the codeblock, intercepting any generated events in the
process. It will return an array reference with all the generated event
objects. All events should be subclasses of [Test2::Event](https://metacpan.org/pod/Test2::Event).

This is a very low-level subtest tool. This is useful for writing tools which
procude subtests. This is not intended for people simply writing tests.

## run\_subtest(...)

Usage:

    run_subtest($NAME, \&CODE, $BUFFERED, @ARGS)

This will run the provided codeblock with the args in `@args`. This codeblock
will be run as a subtest. A subtest is an isolated test state that is condensed
into a single [Test2::Event::Subtest](https://metacpan.org/pod/Test2::Event::Subtest) event, which contains all events
generated inside the subtest.

### ARGUMENTS:

- $NAME

    The name of the subtest.

- \\&CODE

    The code to run inside the subtest.

- $BUFFERED

    If this is true then the subtest will be buffered. In a buffered subtest the
    child events are hidden from the formatter, the formatter will only recieve the
    final [Test2:Event::Subtest](Test2:Event::Subtest) event. In an unbuffered subtest the formatter
    will see all events as they happen, as well as the final one.

- @ARGS

    Any extra arguments you want passed into the subtest code.

# OTHER EXAMPLES

See the `/Examples/` directory included in this distribution.

# SEE ALSO

[Test2::Context](https://metacpan.org/pod/Test2::Context) - Detailed documentation of the context object.

[Test2::Global](https://metacpan.org/pod/Test2::Global) - Interface to global state. This is where to look if you need
a tool to produce a global effect.

[Test2::IPC](https://metacpan.org/pod/Test2::IPC) - The IPC system used for threading/fork support.

[Test2::Formatter](https://metacpan.org/pod/Test2::Formatter) - Formatters such as TAP live here.

[Test2::Event](https://metacpan.org/pod/Test2::Event) - Events live in this namespace.

[Test2::Hub](https://metacpan.org/pod/Test2::Hub) - All events eventually funnel through a hub. Custom hubs are how
`intercept()` and `run_subtest()` are implemented.

# SOURCE

The source code repository for Test2 can be found at
`http://github.com/Test-More/Test2/`.

# MAINTAINERS

- Chad Granum &lt;exodist@cpan.org>

# AUTHORS

- Chad Granum &lt;exodist@cpan.org>

# COPYRIGHT

Copyright 2015 Chad Granum &lt;exodist7@gmail.com>.

This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

See `http://dev.perl.org/licenses/`
