package DBIx::QuickDB::Driver;
use strict;
use warnings;

our $VERSION = '0.000004';

use Carp qw/croak confess/;
use File::Path qw/remove_tree/;
use POSIX ":sys_wait_h";
use Scalar::Util qw/blessed/;
use Time::HiRes qw/sleep/;

use DBIx::QuickDB::Util::HashBase qw{
    pid -root_pid log_file
    -dir
    -_cleanup
    -autostop -autostart
    verbose
    -_log_id
    username
    password
};

sub viable { (0, "socket() is not implemented for the " . $_[0]->name . " driver") }

sub socket         { confess "socket() is not implemented for the " . $_[0]->name . " driver" }
sub load_sql       { confess "load_sql() is not implemented for the " . $_[0]->name . " driver" }
sub bootstrap      { confess "bootstrap() is not implemented for the " . $_[0]->name . " driver" }
sub connect_string { confess "connect_string() is not implemented for the " . $_[0]->name . " driver" }
sub start_command  { confess "start_command() is not implemented for the " . $_[0]->name . " driver" }
sub shell_command  { confess "shell_command() is not implemented for the " . $_[0]->name . " driver" }

sub name {
    my $in = shift;
    my $type = blessed($in) || $in;

    $in =~ s/^DBIx::QuickDB::Driver:://;

    return $in;
}

sub init {
    my $self = shift;

    confess "'dir' is a required attribute" unless $self->{+DIR};

    $self->{+ROOT_PID} = $$;
    $self->{+_CLEANUP} = delete $self->{cleanup};

    $self->{+USERNAME} = '' unless defined $self->{+USERNAME};
    $self->{+PASSWORD} = '' unless defined $self->{+PASSWORD};

    return;
}

sub run_command {
    my $self = shift;
    my ($cmd, $params) = @_;
    my $pid = fork();
    croak "Could not fork" unless defined $pid;

    my $no_log = $self->{+VERBOSE} || $params->{no_log} || $ENV{DB_VERBOSE};
    my $log_file = $no_log ? undef : $self->{+DIR} . "/cmd-log-" . $self->{+_LOG_ID}++;

    if ($pid) {
        return ($pid, $log_file) if $params->{no_wait};
        local $?;
        my $ret = waitpid($pid, 0);
        my $exit = $?;
        die "waitpid returned $ret" unless $ret == $pid;

        return unless $exit;

        my $log = "";
        unless ($no_log) {
            open(my $fh, '<', $log_file) or warn "Failed to open log: $!";
            $log = eval { join "" => <$fh> };
        }
        croak "Failed to run command '" . join(' ' => @$cmd) . "' ($exit)\n$log";
    }

    unless ($no_log) {
        open(my $log, '>', $log_file) or die "Could not open log file ($log_file): $!";
        close(STDOUT);
        open(STDOUT, '>&', $log);
        close(STDERR);
        open(STDERR, '>&', $log);
    }

    if (my $file = $params->{stdin}) {
        close(STDIN);
        open(STDIN, '<', $file) or die "Could not open new STDIN ($file): $!";
    }

    exec(@$cmd);
}

sub should_cleanup { shift->{+_CLEANUP} }

sub cleanup {
    my $self = shift;
    remove_tree($self->{+DIR}, {safe => 1});
    return;
}

sub connect {
    my $self = shift;
    my ($db_name, %params) = @_;

    %params = (AutoCommit => 1) unless @_ > 1;

    my $cstring = $self->connect_string($db_name);

    require DBI;
    return DBI->connect($cstring, $self->username, $self->password, \%params);
}

sub start {
    my $self = shift;
    my @args = @_;

    my $dir = $self->{+DIR};
    my $socket = $self->socket;

    return if $self->{+PID} || -S $socket;

    my ($pid, $log_file) = $self->run_command([$self->start_command, @args], {no_wait => 1});

    my $start = time;
    until (-S $socket) {
        my $waited = time - $start;
        my $dump = 0;

        if ($waited > 10) {
            kill('QUIT', $pid);
            waitpid($pid, 0);
            $dump = "Timeout waiting for server:";
        }

        if (waitpid($pid, WNOHANG) == $pid) {
            $dump = "Server failed to start ($?):"
        }

        if ($dump) {
            open(my $fh, '<', $log_file) or warn "Failed to open log: $!";
            my $data = eval { join "" => <$fh> };
            confess "$dump\n$data\nAborting";
        }

        sleep 0.01;
    }

    $self->{+LOG_FILE} = $log_file;
    $self->{+PID}      = $pid;

    return;
}

sub stop {
    my $self = shift;

    my $pid = $self->{+PID} or return;

    local $?;
    kill('TERM', $pid);
    my $ret = waitpid($pid, 0);
    my $exit = $?;
    die "waitpid returned $ret (expected $pid)" unless $ret == $pid;

    if ($exit) {
        my $name = $self->name;
        my $msg = "";
        if (my $lf = $self->{+LOG_FILE}) {
            if (open(my $fh, '<', $lf)) {
                $msg = "\n" . join "" => <$fh>;
            }
            else {
                $msg = "\nCould not open $name log file '$lf': $!";
            }
        }
        warn "$name exited badly ($exit)$msg";
    }

    delete $self->{+LOG_FILE};
    delete $self->{+PID};

    return;
}

sub shell {
    my $self = shift;
    my ($db_name) = @_;
    $db_name = 'quickdb' unless defined $db_name;

    system($self->shell_command($db_name));
}

sub DESTROY {
    my $self = shift;
    return unless $self->{+ROOT_PID} && $self->{+ROOT_PID} == $$;

    $self->stop    if $self->{+AUTOSTOP} || $self->{+_CLEANUP};
    $self->cleanup if $self->{+_CLEANUP};

    return;
}

1;

__END__


=pod

=encoding UTF-8

=head1 NAME

DBIx::QuickDB::Driver - Base class for DBIx::QuickDB drivers.

=head1 DESCRIPTION

Base class for DBIx::QuickDB drivers.

=head1 SYNOPSIS

    package DBIx::QuickDB::Driver::MyDriver;
    use strict;
    use warnings;

    use parent 'DBIx::QuickDB::Driver';

    use DBIx::QuickDB::Util::HashBase qw{ ... };

    sub viable { ... ? 1 : (0, "This driver will not work because ...") }

    sub init {
        my $self = shift;

        $self->SUPER::init();

        ...
    }

    # Methods most drivers should implement

    sub socket         { ... }
    sub load_sql       { ... }
    sub bootstrap      { ... }
    sub connect_string { ... }
    sub start_command  { ... }
    sub shell_command  { ... }

    1;

=head1 METHODS PROVIDED HERE

=item $bool = $db->autostart

True if this db was created with 'autostart' requested.

=item $bool = $db->autostop

True if this db was created with 'autostop' requested.

=item $db->cleanup

This will completely delete the database directory. B<BE CAREFUL>.

=item $dbh = $db->connect()

=item $dbh = $db->connect($db_name)

=item $dbh = $db->connect($db_name, %connect_params)

Connect to the database server. If no C<%connect_params> are specified then
C<< (AutoCommit => 1) >> will be used.

Behavior for an undef (or omitted) C<$db_name> is driver specific.

This will use the username in C<username()> and the password in C<password()>.
The connection string is defined by C<connect_string()> which must be overriden
in each driver subclass.

=item $path = $db->dir

Get the path to the database directory.

=item $db->init

This is called automatically during object construction. You B<SHOULD NOT> call
this directly, except in a subclass which overrides C<init()>.

=item $path = $db->log_file

If the database is running this will point to the log file. If the database is
not yet running, or has been stopped, this will be undef.

=item $driver_name = $db->name

Get the short name of the driver ('DBIx::QuickDB::Driver::' has been stripped).

=item $pw = $db->password

=item $db->password($pw)

Get/Set the password to use when calling C<connect()>.

=item $pid = $db->pid

=item $db->pid($pid)

If the server is running then this will have the pid. If the server is stopped
this will be undef.

B<NOTE:> This will also be undef if the server is running independantly of this
object, if the server is running, but this is undef, it means another
object/process is in control of it.

=item $pid = $db->root_pid

This should contain the original pid of the process in which the instance was
created.

=item $db->run_command(\@cmd)

=item $db->run_command(\@cmd, \%params)

=item ($pid, $logfile) = $db->run_command(\@cmd, {no_wait => 1})

This will execute the command specified in C<@cmd>. If the command fails an
exception will be thrown. By default all output will be captured into log files
and ignored. If the command fails the output will be attached to the exception.
Normally this will block until the command exits. if C<verbose()> is set then
all output is always shown.

Normally there is no return value. If the 'no_wait' param is specified then
the command will be run non-blocking and the pid and log file will be returned.

Allowed params:

=over 4

=item no_log => bool

Show the output in realtime, do not redirect it.

=item no_wait => bool

Do not block, instead return the pid and log file to use later.

=item stdin => path_to_file

Run the command with the specified file is input.

=back

=item $db->shell

Launch a database shell. This depends on the C<shell_command> method, which
drivers should provide. Not all driver may support this.

=item $bool = $db->should_cleanup

True if the instance was created with the 'cleanup' specification. If this is
true then the database directory will be deleted when the program ends.

=item $db->start

Start the database. Most drivers will make this a no-op if the db is already
running.

=item $db->stop

Stop the database. Most drivers will make this a no-op if the db is already
stopped.

=item $user = $db->username

=item $db->username($user)

Get/set the username to use in C<connect()>.

=item $bool = $db->verbose

=item $db->verbose($bool)

If this is true then all output from C<run_command> will be shown at all times.

=item $db->DESTROY

Used to stop the server and delete the data dir (if desired) when the program
exits.

=back

=head1 METHODS SUBCLASSES SHOULD PROVIDE

=over

=item ($bool, $why) = $db->viable()

=item ($bool, $why) = $db->viable(\%spec)

This should check if it is possible to launch this db type on the current
system with the given spec.

See L<DBIx::QuickDB/"SPEC HASH"> for what might be in C<%spec>.

The first return value is a simple boolean, true if the driver is viable, false
if it is not. The second value should be an explanation as to why the driver is
not viable (in cases where it is not).

=item $socket = $db->socket()

Unix Socket used to communicate with the db. If the db type does not use
sockets (such as SQLite) then this can be skipped. B<NOTE:> If you skip this
you will need to override C<stop()> and C<start()> to account for it. See
L<DBIx::QuickDB::Driver::SQLite> for an example.

=item $db->load_sql($db_name, $file)

Load the specified sql file into the specified db. It is possible that
C<$db_name> will be undef in some drivers.

=item $db->bootstrap()

Initialize the database server and create the 'quickdb' database.

=item $string = $db->connect_string()

=item $string $db->connect_string($db_name)

String to pass into C<< DBI->connect >>.

Example: C<"dbi:Pg:dbname=$db_name;host=$socket">

=item @cmd = $db->start_command()

Command used to start the server.

=item @cmd = $db->shell_command()

Command used to launch a shell into the database.

=back

=head1 SOURCE

The source code repository for DBIx-QuickDB can be found at
F<https://github.com/exodist/DBIx-QuickDB/>.

=head1 MAINTAINERS

=over 4

=item Chad Granum E<lt>exodist@cpan.orgE<gt>

=back

=head1 AUTHORS

=over 4

=item Chad Granum E<lt>exodist@cpan.orgE<gt>

=back

=head1 COPYRIGHT

Copyright 2018 Chad Granum E<lt>exodist7@gmail.comE<gt>.

This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

See F<http://dev.perl.org/licenses/>

=cut
