/*
 * IBPerl -- a Perl 5 extension for SQL RDBMS programming
 *   with InterBase client library.
 *
 * IBPerl.xs
 *
 * Copyright (c) 1996-1999 Bill Karwin
 *
 * This is unsupported, free software.  Neither Bill Karwin,
 * InterBase Software Corporation, nor Inprise Corporation are
 * responsible for damages resulting from the use or misuse of
 * this software.  Test early, test well, test often.
 *
 * You may distribute under the terms of either the GNU General
 * Public License or the Artistic License, as specified in the
 * Perl README file.
 */

#include "EXTERN.h"
#include "perl.h"
#include "XSUB.h"
#include "IBPerl.h"

#ifdef  __BCPLUSPLUS__
extern "C" __declspec(dllexport) void XS_IBPerl__Connection_IB_connect(CV *);
extern "C" __declspec(dllexport) void XS_IBPerl__Connection_IB_create(CV *);
extern "C" __declspec(dllexport) void XS_IBPerl__Connection_IB_disconnect(CV *);
extern "C" __declspec(dllexport) void XS_IBPerl__Transaction_IB_start_transaction(CV *);
extern "C" __declspec(dllexport) void XS_IBPerl__Transaction_IB_commit_transaction(CV *);
extern "C" __declspec(dllexport) void XS_IBPerl__Transaction_IB_rollback_transaction(CV *);
extern "C" __declspec(dllexport) void XS_IBPerl__Statement_IB_prepare(CV *);
extern "C" __declspec(dllexport) void XS_IBPerl__Statement_IB_execute(CV *);
extern "C" __declspec(dllexport) void XS_IBPerl__Statement_IB_open(CV *);
extern "C" __declspec(dllexport) void XS_IBPerl__Statement_IB_fetch(CV *);
extern "C" __declspec(dllexport) void XS_IBPerl__Statement_IB_update_current(CV *);
extern "C" __declspec(dllexport) void XS_IBPerl__Statement_IB_delete_current(CV *);
extern "C" __declspec(dllexport) void XS_IBPerl__Statement_IB_close(CV *);
extern "C" __declspec(dllexport) void XS_boot_IBPerl(CV *);
#endif /* __BCPLUSPLUS__ */

/*
 * Error_String() - 
 */
static void
Error_String(HV *hash, char *error_string)
{
    if (hash != NULL
	&& error_string != NULL
	&& *error_string != '\0')
    {
	hv_store(hash, "Error", 5,
	    newSVpv(error_string, strlen(error_string)), 0);
    }
    return;
}

/*
 * Error_Vector() - Given an InterBase status vector,
 * build a string with the full error message.
 */
static char *
Error_Vector(long *status_vector)
{
    char *p;
    long sql_code;
#ifdef IBPERL_THREAD_SAFE
    char *p_base;

    /*
     * XXX is this a memory leak? Check into it.
     * Consider instead using Perl's own XS interface
     * for allocating objects that are subject to
     * garbage collection.  That's so cool.
     */
    if ((p_base = malloc(1024 * sizeof(char))) == NULL)
    {
	return NULL;
    }

#else
    /* 
     * XXX This is an experimental static array
     * for the error buffer.  This isn't thread-safe
     * but it will reduce the risk of a memory leak.
     */
    static char p_base[1024];
#endif /* IBPERL_THREAD_SAFE */

#if 0
    /*
     * XXX Here's some code notes to myself for version 0.8.
     * I'm too close to releasing 0.7 to rock the boat.
     */
    /*
     * Allocate a "mortal" scalar so it'll be garbage-collected
     * and we won't have to worry about memory leaks.
     * XXX consider doing this for all malloc'ed scalars in
     * IBPerl...
     */
     p_base = sv_newmortal();
    /*
     * Use it like an SV * later in the code.  That is,
     * use sv_catpv(p_base, char *) to append strings to
     * this mortal scalar.
     */
#endif

    p = p_base;

#ifdef BUG_10378_FIXED
    /*
     * First the SQL error message.
     *
     * XXX There seems to be a problem with isc_sql_interprete()
     * causing the IB client to crash on Solaris when reporting
     * some error messages.  I have logged this as bug #10378.
     * Until it is fixed, I'm suppressing the use of isc_sql_interprete().
     */
    if ((sql_code = isc_sqlcode(status_vector)) != 0)
    {
	isc_sql_interprete(sql_code, p, 1024);
	while (*p) p++;
	*p++ = '\n';
	*p++ = '-';
    }
#endif

    while (isc_interprete(p, &status_vector))
    {
	while (*p) p++;
	*p++ = '\n';
	*p++ = '-';
    }
    *--p = '\0'; /* null-terminate buffer */

    return p_base;
}

#ifdef IBPERL_DEBUG
static void
IB_dump_XSQLDA(XSQLDA *da)
{
int i;
XSQLVAR *var;

    if (!da) return;

    printf("XSQLDA\n VERSION 0x%x = %d\n SQLDAID 0x%x = \"%.8s\"\n SQLDABC 0x%x = %d\n SQLN 0x%x = %d\n SQLD 0x%x = %d\n",
	&(da->version), da->version,
	&(da->sqldaid), da->sqldaid? da->sqldaid: "null",
	&(da->sqldabc), da->sqldabc,
	&(da->sqln), da->sqln,
	&(da->sqld), da->sqld
    );

    for (i = 0, var = &da->sqlvar[i]; i < da->sqld; ++i, ++var)
    {
    unsigned short dtype;

	printf(" XSQLVAR[%d]: 0x%x\n", i, var); 

	printf("  SQLSCALE 0x%x = %d\n", &(var->sqlscale), var->sqlscale); 
	printf("  SQLLEN 0x%x = %d\n", &(var->sqllen), var->sqllen); 
	printf("  SQLTYPE 0x%x = ", &(var->sqltype));
	dtype = var->sqltype & ~1;
	switch (dtype)
	{
	case SQL_VARYING:
	    printf("SQL_VARYING\n");
	    printf("  SQLDATA 0x%x = ", &(var->sqldata));
	    if (var->sqldata)
	    {
		VARY *vary = (VARY *) var->sqldata;
		printf("'%.*s'\n", vary->vary_length, vary->vary_string);
	    } else
		printf("NULL\n");
	    break;
	case SQL_TEXT:
	    printf("SQL_TEXT\n");
	    printf("  SQLDATA 0x%x = ", &(var->sqldata));
	    if (var->sqldata)
		printf("'%s'\n", var->sqldata);
	    else
		printf("NULL\n");
	    break;
	case SQL_SHORT:
	    printf("SQL_SHORT\n");
	    printf("  SQLDATA 0x%x = ", &(var->sqldata));
	    if (var->sqldata)
		printf("%d\n", *(short *) (var->sqldata));
	    else
		printf("NULL\n");
	    break;
	case SQL_LONG:
	    printf("SQL_LONG\n");
	    printf("  SQLDATA 0x%x = ", &(var->sqldata));
	    if (var->sqldata)
		printf("%d\n", *(long *) (var->sqldata));
	    else
		printf("NULL\n");
	    break;
	case SQL_FLOAT:
	    printf("SQL_FLOAT\n");
	    printf("  SQLDATA 0x%x = ", &(var->sqldata));
	    if (var->sqldata)
		printf("%f\n", *(float *) (var->sqldata));
	    else
		printf("NULL\n");
	    break;
	case SQL_DOUBLE:
	    printf("SQL_DOUBLE\n");
	    printf("  SQLDATA 0x%x = ", &(var->sqldata));
	    if (var->sqldata)
		printf("%g\n", *(double *) (var->sqldata));
	    else
		printf("NULL\n");
	    break;
	case SQL_DATE:
	    printf("SQL_DATE\n");
	    printf("  SQLDATA 0x%x = 0x%x\n", &(var->sqldata), var->sqldata);
	    break;
	case SQL_BLOB:
	    printf("SQL_BLOB\n");
	    printf("  SQLSUBTYPE 0x%x = %d\n", &(var->sqlsubtype), var->sqlsubtype); 
	    printf("  SQLDATA 0x%x = 0x%x\n", &(var->sqldata), var->sqldata); 
	    break;
	case SQL_ARRAY:
	    printf("SQL_ARRAY\n");
	    printf("  SQLDATA 0x%x = 0x%x\n", &(var->sqldata), var->sqldata); 
	    break;
	case SQL_QUAD:
	    printf("SQL_QUAD\n");
	    printf("  SQLDATA 0x%x = 0x%x\n", &(var->sqldata), var->sqldata); 
	    break;
	default:
	    printf("unknown %d\n", dtype);
	    break;
	}

	printf("  SQLIND 0x%x = 0x%x\n", &(var->sqlind), var->sqlind);
	printf("  SQLNAME 0x%x = \"%.31s\" (%d)\n",
	    &(var->sqlname),
	    var->sqlname? var->sqlname: "NULL", var->sqlname_length); 
	printf("  RELNAME 0x%x = \"%.31s\" (%d)\n",
	    &(var->relname),
	    var->relname? var->relname: "NULL", var->relname_length); 
	printf("  OWNNAME 0x%x = \"%.31s\" (%d)\n",
	    &(var->ownname),
	    var->ownname? var->ownname: "NULL", var->ownname_length); 
	printf("  ALIASNAME 0x%x = \"%.31s\" (%d)\n",
	    &(var->aliasname),
	    var->aliasname? var->aliasname: "NULL", var->aliasname_length); 
	fflush(stdout);
    }
}
#endif

static int
IB_fill_isqlda(AV *parms, ST_info *st, HV *sthash)
{
int RETVAL;
int i;
unsigned int len;
XSQLVAR *ivar;
SV **scalar;

    for (i=0, ivar = st->in_sqlda->sqlvar, scalar = av_fetch(parms, i, 0);
	i < st->in_sqlda->sqld;
	i++, ivar++, scalar = av_fetch(parms, i, 0))
    {
	switch (ivar->sqltype & ~1)
	{
	case SQL_VARYING:
	    len = SvCUR(*scalar);
	    if ((ivar->sqldata = (char *)
		malloc( sizeof(char) * len + sizeof(short) + 1))
		== NULL)
	    {
		fprintf(stderr, "Cannot allocate buffer for TEXT input parameter #%d\n", i);
		RETVAL = FAILURE;
		goto end;
	    }

	    /* The first word of VARCHAR sqldata is the length */
	    *((short *) ivar->sqldata) = len;
	    memcpy(ivar->sqldata + sizeof(short), SvPV(*scalar, na), len);
	    ivar->sqldata[len + sizeof(short)] = '\0';

	    ivar->sqllen = len + sizeof(short);
	    break;

	case SQL_TEXT:
	    len = SvCUR(*scalar);
	    if ((ivar->sqldata = (char *)
		malloc(sizeof(char) * len + 1)) == NULL)
	    {
		fprintf(stderr, "Cannot allocate buffer for TEXT input parameter #%d\n", i);
		RETVAL = FAILURE;
		goto end;
	    }

	    memcpy(ivar->sqldata, SvPV(*scalar, na), len);
	    ivar->sqldata[len] = '\0';

	    ivar->sqllen = len;
	    break;

	case SQL_SHORT:
	    if ((ivar->sqldata = (char *) malloc(sizeof(short))) == NULL)
	    {
		fprintf(stderr, "Cannot allocate buffer for SHORT input parameter #%d\n", i);
		RETVAL = FAILURE;
		goto end;
	    }
	    *(short *) (ivar->sqldata) = SvIV(*scalar);
	    break;

	case SQL_LONG:
	    if ((ivar->sqldata = (char *) malloc(sizeof(long))) == NULL)
	    {
		fprintf(stderr, "Cannot allocate buffer for LONG input parameter #%d\n", i);
		RETVAL = FAILURE;
		goto end;
	    }
	    *(long *) (ivar->sqldata) = SvIV(*scalar);
	    break;

	case SQL_FLOAT:
	    if ((ivar->sqldata = (char *) malloc(sizeof(float))) == NULL)
	    {
		fprintf(stderr, "Cannot allocate buffer for FLOAT input parameter #%d\n", i);
		RETVAL = FAILURE;
		goto end;
	    }
	    *(float *) (ivar->sqldata) = SvNV(*scalar);
	    break;

	case SQL_DOUBLE:
	    if ((ivar->sqldata = (char *) malloc(sizeof(double))) == NULL)
	    {
		fprintf(stderr, "Cannot allocate buffer for DOUBLE input parameter #%d\n", i);
		RETVAL = FAILURE;
		goto end;
	    }
	    *(double *) (ivar->sqldata) = SvNV(*scalar);
	    break;

	case SQL_DATE:
	{
	char *datestring;

	    /*
	     * Coerce the date literal into a CHAR string, so as
	     * to allow InterBase's internal date-string parsing
	     * to interpret the date.
	     * XXX what this doesn't allow for is to input a list
	     * of elements localtime() style.  Something for later.
	     */
	    ivar->sqltype = SQL_TEXT;
	    datestring = SvPV(*scalar, len);
	    ivar->sqllen = len;
	    if ((ivar->sqldata = (char *)
		malloc( sizeof(char)*ivar->sqllen)) == NULL)
	    {
		fprintf(stderr, "Cannot allocate buffer for DATE input parameter #%d\n", len);
		RETVAL = FAILURE;
		goto end;
	    }
	    memcpy(ivar->sqldata, datestring, len);
	}
	    break;

	case SQL_BLOB:
	{
	isc_blob_handle blob_handle = NULL;
	ISC_QUAD blob_id;
	int blob_stat;
	long total_length, t;
	char *p;

	    if ((total_length = SvCUR(*scalar)) >= MAX_SAFE_BLOB_LENGTH)
	    {
		RETVAL = FAILURE;
		goto end;
	    }
	    if ((ivar->sqldata = (char *) malloc(sizeof(ISC_QUAD))) == NULL)
	    {
		fprintf(stderr, "Cannot allocate buffer for Blob input parameter #%d\n", i);
		RETVAL = FAILURE;
		goto end;
	    }

	    /*
	     * Create a Blob ID and handle
	     */
	    isc_create_blob(
		st->status,
		&(st->trans->db->handle),
		&(st->trans->handle),
		&blob_handle,            /* set by this function */
		&blob_id                 /* set by this function */
	    );
	    ERRCHECK(st->status, sthash);

	    t = total_length;
	    p = SvPV(*scalar, na);
		
	    while (1) /* loop on segments */
	    {
	    char blob_segment_buffer[MAX_BLOB_SEGMENT];
	    char *q;

		q = blob_segment_buffer;
		if (ivar->sqlsubtype == isc_bpb_type_stream /* text */)
		{
		    /*
		     * Copy segment, up to and including newline,
		     * or MAX_BLOB_SEGMENT bytes, whichever comes first.
		     */
		    while (q - blob_segment_buffer <= MAX_BLOB_SEGMENT && t > 0)
		    {
			--t;
			if ((*q++ = *p++) == '\n') break;
		    }
		} else /* subtype not text */ {
		    /*
		     * Copy segment, up to MAX_BLOB_SEGMENT bytes.
		     */
		    while (q - blob_segment_buffer <= MAX_BLOB_SEGMENT && t > 0)
		    {
			--t;
			*q++ = *p++;
		    }
		}
		blob_stat = isc_put_segment(
		    st->status,
		    &blob_handle,
		    (unsigned short) (q - blob_segment_buffer),
		    blob_segment_buffer
		);
		ERRCHECK(st->status, sthash);
		if (t <= 0) break;
	    }

	    /*
	     * Clean up after ourselves.
	     */
	    isc_close_blob(st->status, &blob_handle);
	    ERRCHECK(st->status, sthash);

	    memcpy((ISC_QUAD *) ivar->sqldata, &blob_id, sizeof(ISC_QUAD));
	}
	    break;

	case SQL_ARRAY:
#ifdef ARRAY_SUPPORT
	    !!! NOT IMPLEMENTED YET !!!
#endif
	    break;

	default:
	    break;
	}

	/*
	 * NULL indicator
	 */
	ivar->sqlind = &st->isqlind[i];

	if (ivar->sqltype & 1)
	{
	    /*
	     * XXX Future enhancement will allow NULLs to be passed by
	     * scripts, and here's where IBPerl will take care of them.
	     * But for now, I'm not going to support NULL parms.
	     * Why bother entering a parameter if you're just going to
	     * pass a NULL?
	     */
	    *(ivar->sqlind) = 0;
	}
    }
#ifdef IBPERL_DEBUG
    printf("IB_fill_isqlda: Print input XSQLDA:\n");
    IB_dump_XSQLDA(st->in_sqlda);
    printf("IB_fill_isqlda: Print input XSQLDA done.\n");
    fflush(stdout);
#endif
    end:
    return RETVAL;
}


MODULE = IBPerl PACKAGE = IBperl

# VERSIONCHECK: ENABLE

# PROTOTYPES: ENABLE

# ======================================================================


MODULE = IBPerl PACKAGE = IBPerl::Connection

#
# IB_connect () - Connect to a database.
# Return 0 on success, -1 on failure.
# Stores errors and handles in the dbhash.
#
int
IB_connect(ref)
    SV *ref
    CODE:
    {
    HV *dbhash;
    SV **Database, **User, **Password, **Cache, **Charset, **Role;
    char *database, *user, *password, *charset;
    unsigned int database_len, user_len, password_len, charset_len;
    int cache;
    DB_info *db;
    char *dpb;

	if (!SvROK(ref))
	{
	    RETVAL = FAILURE;
	    goto end;
	}
	dbhash = (HV *) SvRV(ref);

	/*
	 * Assume failure: set up a -1 handle that will be
	 * overwritten with a real handle if we succeed.
	 */
	if (hv_store(dbhash, "Handle", 6, newSViv(-1), 0) == NULL)
	{
	    Error_String(dbhash, "IB_connect: cannot return database handle.");
	    RETVAL = FAILURE;
	    goto end;
	}

	/*
	 * Decode the dbhash, get some fields out of it for
	 * database, user, password, cache, charset.
	 */

	if ((Database  = hv_fetch(dbhash, "Connect_String", 14, 0)) == NULL
	    || !SvPOK(*Database))
	    { RETVAL = FAILURE; goto end; }

	if ((User      = hv_fetch(dbhash, "User",      4, 0)) == NULL
	    || !SvPOK(*User))
	    { RETVAL = FAILURE; goto end; }

	if ((Password  = hv_fetch(dbhash, "Password",  8, 0)) == NULL
	    || !SvPOK(*Password))
	    { RETVAL = FAILURE; goto end; }

	if ((Cache     = hv_fetch(dbhash, "Cache",     5, 0)) == NULL
	    || !SvIOK(*Cache))
	    { RETVAL = FAILURE; goto end; }

	if ((Charset     = hv_fetch(dbhash, "Charset", 7, 0)) == NULL
	    || !SvPOK(*Charset))
	    { RETVAL = FAILURE; goto end; }

	/*
	 * Convert HV member references into scalars
	 */
	database  = SvPV(*Database, database_len);
	user      = SvPV(*User, user_len);
	password  = SvPV(*Password, password_len);
	cache     = SvIV(*Cache);
	charset   = SvPV(*Charset, charset_len);

	db = IB_new_DB();
	if (db->id <= 0)
	{
	    RETVAL = FAILURE;
	    goto end;
	}

	/*
	 * Fill the Database Parameter Block
	 * with nifty options.
	 */
	dpb = db->dpb_buffer;
	DPB_FILL( isc_dpb_version1 );

	/*
	 * The number of cache pages.  Zero is a special value meaning,
	 * "default: let the server determine cache".  Different versions
	 * of InterBase server have different defaults.
	 */
	if (cache > 0)
	{
	    /*
	     * Safety check:
	     * Do not allocate a cache buffer greater than 10000 pages,
	     * so we don't exhaust memory inadvertently.
	     */
	    if (cache > 10000) cache = 10000;
	    DPB_FILL( isc_dpb_num_buffers );
	    DPB_FILL_INT( cache );
	}

	/*
	 * The DPB also contains the User and Password
	 * we are connecting as.  This convenience function
	 * adds these parameters into the dpb, and may
	 * expand the memory allocated for the dpb,
	 * if necessary.
	 */
	db->dpb_length = dpb - db->dpb_buffer;
	isc_expand_dpb(
	    &(db->dpb_buffer),
	    (short *) &(db->dpb_length),
	    isc_dpb_user_name, user,
	    isc_dpb_password, password,
	    isc_dpb_lc_ctype, charset,
	    (char *) NULL);

	/* 
	 * Optional parameters; do not die if these aren't set.
	 */
	Role = hv_fetch(dbhash, "Role", 4, 0);

	/*
	 * Role is optional
	 */
	if (Role != NULL && SvPOK(*Role))
	{
	char *role;
	unsigned int role_len;

	    role = SvPV(*Role, role_len);
	    /*
	     * You must compile with a V5 client for
	     * the following code to work.
	     */
#ifdef isc_dpb_sql_role_name
	    isc_expand_dpb(
		&(db->dpb_buffer),
		(short *) &(db->dpb_length),
		isc_dpb_sql_role_name, role,
		(char *) NULL);
#endif
	}

	isc_attach_database(
	    db->status,				/* status vector */
	    0,			/* connect string is null-terminated */
	    database,				/* connect string */
	    (isc_db_handle *) &(db->handle),	/* ref to db handle */
	    db->dpb_length,			/* length of dpb */
	    db->dpb_buffer);			/* connect options */
	ERRCHECK(db->status, dbhash);

	if (hv_store(dbhash, "Handle", 6, newSViv(db->id), 0) == NULL)
	{
	    Error_String(dbhash, "IB_connect: cannot return database handle.");
	    RETVAL = FAILURE;
	}
	else
	{
	    RETVAL = SUCCESS;
	}

	end: ;
    }
    OUTPUT:
    RETVAL

#
# IB_create () - Create a new database and attach
# to the database.  Return 0 on success, -1 on failure.
# Stores errors and handles in the dbhash.
#
int
IB_create(ref)
    SV *ref
    CODE:
    {
    HV *dbhash;
    SV **Database, **User, **Password, **Page_Size, **Charset;
    SV *create_db_stmt;
    isc_db_handle dummy_db_handle;
    isc_tr_handle dummy_tr_handle;
    long status[20];
    DB_info *db;

	if (!SvROK(ref))
	{
	    RETVAL = FAILURE;
	    goto end;
	}
	dbhash = (HV *) SvRV(ref);

	/*
	 * Decode the dbhash, get some fields out of it for
	 * database, user, password, page_size.
	 */

	Database  = hv_fetch(dbhash, "Connect_String", 14, 0);
	User      = hv_fetch(dbhash, "User", 4, 0);
	Password  = hv_fetch(dbhash, "Password", 8, 0);
	Page_Size = hv_fetch(dbhash, "Page_Size", 9, 0);
	Charset   = hv_fetch(dbhash, "Charset", 7, 0);

	/*
	 * This string has the potential to grow large so I am
	 * making use of Perl's functions newSVpv(), sv_catpvn() and
	 * sv_catpvf() to create a string that grows dynamically.
	 * This model allows me to add features in the future like
	 * a multiple-file list and a shadow-file list.
	 */
	create_db_stmt = newSVpv("CREATE DATABASE ", 0);
	if (Database != NULL && SvPOK(*Database))
	{
	    sv_catpv(create_db_stmt, " \"");
	    sv_catsv(create_db_stmt, *Database);
	    sv_catpv(create_db_stmt, "\" ");
	}
	else
	{
	    Error_String(dbhash, "IB_create: mandatory connect string not supplied.");
	    RETVAL = FAILURE;
	    goto end;
	}
	if (Page_Size != NULL
	    && SvIOK(*Page_Size)
	    && SvIV(*Page_Size) >= 1024) /* optional */
	{
	    sv_catpv(create_db_stmt, " PAGE_SIZE ");
	    sv_catsv(create_db_stmt, *Page_Size);
	}
	if (User != NULL && SvPOK(*User)
	    && Password != NULL && SvPOK(*Password)) /* optional */
	{
	    sv_catpv(create_db_stmt, " USER \"");
	    sv_catsv(create_db_stmt, *User);
	    sv_catpv(create_db_stmt, "\" PASSWORD \"");
	    sv_catsv(create_db_stmt, *Password);
	    sv_catpv(create_db_stmt, "\" ");
	}
	if (Charset != NULL && SvPOK(*Charset))
	{
	    sv_catpv(create_db_stmt, " DEFAULT CHARACTER SET ");
	    sv_catsv(create_db_stmt, *Charset);
	}
	sv_catpv(create_db_stmt, "\000"); /* NULL terminate the string */

	dummy_db_handle = (isc_db_handle) NULL;
	dummy_tr_handle = (isc_tr_handle) NULL;
	isc_dsql_execute_immediate(status,
	    (isc_db_handle *) &(dummy_db_handle),
	    (isc_tr_handle *) &dummy_tr_handle,
	    0,
	    SvPV(create_db_stmt, na),
	    IBPERL_DEFAULT_SQL_DIALECT,
	    NULL);
	ERRCHECK(status, dbhash);

	/*
	 * Michael (michael@geocom.ru) says I should comment this out:
	 *
	 * isc_commit_transaction(status, &dummy_tr_handle);
	 * ERRCHECK(status, dbhash);
	 */
#if 0
	isc_detach_database(status, &dummy_db_handle);
	ERRCHECK(status, dbhash);

	RETVAL = IB_connect(dbhash);
#else
	db = IB_new_DB();
	if (db->id <= 0)
	{
	    RETVAL = FAILURE;
	    goto end;
	}

	db->handle = dummy_db_handle;

	if (hv_store(dbhash, "Handle", 6, newSViv(db->id), 0) == NULL)
	{
	    Error_String(dbhash, "IB_create: cannot return database handle.");
	    RETVAL = FAILURE;
	}
	else
	{
	    RETVAL = SUCCESS;
	}
#endif
	end: ;
    }
    OUTPUT:
    RETVAL

#
# IB_disconnect() - Detach from a database.
# Return 1 on error, 0 on success.
#
int
IB_disconnect(ref)
    SV *ref
    CODE:
    {
    HV *dbhash;
    DB_info *db;
    SV **id;

	if (!SvROK(ref))
	{
	    RETVAL = FAILURE;
	    goto end;
	}
	dbhash = (HV *) SvRV(ref);

	if ((id = hv_fetch(dbhash, "Handle", 6, 0)) == NULL || !SvIOK(*id))
	{
	    RETVAL = FAILURE;
	    goto end;
	}

	if (SvIV(*id) < 0)
	{
	    RETVAL = SUCCESS;
	    goto end;
	}

	if ((db = IB_get_DB(SvIV(*id))) == NULL)
	{
	    RETVAL = FAILURE;
	    goto end;
	}

	if (db->inuse != TAKEN)
	{
	    Error_String(dbhash, "IB_disconnect: uninitialized database handle.");
	    RETVAL = FAILURE;
	    goto end;
	}

	isc_detach_database(db->status, (isc_db_handle *) &(db->handle));
	ERRCHECK(db->status, dbhash);
	
	RETVAL = SUCCESS;

	IB_free_DB(db); 

	end: ;
    }
    OUTPUT:
    RETVAL

# ======================================================================

MODULE = IBPerl PACKAGE = IBPerl::Transaction

#
# IB_start_transaction() - start a transaction on the
# current database.  Return 0 on success, -1 on failure.
# Stores errors and handles in the trhash.
#
int
IB_start_transaction(ref)
    SV *ref
    CODE:
    {
#if 0
    /*
     * Options for the future.
     */
    char *mode, *resolution, *isolation;
#endif
    HV *trhash, *dbhash;
    TR_info *tr;
    SV **dbref, **dbhandle;

	if (!SvROK(ref))
	{
	    RETVAL = FAILURE;
	    goto end;
	}
	trhash = (HV *) SvRV(ref);

	/*
	 * Assume failure: set up a -1 handle that will be
	 * overwritten with a real handle if we succeed.
	 */
	if (hv_store(trhash, "Handle", 6, newSViv(-1), 0) == NULL)
	{
	    Error_String(trhash, "IB_start_transaction: cannot return database handle.");
	    RETVAL = FAILURE;
	    goto end;
	}

	if ((dbref = hv_fetch(trhash, "Database", 8, 0)) == NULL
	    || !SvROK(*dbref))
	{
	    RETVAL = FAILURE;
	    goto end;
	}

	dbhash = (HV *) SvRV(*dbref);

	if ((dbhandle = hv_fetch(dbhash, "Handle", 6, 0)) == NULL
	    || !SvIOK(*dbhandle)
	    || SvIV(*dbhandle) == -1)
	{
	    Error_String(trhash, "IB_start_transaction: invalid database handle.");
	    RETVAL = FAILURE;
	    goto end;
	}

	if ((tr = IB_new_TR(SvIV(*dbhandle))) == NULL)
	{
	    Error_String(trhash, "IB_start_transaction: cannot create transaction.");
	    RETVAL = FAILURE;
	    goto end;
	}

	/*
	 * XXX  I will implement different mode, resolution and
	 * isolation level sometime.  Also the RESERVING clause,
	 * needs to be handled.  The USING clause should be
	 * handled by using isc_start_multiple() and passing a
	 * list of dbhandles to the current function.  I'll try
	 * to do that before IBPerl 1.0.
	 */
	isc_start_transaction(tr->status,
	    (isc_tr_handle *) &(tr->handle),
	    1,
	    (isc_db_handle *) &(tr->db->handle),
	    0,
	    NULL);
	ERRCHECK(tr->status, trhash);

	if (hv_store(trhash, "Handle", 6, newSViv(tr->id), 0) == NULL)
	{
	    Error_String(trhash, "IB_start_transaction: cannot return transaction handle.");
	    RETVAL = FAILURE;
	}
	else
	{
	    RETVAL = SUCCESS;
	}

	end: ;
    }
    OUTPUT:
    RETVAL

#
# IB_commit_transaction() - Commit the specified transaction.
# Return 0 on success, -1 on failure.
# Stores errors in the trhash.
#
int
IB_commit_transaction(ref)
    SV *ref
    CODE:
    {
    HV *trhash;
    TR_info *tr;
    SV **id;

	if (!SvROK(ref))
	{
	    RETVAL = FAILURE;
	    goto end;
	}
	trhash = (HV *) SvRV(ref);

	if ((id = hv_fetch(trhash, "Handle", 6, 0)) == NULL || !SvIOK(*id))
	{
	    RETVAL = FAILURE;
	    goto end;
	}

	if (SvIV(*id) < 0)
	{
	    RETVAL = SUCCESS;
	    goto end;
	}

	if ((tr = IB_get_TR(SvIV(*id))) == NULL)
	{
	    RETVAL = FAILURE;
	    goto end;
	}

	if (tr->inuse != TAKEN)
	{
	    char err[80];
	    sprintf(err, "IB_commit_transaction: uninitialized transaction handle %d.", id);
	    Error_String(trhash, err);
	    RETVAL = FAILURE;
	    goto end;
	}

	/*
	 * XXX: implement COMMIT RETAINING?
	 * Something for the future.
	 */
	isc_commit_transaction(tr->status,
	    (isc_tr_handle *) &(tr->handle));
	ERRCHECK(tr->status, trhash);

	(void) hv_store(trhash, "Handle", 6, newSViv(0), 0);
	RETVAL = SUCCESS;

	IB_free_TR(tr);

	end: ;
    }
    OUTPUT:
    RETVAL

#
# IB_commit_transaction() - Roll back the specified transaction.
# Return 0 on success, -1 on failure.
# Stores errors in the trhash.
#
int
IB_rollback_transaction(ref)
    SV *ref
    CODE:
    {
    HV *trhash;
    TR_info *tr;
    SV **id;

	if (!SvROK(ref))
	{
	    RETVAL = FAILURE;
	    goto end;
	}
	trhash = (HV *) SvRV(ref);

	if ((id = hv_fetch(trhash, "Handle", 6, 0)) == NULL || !SvIOK(*id))
	{
	    RETVAL = FAILURE;
	    goto end;
	}

	if (SvIV(*id) < 0)
	{
	    RETVAL = SUCCESS;
	    goto end;
	}

	if ((tr = IB_get_TR(SvIV(*id))) == NULL)
	{
	    RETVAL = FAILURE;
	    goto end;
	}

	if (tr->inuse != TAKEN)
	{
	    char err[80];
	    sprintf(err, "IB_rollback_transaction: uninitialized transaction handle %d.", id);
	    Error_String(trhash, err);
	    RETVAL = FAILURE;
	    goto end;
	}

	/*
	 * XXX: implement ROLLBACK RETAINING (6.0 only)?
	 * Something for the future.
	 */
	isc_rollback_transaction(tr->status,
	    (isc_tr_handle *) &(tr->handle));
	ERRCHECK(tr->status, trhash);

	(void) hv_store(trhash, "Handle", 6, newSViv(0), 0);
	RETVAL = SUCCESS;

	IB_free_TR(tr);

	end: ;
    }
    OUTPUT:
    RETVAL
# ======================================================================

MODULE = IBPerl PACKAGE = IBPerl::Statement

#
# IB_prepare() - Set up a new DSQL statement.
# Return 0 on success, -1 on failure.
# Stores errors and handle in the sthash.
#
int
IB_prepare(ref)
    SV *ref
    CODE:
    {
    HV *sthash, *trhash;
    static char stmt_info[1];
    char info_buffer[20], count_item = 0, *query;
    ST_info *st;
    SV **trref, **trhandle, **Statement, **dialectref;
    unsigned int stmt_len;

	if (!SvROK(ref))
	{
	    RETVAL = FAILURE;
	    goto end;
	}
	sthash = (HV *) SvRV(ref);

	/*
	 * Assume failure: set up a -1 handle that will be
	 * overwritten with a real handle if we succeed.
	 */
	if (hv_store(sthash, "Handle", 6, newSViv(-1), 0) == NULL)
	{
	    Error_String(sthash, "IB_prepare: cannot return statement handle.");
	    RETVAL = FAILURE;
	    goto end;
	}

	if ((trref = hv_fetch(sthash, "Transaction", 11, 0)) == NULL)
	{
	    Error_String(sthash, "IB_prepare: invalid transaction.");
	    RETVAL = FAILURE;
	    goto end;
	}

	if (!SvROK(*trref))
	{
	    RETVAL = FAILURE;
	    goto end;
	}
	trhash = (HV *) SvRV(*trref);

	if ((trhandle = hv_fetch(trhash, "Handle", 6, 0)) == NULL
	    || !SvIOK(*trhandle)
	    || SvIV(*trhandle) == -1)
	{
	    Error_String(sthash, "IB_prepare: invalid transaction handle.");
	    RETVAL = FAILURE;
	    goto end;
	}

	if ((st = IB_new_ST(SvIV(*trhandle))) == NULL)
	{
	    Error_String(sthash, "IB_prepare: cannot create statement.");
	    RETVAL = FAILURE;
	    goto end;
	}

	/*
	 * Allocate memory for a Statement data structure.
	 */
	isc_dsql_alloc_statement2(st->status,
	    (isc_db_handle *) &(st->trans->db->handle),
	    (isc_stmt_handle *) &(st->handle));
	ERRCHECK(st->status, sthash);

	/*
	 * Get the SQL statement out of the statement object.
	 */
	if ((Statement = hv_fetch(sthash, "Stmt", 4, 0)) == NULL
	    || !SvPOK(*Statement))
	{
	    Error_String(sthash, "IB_prepare: cannot determine statement.");
	    RETVAL = FAILURE;
	    goto end;
	}

	if ((query = SvPV(*Statement, stmt_len)) == NULL)
	{
	    Error_String(sthash, "IB_prepare: cannot determine statement.");
	    RETVAL = FAILURE;
	    goto end;
	}
	query[stmt_len] = '\0';

	if ((dialectref = hv_fetch(trhash, "SQLDialect", 10, 0)) != NULL
	    && SvIOK(*dialectref))
	{
	    st->sqldialect = SvIV(*dialectref);
	}


	/*
	 * This is supposed to prepare the SQL statement and
	 * set up appropriate values in out_sqlda.
	 */
	isc_dsql_prepare(st->status,
	    (isc_tr_handle *) &(st->trans->handle),
	    (isc_stmt_handle *) &(st->handle),
	    0,
	    query,
	    st->sqldialect,
	    st->out_sqlda);
	ERRCHECK(st->status, sthash);

	/*
	 * What is the statement type of this statement? 
	 * stmt_info is a 1 byte info request.  info_buffer is a buffer
	 * large enough to hold the returned info packet
	 * The info_buffer returned contains a isc_info_sql_stmt_type in the
	 * first byte,  two bytes of length, and a statement_type token.
	 */
	stmt_info[0] = isc_info_sql_stmt_type;
	isc_dsql_sql_info(st->status,
	    (isc_stmt_handle *) &(st->handle),
	    sizeof (stmt_info),   stmt_info,
	    sizeof (info_buffer), info_buffer);
	ERRCHECK(st->status, sthash);

	{
	    short l = (short) isc_vax_integer((char *) info_buffer + 1, 2);
	    st->stmt_type = isc_vax_integer((char *) info_buffer + 3, l);
	}

	switch (st->stmt_type)
	{
	/*
	 * Implemented statement types.
	 */
	case isc_info_sql_stmt_select:
	case isc_info_sql_stmt_select_for_upd:
	    {
	    static char plan_info [] = { isc_info_sql_get_plan };
	    char *plan;
	    
		/*
		 * XXX This is bad design, allowing only 512 bytes
		 * for the PLAN.  But I'm lazy right now.  I'll fix
		 * it another time.  Best plan would be to use
		 * a mortal scalar.
		 */
		if ((plan = (char *) malloc(512)) == NULL)
		{
		    fprintf(stderr, "Cannot allocate buffer for statement plan.\n");
		    RETVAL = FAILURE;
		    goto end;
		}

		isc_dsql_sql_info(
		    st->status,
		    (isc_stmt_handle *) &(st->handle),
		    sizeof(plan_info), plan_info,
		    512, plan);
		ERRCHECK(st->status, sthash);

		if (plan[0] == isc_info_sql_get_plan)
		{
		    short l = isc_vax_integer(plan+1, 2);
		    hv_store(sthash, "Plan", 4, newSVpv(plan+3, l), 0);
		}
 		free(plan);
	    }
	    /* count_item = isc_info_req_select_count; */
	    /*
	     * Unfortunately, select count item doesn't work
	     * in current versions of InterBase. 
	     */
	    break;

	case isc_info_sql_stmt_insert:
	    count_item = isc_info_req_insert_count;
	    break;

	case isc_info_sql_stmt_update:
	    count_item = isc_info_req_update_count;
	    break;

	case isc_info_sql_stmt_delete:
	    count_item = isc_info_req_delete_count;
	    break;

	case isc_info_sql_stmt_ddl:
	    break;

	/*
	 * XXX This is experimental and doesn't currently work.
	 * I will disable it for now and address it in a future
	 * release.
	 */
#ifdef IBPERL_EXEC_PROC
	case isc_info_sql_stmt_exec_procedure:
	    break;
#endif

	/*
	 * Unimplemented statement types.
	 * Some may be implemented in the future.
	 */
	case isc_info_sql_stmt_get_segment:
	case isc_info_sql_stmt_put_segment:
	case isc_info_sql_stmt_start_trans:
	case isc_info_sql_stmt_commit:
	case isc_info_sql_stmt_rollback:
	default:
	    Error_String(sthash, "IB_prepare: statement type is not implemented in this version of IBPerl.");
	    hv_store(sthash, "Stmt_type_no", 12, newSViv(0), 0);
	    RETVAL = FAILURE;
	    goto end;
	    break;
	}

	hv_store(sthash, "Stmt_type_no", 12, newSViv(st->stmt_type), 0);
	RETVAL = SUCCESS;
	    

	if (hv_store(sthash, "Handle", 6, newSViv(st->id), 0) == NULL)
	{
	    Error_String(sthash, "IB_prepare: cannot return statement handle.");
	    RETVAL = FAILURE;
	    goto end;
	}

	/*
	 * Reallocate output XSQLDA, if we failed to make
	 * it big enough in the first place.
	 */
	if (st->out_sqlda->sqld > st->out_sqlda->sqln)
	{
	    int n = st->out_sqlda->sqld;
	    if ((st->out_sqlda = (XSQLDA *) realloc(st->out_sqlda, XSQLDA_LENGTH(n))) == NULL)
	    {
		fprintf(stderr, "Cannot allocate output XSQLDA structure.\n");
		exit(1);
	    }
	    memset(st->out_sqlda, '\0', XSQLDA_LENGTH(n));

	    st->out_sqlda->sqln = st->out_sqlda->sqld = n;
	    st->out_sqlda->version = SQLDA_VERSION1;
	    isc_dsql_describe(st->status,
		&(st->handle),
		st->sqldialect,
		st->out_sqlda);
	    ERRCHECK(st->status, sthash);
	}

	/*
	 * Reallocate input XSQLDA, if we failed to make
	 * it big enough in the first place.
	 */
	isc_dsql_describe_bind(
	    st->status,
	    &(st->handle),
	    st->sqldialect,
	    st->in_sqlda);
	ERRCHECK(st->status, sthash);
	if (st->in_sqlda->sqld > st->in_sqlda->sqln)
	{
	    int n = st->in_sqlda->sqld;
	    if ((st->in_sqlda = (XSQLDA *) realloc(st->in_sqlda, XSQLDA_LENGTH(n))) == NULL)
	    {
		fprintf(stderr, "Cannot allocate input XSQLDA structure.\n");
		exit(1);
	    }
	    memset(st->in_sqlda, '\0', XSQLDA_LENGTH(n));

	    st->in_sqlda->sqln = st->in_sqlda->sqld = n;
	    st->in_sqlda->version = SQLDA_VERSION1;

	    isc_dsql_describe_bind(
		st->status,
		&(st->handle),
		st->sqldialect,
		st->in_sqlda);
	    ERRCHECK(st->status, sthash);
	}

	if (count_item)
	{
	    /*
	     * How many rows were affected with this statement?
	     * stmt_info is a 1 byte info request.  info_buffer
	     * is a buffer large enough to hold the returned
	     * info packet.  The info_buffer returned contains
	     * an isc_info_req_*_count in the first byte, two
	     * bytes of length, and a statement_type token.
	     *
	     * Unfortunately, this feature does not work for
	     * SELECT counts in current versions of InterBase
	     * (as of V5.5.0).
	     */
	    stmt_info[0] = count_item;
	    isc_dsql_sql_info(st->status,
		(isc_stmt_handle *) &(st->handle),
		sizeof (stmt_info),   stmt_info,
		sizeof (info_buffer), info_buffer);
	    ERRCHECK(st->status, sthash);

	    {
		short l = (short) isc_vax_integer((char *) info_buffer + 1, 2);
		st->count = isc_vax_integer((char *) info_buffer + 3, l);
	    }

	    if (hv_store(sthash, "Count", 5, newSViv(st->count), 0) == NULL)
	    {
		Error_String(sthash, "IB_prepare: cannot return number of rows.");
		RETVAL = FAILURE;
		goto end;
	    }
	}

	end: ;
    }
    OUTPUT:
    RETVAL

#
# IB_execute() - Run non-SELECT statements.
# Return 0 on success, -1 on failure.
# Stores errors in the sthash.
#
int
IB_execute(ref)
    SV *ref
    CODE:
    {
    HV *sthash;
    ST_info *st;
    SV **id;

	if (!SvROK(ref))
	{
	    RETVAL = FAILURE;
	    goto end;
	}
	sthash = (HV *) SvRV(ref);

	if ((id = hv_fetch(sthash, "Handle", 6, 0)) == NULL || !SvIOK(*id))
	{
	    RETVAL = FAILURE;
	    goto end;
	}

	if ((st = IB_get_ST(SvIV(*id))) == NULL)
	{
	    RETVAL = FAILURE;
	    goto end;
	}

	if (st->inuse != TAKEN)
	{
	    Error_String(sthash, "IB_execute: uninitialized statement handle.");
	    RETVAL = FAILURE;
	    goto end;
	}

	if (st->out_sqlda && st->out_sqlda->sqld)
	{
	    Error_String(sthash, "IB_execute: out_sqlda is not null.");
	    RETVAL = FAILURE;
	    goto end;
	}

	/*
	 * fill in input parms
	 */
	if (st->in_sqlda && st->in_sqlda->sqld > 0)
	{
	    SV **parmsref;
	    AV *parms;

	    if ((parmsref = hv_fetch(sthash, "Parms", 5, 0)) != NULL
		|| !SvROK(*parmsref))
	    {
		if ((st->isqlind = (short *) malloc(st->in_sqlda->sqld * sizeof(short))) == NULL)
		{
		    fprintf(stderr, "Cannot allocate input XSQLDA null indicators.\n");
		    exit(1);
		}
		IB_fill_isqlda((AV *) SvRV(*parmsref),
		    (ST_info *) st,
		    (HV *) sthash);
	    }
	}

	isc_dsql_execute(st->status,
	    (isc_tr_handle *) &(st->trans->handle),
	    (isc_stmt_handle *) &(st->handle),
	    st->sqldialect,
	    st->in_sqlda && st->in_sqlda->sqld > 0?
		st->in_sqlda: NULL);
	ERRCHECK(st->status, sthash);

	RETVAL = SUCCESS;

	end: ;
    }
    OUTPUT:
    RETVAL

#
# IB_open() - Create a cursor for a SELECT statement.
# Return 0 on success, -1 on failure.
# Stores errors in the sthash.
#
int
IB_open(ref)
    SV *ref
    CODE:
    {
    HV *sthash;
    ST_info *st;
    XSQLVAR *ovar;
    int i;
    SV **id;

	if (!SvROK(ref))
	{
	    RETVAL = FAILURE;
	    goto end;
	}
	sthash = (HV *) SvRV(ref);

	if ((id = hv_fetch(sthash, "Handle", 6, 0)) == NULL || !SvIOK(*id))
	{
	    Error_String(sthash, "IB_open: cannot get Handle of statement.");
	    RETVAL = FAILURE;
	    goto end;
	}

	if ((st = IB_get_ST(SvIV(*id))) == NULL)
	{
	    char err[80];
	    sprintf(err, "IB_open: cannot retrieve statement %d.", id);
	    Error_String(sthash, err);
	    RETVAL = FAILURE;
	    goto end;
	}

	if (st->inuse != TAKEN)
	{
	    Error_String(sthash, "IB_open: statement is not initialized.");
	    RETVAL = FAILURE;
	    goto end;
	}

	if (!st->out_sqlda || st->out_sqlda->sqld == 0)
	{
	    Error_String(sthash, "IB_open: sqld is 0.");
	    RETVAL = FAILURE;
	    goto end;
	}

	/*
	 * fill in input parms
	 */
	if (st->in_sqlda && st->in_sqlda->sqld > 0)
	{
	    SV **parmsref;

	    if ((parmsref = hv_fetch(sthash, "Parms", 5, 0)) != NULL
		|| !SvROK(*parmsref))
	    {
		IB_fill_isqlda((AV *) SvRV(*parmsref),
		    (ST_info *) st,
		    (HV *) sthash);
	    }
	}

	/*
	 * Allocate a buffer for the output XSQLVAR null indicators.
	 */
	if ((st->osqlind = (short *) malloc(st->out_sqlda->sqld * sizeof(short))) == NULL)
	{
	    fprintf(stderr, "Cannot allocate output XSQLDA null indicators.\n");
	    exit(1);
	}

	/*
	 * XXX This is experimental and doesn't currently work.
	 * I will disable it for now and address it in a future
	 * release.
	 */
#ifdef IBPERL_EXEC_PROC
	if (st->stmt_type == isc_info_sql_stmt_exec_procedure)
	{
	    int i;
	    XSQLVAR *ovar;

	    /*
	     * We actually fetch the result set here in open(),
	     * not later when we call fetch().  So we've got to
	     * allocate the XSQLVARs.
	     */
	    for (i=0, ovar = st->out_sqlda->sqlvar;
		i < st->out_sqlda->sqld;
		i++, ovar++)
	    {
		ovar->sqldata = (char *) malloc(ovar->sqllen +
		    ((ovar->sqltype & ~1) == SQL_VARYING ? 2 : 0) );
		ovar->sqlind = &st->osqlind[i];
	    }

	    isc_dsql_execute2(st->status,
		(isc_tr_handle *) &(st->trans->handle),
		(isc_stmt_handle *) &(st->handle),
		st->sqldialect,
		st->in_sqlda && st->in_sqlda->sqld > 0?
		    st->in_sqlda: NULL,
		st->out_sqlda && st->out_sqlda->sqld > 0?
		    st->out_sqlda: NULL);
	    ERRCHECK(st->status, sthash);
	}
	else /* A regular SELECT, not an exec procedure */
#endif /* IBPERL_EXEC_PROC */
	{
	    isc_dsql_execute(st->status,
		(isc_tr_handle *) &(st->trans->handle),
		(isc_stmt_handle *) &(st->handle),
		st->sqldialect,
		st->in_sqlda && st->in_sqlda->sqld > 0?
		    st->in_sqlda: NULL);
	    ERRCHECK(st->status, sthash);
	}

	RETVAL = SUCCESS;

	for (i=0, ovar = st->out_sqlda->sqlvar;
	    i < st->out_sqlda->sqld;
	    i++, ovar++)
	{
	    ovar->sqldata = (char *) malloc(ovar->sqllen +
		((ovar->sqltype & ~1) == SQL_VARYING ? 2 : 0) );
	    ovar->sqlind = &st->osqlind[i];
	}

	/*
	 * Declare a unique cursor for this query
	 */
	if (st->stmt_type == isc_info_sql_stmt_select_for_upd)
	{
	    if ((st->cursor_name = (char *) malloc(10)) == NULL)
	    {
		fprintf(stderr, "Cannot allocate cursor name.\n");
		exit(1);
	    }
	    /*
	     * Use the statement handle to make a unique cursor name!
	     * The handle is unique across the whole server, not just
	     * for one process or attachment.  How clever.
	     */
	    sprintf(st->cursor_name, "perl%06.6x", st->handle);
	    isc_dsql_set_cursor_name(
		st->status,
		(isc_stmt_handle *) &(st->handle),
		st->cursor_name,
		(unsigned short) NULL);
	    ERRCHECK(st->status, sthash);
	}
	end:
	;
    }
    OUTPUT:
    RETVAL

#
# IB_fetch() - Get one row of output from a SELECT.
# Return 0 on success, -1 on failure, 100 on end-of-records.
# Stores errors in the sthash.
#
int
IB_fetch(ref)
    SV *ref
    CODE:
    {
    HV *sthash;
    ST_info *st;
    SV **id, **valuesref, **columnsref, **nullsref, **lengthsref;
    AV *values, *columns, *nulls, *lengths;
    int i, null;
    ISC_STATUS fetchresult;

	RETVAL = SUCCESS;

	if (!SvROK(ref))
	{
	    RETVAL = FAILURE;
	    goto end;
	}
	sthash = (HV *) SvRV(ref);

	if ((id = hv_fetch(sthash, "Handle", 6, 0)) == NULL || !SvIOK(*id))
	{
	    Error_String(sthash, "IB_fetch: Cannot find statement handle.");
	    RETVAL = FAILURE;
	    goto end;
	}

	if ((st = IB_get_ST(SvIV(*id))) == NULL)
	{
	    char err[80];
	    sprintf(err, "IB_fetch: cannot find statement %d.", id);
	    Error_String(sthash, err);
	    RETVAL = FAILURE;
	    goto end;
	}

	if (st->inuse != TAKEN)
	{
	    Error_String(sthash, "IB_fetch: uninitialized statement handle.");
	    RETVAL = FAILURE;
	    goto end;
	}

	valuesref  = hv_fetch(sthash, "Values",  6, 1);
	columnsref = hv_fetch(sthash, "Columns", 7, 1);
	nullsref   = hv_fetch(sthash, "Nulls",   5, 1);
	lengthsref = hv_fetch(sthash, "Lengths", 7, 1);

	if (valuesref == NULL
	    || columnsref == NULL
	    || nullsref == NULL
	    || lengthsref == NULL)
	{
	    Error_String(sthash, "IB_fetch: array(s) are NULL.");
	    RETVAL = FAILURE;
	    goto end;
	}

	if ( ! (
		SvROK(*valuesref)
		&& SvROK(*columnsref)
		&& SvROK(*nullsref)
		&& SvROK(*lengthsref)
	    ) )
	{
	    Error_String(sthash, "IB_fetch: array(s) are not references.");
	    RETVAL = FAILURE;
	    goto end;
	}

	if (!(SvTYPE(SvRV(*valuesref))  == SVt_PVAV &&
	      SvTYPE(SvRV(*columnsref)) == SVt_PVAV &&
	      SvTYPE(SvRV(*nullsref))   == SVt_PVAV &&
	      SvTYPE(SvRV(*lengthsref)) == SVt_PVAV))
	{
	    Error_String(sthash, "IB_fetch: array(s) are not references to arrays.");
	    RETVAL = FAILURE;
	    goto end;
	}
#ifdef IBPERL_EXEC_PROC
	/*
	 * XXX This is experimental and doesn't currently work.
	 * I will disable it for now and address it in a future
	 * release.
	 */
	if (st->stmt_type == isc_info_sql_stmt_exec_procedure)
	{
	    /*
	     * Here's a hack to prevent someone from calling fetch() in a loop
	     * for the results of a stored procedure, and never exiting the
	     * loop.  Once we give the results back once, successive fetch()'s
	     * always return 100.
	     */
	    if (st->sp_fetched != 0)
	    {
		RETVAL = 100;
		goto end;
	    }
	    fetchresult = 0;
	    st->sp_fetched = 1;
	}
	else /* A regular SELECT, not an execute procedure. */
#endif /* IBPERL_EXEC_PROC */
	{
	    fetchresult = isc_dsql_fetch(st->status,
		(isc_stmt_handle *) &(st->handle),
		st->sqldialect,
		st->out_sqlda);
	}

	switch (fetchresult)
	{
	/*
	 * Got another row successfully.
	 */
	case 0: {
	    XSQLVAR *ovar;

	    values  = (AV *) SvRV(*valuesref);  av_clear(values);
	    columns = (AV *) SvRV(*columnsref); av_clear(columns);
	    nulls   = (AV *) SvRV(*nullsref);   av_clear(nulls);
	    lengths = (AV *) SvRV(*lengthsref); av_clear(lengths);

	    for (i=0, ovar = st->out_sqlda->sqlvar;
		i < st->out_sqlda->sqld;
		i++, ovar++)
	    {
		short dtype;
		SV *val = 0;

		dtype = ovar->sqltype & ~1;
		if (ovar->sqltype & 1 && ovar->sqlind && *ovar->sqlind < 0)
		{
		    /*
		     * This field has no value.
		     */
		    val = &sv_undef;
		    null = 1;
		}
		else
		{
		    /*
		     * Got a non-null field.  Got to pass it back to the
		     * application, which means some datatype dependant code.
		     */
		    null = 0;

		    switch (dtype)
		    {
		    case SQL_SHORT:
			if (ovar->sqlscale) /* handle NUMERICs */
			{
			    double numeric;

			    numeric = ((double) (*(short *) ovar->sqldata))
				/ pow(10.0, (double) -ovar->sqlscale);
			    val = newSVnv(numeric);
			}
			else
			    val = newSViv(*(short *) (ovar->sqldata));
			break;

		    case SQL_LONG:
			if (ovar->sqlscale) /* handle NUMERICs */
			{
			    double numeric;

			    numeric = ((double) (*(long *) ovar->sqldata))
				/ pow(10.0, (double) -ovar->sqlscale);
			    val = newSVnv(numeric);
			}
			else
			    val = newSViv(*(long *) (ovar->sqldata));
			break;

		    case SQL_FLOAT:
			val = newSVnv((double)(*(float *) (ovar->sqldata)));
			break;

		    case SQL_DOUBLE:
			if (ovar->sqlscale) /* handle NUMERICs */
			{
			    double d = *(double *)ovar->sqldata;
			    short q = -ovar->sqlscale;

			    val = newSVnv( d > 0?
				floor(d * pow(10.0, (double) q))
				    / pow(10.0, (double) q):
				ceil(d * pow(10.0, (double) q))
				    / pow(10.0, (double) q));
			}
			else
			    val = newSVnv(*(double *) (ovar->sqldata));
			break;

		    case SQL_TEXT:
			ovar->sqldata[ovar->sqllen] = '\0';
			val = newSVpv(ovar->sqldata, ovar->sqllen);
			break;

		    case SQL_VARYING:
		    {
			VARY *vary = (VARY *) ovar->sqldata;
			char *string;
			short len;

			len = vary->vary_length;
			string = vary->vary_string;
			val = newSV(len+1);
			sv_setpvn(val, string, len);
			break;
		    }

		/*
		 * If user specifies a DateFormat property of the Statement
		 * class, then that string is the format string for strftime().
		 *
		 * If the user doesn't specify a DateFormat, then format is
		 * NULL, strftime interprets as %c, defined in
		 * /usr/lib/locale/<locale>/LC_TIME/time, where <locale> is
		 * the host's chosen locale.
		 */

		    case SQL_DATE:
		    {
		    SV **format_sv;
		    char s[100], *format;
		    unsigned int len;
		    struct tm times;

			isc_decode_date((ISC_QUAD *) ovar->sqldata, &times);
			(void) mktime(&times); /* normalize */

			format_sv = hv_fetch(sthash, "DateFormat", 10, 0);

			if (format_sv != (SV **) NULL
			    && SvPOK(*format_sv)
			    && (len = SvCUR(*format_sv)) > 0)
			{
			    format = SvPV(*format_sv, na);
			    format[len] = '\0';
			}
			else
			{
			    format = NULL; /* means %C to strftime() */
			}

			/*
			 * If the DateFormat property is literally 'tm', then
			 * don't use strftime.  Instead create an array like
			 * that returned by Perl's builtin localtime function.
			 */
			if (format && !strcmp(format, "tm"))
			{
			    SV *item;
			    AV *list;

			    list = newAV();

			    item = newSViv(times.tm_sec);   av_push(list, item);
			    sv_setiv(item, times.tm_min);   av_push(list, item);
			    sv_setiv(item, times.tm_hour);  av_push(list, item);
			    sv_setiv(item, times.tm_mday);  av_push(list, item);
			    sv_setiv(item, times.tm_mon);   av_push(list, item);
			    sv_setiv(item, times.tm_year);  av_push(list, item);
			    sv_setiv(item, times.tm_wday);  av_push(list, item);
			    sv_setiv(item, times.tm_yday);  av_push(list, item);
			    sv_setiv(item, times.tm_isdst); av_push(list, item);
			    SvREFCNT_dec(item);

			    /*
			     * val becomes a _reference_ to this list.
			     */
			    val = newRV_noinc((SV *) list);
			}
			else
			{
			    strftime(s, sizeof(s), format, &times);
			    val = newSVpv(s, strlen(s));
			}

			break;
		    }

		    case SQL_BLOB:
		    {
			isc_blob_handle blob_handle = NULL;
			int blob_stat;
			char blob_info_buffer[32], *p,
			    blob_segment_buffer[MAX_BLOB_SEGMENT];
			char blob_info_items[] =
			    {
				isc_info_blob_max_segment,
				isc_info_blob_total_length
			    };
			long max_segment = -1L, total_length = -1L, t;
			unsigned short seg_length;

			/*
			 * Open the Blob according to the Blob id.
			 * Here's where my internal st, tr, and db
			 * data structures are really starting to pay off.
			 */
			isc_open_blob2(
			    st->status,
			    &(st->trans->db->handle),
			    &(st->trans->handle),
			    &blob_handle,
			    (ISC_QUAD *) ovar->sqldata,
			    (short) 0,		/* no filter */
			    (char *) NULL		/* no filter */
			);
			ERRCHECK(st->status, sthash);

			/*
			 * To find out the segment size in the proper way,
			 * I must query the blob information.
			 */
			isc_blob_info(
			    st->status,
			    &blob_handle,
			    sizeof(blob_info_items),
			    blob_info_items,
			    sizeof(blob_info_buffer),
			    blob_info_buffer
			);
			ERRCHECK(st->status, sthash);

			/*
			 * Get the information out of the info buffer.
			 */
			for (p = blob_info_buffer; *p != isc_info_end; )
			{
			    short length;
			    char datum = *p++;

			    length = (short) isc_vax_integer(p, 2);
			    p += 2;
			    switch (datum)
			    {
			    case isc_info_blob_max_segment:
				max_segment = isc_vax_integer(p, length);
				break;
			    case isc_info_blob_total_length:
				total_length = isc_vax_integer(p, length);
				break;
			    }
			    p += length;
			}

			if (max_segment == -1L || total_length == -1L)
			{
			    Error_String(sthash, "IB_fetch: cannot determine Blob dimensions");
			    RETVAL = FAILURE;
			    break;
			}

			if (total_length >= MAX_SAFE_BLOB_LENGTH)
			{
			    Error_String(sthash,"IB_fetch: Blob exceeds maximum length");
			    val = newSVpv("<Blob exceeds maximum safe length>", 34);
			    /*
			     * I deliberately don't set FAILURE based on this.
			     */
			    RETVAL = SUCCESS;
			    break;
			}

			/*
			 * Create a zero-length string.
			 */
			val = newSV(total_length);
			sv_setpv(val, "");

			t = total_length;
			while (1)
			{
			    blob_stat = isc_get_segment(
				st->status,
				&blob_handle,
				&seg_length,
				(short) MAX_BLOB_SEGMENT,
				blob_segment_buffer
			    );
			    if (st->status[1] != isc_segment)
				ERRCHECK(st->status, sthash);
			    if (st->status[1] == isc_segstr_eof)
				break;

    /*
     * As long as the fetch was successful, concatenate the segment we fetched
     * into the growing Perl scalar.  
     */
    /*
     * XXX This is dangerous if the Blob is enormous.  But Perl is supposed
     * to be able to grow scalars indefinitely as far as resources allow,
     * so what the heck.  Besides, I limited the max length of a Blob earlier
     * to MAX_SAFE_BLOB_LENGTH.
     */

			    sv_catpvn(val, blob_segment_buffer, seg_length);
			    t -= seg_length;

			    if (t <= 0) break;
			    if (blob_stat == 100) break;
			}

			/*
			 * Clean up after ourselves.
			 */
			isc_close_blob(st->status, &blob_handle);
			ERRCHECK(st->status, sthash);

			RETVAL = SUCCESS;
		    }
			break;

		    case SQL_ARRAY:
#ifdef ARRAY_SUPPORT
			!!! NOT IMPLEMENTED YET !!!
#else
			val = newSVpv("<array>", 8);
#endif
			break;

		    default:
			val = newSVpv("<unknown>", 10);
		    }
		}

		av_push(values, val);
		av_push(lengths, newSViv(ovar->sqllen));

		/*
		 * I just use the column's alias name because in the absence
		 * of an alias, it contains the column name anyway.  Only if
		 * the alias AND the column names are zero-length do I want to
		 * use a generic "COLUMN%d" header.  This happens, for example,
		 * when the column is a computed field.
		 */
		if (ovar->aliasname_length > 0)
		{
		    av_push(columns, newSVpv(ovar->aliasname, ovar->aliasname_length));
		}
		else
		{
		    char s[20];
		    sprintf(s, "COLUMN%d", i);
		    av_push(columns, newSVpv(s, strlen(s)));
		}

		av_push(nulls, newSViv(null));
	    }
#ifdef IBPERL_DEBUG
	    printf("IB_fetch(): Print output XSQLDA:\n");
	    IB_dump_XSQLDA(st->out_sqlda);
	    printf("IB_fetch(): Print output XSQLDA done.\n");
	    fflush(stdout);
#endif
	    break;
	}

	/*
	 * Code 100 means we've reached the end of the set
	 * of rows that the SELECT will return.
	 */
	case 100:
	    /*
	     * To put an empty list on the return stack,
	     * just don't push anything on.  This works
	     * because we're in a PP CODE: block.
	     */
	    RETVAL = 100;
	    break;

	/*
	 * An error has occurred.
	 */
	default:
	    {
		char *err = Error_Vector(st->status);
		if (err)
		{
		    Error_String(sthash, err);
 		    free(err);
		}
		else
		{
		    Error_String(sthash, "IB_fetch: unknown fetch error");
		}

		/*
		 * put 'undef' on the return stack
		 */
		RETVAL = FAILURE;
		break;
	    }
	}

	end: ;

    }
    OUTPUT:
    RETVAL

#
# IB_update() - 
# Return 0 on success, -1 on failure.
# Stores errors in the sthash.
#
int
IB_update_current(ref)
    SV *ref
    CODE:
    {
    HV *sthash, *changes_hash;
    ST_info *st;
    SV **id, *value, **changes_ref;
    char *key, stmtbuf[1024], *p, *tablename;
    int keylength, i;

	RETVAL = SUCCESS;

	if (!SvROK(ref))
	{
	    RETVAL = FAILURE;
	    goto end;
	}
	sthash = (HV *) SvRV(ref);

	if ((id = hv_fetch(sthash, "Handle", 6, 0)) == NULL || !SvIOK(*id))
	{
	    Error_String(sthash, "IB_update_where_current: cannot find statement handle.");
	    RETVAL = FAILURE;
	    goto end;
	}

	if ((st = IB_get_ST(SvIV(*id))) == NULL)
	{
	    char err[80];
	    sprintf(err, "IB_update_where_current: cannot find statement %d.", id);
	    Error_String(sthash, err);
	    RETVAL = FAILURE;
	    goto end;
	}

	if (st->inuse != TAKEN)
	{
	    Error_String(sthash, "IB_update_where_current: uninitialized statement handle.");
	    RETVAL = FAILURE;
	    goto end;
	}

	if (((changes_ref = hv_fetch(sthash, "Changes", 7, 0)) == NULL) 
	    || (!SvROK(*changes_ref)))
	{
	    Error_String(sthash, "IB_update_where_current: Cannot find positioned update changes.");
	    RETVAL = FAILURE;
	    goto end;
	}

	changes_hash = (HV *) SvRV(*changes_ref);

	tablename = st->out_sqlda->sqlvar[0].relname;

	/*
	 * If we find any table that isn't the same as the first table,
	 * then we can't do the update.
	 */
	/*
	 * XXX Future enhancement: permit multiple tables
	 * by creating a separate UPDATE for each relname.
	 */
	for (i = 1; i < st->out_sqlda->sqld; ++i)
	{
	    if (strcmp(st->out_sqlda->sqlvar[i].relname, tablename))
	    {
		Error_String(sthash, "IB_update_where_current: Cannot update query with multiple tables.");
		RETVAL = FAILURE;
		goto end;
	    }
	}

	/*
	 * Loop over all tablenames mentioned in XSQLVARs of st
	 */
	{
	    strcpy(stmtbuf, "UPDATE ");
	    strcat(stmtbuf, tablename);
	    strcat(stmtbuf, " SET ");
	    p = stmtbuf+strlen(stmtbuf);

	    hv_iterinit(changes_hash);
	    while ((value = hv_iternextsv(changes_hash, &key, (I32 *) &keylength)) != NULL)
	    {
	    char quote;

		p += sprintf(p, "%s = %c%s%c,",
		    key,
		    1? '\'': ' ',
		    SvPV(value, na),
		    1? '\'': ' '
		);
	    }
	    p--; /* back up over the last comma */

	    strcpy(p, " WHERE CURRENT OF ");
	    strcat(stmtbuf, st->cursor_name);

	    isc_dsql_execute_immediate(
		st->status,
		&(st->trans->db->handle),
		&(st->trans->handle),
		0,
		stmtbuf,
		st->sqldialect,
		(XSQLDA *) NULL
	    );
	    ERRCHECK(st->status, sthash);
	}


	end: ;

    }
    OUTPUT:
    RETVAL

#
# IB_delete() - 
# Return 0 on success, -1 on failure.
# Stores errors in the sthash.
#
int
IB_delete_current(ref)
    SV *ref
    CODE:
    {
    HV *sthash;
    ST_info *st;
    SV **id, *value, **changes_ref;
    char *key, stmtbuf[1024], *p, *tablename;
    int keylength, i;

	RETVAL = SUCCESS;

	if (!SvROK(ref))
	{
	    RETVAL = FAILURE;
	    goto end;
	}
	sthash = (HV *) SvRV(ref);

	if ((id = hv_fetch(sthash, "Handle", 6, 0)) == NULL || !SvIOK(*id))
	{
	    Error_String(sthash, "IB_update_where_current: cannot find statement handle.");
	    RETVAL = FAILURE;
	    goto end;
	}

	if ((st = IB_get_ST(SvIV(*id))) == NULL)
	{
	    char err[80];
	    sprintf(err, "IB_update_where_current: cannot find statement %d.", id);
	    Error_String(sthash, err);
	    RETVAL = FAILURE;
	    goto end;
	}

	if (st->inuse != TAKEN)
	{
	    Error_String(sthash, "IB_update_where_current: uninitialized statement handle.");
	    RETVAL = FAILURE;
	    goto end;
	}

	tablename = st->out_sqlda->sqlvar[0].relname;

	/*
	 * If we find any table that isn't the same as the first table,
	 * then we can't do the delete.
	 */
	/*
	 * XXX Future enhancement: permit multiple tables
	 * by creating a separate DELETE for each relname.
	 */
	for (i = 1; i < st->out_sqlda->sqld; ++i)
	{
	    if (strcmp(st->out_sqlda->sqlvar[i].relname, tablename))
	    {
		Error_String(sthash, "IB_update_where_current: Cannot update query with multiple tables.");
		RETVAL = FAILURE;
		goto end;
	    }
	}

	/*
	 * Loop over all tablenames mentioned in XSQLVARs of st
	 */
	{
	    strcpy(stmtbuf, "DELETE ");
	    strcat(stmtbuf, tablename);
	    strcat(stmtbuf, " WHERE CURRENT OF ");
	    strcat(stmtbuf, st->cursor_name);

	    isc_dsql_execute_immediate(
		st->status,
		&(st->trans->db->handle),
		&(st->trans->handle),
		0,
		stmtbuf,
		st->sqldialect,
		(XSQLDA *) NULL
	    );
	    ERRCHECK(st->status, sthash);
	}


	end: ;

    }
    OUTPUT:
    RETVAL

#
# IB_close() - close a SELECT cursor that we are done with.
# Return 0 on success, -1 on failure.
# Stores errors in the sthash.
#
int
IB_close(ref)
    SV *ref
    CODE:
    {
    HV *sthash;
    ST_info *st;
    XSQLVAR *var;
    int i;
    SV **id;

	if (!SvROK(ref))
	{
	    RETVAL = FAILURE;
	    goto end;
	}
	sthash = (HV *) SvRV(ref);

	if ((id = hv_fetch(sthash, "Handle", 6, 0)) == NULL || !SvIOK(*id))
	{
	    Error_String(sthash, "IB_close: cannot fetch statement Handle.");
	    RETVAL = FAILURE;
	    goto end;
	}

	if (SvIV(*id) < 0)
	{
	    RETVAL = SUCCESS;
	    goto end;
	}

	if ((st = IB_get_ST(SvIV(*id))) == NULL)
	{
	    char err[80];
	    sprintf(err, "IB_close: cannot retrieve Statement %d.", id);
	    Error_String(sthash, err);
	    RETVAL = FAILURE;
	    goto end;
	}


	if (st->inuse != TAKEN)
	{
	    Error_String(sthash, "IB_close: uninitialized statement handle.");
	    RETVAL = FAILURE;
	    goto end;
	}

	/*
	 * Close the cursor.
	 */
	isc_dsql_free_statement(st->status,
	    (isc_stmt_handle *) &(st->handle),
	    DSQL_close);
	ERRCHECK(st->status, sthash);
	
	RETVAL = SUCCESS;

	end: ;
    }
    OUTPUT:
    RETVAL

#
# IB_close() - close a SELECT cursor that we are done with.
# Return 0 on success, -1 on failure.
# Stores errors in the sthash.
#
int
IB_destroy_ST(ref)
    SV *ref
    CODE:
    {
    HV *sthash;
    ST_info *st;
    XSQLVAR *var;
    int i;
    SV **id;

	if (!SvROK(ref))
	{
	    RETVAL = FAILURE;
	    goto end;
	}
	sthash = (HV *) SvRV(ref);

	if ((id = hv_fetch(sthash, "Handle", 6, 0)) == NULL || !SvIOK(*id))
	{
	    Error_String(sthash, "IB_close: cannot fetch statement Handle.");
	    RETVAL = FAILURE;
	    goto end;
	}

	if (SvIV(*id) < 0)
	{
	    RETVAL = SUCCESS;
	    goto end;
	}

	if ((st = IB_get_ST(SvIV(*id))) == NULL)
	{
	    Error_String(sthash, "IB_close: cannot retrieve Statement.");
	    RETVAL = FAILURE;
	    goto end;
	}

	if (st->inuse != TAKEN)
	{
	    Error_String(sthash, "IB_close: uninitialized statement handle.");
	    RETVAL = FAILURE;
	    goto end;
	}

	/*
	 * Drop the statement and free the resource
	 */
	IB_free_ST(st);
	isc_dsql_free_statement(st->status,
	    (isc_stmt_handle *) &(st->handle),
	    DSQL_drop);
	ERRCHECK(st->status, sthash);
	
	RETVAL = SUCCESS;


	end: ;
    }
    OUTPUT:
    RETVAL

# ======================================================================
