package Oak::Application;

use strict;
use Error qw(:try);
use base qw(Oak::Object);

=head1 NAME

Oak::Application - Class for creating applications in Oak

=head1 SYNOPSIS

  my $app = new Oak::Application
    (
     "MyApp::TopLevel1" => "TopLevel1.xml",
     "MyApp::TopLevel2" => "TopLevel2.xml",
     "MyApp::TopLevel3" => "TopLevel3.xml",
     "default" => "MyApp::TopLevel1"
    );

  $app->run;			# abstract in Oak::Application

=head1 DESCRIPTION

This is the class that will be used to create real applications,
the executable file will launch it.

=head1 METHODS

=over

=item constructor

This method was overwrited to create all the objects passed to new.
The objects it creates will be available in the namespace ::TL. ie:
If one toplevel component have the name pagLogin, its object will
be available as $::TL::pagLogin.

The exceptions generated by Oak::Component wont be treated here, so,
they will propagate to the first level.

Exceptions generated by Oak::Application
  Oak::Application::Error::DuplicatedTopLevel - toplevel components have the same name
  Oak::Application::Error::ClassNotFound - toplevel class not found in lib

=back

=cut

sub constructor {
	my $self = shift;
	my %parms = @_;
	$self->SUPER::constructor(%parms);
	$self->set('default' => $parms{default});
	$self->{names} = [];
	delete $parms{default};
	foreach my $class (keys %parms) {
		eval "require $class" || throw Oak::Application::Error::ClassNotFound;
		my $obj = $class->new(RESTORE_TOPLEVEL => $parms{$class});
		my $name = $obj->get('name');
		eval 'if (defined $::TL::'.$name.') { die } else { return 1 }' ||
		  throw Oak::Application::Error::DuplicatedTopLevel;
		push @{$self->{names}}, $name;
		eval '$::TL::'.$name.' = $obj';
		if ($self->get('default') eq $class) {
			eval '$::TL::default = $obj';
		}
	}
}

=over

=item DESTROY

Overwrited to undef the objects created in the ::TL namespace.

=back

=cut

sub DESTROY {
	my $self = shift;
	eval '$::TL::default = undef';
	foreach my $name (@{$self->{names}}) {
		eval '$::TL::'.$name.' = undef';
	}
	return $self->SUPER::DESTROY;
}

=over

=item run

Abstract in Oak::Application, each type of application will
implement how they run.

=back

=cut

sub run {
	# Abstract in Oak::Application
	return 1
}

=head1 EXCEPTIONS

The following exceptions are introduced by Oak::Application

=over

=item Oak::Application::Error::ClassNotFound

This error is throwed when the class passed as parameter to
the new failed to be required.

=back

=cut

package Oak::Application::Error::ClassNotFound;
use base qw (Error);

sub stringify {
	return "Class not found in lib";
}

=over

=item Oak::Application::Error::DuplicatedTopLevel

This error is throwed when two toplevel components
have the same name

=back

=cut

package Oak::Application::Error::DuplicatedTopLevel;
use base qw (Error);

sub stringify {
	return "Two toplevels with the same name";
}


1;

__END__

=head1 BUGS

Too early to determine. :)

=head1 COPYRIGHT

Copyright (c) 2001
Daniel Ruoso <daniel@ruoso.com>
All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.
