package Code::TidyAll::Plugin::JSON;

use strict;
use warnings;

use JSON::MaybeXS ();
use Moo;

our $VERSION = '0.31';

extends 'Code::TidyAll::Plugin';

has 'ascii' => ( is => 'ro', default => 0 );

sub transform_source {
    my $self   = shift;
    my $source = shift;

    my $json = JSON::MaybeXS->new(
        canonical => 1,
        pretty    => 1,
        relaxed   => 1,
        utf8      => 1,
    );

    $json = $json->ascii if $self->ascii;

    return $json->encode( $json->decode($source) );
}

1;

# ABSTRACT: Use the JSON::MaybeXS module to tidy JSON documents with tidyall

__END__

=pod

=head1 NAME

Code::TidyAll::Plugin::JSON - Use the JSON::MaybeXS module to tidy JSON
documents with tidyall

=head1 VERSION

version 0.31

=head1 SYNOPSIS

   In configuration:

   [JSON]
   select = **/*.json
   ascii = 1

=head1 DESCRIPTION

Uses L<JSON::MaybeXS> to format JSON files. Files are put into a canonical
format with the keys of objects sorted.

=head1 CONFIGURATION

=over

=item ascii

Escape non-ASCII characters. The output file will be valid ASCII.

=back

=head1 SEE ALSO

L<Code::TidyAll>

=head1 AUTHORS

=over 4

=item *

Jonathan Swartz <swartz@pobox.com>

=item *

Dave Rolsky <autarch@urth.org>

=back

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2011 - 2015 by Jonathan Swartz.

This is free software; you can redistribute it and/or modify it under the same
terms as the Perl 5 programming language system itself.

=cut
