use Test::More tests => 124;

BEGIN { 
   unshift @INC, qw(lib ../lib);
   use_ok('Locale::Maketext::Utils');
};

package TestApp::Localize;
use Locale::Maketext::Utils;
use base 'Locale::Maketext::Utils';

our $Encoding = 'utf8';

our %Lexicon = (
    '_AUTO'    => 42, 
    'Fallback' => 'Fallback orig',
    'One Side' => 'I am not one sides',
);

__PACKAGE__->make_alias('i_alias1', 1);

sub output_test {
    my ($lh, $string) = @_;
    return "TEST $string TEST";
}

package TestApp::Localize::en;
use base 'TestApp::Localize';

package TestApp::Localize::en_us;
use base 'TestApp::Localize';

package TestApp::Localize::i_default;
use base 'TestApp::Localize';

package TestApp::Localize::i_oneside;
use base 'TestApp::Localize';

__PACKAGE__->make_alias( [qw(i_alias2 i_alias3)], 0 ); 
our $Onesided = 1;
our %Lexicon = (
    'One Side' => '',
);

package TestApp::Localize::fr;
use base 'TestApp::Localize';

our $Encoding = 'utf7';

our %Lexicon = (
    'Hello World' => 'Bonjour Monde',
);

sub init {
    my ($lh) = @_;
    $lh->SUPER::init();
    $lh->{'numf_comma'} = 1; # Locale::Maketext numf()
    $lh->{'list_seperator'}   = '. ';
    $lh->{'oxford_seperator'} = '';
    $lh->{'list_default_and'} = '&&';
    return $lh;
}

package main;

{
    local $ENV{'maketext_obj_skip_env'} = 1;
    local $ENV{'maketext_obj'} = 'CURRENT VALUE';
    my $noarg = TestApp::Localize->get_handle();
    # depending on their Locales::Base may not have one of these
    ok($noarg->language_tag() eq 'en' || $noarg->language_tag() eq 'en-us', 'get_handle no arg');

    my $first_lex = (@{ $noarg->_lex_refs() })[0];
    ok(!exists $first_lex->{'_AUTO'}, '_AUTO removal/remove_key_from_lexicons()');
    ok($noarg->{'_removed_from_lexicons'}{'0'}{'_AUTO'} eq '42', 
       '_AUTO removal archive/remove_key_from_lexicons()');

    ok($ENV{'maketext_obj'} ne $noarg, 'ENV maketext_obj_skip_env true');
}

my $no_arg = TestApp::Localize->get_handle();
ok(ref($no_arg) eq 'TestApp::Localize::en_us', 'no argument has highest level langtag NS');

my $en = TestApp::Localize->get_handle('en');
ok($ENV{'maketext_obj'} eq $en, 'ENV maketext_obj_skip_env false');

ok($en->get_language_class() eq 'TestApp::Localize::en', 'get_language_class() obj method');
ok(TestApp::Localize::fr->get_language_class() eq 'TestApp::Localize::fr', 'get_language_class() class method');
ok(!defined $en->get_base_class_dir(), 'get_base_class_dir() returns undefined for non .pm based base class');
ok(!defined $en->list_available_locales(), 'list_available_locales() returns undefined for non .pm based base class');


my $has_sub_todo = eval { require Sub::Todo } ? 1 : 0;
$! = 0; # just to be sure
ok(!$en->add_lexicon_override_hash('en', 'before', {a=>1}), "add_lexicon_override_hash() returns false with non Tie::Hash::ReadonlyStack compat Lexicon");
SKIP: {
    skip "Sub::Todo required to test for 'not implemented' status", 1 if !$has_sub_todo;
    ok( $! > 0, 'add_lexicon_override_hash() + Sub::Todo sets $! with non Tie::Hash::ReadonlyStack compat Lexicon');
    $! = 0;
};
SKIP: {
    skip "Sub::Todo must not be installed to test for 'no Sub::Todo not implemented' status", 1 if $has_sub_todo;
    ok( $! == 0, 'add_lexicon_override_hash() w/ out Sub::Todo does not get $! set with non Tie::Hash::ReadonlyStack compat Lexicon');
    $! = 0;
};

ok(!$en->add_lexicon_fallback_hash('en', 'after', {b=>1}), "add_lexcion_fallback_hash() returns false with non Tie::Hash::ReadonlyStack compat Lexicon");
SKIP: {
    skip "Sub::Todo required to test for 'not implemented' status", 1 if !$has_sub_todo;
    ok( $! > 0, 'add_lexicon_fallback_hash() + Sub::Todo sets $! with non Tie::Hash::ReadonlyStack compat Lexicon');
    $! = 0;
};
SKIP: {
    skip "Sub::Todo must not be installed to test for 'no Sub::Todo not implemented' status", 1 if $has_sub_todo;
    ok( $! == 0, 'add_lexicon_fallback_hash() w/ out Sub::Todo does not get $! set with non Tie::Hash::ReadonlyStack compat Lexicon');
    $! = 0;
};

ok(!$en->del_lexicon_hash('en', 'before'), "del_lexicon_hash() returns false with non Tie::Hash::ReadonlyStack compat Lexicon");
SKIP: {
    skip "Sub::Todo required to test for 'not implemented' status", 1 if !$has_sub_todo;
    ok( $! > 0, 'del_lexicon_hash() + Sub::Todo sets $! with non Tie::Hash::ReadonlyStack compat Lexicon');
    $! = 0;
};
SKIP: {
    skip "Sub::Todo must not be installed to test for 'no Sub::Todo not implemented' status", 1 if $has_sub_todo;
    ok( $! == 0, 'del_lexicon_hash() w/ out Sub::Todo does not get $! set with non Tie::Hash::ReadonlyStack compat Lexicon');
    $! = 0;
};

ok(!$en->del_lexicon_hash('*', 'after'), "del_lexicon_hash() returns false w/ * even with non Tie::Hash::ReadonlyStack compat Lexicon");
SKIP: {
    skip "Sub::Todo required to test for 'not implemented' status", 1 if !$has_sub_todo;
    ok( $! > 0, 'del_lexicon_hash() + Sub::Todo sets $! with non Tie::Hash::ReadonlyStack compat Lexicon');
    $! = 0;
};
SKIP: {
    skip "Sub::Todo must not be installed to test for 'no Sub::Todo not implemented' status", 1 if $has_sub_todo;
    ok( $! == 0, 'del_lexicon_hash() + * w/ out Sub::Todo does not get $! set with non Tie::Hash::ReadonlyStack compat Lexicon');
    $! = 0;
};

ok($en->language_tag() eq 'en', 'get_handle en');
ok($en->langtag_is_loadable('invalid') eq '0', 'langtag_is_loadable() w/ unloadable tag');
ok(ref $en->langtag_is_loadable('fr') eq 'TestApp::Localize::fr', 
   'langtag_is_loadable() w/ loadable tag');

my $is_singleton = TestApp::Localize->get_handle('en');
ok($en eq $is_singleton, 'same args result in singleton behavior');

my $one = TestApp::Localize->get_handle('en','fr');
my $two = TestApp::Localize->get_handle('en','fr');
my $three = TestApp::Localize->get_handle('fr', 'en');

ok($one eq $two, 'singleton same order is the same obj');
ok($two ne $three, 'singleton different order is not the same obj');

ok($en->encoding() eq 'utf8', 'base $Encoding');   
$en->{'_get_key_from_lookup'} = sub {
     return 'look up version';
};
ok($en->maketext('Needs looked up') eq 'look up version', '_get_key_from_lookup');

my $bad = TestApp::Localize->get_handle('bad');
ok($bad->language_tag() eq 'en', 'invalid get_handle arg');
$bad->{'_log_phantom_key'} = sub {
    $ENV{'_log_phantum_key'} = 'done';    
};
ok($bad->maketext('Not in Lexicon') eq 'Not in Lexicon'
   && $ENV{'_log_phantum_key'} eq 'done', '_log_phantom_key');

my $oneside = TestApp::Localize->get_handle('i_oneside');

ok($TestApp::Localize::i_oneside::Lexicon{'One Side'} eq '', '$Onesided untouched initially');
ok($oneside->maketext('One Side') eq 'One Side', 'Once used $Onesided returns proper value');
ok(ref $TestApp::Localize::i_oneside::Lexicon{'One Side'} eq 'SCALAR', 'Once used $Onesided does lexicon (sanity check that it is not just falling back)');

my $alias1 = TestApp::Localize->get_handle('i_alias1');
ok($alias1->get_language_tag() eq 'i_alias1', '$Aliaspkg w/ string');
my $alias2 = TestApp::Localize->get_handle('i_alias2');
ok($alias2->get_language_tag() eq 'i_alias2', '$Aliaspkg w/ array ref 1');
my $alias3 = TestApp::Localize->get_handle('i_alias3');
ok($alias3->get_language_tag() eq 'i_alias3', '$Aliaspkg w/ array ref 2');

ok($alias1->fetch('One Side') eq 'I am not one sides', 'Base class make_alias');
ok($alias2->fetch('One Side') eq 'One Side', 'Extended class make_alias');

my $en_US = TestApp::Localize->get_handle('en-US');
ok($en_US->language_tag() eq 'en-us', 'get_handle en-US');
ok($en_US->get_language_tag() eq 'en_us', 'get_language_tag()');

my $fr = TestApp::Localize->get_handle('fr');
ok($fr->language_tag() eq 'fr', 'get_handle fr');
ok($fr->get_base_class() eq 'TestApp::Localize', 'get_base_class()');
ok($fr->fetch('Hello World') eq 'Bonjour Monde', 'fetch() method'); 
ok($fr->{'numf_comma'} eq '1', 'init set value ok');

# safe to assume print() will work to if fetch() does...

{
    local $/ = "\n"; # just to be sure we're testing consistently...
    ok($fr->get('Hello World') eq "Bonjour Monde\n", 'get() method'); 
    # safe to assume say() will work to if get() does...
}

# this was a bad bad experimental idea (which is why it was undocumented and finally removed in 0.13)
# ## test AUTOLOAD:
# ok($fr->fetch_p('Hello World') eq '<p>Bonjour Monde</p>', 'AUTOLOAD tag');
# ok($fr->fetch_p_open('Hello World') eq '<p>Bonjour Monde', 'AUTOLOAD tag open');
# ok($fr->fetch_p_close('Hello World') eq 'Bonjour Monde</p>', 'AUTOLOAD tag close');
# 
# ok($fr->fetch_p_err('Hello World') eq '<p class="err">Bonjour Monde</p>', 'AUTOLOAD tag class');
# ok($fr->fetch_p_err_open('Hello World') eq '<p class="err">Bonjour Monde', 'AUTOLOAD tag class open');
# ok($fr->fetch_p_err_close('Hello World') eq 'Bonjour Monde</p>', 'AUTOLOAD tag class close');
# ok(!$fr->mistyped_non_existant_no_args(), 'AUTOLOAD no-arg not fatal'); # what about mistyped ones that do have args, yikes...
# SKIP: {
#     skip "Sub::Todo required to test for 'not implemented' status", 1 if !$has_sub_todo;
#     ok( $! > 0, 'AUTOLOAD mistyped_non_existant_no_args() + Sub::Todo sets $! with non Tie::Hash::ReadonlyStack compat Lexicon');
#     $! = 0;
# };
# SKIP: {
#     skip "Sub::Todo must not be installed to test for 'no Sub::Todo not implemented' status", 1 if $has_sub_todo;
#     ok( $! == 0, 'AUTOLOAD mistyped_non_existant_no_args() w/ out Sub::Todo does not get $! set with non Tie::Hash::ReadonlyStack compat Lexicon');
#     $! = 0;
# };
# # end AUTOLOAD tests

ok($fr->encoding() eq 'utf7', 'class $Encoding'); 
ok($fr->fetch('Fallback') eq 'Fallback orig', 'fallback  behavior');
ok($fr->fetch('Thank you') eq 'Thank you', 'fail_with _AUTO behavior');

$fr->append_to_lexicons({
    '_'  => {
        'Fallback' => 'Fallback new',  
    },
    'fr' => {
        'Thank you' => 'Merci',
    },
});

ok($fr->fetch('Thank you') eq 'Merci', 'append_to_lexicons()');
ok($fr->fetch('Fallback') eq 'Fallback new', 'fallback behavior after append');

my $fr_hr = $fr->lang_names_hashref('en-uk', 'it', 'xxyyzz');
ok($fr_hr->{'en'} eq 'Anglais', 'names default');
ok($fr_hr->{'en-uk'} eq 'Anglais (UK)', 'names suffix');
ok($fr_hr->{'it'} eq 'Italien', 'names normal');
ok($fr_hr->{'xxyyzz'} eq 'xxyyzz', 'names fake');

my $sig_warn = exists $SIG{__WARN__} && defined $SIG{__WARN__} ? $SIG{__WARN__} : 'no exists/defined';
my $base_sig_warn = exists $Locales::Base::SIG{__WARN__} && defined $Locales::Base::SIG{__WARN__} ? $Locales::Base::SIG{__WARN__} : 'no exists/defined';
my ($loc_hr, $nat_hr) = $fr->lang_names_hashref('en-uk', 'it', 'xxyyzz');
my $sig_warn_aft = exists $SIG{__WARN__} && defined $SIG{__WARN__} ? $SIG{__WARN__} : 'no exists/defined';
my $base_sig_warn_aft = exists $Locales::Base::SIG{__WARN__} && defined $Locales::Base::SIG{__WARN__} ? $Locales::Base::SIG{__WARN__} : 'no exists/defined';
ok($sig_warn eq $sig_warn_aft, 'main sig warn unchanged by lang_names_hashref()');
ok($base_sig_warn eq $base_sig_warn_aft, 'locale::base sig warn unchanged by lang_names_hashref()');

ok($loc_hr->{'en'} eq 'Anglais', 'array context handle locale names default');
ok($loc_hr->{'en-uk'} eq 'Anglais (UK)', 'array context handle locale names suffix');
ok($loc_hr->{'it'} eq 'Italien', 'array context handle locale names normal');
ok($loc_hr->{'xxyyzz'} eq 'xxyyzz', 'array context handle locale  names fake');

ok($nat_hr->{'en'} eq 'English', 'array context native names default');
ok($nat_hr->{'en-uk'} eq 'English (UK)', 'array context native names suffix');
ok($nat_hr->{'it'} eq 'Italian', 'array context native names normal');
ok($nat_hr->{'xxyyzz'} eq 'xxyyzz', 'array context native names fake');

my $loadable_hr = $fr->loadable_lang_names_hashref('en-uk', 'it', 'xxyyzz', 'fr');

ok( (keys %{ $loadable_hr }) == 2
    && exists $loadable_hr->{'en'}
    && exists $loadable_hr->{'fr'}, 'loadable names');

# prepare 
my $dir = './my_lang_pm_search_paths_test';
mkdir $dir;
mkdir "$dir/TestApp";
mkdir "$dir/TestApp/Localize";
die "mkdir $@" if !-d "$dir/TestApp/Localize";

open my $pm, '>', "$dir/TestApp/Localize/it.pm" or die "open $!";
    print {$pm} <<'IT_END';
package TestApp::Localize::it;
use base 'TestApp::Localize';

__PACKAGE__->make_alias('it_us');

our %Lexicon = (
    'Hello World' => 'Ciao Mondo',  
);

1;
IT_END
close $pm;

require "$dir/TestApp/Localize/it.pm";
my $it_us = TestApp::Localize->get_handle('it_us');
ok($it_us->fetch('Hello World') eq 'Ciao Mondo', '.pm file alias test');

# _lang_pm_search_paths
$en->{'_lang_pm_search_paths'} = [$dir];
my $dir_hr = $en->lang_names_hashref();
ok( (keys %{ $dir_hr }) == 2
    && exists $dir_hr->{'en'}
    && exists $dir_hr->{'it'}, '_lang_pm_search_paths names');

# @INC
unshift @INC, $dir;
my $inc_hr = $fr->lang_names_hashref();
ok( (keys %{ $inc_hr }) == 2
    && exists $inc_hr->{'en'}
    && exists $inc_hr->{'it'}, '@INC names');

delete $en->{'_get_key_from_lookup'}; #  don't this anymore

# datetime

ok( $en->maketext('[datetime]') =~ m{ \A \w+ \s \d+ [,] \s \d+ \z }xms, 'undef 1st undef 2nd');
my $dt_obj = DateTime->new('year'=> 1978); # DateTime already brought in by prev [datetime] call
ok( $en->maketext('[datetime,_1]', $dt_obj)  =~ m{^January 1, 1978$}i, '1st arg object');
ok( $en->maketext('[datetime,_1,_2]', {'year'=>1977}, '')  =~ m{^January 1, 1977$}i, '1st arg hashref');
ok( $en->maketext('[datetime,_1,_2]', {'year'=>1977}, '%Y') eq '1977', '2nd arg string');
ok( $en->maketext('[datetime,_1,_2]', {'year'=>1977}, sub { $_[0]->{'locale'}->long_datetime_format }) =~ m{^January 1, 1977 12:00:00 AM .*$}i, '2nd arg coderef');
ok( $en->maketext('[datetime,_1,_2]', {'year'=>1978, 'month'=>11, 'day'=>13}, sub { $_[0]->{'locale'}->long_datetime_format }) =~ m{^November 13, 1978 12:00:00 AM .*$}i ,'[datetime] English');
ok( $fr->maketext('[datetime,_1,_2]', {'year'=>1999, 'month'=>7, 'day'=>17}, sub { $_[0]->{'locale'}->long_datetime_format }) =~ m{^17 juillet 1999 00:00:00 .*$}i ,'[datetime] French');

ok( $en->maketext('[datetime,_1,short_datetime_format]', {'year'=>1977} ) eq '1/1/77 12:00 AM', '2nd arg DateTime::Locale method name');
ok( $en->maketext('[datetime,_1,_2]', {'year'=>1977}, 'invalid' ) eq 'invalid', '2nd arg DateTime::Locale method name invalid');

my $epoch = time;
my $epoch_utc = DateTime->from_epoch( 'epoch' => $epoch, 'time_zone' => 'UTC');
ok( $en->maketext('[datetime,_1,%s]','UTC') >= $epoch , '1st arg TZ');
ok( $en->maketext('[datetime,_1,long_datetime_format]', $epoch) eq $epoch_utc->strftime($epoch_utc->{'locale'}->long_datetime_format), '1st arg Epoch');
ok( $en->maketext('[datetime,_1,long_datetime_format]',"$epoch:UTC") eq $epoch_utc->strftime($epoch_utc->{'locale'}->long_datetime_format), '1st arg Epoch:TZ');

# numf w/ decimal support 

my $pi = 355/113;
ok( $en->maketext("pi is [numf,_1]",$pi) eq 'pi is 3.14159', 'default decimal behavior');
ok( $en->maketext("pi is [numf,_1,_2]",$pi,'') eq 'pi is 3.14159292035398', 'w/ empty');
ok( $en->maketext("pi is [numf,_1,_2]",$pi,0) eq 'pi is 3', 'w/ zero');
ok( $en->maketext("pi is [numf,_1,_2]",$pi,6) eq 'pi is 3.141592', 'w/ number');
ok( $en->maketext("pi is [numf,_1,_2]",$pi,-6) eq 'pi is 3.141592', 'w/ negative');
ok( $en->maketext("pi is [numf,_1,_2]",$pi,6.2) eq 'pi is 3.141592', 'w/ decimal');
ok( $en->maketext("pi is [numf,_1,_2]",$pi,'%.3f') eq 'pi is 3.142', 'w/ no numeric');

ok( $en->maketext("pi is [numf,_1,]",$pi) eq 'pi is 3.14159292035398', 'bn: w/ empty');
ok( $en->maketext("pi is [numf,_1,0]",$pi) eq 'pi is 3', 'bn: w/ zero');
ok( $en->maketext("pi is [numf,_1,6]",$pi) eq 'pi is 3.141592', 'bn: w/ number');
ok( $en->maketext("pi is [numf,_1,-6]",$pi) eq 'pi is 3.141592', 'bn: w/ negative');
ok( $en->maketext("pi is [numf,_1,6.2]",$pi) eq 'pi is 3.141592', 'bn: w/ decimal');
ok( $en->maketext("pi is [numf,_1,_2]",$pi,'%.3f') eq 'pi is 3.142', 'bn: w/ no numeric');

# range

ok( $en->maketext("[_1] [_2.._#]",1,2,3,4) eq '1 234', 'basic range' );
ok( $en->maketext("[_2] [_-1.._#]",1,2,3,4) eq '2 41234', 'no zero range' );
ok( $en->maketext("[_2] [_2.._3] [_4]",1,2,3,4) eq '2 23 4', 'specific range' );
ok( $en->maketext("[_1] [_2.._#]",1,2) eq '1 2', 'range goes to 1' );

# join

ok( $en->maketext("[join,~,,_*]",1,2,3,4) eq '1,2,3,4', "join all");
ok( $en->maketext("[join,,_*]",1,2,3,4) eq '1234', "blank sep");
ok( $en->maketext("[join,_*]",1,2,3,4) eq '21314', "no sep");
ok( $en->maketext("[join,-,_2,_4]",1,2,3,4) eq '2-4', "join specifc");
ok( $en->maketext("[join,-,_2.._#]",1,2,3,4) eq '2-3-4', "join range");

# list

ok( $en->maketext("[_1] is [list,and,_2.._#]",qw(a)) eq 'a is ','list no arg');
ok( $en->maketext("[_1] is [list,and,_2.._#]",qw(a b)) eq 'a is b','list one arg "and" sep');
ok( $en->maketext("[_1] is [list,&&,_2.._#]",qw(a b c)) eq 'a is b && c','list 2 arg special sep');
ok( $en->maketext("[_1] is [list,,_2.._#]",qw(a b c d)) eq 'a is b, c, & d','list 3 arg undef sep');
ok( $en->maketext("[_1] is [list,or,_2.._#]",qw(a b c d e)) eq 'a is b, c, d, or e','list 4 arg "or" sep');
ok( $en->maketext("[_1] is [list,and,_2.._#]",qw(a b c d e)) eq 'a is b, c, d, and e','list 4 arg "and" sep');

ok( $fr->maketext("[_1] is [list,,_2.._#]",qw(a b c d e)) eq 'a is b. c. d && e','specials set by class');

# boolean

ok($en->maketext('boolean [boolean,_1,true,false] x',1) eq 'boolean true x', 'boolean 2 arg true');
ok($en->maketext('boolean [boolean,_1,true,false] x',0) eq 'boolean false x', 'boolean 2 arg false');
ok($en->maketext('boolean [boolean,_1,true,false] x',undef) eq 'boolean false x', 'boolean 2 arg undef');
ok($en->maketext('boolean [boolean,_1,true,false,null] x',1) eq 'boolean true x', 'boolean 3 arg true');
ok($en->maketext('boolean [boolean,_1,true,false,null] x',0) eq 'boolean false x', 'boolean 3 arg false');
ok($en->maketext('boolean [boolean,_1,true,false,null] x',undef) eq 'boolean null x', 'boolean 3 arg undef');

# output 

ok($en->maketext('hello [output,test,hello world]') eq 'hello TEST hello world TEST', "output() with existing function");
ok($en->maketext('hello [output,notexists,hello world]') eq 'hello hello world', "output() with non existant function");

SKIP: {
    skip "Sub::Todo required to test for 'not implemented' status", 1 if !$has_sub_todo;
    ok( $! > 0, 'output() with non existant function + Sub::Todo sets $!');
    $! = 0;
};
SKIP: {
    skip "Sub::Todo must not be installed to test for 'no Sub::Todo not implemented' status", 1 if $has_sub_todo;
    ok( $! == 0, 'output() with non existant function w/ out Sub::Todo does not get $! set');
    $! = 0;
};

# convert

SKIP: {
    eval 'use Math::Units';
    skip 'Math::Units required for testing convert()', 1 if $@;
    ok( $en->maketext("[convert,_*]",1,'ft','in') eq '12', 'convert() method');
};

# format_bytes

SKIP: {
    eval 'use Number::Bytes::Human';
    skip 'Number::Bytes::Human required for testing format_bytes()', 1 if $@;
    ok( $en->maketext("[format_bytes,_*]",1024) eq '1.0K', 'format_bytes() method');
};


# cleanup 
unlink "$dir/TestApp/Localize/it.pm";
rmdir "$dir/TestApp/Localize";
rmdir "$dir/TestApp";
rmdir $dir;
warn "Could not cleanup $dir" if -d $dir;