package Archive::Har::Entry::Timings;

use warnings;
use strict;
use Carp();
use Archive::Har::Entry::Cache::Request();

our $VERSION = '0.11';

sub new {
	my ($class, $params) = @_;
	my $self = {};
	bless $self, $class;
	if (defined $params) {
		if (defined $params->{blocked}) {
			$self->blocked($params->{blocked});
		}
		if (defined $params->{dns}) {
			$self->dns($params->{dns});
		}
		if (defined $params->{connect}) {
			$self->connect($params->{connect});
		}
		$self->send($params->{send});
		$self->wait($params->{wait});
		$self->receive($params->{receive});
		if (defined $params->{ssl}) {
			$self->ssl($params->{ssl});
		}
		if (defined $params->{comment}) {
			$self->comment($params->{comment});
		}
		foreach my $key (sort { $a cmp $b } keys %{$params}) {
			if ($key =~ /^_[[:alnum:]]+$/smx) { # private fields
				$self->$key($params->{$key});
			}
		}
	}
	return $self;
}

sub blocked {
	my ($self, $new) = @_;
	my $old = $self->{blocked};
	if (@_ > 1) {
		if (defined $new) {
			$self->{blocked} = $new + 0;
		} else {
			$self->{blocked} = -1;
		}
	}
	if ((defined $old) && ($old eq -1)) {
		return;
	} else {
		return $old;
	}
}

sub dns {
	my ($self, $new) = @_;
	my $old = $self->{dns};
	if (@_ > 1) {
		if (defined $new) {
			$self->{dns} = $new + 0;
		} else {
			$self->{dns} = -1;
		}
	}
	if ((defined $old) && ($old eq -1)) {
		return;
	} else {
		return $old;
	}
}

sub connect {
	my ($self, $new) = @_;
	my $old = $self->{connect};
	if (@_ > 1) {
		if (defined $new) {
			$self->{connect} = $new + 0;
		} else {
			$self->{connect} = -1;
		}
	}
	if ((defined $old) && ($old eq -1)) {
		return;
	} else {
		return $old;
	}
}

sub send {
	my ($self, $new) = @_;
	my $old = $self->{send};
	if (@_ > 1) {
		$self->{send} = $new;
	}
	return $old;
}

sub wait {
	my ($self, $new) = @_;
	my $old = $self->{wait};
	if (@_ > 1) {
		$self->{wait} = $new;
	}
	return $old;
}

sub receive {
	my ($self, $new) = @_;
	my $old = $self->{receive};
	if (@_ > 1) {
		$self->{receive} = $new;
	}
	return $old;
}

sub ssl {
	my ($self, $new) = @_;
	my $old = $self->{ssl};
	if (@_ > 1) {
		if (defined $new) {
			$self->{ssl} = $new;
		} else {
			$self->{ssl} = -1;
		}
	}
	if ((defined $old) && ($old eq -1)) {
		return;
	} else {
		return $old;
	}
}

sub comment {
	my ($self, $new) = @_;
	my $old = $self->{comment};
	if (@_ > 1) {
		$self->{comment} = $new;
	}
	return $old;
}

sub AUTOLOAD {
	my ($self, $new) = @_;

	my $name = $Archive::Har::Entry::Timings::AUTOLOAD;
	$name =~ s/.*://smx;   # strip fully-qualified portion

	my $old;
	if ($name =~ /^_[[:alnum:]]+$/smx) { # private fields
		$old = $self->{$name};
		if (@_ > 1) {
			$self->{$name} = $new;
		}
	} else {
		Carp::croak("$name is not specified in the HAR 1.2 spec and does not start with an underscore");
	}
	return $old;
}

sub TO_JSON {
	my ($self) = @_;
	my $json = {};
	if (defined $self->blocked()) {
		$json->{blocked} = $self->blocked() + 0;
	} else {
		$json->{blocked} = -1;
	}
	if (defined $self->dns()) {
		$json->{dns} = $self->dns() + 0;
	} else {
		$json->{dns} = -1;
	}
	if (defined $self->connect()) {
		$json->{connect} = $self->connect() + 0;
	} else {
		$json->{connect} = -1;
	}
	$json->{send} = ($self->send() || 0) + 0;
	$json->{wait} = ($self->wait() || 0) + 0;
	$json->{receive} = ($self->receive() || 0) + 0;
	if (defined $self->ssl()) {
		$json->{ssl} = $self->ssl() + 0;
	} else {
		$json->{ssl} = -1;
	}
	if (defined $self->comment()) {
		$json->{comment} = $self->comment();
	}
	foreach my $key (sort { $a cmp $b } keys %{$self}) {
		next if (!defined $self->{$key});
		if ($key =~ /^_[[:alnum:]]+$/smx) { # private fields
			$json->{$key} = $self->{$key};
		}
	}
	
	return $json;
}

1;
__END__
=head1 NAME

Archive::Har::Entry::Timings - Represents the timings for the individual phases during a request/response pair inside the HTTP Archive

=head1 VERSION

Version '0.11'

=head1 SYNOPSIS

    use Archive::Har();

    my $http_archive_string = '"log": { "version": "1.1", .... ';
    my $har = Archive::Har->new();
    $har->string($http_archive_string);
    foreach my $entry ($har->entries()) {
        my $timings = $entry->timings();
	print "Blocked: " . $timings->blocked() . "\n";
	print "DNS: " . $timings->dns() . "\n";
	print "Connect: " . $timings->connect() . "\n";
	print "Send: " . $timings->send() . "\n";
	print "Wait: " . $timings->wait() . "\n";
	print "Receive: " . $timings->receive() . "\n";
	print "Ssl: " . $timings->ssl() . "\n";
	print "Comment: " . $timings->comment() . "\n";
    }

=head1 SUBROUTINES/METHODS

=head2 new

returns a new Timings object

=head2 blocked

returns the time in milliseconds spent waiting for a network connection.  Ihe function will return undef if it does not apply to the current request

=head2 dns

returns the time in milliseconds spent in DNS resolution of the host name.  Ihe function will return undef if it does not apply to the current request

=head2 connect

returns the time in milliseconds spent making the TCP connection.  Ihe function will return undef if it does not apply to the current request

=head2 send

returns the time in milliseconds spent sending the request to the server.

=head2 wait

returns the time in milliseconds spent waiting for a response from the server.

=head2 receive

returns the time in milliseconds spent reading the response from the server.

=head2 ssl

returns the time in milliseconds spent negotiating the SSL/TLS session.  Ihe function will return undef if it does not apply to the current request

=head2 comment

returns the comment about the page timings

