#!/usr/bin/env perl

use Test::Most;

use autodie;
use feature qw(say);

use List::AllUtils;
use Module::Runtime qw(use_module);
use Path::Class qw(file);

use Bio::MUST::Core;
use Bio::MUST::Drivers;


say 'Note: tests designed for: exonerate from exonerate version 2.2.0';

my $class = 'Bio::MUST::Drivers::Exonerate';

# Note: provisioning system is not enabled to help tests to pass on CPANTS
my $app = use_module('Bio::MUST::Provision::Exonerate')->new;
unless ( $app->condition ) {
    plan skip_all => <<"EOT";
skipped all Exonerate tests!
If you want to use this module you need to install the Exonerate executable:
https://www.ebi.ac.uk/about/vertebrate-genomics/software/exonerate
If you --force installation, I will eventually try to install Exonerate with brew:
https://brew.sh/
EOT
}

my $dna_str = <<'IN';
CCTCCTCCCTCAGGCTGGGACGCAGAGACCGGCAGCGATTCGGAGGACGGGCTGCGGCCT
GACGTGCTGGTGTCTCTCGCGGCGCCCAAGCGCTGCGCTGGCCGCTTCTCCGGGCGCCAC
CACTTCGTGGCCGGCAGGTTCGTGCCCGATGACGTGCGCCGCAAGTTCGCTCTGCGCCTG
CCGGGATACACGGGCACCGACTGCGTCGCGGCACTGTGACCGCCACCCGCGGCCACACCG
CAGGGACCCTCGCCAATAAACAGCCCTCCCACCACCGCCGCCTCGCCTCCGCCTCCTTTG
TGCACCGGGCCCGCTGCGGGGGTGGGCACACTCGAAGAGGGACTTCACACGGTCCCTGGG
TGATCCAAGGCGGGGAGCATGAGAAGGGCGGTGGAGTGGGACTTCCCGCTGGCCTAGAAA
ACTTCAGCTAGGGCTGGGGGCGGTGGCTCCTGCCTGTAATCTCAGCACTTTGGGAGGCTG
AGGCTGGAGGATCGCTTAAGGCCAGGAGTTTGGGACCAGCCTGGGCAACATAGCAAGATC
CCGACTCTACAAATTAAAAGAAAAAGAAGAAAGAAAACTCGAGCCAGAAGAATGGAGGGT
AGATTCAGGGTGGGACTCCCAGACTGTTTGGGGTGGCCTAAAGCCTATAGGATCGAAGTG
AGACGCGAGACAGGACTTCCAGCCACCTAAAGCGGGTCCGAGCAGCTTGAGGCTGGATTC
CAAGTGACACTTCCCTCAGGCAGCTTGGGATGCAGCTTGGAGGACTGAGTGGGCTCGGGG
AGGGAGTTCCAGCAGGCCACGGGGAACCCAGGACAGCCTGAGGACGGAGGCACAGCCAGA
CGCACCAAGGGGGGCGACCCAGAGGGGTTGCTCGAGCGCCGTCACCCAGGGCCAGGGGGC
GTGGCCTGAATGGGGCGTGGCCAGGGCGGGATCTGCGGTCGCATCCCGCCCCCACTCTCA
GTCCCAGCGGCCGCCAGACCCGCCGGAGTTGGACCCGAGCACGCCGCGGAGCCCGGACCC
TCCCTCGGACGCTCTGCCCCGGCCATGGCGTCGCTGCTGCCACTGCTCTGTCTCTGTGTC
GTCGCTGCGCACCTGGCGGGGGCCCGAGGTGAGGCGCCTCCAGCCCCCGCGCCCAGCCCC
TGAGGCTCCCGAGGGCCTGGCAGCCGGCCTTGGGTGAAGCCGCACTCGGGGAGAGAACCC
AGGGACCCACGCGCTTCCCCGGGGACTGGGTCCCCTTCACCAGTCGGATCCCCGCGACCA
CCTGGGGCGCGGAGCGGGCCTCATCCATTCTCTTCCTTGGCTAAGCCGGGAGTGTGGATA
AGAACCTAGGGTTCCTCCCGGGGTTTGGACTCTGAATCACCGGCCAGGTCTCCCACCCTG
GACCGAAGGGGTCGGAACTGGCCCCTCCCACCTAACTCCCACACACGCACTCACTCCAAC
CGCTGGAGAAAGAAATGCCCATCCCTGGAACACTCAATGGGGCGGGGGACGGGCCCACTT
CTCAAGTGAGGAGACTGAGGCACCCAGAAGGAAGGGGCTTGTCTGACCTCATCACTCTGC
TGGGAGTATCTTGGACAGCTCCTATCTTGGAGAAGCCAGTCTAGATTCCAAGGACTTCCC
GGCAGCTCGGACTGTAGTACAGGGATGGAAGACAGACTAAGGGAGGGGCTTCCAGCAGGC
CAAGTGGATGAGGGGGAGCCCAGCCTGAGGGCAGAGGCACAGCCAGCTGAACTTGATTGA
TCTCGCGGCAGCTGAACATAGTGGGAGGGAGACTCTAGCCTAGGGGAACTGGGTGGTACT
TGCACTTCCCCAGCCGGGGGGTTAGGGCTAGGTCAGAGAGAGGTACCTGGCTTTGGGGGC
CTGGTGGTGGTCATGGGCAGCGCGCTGGAGGAGAGTGTATGGCAGCCCTTGGGGCTAGAG
CTGGGTTCTGGCCTCTCAGAGCCTCAATTTTCCCTCCTGTTAAAATGGGGATCATGGGCC
AAGCACGGTGGCTCACGCCTGTAATCCCAGCACTTTGGGAGGCCGAGGCGGGCGGACTAC
CTGAGGTCAGGAGTTTGAGACCAACCTGGCCAACATGGTGAAACCCCGTCTCTACTAAAA
ATATAAAAATTAGCTAGGCCTGGTGGAGCGTGCCTGTAATCCCAGCTACTCAGGAGGCTG
AGGTGTGAGAATCACTTGAACCCAGGAGGCAGAGGCTGCAGTGAGCCGAGATCGTGCCAC
TGCACTCCAGCCTGGGTGACAGAGGGAGATTCCGTCTCAAAAAAAAAAAAAAAAAGGGGG
GGGATGATCGTGGCTGCACCTTGCCTAAAAGGAGGCTCCCAGCGGTCCCCATCAGTAGCA
GCGTGCTTCCTTCTCCCCACTCCTCACCCACAGACGCCACCCCCACCGAGGAGCCAATGG
CGACTGCACTGGGCCTGGAAAGACGGTCCGTGTACACCGGCCAGCCCTCACCAGCCCTGG
AGGACTGGGAAGGTGAGTTAGCCTGCCTCGGAGTCCCGTCTCTGCTGCGGGCTTGTTGGG
GGTGGGGCTTGGGCTACTGGGGTCGTGTGAGGAGAGGGCACACACTAGACGAAGGGCCCG
GCGCCGTCCTGGTCCCCGGACATGACAGCCCCTGGAGGTGATGCCCCAAGTTCAAGATCT
AAGTGAGAGGCCGGTCAGACAGAGGCAAGAGCTCAGCGCACCGGGATGGACCAGGTCAGG
CCCTGGGCGGCAGAACTGGGGTCGCGGGGAACCCAGTCTGCCCTGCACCTGTTTCAGGCC
GCTGGCTCGGGTCGTGGGCGCGCTCGGCTAGCCGGTGCCCACCGGGGGAGGGGGCTGAGA
CAGCAAGTAAGGCCTTTGCACGCATGCATGGGGGCCTACAGGCCGCCGCCCTGGTCCCAG
CGCGTGCGGTGCCCGCAGAGGCCAGCGAGTGGACGTCCTGGTTCAACGTGGACCACCCCG
GAGGCGACGGCGACTTCGAGAGCCTGGCTGCCATCCGCTTCTACTACGGGCCAGCGCGCG
TGTGCCCGCGACCGCTGGCGCTGGAAGCGCGCACCACGGACTGGGCCCTGCCGTCCGCCG
TCGGCGAGCGCGTGCACTTGAACCCCACGCGCGGCTTCTGGTGCCTCAACCGCGAGCAAC
CGCGTGGCCGCCGCTGCTCCAACTACCACGTGCGCTTCCGCTGCCCACTAGGTGAGGGCG
GGGCTGGATGACGGGGGCGGGGCTTTGAATGGGCGGGGCTGGCGAACGCCGGGAAGAGTC
GTGGTGGGTGGGGCTGGGAGGGGCAGGACCGTGTGGGTGTGGGCGTGGCTGGGAAGAGCC
GGGGGACCCTTAGGCGTAGGACGACGTCAGGGGGCGGGGCCAGGCAGGGCGGGGCTTTTG
AAGAGAGGCGCTGGGAACAGCGCGGGGCCCAAGTGCACCGTCAGGAGGCCCAGCGGGTCA
GGACCCATGTCCGGCCCCCTGGAACGCATCTCCCTCCCGCTGCCCTCATTTGAGCATCTG
TAAGAGCTCAGTTCTGTGGAGTGTCTTGAGCCCAAATGCGTGCAGGATGCAGGGAGAAAA
GCAGGACTGATCCCCGGGACCTTATGCGGGGGTGGGGAGGTGGGTCGGTACCTAAACTAT
GTCTTCGACAGGCGCGCCTAGGGGAGCATGGGTGGGGGCGATGGCAGATCTGCTCTCTGG
AGTGTCCGCCCTTGGATGCACATATGTGAGGGGGTTGGGGAAGAAGGGTCCTTTAGTCCC
AGGTTACCCGGAGGCGCAGTTCCTGGGGAGGGCGGCCACCTGATCTCCGTCCCTGCCTTC
CGCAGAAGCCTCGTGGGGCGCGTGGGGCCCGTGGGGTCCCTGCTCGGGGAGCTGTGGGCC
AGGCCGTCGCTTGCGCCGCCGCCACTGCCCAAGCCCCGCTGGGGATGCGTGTCCCGGGCG
TCCTCTGGAGGCGCAGAAGTGCGTGCGGCCTCGGTGTCCAGGTAGGAGGGGCGGGGTCTG
GAGGCTGGGACCTGTACAGAGGGGAGGAAGGGGAGGTTAGGGGCTGAGCCTGGGAGAGAG
GGTGGAGTTAGAGGCGGTGCCTGGATGTAGAGAGGCAGTGCTTCTCCTGAGCGATGCCTA
GAGGGGAGGAGCTAGTTGGGGGCGGGGCCTCGTGGACGGGGCTGTGAAGGGCGGGGCCTG
GTGCGGGGGCGGAGCCTGGAGGACCCAAACTCTGGAGCCAGGTGGAGACCAGCCCTTCCT
TCCCTCCTGGGCCCAAGGAGCACTGGTAACTATGCACATCTGGACAGGTGGATGCCAGGG
GGCTCTTGTGCAGGCAGACATGGGCAGATAGGAGGGGGTGCTTCCCCTGTGTGCATTCCA
GACTGTACAGATGTCTTCGTGGGGAAACAGAATTCAGTTCATGCACTCAGGGTGCACACG
TGCATGGCATTTGCAGCCCCCACCCCGGTCCCCCACGCCCACACAGACACCCGCTCACAG
CCTGGCAGACATGATGCCTAGGCCCTCCCAGGGCCAGACACCGGGCAAGCCAGGAGGAGA
CTGGAAATTCCGCTAGCGCCTGGGCTCCACAAACACGGCCTGGAGGCTGCTCCGGGGAGC
CCGTCACAGGGTGGGGGCACATGGTGGCCAATGGGGCGTTGGGGTCCCCACCCCATCGCG
CCTGGCTCTGCTCTCCCGGTGCCTGTGCCCTGAAGTTTCCAGTGGAATTTTCCATTTTAT
GGAGAAACCGCAGGATTGGGAAGGGGGGCAGTTGTTCTCAGCCCAAGATCCAGCCCCTTA
CTGGGCCCCCAGCCCCTGTCCAGACATTTTCTCTTCTGCTGGAAAGGGCAGTGCCATATT
TTGAGACTTACCTCTCCTGTCCCCTCCTATAGGACCTCTTCCCTACGTCTCTGGATCCCA
GTTCCTATGCCCTGGCCCAGTCCTGACCACTTAGCCCATCTCCCCTTCACACTAGGGACT
GCTCCAAGCAGGGCCATGCTGAGGCCTCTGAGAGGCACCTTATGGGGGGGCCCCAGGCAT
ATTTGTTGAGTGACTGATTGAATGGAAAGCCCTCCTCAGGAGGGAGTGTGAAGGTTTCTC
TGTCCTGCTTCCTCTCCATATCCCCCCAGGGTGCAGCCTTGACACCTGTGAATGCCCGGA
CCACATCCTCCTGGGCTCGGTGGTCACCCCATCTGGGCAACCACTGCTAGGAGCCAGGGT
CTCCCTGCGAGACCAGCCTGGCACTGTGGCCACCAGCGATGCTCACGGAACCTTCCGGGT
GCCTGGTGTCTGTGCTGACAGCCGCGCCAACATCAGGGCCCAGATGGATGGCTTCTCTGC
AGGGGAGGCCCAGGCCCAGGCCAACGGATCCATCTCTGTGGTCACCATCATCCTTGATAA
GTTGGGTAAGCACCCTTGCAACATGGGGCATGAAGGGGCTGAGGATCTGGGGAGGAAGTT
CTAGCACTCGATCAAGAGAGGAAGAAATGAGATGTGACACAGTCATTGGCGAAGGCACAG
AGTGAGTCGGCATTTCTGGGAGAAAGGAGAGGATGGGGACGTTGCTCCTAGGAAGGGTGG
GTGGGAGAAGCTCTGATCTTTACCACCTTTGACCCCAGAGAAGCCGTACCTGGTGAAACA
CCCTGAGTCCCGAGTGCGAGAGGCTGGCCAGAATGTGACTTTCTGCTGCAAAGCCTCCGG
GACCCCCATGCCCAAGAAATACTCCTGGTGAGCGCCCGCCCCGGGCTCAGGGGCATCTTC
TGTGGCTTTGGGGTTAGATTCAATTTCATGTTGTTATGTGGTTTCCCATCTGGAGAGTGG
GGAAAGCGTTAACGGGGAATGAATTGGGTGGGAGAGGGACTACCAAGGAGGCTGAATGGG
GCCAGTCTTGCTGGGGAGGAGGCTGACTTTGGCAAAGCCTCTCTGCAGCTCCCCAGCCTG
CAGACTGGGGGCAACACAGCCCCTCCCACTGAGTCCTATGGTGTCGCTCACCTGCCATCC
AGGTTCCACAATGGGACCCTGCTGGACAGGCGAGCTCATGGGTACGGGGCCCACCTGGAG
CTGCGGGGACTGCGCCCAGACCAGGCTGGCATCTACCACTGCAAGGCATGGAATGAGGCG
GGTGCCGTGCGCTCGGGCACTGCCCGGCTCACTGTACTTGGTGAGTGTCCTTGGCCACAG
CCCCGAGCAAGCCTTCACCCAAACGGAACCCGGACTGTAGCTAAGCTCAACCCCAAATGG
AGCCCCCACTTCAGACTGATCCAATCCTAGGCCTCCACTGAGCCCCCATACCACTCTGAG
CACCAACCTTCAGTCCTTACCCTGGGCAGACCCCTGTCCCTTAGGTGGAGTCCTTGACTG
CAGACAGAGTCCCCAGCCTTGGCCTCCATGAAGTCCTCCTCCCTGCCCTGACCTGCATGT
TTTCTTTGTCCCCAGCCCCAGGCCAGCCAGCCTGCGACCCCCGGCCCCGAGAGTACCTGA
TCAAGCTCCCTGAGGACTGTGGTCAGCCAGGTAGTGGCCCTGCCTACCTGGATGTGGGCC
TCTGTCCCGACACCCGCTGCCCCAGCCTGGCAGGCTCCAGCCCCCGCTGCGGGGACGCCA
GCTCCCGCTGCTGCTCTGTGCGCCGTCTGGAGAGAAGGGAGATTCACTGCCCTGGCTACG
TCCTCCCAGTGAAGGTGGTGGCAGAGTGTGGCTGCCAGAAGTGTCTGCCCCCTCGGGGGC
TGGTCCGGGGCCGTGTTGTGGCTGCTGACTCCGGGGAGCCGCTACGCTTCGCCAGGATTC
TGCTGGGCCAGGAGCCCATCGGCTTCACCGCCTACCAGGGCGACTTTACCATTGAGGTGC
CGCCCTCCACCCAGCGGCTGGTGGTGACTTTTGTGGACCCCAGCGGTGAGTTCATGGACG
CTGTCCGGGTCTTGCCTTTTGATCCTCGAGGTGCCGGCGTGTACCACGAGGTCAAGGCCA
TGCGGAAGAAAGCCCCGGTCATTTTACATACCAGCCAGAGCAACACGATCCCCCTGGGCG
AGCTGGAAGATGAGGCGCCCCTGGGCGAGCTGGTCCTGCCTTCTGGCGCTTTCCGCAGAG
CCGACGGCAAACCCTACTCGGGGCCTGTGGAGGCCCGGGTGACGTTCGTGGACCCCCGAG
ACCTCACCTCGGCGGCGTCTGCCCCCAGTGACCTGCGCTTCGTGGACAGCGACGGCGAGC
TGGCTCCACTGCGCACCTACGGCATGTTCTCCGTGGACCTCCGTGCGCCCGGCTCCGCGG
AGCAGCTGCAGGTGGGGCCGGTGGCCGTGCGGGTGGCCGCCAGCCAGATCCACATGCCAG
GCCACGTGGAGGCCCTCAAGCTGTGGTCGCTGAACCCCGAGACCGGCTTGTGGGAGGAGG
AGAGCGGCTTCCGGCGCGAGGGGTCCTCGGGCCCCCGGGTGCGCCGGGAGGAGCGCGTCT
TCCTGGTGGGCAACGTGGAGATCCGGGAGCGGCGCCTGTTCAATCTGGACGTGCCTGAGC
GCCGCCGCTGCTTCGTGAAGGTGCGCGCCTACGCCAACGACAAGTTCACCCCCAGCGAGC
AGGTGGAGGGCGTGGTGGTCACGCTGGTCAATCTGGAGCCCGCCCCCGGCTTCTCCGCCA
ACCCCCGTGCCTGGGGCCGCTTTGACAGCGCGGTCACCGGCCCCAATGGCGCCTGCCTCC
CCGCCTTCTGCGACGCCGACAGGCCAGACGCCTACACCGCCCTGGTCACCGCCACCCTGG
GCGGCGAGGAGCTGGAGCCGGCCCCTTCCTTGCCCCGCCCACTCCCGGCCACCGTGGGCG
TCACCCAGCCCTACCTGGACAGGCTGGGGTACCGTCGGACGGACCACGACGATCCCGCCT
TCAAGCGTAACGGCTTCCGCATCAACCTCGCCAAGCCCAGGCCAGGTGACCCCGCCGAGG
CCAATGGGCCTGTGTACCCGTGGCGCAGCCTGCGGGAATGCCAGGGGGCCCCGGTGACTG
CCAGCCACTTCCGCTTCGCCAGGGTGGAGGCGGACAAGTACGAGTACAACGTGGTCCCCT
TCCGAGAGGGCACACCTGCCTCCTGGACTGGCGATCTCCTGGCCTGGTGGCCCAACCCGC
AGGAGTTCCGGGCCTGCTTCCTCAAGGTGAAGATCCAGGGTCCCCAGGAGTATATGGTCC
GCTCCCACAACGCAGGGGGCAGCCACCCACGCACCCGCGGCCAGCTCTACGGACTTCGGG
ATGCCCGGAGTGTGCGAGACCCCGAGCGTCCGGGCACCTCGGCAGCCTGCGTGGAGTTCA
AGTGCAGCGGGATGCTGTTCGACCAGCGGCAGGTGGACAGGACGCTGGTGACCATTATGC
CCCAGGGCAGCTGCCGGCGCGTGGCCGTCAACGGACTCCTTCGGGATTACCTGACCCGGC
ACCCCCCACCGGTGCCCGCGGAGGACCCAGCTGCCTTCTCCATGCTGGCCCCCCTAGACC
CTCTGGGCCACAACTATGGCGTCTACACTGTCACTGACCAGAGCCCACGCTTGGCCAAGG
AGATCGCCATTGGCCGCTGCTTTGATGGTTCCTCTGACGGCTTCTCCAGAGAGATGAAGG
CTGATGCCGGCACAGCCGTCACCTTCCAGTGCCGGGAGCCACCGGCCGGACGACCCAGCC
TCTTCCAGAGGCTGCTGGAGTCCCCGGCGACAGCACTTGGTGACATCCGCAGGGAGATGA
GCGAGGCGGCGCAGGCACAGGCCCGGGCCTCAGGTCCCCTCCGCACCCGCCGGGGTAGGG
TCCGGCAGTGACCTGGGCAGGGGCCTCGCTTTCCCACCTCCCTCCAGACTCCTTTGACCC
CAGGAAGTTTTGCCCCTCCTTCTTCTCCAGACAGCCCCCTCCCCAGGTGTCTGGGTCCCC
TTTCCCGCCCCTTTCCAGAACTCAGAGTCAGACAAGAACCCAGAGCATCCGATGGTAGAA
ACACCAGGAAGACAATTGTTGCTGTGTGGTATGGAATGGAGTTTGCGGTGACTCTGGGGC
CAGCACCCAGGGGACGACGTTCAACCCTAGCCTGAAGGGACCCGCTCCCAGCTCAGAAGC
CGTCTCTGACTTCTCGTGCGTATTTTGACCCTGATTTCAATCTTCTACCCTTGGGAGTTC
TGGCGTTTGGCACAAAGTCCCCTCTGCCTGTTTGGAGCTCAGTGCTAGACCAGGTCCCCT
GCCCCGAGCTTTGTTTTTGGGGTTATTTATTGAAACAAAGTGTGGGGAGCTGGTTGTGGG
TGTGAGTGGGGGTGTGGGGTCCAGGCTGGGCCCAGTGAAAAGGAGGAAGGGGTTCCCATG
CGGGGGAGGCTCTGGGGCTGAGGGGAACAATTCTCACGTGTTTGGTGCTTAGAGACCTGC
CCGGGGCGTTGGGCAGGCCCTCCGGGGGCTGAATTAAAAATGCTTTATTTCCAACTCATG
GTGTCAGGCTCTAGGGAGGAAGGATGGAGGCTGGGGCTGCCCGGCCCAGATACACCTTTC
AGGCAGAAACACGCGGAGGCACACGCGGGTGCCGGGGGCAAGAAGCACGGGCGGCTGTGA
CCGCTGAACACAGTGGCAGCCTGCTGCTCTGGCCTGGACTCCTTGGCTGTGCCTGGCCGT
ATGCCCCGGGGCTACAGCAGGCAGGGAGGGGCCCTGGCAGGGGAGCCTCTGGGGCAGGCA
ACAGCAAGCCAGGCCACCCTGGACCTCCTCAGTCGAGTCCACAGCAGCCCCACCCTGGGC
CCCACCCACCTGAACCCCGGCCACACCTCGGTCATGTGGCGACGGGGCTGCCCGAGGCCA
GGGCTTTCTGACCTTGCCAGGACCCTGGGTCTGGAGGCGGCAGAATCACGGCTGGCTGGC
AGTCTGTGT
IN

my $pep_str = <<'IN';
MAAIKTWVFSFLVLEVTTVLGRQTMLAQSVRRVQPVKRTPKTLAKPADSQESPGEWTTWF
NIDHPGGQGDYERLDAIRFYYGERVCARPLRLEARTTDWMPAGSTGQVVHGSPREGFWCL
NREQRPGQNCSNYTVRFLCPPGSLRGDAEHIWSSWSPWSKCSAACGHTGVQTRTRTCLAQ
TVSLCSEATEEGQLCMSQACTACDLTCPMGQVNADCDACMCQDFMLHGAISLPGGGPAPG
AAVYLLAKAPKMLTRTDSSGRFRVPGLCPDGKTILKITKTKFAPIMITMPKTSLKSATIN
AEFVRAETPYIVMNPEMKARRAGQSVSLCCKATGKPSPDKYFWYHNNTLLDPSLYKHESK
LVLRNLQQDQAGEYFCKAQSDAGAVKSKVTQLTVIAHDETPCNPTPESYLIRLPHDCFQN
ASNSFYYDVGRCPIKTCAGQQDNGIRCRDAVENCCGISRTEEREIQCSGYTLPTKVAVEC
SCQRCAETRSIVRGRVTAADNGEPMRFGHVYMGNNRVSMTGYKGTFTLHIPQDTERLVLT
FVDRLQKFVNTTKVLPFNKKGSAVFHEIKMLRQKEPITLEAMETNIIPLGEVIGEDPVAE
LEIPSKSFYRQNGEPFTGKVKASVTFLDPRNISTATAAQSDLNFINDEGDTFPLRTYGMF
SVDFRDEATSESLNAGKVKVHLDSTQVKMPEHVPAMKLWSLNPDTGLWEEEGDFKFESQR
RNKREERTFLVGNMEIRERRLFNLDVPESRRCFIKVRTYRSERFLPSEQIQGVVVSVINL
EPRTGFSSNPRAWGRFDSVITGPNGACLPAFCDDQSPDAYSVYVLASLSGEELEAVESSP
KFNPNAIGVPQPYLNKLKYRRTDHEDPRVKKTAFQISMAKPRPNSAEESNGPIYAFENLR
ACEEAPPSAAHFRFYQIEGDRYDYNTVPFNEDDPMSWTEDYLAWWPKPMEFRACYIKVKI
VGPLEVNVRSRNMGGTHRQTVGKLYGIRDVKSTRDRDQPNVSSACLEFKCSGMLYDQDRV
DRTLVKVIPQGSCHRASVNSMLHEYLVNHLPLAVNNDTSEYTMLAPLDPLGHNYGIYTVT
DQDPRTAKEIALGRCFDGTSDGSSRIMKSNVGVALTFNCAERQVGRQSAFQYLQSTPARS
PATGTVQGRVPAMRQQRASRGGLRRRGSMGPSEIFWCCSTTSEQLSLVVLSPFCPSHVTA
VVRLMHRLSLVNLNISGLVQFVSLCLYLLSLPHDADCHRAPTMTKQRFSS
IN

my @exp_seqs = (
[<<'CDS',
GACACCTGTGAATGCCCGGACCACATCCTCCTGGGCTCGGTGGTCACCCCATCTGGGCAA
CCACTGCTAGGAGCCAGGGTCTCCCTGCGAGACCAGCCTGGCACTGTGGCCACCAGCGAT
GCTCACGGAACCTTCCGGGTGCCTGGTGTCTGTGCTGACAGCCGCGCCAACATCAGGGCC
CAGATGGATGGCTTCTCTGCAGGGGAGGCCCAGGCCCAGGCCAACGGATCCATCTCTGTG
GTCACCATCATCCTTGATAAGTTGGCCCCAGAGAAGCCGTACCTGGTGAAACACCCTGAG
TCCCGAGTGCGAGAGGCTGGCCAGAATGTGACTTTCTGCTGCAAAGCCTCCGGGACCCCC
ATGCCCAAGAAATACTCCTGGTTCCACAATGGGACCCTGCTGGACAGGCGAGCTCATGGG
TACGGGGCCCACCTGGAGCTGCGGGGACTGCGCCCAGACCAGGCTGGCATCTACCACTGC
AAGGCATGGAATGAGGCGGGTGCCGTGCGCTCGGGCACTGCCCGGCTCACTGTACTTGCC
CCAGGCCAGCCAGCCTGCGACCCCCGGCCCCGAGAGTACCTGATCAAGCTCCCTGAGGAC
TGTGGTCAGCCAGGTAGTGGCCCTGCCTACCTGGATGTGGGCCTCTGTCCCGACACCCGC
TGCCCCAGCCTGGCAGGCTCCAGCCCCCGCTGCGGGGACGCCAGCTCCCGCTGCTGCTCT
GTGCGCCGTCTGGAGAGAAGGGAGATTCACTGCCCTGGCTACGTCCTCCCAGTGAAGGTG
GTGGCAGAGTGTGGCTGCCAGAAGTGTCTGCCCCCTCGGGGGCTGGTCCGGGGCCGTGTT
GTGGCTGCTGACTCCGGGGAGCCGCTACGCTTCGCCAGGATTCTGCTGGGCCAGGAGCCC
ATCGGCTTCACCGCCTACCAGGGCGACTTTACCATTGAGGTGCCGCCCTCCACCCAGCGG
CTGGTGGTGACTTTTGTGGACCCCAGCGGTGAGTTCATGGACGCTGTCCGGGTCTTGCCT
TTTGATCCTCGAGGTGCCGGCGTGTACCACGAGGTCAAGGCCATGCGGAAGAAAGCCCCG
GTCATTTTACATACCAGCCAGAGCAACACGATCCCCCTGGGCGAGCTGGAAGATGAGGCG
CCCCTGGGCGAGCTGGTCCTGCCTTCTGGCGCTTTCCGCAGAGCCGACGGCAAACCCTAC
TCGGGGCCTGTGGAGGCCCGGGTGACGTTCGTGGACCCCCGAGACCTCACCTCGGCGGCG
TCTGCCCCCAGTGACCTGCGCTTCGTGGACAGCGACGGCGAGCTGGCTCCACTGCGCACC
TACGGCATGTTCTCCGTGGACCTCCGTGCGCCCGGCTCCGCGGAGCAGCTGCAGGTGGGG
CCGGTGGCCGTGCGGGTGGCCGCCAGCCAGATCCACATGCCAGGCCACGTGGAGGCCCTC
AAGCTGTGGTCGCTGAACCCCGAGACCGGCTTGTGGGAGGAGGAGAGCGGCTTCCGGCGC
GAGGGGTCCTCGGGCCCCCGGGTGCGCCGGGAGGAGCGCGTCTTCCTGGTGGGCAACGTG
GAGATCCGGGAGCGGCGCCTGTTCAATCTGGACGTGCCTGAGCGCCGCCGCTGCTTCGTG
AAGGTGCGCGCCTACGCCAACGACAAGTTCACCCCCAGCGAGCAGGTGGAGGGCGTGGTG
GTCACGCTGGTCAATCTGGAGCCCGCCCCCGGCTTCTCCGCCAACCCCCGTGCCTGGGGC
CGCTTTGACAGCGCGGTCACCGGCCCCAATGGCGCCTGCCTCCCCGCCTTCTGCGACGCC
GACAGGCCAGACGCCTACACCGCCCTGGTCACCGCCACCCTGGGCGGCGAGGAGCTGGAG
CCGGCCCCTTCCTTGCCCCGCCCACTCCCGGCCACCGTGGGCGTCACCCAGCCCTACCTG
GACAGGCTGGGGTACCGTCGGACGGACCACGACGATCCCGCCTTCAAGCGTAACGGCTTC
CGCATCAACCTCGCCAAGCCCAGGCCAGGTGACCCCGCCGAGGCCAATGGGCCTGTGTAC
CCGTGGCGCAGCCTGCGGGAATGCCAGGGGGCCCCGGTGACTGCCAGCCACTTCCGCTTC
GCCAGGGTGGAGGCGGACAAGTACGAGTACAACGTGGTCCCCTTCCGAGAGGGCACACCT
GCCTCCTGGACTGGCGATCTCCTGGCCTGGTGGCCCAACCCGCAGGAGTTCCGGGCCTGC
TTCCTCAAGGTGAAGATCCAGGGTCCCCAGGAGTATATGGTCCGCTCCCACAACGCAGGG
GGCAGCCACCCACGCACCCGCGGCCAGCTCTACGGACTTCGGGATGCCCGGAGTGTGCGA
GACCCCGAGCGTCCGGGCACCTCGGCAGCCTGCGTGGAGTTCAAGTGCAGCGGGATGCTG
TTCGACCAGCGGCAGGTGGACAGGACGCTGGTGACCATTATGCCCCAGGGCAGCTGCCGG
CGCGTGGCCGTCAACGGACTCCTTCGGGATTACCTGACCCGGCACCCCCCACCGGTGCCC
GCGGAGGACCCAGCTGCCTTCTCCATGCTGGCCCCCCTAGACCCTCTGGGCCACAACTAT
GGCGTCTACACTGTCACTGACCAGAGCCCACGCTTGGCCAAGGAGATCGCCATTGGCCGC
TGCTTTGATGGTTCCTCTGACGGCTTCTCCAGAGAGATGAAGGCTGATGCCGGCACAGCC
GTCACCTTCCAGTGCCGGGAGCCACCGGCCGGACGACCCAGCCTCTTCCAGAGGCTGCTG
GAGTCCCCGGCGACAGCACTTGGTGACATCCGCAGGGAGATGAGCGAGGCGGCGCAGGCA
CAGGCCCGGGCCTCAGGTCCCCTCCGCACCCGCCGGGGT
CDS
],
[<<'CDS',
AGCGAGTGGACGTCCTGGTTCAACGTGGACCACCCCGGAGGCGACGGCGACTTCGAGAGC
CTGGCTGCCATCCGCTTCTACTACGGGCCAGCGCGCGTGTGCCCGCGACCGCTGGCGCTG
GAAGCGCGCACCACGGACTGGGCCCTGCCGTCCGCCGTCGGCGAGCGCGTGCACTTGAAC
CCCACGCGCGGCTTCTGGTGCCTCAACCGCGAGCAACCGCGTGGCCGCCGCTGCTCCAAC
TACCACGTGCGCTTCCGCTGCCCACTAGGT
CDS
]
);

my @exp_order = (1, 0);

my @exp_str_in_order = (
<<'CDS',
AGCGAGTGGACGTCCTGGTTCAACGTGGACCACCCCGGAGGCGACGGCGACTTCGAGAGC
CTGGCTGCCATCCGCTTCTACTACGGGCCAGCGCGCGTGTGCCCGCGACCGCTGGCGCTG
GAAGCGCGCACCACGGACTGGGCCCTGCCGTCCGCCGTCGGCGAGCGCGTGCACTTGAAC
CCCACGCGCGGCTTCTGGTGCCTCAACCGCGAGCAACCGCGTGGCCGCCGCTGCTCCAAC
TACCACGTGCGCTTCCGCTGCCCACTAGGT
CDS
,
<<'CDS',
GACACCTGTGAATGCCCGGACCACATCCTCCTGGGCTCGGTGGTCACCCCATCTGGGCAA
CCACTGCTAGGAGCCAGGGTCTCCCTGCGAGACCAGCCTGGCACTGTGGCCACCAGCGAT
GCTCACGGAACCTTCCGGGTGCCTGGTGTCTGTGCTGACAGCCGCGCCAACATCAGGGCC
CAGATGGATGGCTTCTCTGCAGGGGAGGCCCAGGCCCAGGCCAACGGATCCATCTCTGTG
GTCACCATCATCCTTGATAAGTTGGCCCCAGAGAAGCCGTACCTGGTGAAACACCCTGAG
TCCCGAGTGCGAGAGGCTGGCCAGAATGTGACTTTCTGCTGCAAAGCCTCCGGGACCCCC
ATGCCCAAGAAATACTCCTGGTTCCACAATGGGACCCTGCTGGACAGGCGAGCTCATGGG
TACGGGGCCCACCTGGAGCTGCGGGGACTGCGCCCAGACCAGGCTGGCATCTACCACTGC
AAGGCATGGAATGAGGCGGGTGCCGTGCGCTCGGGCACTGCCCGGCTCACTGTACTTGCC
CCAGGCCAGCCAGCCTGCGACCCCCGGCCCCGAGAGTACCTGATCAAGCTCCCTGAGGAC
TGTGGTCAGCCAGGTAGTGGCCCTGCCTACCTGGATGTGGGCCTCTGTCCCGACACCCGC
TGCCCCAGCCTGGCAGGCTCCAGCCCCCGCTGCGGGGACGCCAGCTCCCGCTGCTGCTCT
GTGCGCCGTCTGGAGAGAAGGGAGATTCACTGCCCTGGCTACGTCCTCCCAGTGAAGGTG
GTGGCAGAGTGTGGCTGCCAGAAGTGTCTGCCCCCTCGGGGGCTGGTCCGGGGCCGTGTT
GTGGCTGCTGACTCCGGGGAGCCGCTACGCTTCGCCAGGATTCTGCTGGGCCAGGAGCCC
ATCGGCTTCACCGCCTACCAGGGCGACTTTACCATTGAGGTGCCGCCCTCCACCCAGCGG
CTGGTGGTGACTTTTGTGGACCCCAGCGGTGAGTTCATGGACGCTGTCCGGGTCTTGCCT
TTTGATCCTCGAGGTGCCGGCGTGTACCACGAGGTCAAGGCCATGCGGAAGAAAGCCCCG
GTCATTTTACATACCAGCCAGAGCAACACGATCCCCCTGGGCGAGCTGGAAGATGAGGCG
CCCCTGGGCGAGCTGGTCCTGCCTTCTGGCGCTTTCCGCAGAGCCGACGGCAAACCCTAC
TCGGGGCCTGTGGAGGCCCGGGTGACGTTCGTGGACCCCCGAGACCTCACCTCGGCGGCG
TCTGCCCCCAGTGACCTGCGCTTCGTGGACAGCGACGGCGAGCTGGCTCCACTGCGCACC
TACGGCATGTTCTCCGTGGACCTCCGTGCGCCCGGCTCCGCGGAGCAGCTGCAGGTGGGG
CCGGTGGCCGTGCGGGTGGCCGCCAGCCAGATCCACATGCCAGGCCACGTGGAGGCCCTC
AAGCTGTGGTCGCTGAACCCCGAGACCGGCTTGTGGGAGGAGGAGAGCGGCTTCCGGCGC
GAGGGGTCCTCGGGCCCCCGGGTGCGCCGGGAGGAGCGCGTCTTCCTGGTGGGCAACGTG
GAGATCCGGGAGCGGCGCCTGTTCAATCTGGACGTGCCTGAGCGCCGCCGCTGCTTCGTG
AAGGTGCGCGCCTACGCCAACGACAAGTTCACCCCCAGCGAGCAGGTGGAGGGCGTGGTG
GTCACGCTGGTCAATCTGGAGCCCGCCCCCGGCTTCTCCGCCAACCCCCGTGCCTGGGGC
CGCTTTGACAGCGCGGTCACCGGCCCCAATGGCGCCTGCCTCCCCGCCTTCTGCGACGCC
GACAGGCCAGACGCCTACACCGCCCTGGTCACCGCCACCCTGGGCGGCGAGGAGCTGGAG
CCGGCCCCTTCCTTGCCCCGCCCACTCCCGGCCACCGTGGGCGTCACCCAGCCCTACCTG
GACAGGCTGGGGTACCGTCGGACGGACCACGACGATCCCGCCTTCAAGCGTAACGGCTTC
CGCATCAACCTCGCCAAGCCCAGGCCAGGTGACCCCGCCGAGGCCAATGGGCCTGTGTAC
CCGTGGCGCAGCCTGCGGGAATGCCAGGGGGCCCCGGTGACTGCCAGCCACTTCCGCTTC
GCCAGGGTGGAGGCGGACAAGTACGAGTACAACGTGGTCCCCTTCCGAGAGGGCACACCT
GCCTCCTGGACTGGCGATCTCCTGGCCTGGTGGCCCAACCCGCAGGAGTTCCGGGCCTGC
TTCCTCAAGGTGAAGATCCAGGGTCCCCAGGAGTATATGGTCCGCTCCCACAACGCAGGG
GGCAGCCACCCACGCACCCGCGGCCAGCTCTACGGACTTCGGGATGCCCGGAGTGTGCGA
GACCCCGAGCGTCCGGGCACCTCGGCAGCCTGCGTGGAGTTCAAGTGCAGCGGGATGCTG
TTCGACCAGCGGCAGGTGGACAGGACGCTGGTGACCATTATGCCCCAGGGCAGCTGCCGG
CGCGTGGCCGTCAACGGACTCCTTCGGGATTACCTGACCCGGCACCCCCCACCGGTGCCC
GCGGAGGACCCAGCTGCCTTCTCCATGCTGGCCCCCCTAGACCCTCTGGGCCACAACTAT
GGCGTCTACACTGTCACTGACCAGAGCCCACGCTTGGCCAAGGAGATCGCCATTGGCCGC
TGCTTTGATGGTTCCTCTGACGGCTTCTCCAGAGAGATGAAGGCTGATGCCGGCACAGCC
GTCACCTTCCAGTGCCGGGAGCCACCGGCCGGACGACCCAGCCTCTTCCAGAGGCTGCTG
GAGTCCCCGGCGACAGCACTTGGTGACATCCGCAGGGAGATGAGCGAGGCGGCGCAGGCA
CAGGCCCGGGCCTCAGGTCCCCTCCGCACCCGCCGGGGT
CDS
);

my $exp_complete_cds_str = <<'CDS';
AGCGAGTGGACGTCCTGGTTCAACGTGGACCACCCCGGAGGCGACGGCGACTTCGAGAGC
CTGGCTGCCATCCGCTTCTACTACGGGCCAGCGCGCGTGTGCCCGCGACCGCTGGCGCTG
GAAGCGCGCACCACGGACTGGGCCCTGCCGTCCGCCGTCGGCGAGCGCGTGCACTTGAAC
CCCACGCGCGGCTTCTGGTGCCTCAACCGCGAGCAACCGCGTGGCCGCCGCTGCTCCAAC
TACCACGTGCGCTTCCGCTGCCCACTAGGTGACACCTGTGAATGCCCGGACCACATCCTC
CTGGGCTCGGTGGTCACCCCATCTGGGCAACCACTGCTAGGAGCCAGGGTCTCCCTGCGA
GACCAGCCTGGCACTGTGGCCACCAGCGATGCTCACGGAACCTTCCGGGTGCCTGGTGTC
TGTGCTGACAGCCGCGCCAACATCAGGGCCCAGATGGATGGCTTCTCTGCAGGGGAGGCC
CAGGCCCAGGCCAACGGATCCATCTCTGTGGTCACCATCATCCTTGATAAGTTGGCCCCA
GAGAAGCCGTACCTGGTGAAACACCCTGAGTCCCGAGTGCGAGAGGCTGGCCAGAATGTG
ACTTTCTGCTGCAAAGCCTCCGGGACCCCCATGCCCAAGAAATACTCCTGGTTCCACAAT
GGGACCCTGCTGGACAGGCGAGCTCATGGGTACGGGGCCCACCTGGAGCTGCGGGGACTG
CGCCCAGACCAGGCTGGCATCTACCACTGCAAGGCATGGAATGAGGCGGGTGCCGTGCGC
TCGGGCACTGCCCGGCTCACTGTACTTGCCCCAGGCCAGCCAGCCTGCGACCCCCGGCCC
CGAGAGTACCTGATCAAGCTCCCTGAGGACTGTGGTCAGCCAGGTAGTGGCCCTGCCTAC
CTGGATGTGGGCCTCTGTCCCGACACCCGCTGCCCCAGCCTGGCAGGCTCCAGCCCCCGC
TGCGGGGACGCCAGCTCCCGCTGCTGCTCTGTGCGCCGTCTGGAGAGAAGGGAGATTCAC
TGCCCTGGCTACGTCCTCCCAGTGAAGGTGGTGGCAGAGTGTGGCTGCCAGAAGTGTCTG
CCCCCTCGGGGGCTGGTCCGGGGCCGTGTTGTGGCTGCTGACTCCGGGGAGCCGCTACGC
TTCGCCAGGATTCTGCTGGGCCAGGAGCCCATCGGCTTCACCGCCTACCAGGGCGACTTT
ACCATTGAGGTGCCGCCCTCCACCCAGCGGCTGGTGGTGACTTTTGTGGACCCCAGCGGT
GAGTTCATGGACGCTGTCCGGGTCTTGCCTTTTGATCCTCGAGGTGCCGGCGTGTACCAC
GAGGTCAAGGCCATGCGGAAGAAAGCCCCGGTCATTTTACATACCAGCCAGAGCAACACG
ATCCCCCTGGGCGAGCTGGAAGATGAGGCGCCCCTGGGCGAGCTGGTCCTGCCTTCTGGC
GCTTTCCGCAGAGCCGACGGCAAACCCTACTCGGGGCCTGTGGAGGCCCGGGTGACGTTC
GTGGACCCCCGAGACCTCACCTCGGCGGCGTCTGCCCCCAGTGACCTGCGCTTCGTGGAC
AGCGACGGCGAGCTGGCTCCACTGCGCACCTACGGCATGTTCTCCGTGGACCTCCGTGCG
CCCGGCTCCGCGGAGCAGCTGCAGGTGGGGCCGGTGGCCGTGCGGGTGGCCGCCAGCCAG
ATCCACATGCCAGGCCACGTGGAGGCCCTCAAGCTGTGGTCGCTGAACCCCGAGACCGGC
TTGTGGGAGGAGGAGAGCGGCTTCCGGCGCGAGGGGTCCTCGGGCCCCCGGGTGCGCCGG
GAGGAGCGCGTCTTCCTGGTGGGCAACGTGGAGATCCGGGAGCGGCGCCTGTTCAATCTG
GACGTGCCTGAGCGCCGCCGCTGCTTCGTGAAGGTGCGCGCCTACGCCAACGACAAGTTC
ACCCCCAGCGAGCAGGTGGAGGGCGTGGTGGTCACGCTGGTCAATCTGGAGCCCGCCCCC
GGCTTCTCCGCCAACCCCCGTGCCTGGGGCCGCTTTGACAGCGCGGTCACCGGCCCCAAT
GGCGCCTGCCTCCCCGCCTTCTGCGACGCCGACAGGCCAGACGCCTACACCGCCCTGGTC
ACCGCCACCCTGGGCGGCGAGGAGCTGGAGCCGGCCCCTTCCTTGCCCCGCCCACTCCCG
GCCACCGTGGGCGTCACCCAGCCCTACCTGGACAGGCTGGGGTACCGTCGGACGGACCAC
GACGATCCCGCCTTCAAGCGTAACGGCTTCCGCATCAACCTCGCCAAGCCCAGGCCAGGT
GACCCCGCCGAGGCCAATGGGCCTGTGTACCCGTGGCGCAGCCTGCGGGAATGCCAGGGG
GCCCCGGTGACTGCCAGCCACTTCCGCTTCGCCAGGGTGGAGGCGGACAAGTACGAGTAC
AACGTGGTCCCCTTCCGAGAGGGCACACCTGCCTCCTGGACTGGCGATCTCCTGGCCTGG
TGGCCCAACCCGCAGGAGTTCCGGGCCTGCTTCCTCAAGGTGAAGATCCAGGGTCCCCAG
GAGTATATGGTCCGCTCCCACAACGCAGGGGGCAGCCACCCACGCACCCGCGGCCAGCTC
TACGGACTTCGGGATGCCCGGAGTGTGCGAGACCCCGAGCGTCCGGGCACCTCGGCAGCC
TGCGTGGAGTTCAAGTGCAGCGGGATGCTGTTCGACCAGCGGCAGGTGGACAGGACGCTG
GTGACCATTATGCCCCAGGGCAGCTGCCGGCGCGTGGCCGTCAACGGACTCCTTCGGGAT
TACCTGACCCGGCACCCCCCACCGGTGCCCGCGGAGGACCCAGCTGCCTTCTCCATGCTG
GCCCCCCTAGACCCTCTGGGCCACAACTATGGCGTCTACACTGTCACTGACCAGAGCCCA
CGCTTGGCCAAGGAGATCGCCATTGGCCGCTGCTTTGATGGTTCCTCTGACGGCTTCTCC
AGAGAGATGAAGGCTGATGCCGGCACAGCCGTCACCTTCCAGTGCCGGGAGCCACCGGCC
GGACGACCCAGCCTCTTCCAGAGGCTGCTGGAGTCCCCGGCGACAGCACTTGGTGACATC
CGCAGGGAGATGAGCGAGGCGGCGCAGGCACAGGCCCGGGCCTCAGGTCCCCTCCGCACC
CGCCGGGGT
CDS

my $exp_translation_str = <<'PEP';
SEWTSWFNVDHPGGDGDFESLAAIRFYYGPARVCPRPLALEARTTDWALPSAVGERVHLN
PTRGFWCLNREQPRGRRCSNYHVRFRCPLGDTCECPDHILLGSVVTPSGQPLLGARVSLR
DQPGTVATSDAHGTFRVPGVCADSRANIRAQMDGFSAGEAQAQANGSISVVTIILDKLAP
EKPYLVKHPESRVREAGQNVTFCCKASGTPMPKKYSWFHNGTLLDRRAHGYGAHLELRGL
RPDQAGIYHCKAWNEAGAVRSGTARLTVLAPGQPACDPRPREYLIKLPEDCGQPGSGPAY
LDVGLCPDTRCPSLAGSSPRCGDASSRCCSVRRLERREIHCPGYVLPVKVVAECGCQKCL
PPRGLVRGRVVAADSGEPLRFARILLGQEPIGFTAYQGDFTIEVPPSTQRLVVTFVDPSG
EFMDAVRVLPFDPRGAGVYHEVKAMRKKAPVILHTSQSNTIPLGELEDEAPLGELVLPSG
AFRRADGKPYSGPVEARVTFVDPRDLTSAASAPSDLRFVDSDGELAPLRTYGMFSVDLRA
PGSAEQLQVGPVAVRVAASQIHMPGHVEALKLWSLNPETGLWEEESGFRREGSSGPRVRR
EERVFLVGNVEIRERRLFNLDVPERRRCFVKVRAYANDKFTPSEQVEGVVVTLVNLEPAP
GFSANPRAWGRFDSAVTGPNGACLPAFCDADRPDAYTALVTATLGGEELEPAPSLPRPLP
ATVGVTQPYLDRLGYRRTDHDDPAFKRNGFRINLAKPRPGDPAEANGPVYPWRSLRECQG
APVTASHFRFARVEADKYEYNVVPFREGTPASWTGDLLAWWPNPQEFRACFLKVKIQGPQ
EYMVRSHNAGGSHPRTRGQLYGLRDARSVRDPERPGTSAACVEFKCSGMLFDQRQVDRTL
VTIMPQGSCRRVAVNGLLRDYLTRHPPPVPAEDPAAFSMLAPLDPLGHNYGVYTVTDQSP
RLAKEIAIGRCFDGSSDGFSREMKADAGTAVTFQCREPPAGRPSLFQRLLESPATALGDI
RREMSEAAQAQARASGPLRTRRG
PEP

my @exp_full_ids = (
	'seq1 216 1167 . seq1 5080 8696 + 2636',
	'seq1 53 142 . seq1 2903 3173 + 331'
);

my @exp_full_ids_rev = (
	'seq1 216 1167 . seq1 4709 1093 - 2636',
	'seq1 53 142 . seq1 6886 6616 - 331'
);

my @exp_sugars_in_order = (
	[ qw( AK081544 AF542080  53  142 1 2903 3173 1  331 ) ],
	[ qw( AK081544 AF542080 216 1167 1 5080 8696 1 2636 ) ],
);

my @exp_sugars_in_order_rev = (
	[ qw( AK081544 AF542080  53  142 1 6886 6616 -1  331 ) ],
	[ qw( AK081544 AF542080 216 1167 1 4709 1093 -1 2636 ) ],
);

my @str_methods = qw(
	query_id target_id
);

my @num_methods = qw(
	 query_start  query_end  query_strand
	target_start target_end target_strand
	score
);

{
	my $dna_seq = Bio::MUST::Core::Seq->new(
		seq_id => 'AF542080',
		seq    => $dna_str,
	);
	my $pep_seq = Bio::MUST::Core::Seq->new(
		seq_id => 'AK081544',
		seq    => $pep_str,
	);

	my $fac = Bio::MUST::Core::GeneticCode::Factory->new(
		tax_dir => 'test/taxdump',
	);
	my $code = $fac->code_for(1);

	my $exo_fwd = $class->new(
		dna_seq      => $dna_seq,
		pep_seq      => $pep_seq,
		genetic_code => $code,
	);
	check_exo($exo_fwd, \@exp_sugars_in_order, \@exp_full_ids);

	my $exo_rev = $class->new(
		dna_seq      => $dna_seq->reverse_complemented_seq,
		pep_seq      => $pep_seq,
		genetic_code => $code,
	);
	check_exo($exo_rev, \@exp_sugars_in_order_rev, \@exp_full_ids_rev);
}


sub check_exo {
	my $exo    = shift;
	my $sugars = shift;
	my $ids	   = shift;

	isa_ok $exo, $class;

	is_deeply [ map { $_->full_id } $exo->all_cds ], $ids,
        'got expected full_ids directly from Seqs';

	cmp_deeply [ $exo->cds_order ], \@exp_order,
		'got expected cds order';

	for my $sugar ($exo->all_sugars_in_order) {
        cmp_deeply [
            ( map {     $sugar->$_ } @str_methods ),
            ( map { 0 + $sugar->$_ } @num_methods )       # numberize results
        ], shift @{ $sugars },
            'got expected values for all methods for sugar: ';
        explain $sugar;
	}

	my @exp_exons_in_order = map {
		Bio::MUST::Core::Seq->new( seq_id => 'seq', seq => $_ )
	} @exp_str_in_order;

	cmp_deeply [ map { $_->seq } $exo->all_exons_in_order ],
			   [ map { $_->seq } @exp_exons_in_order ],
				'got expected exons in order';

	my $exp_complete_cds = Bio::MUST::Core::Seq->new(
		seq_id => 'CDS',
		seq    => $exp_complete_cds_str,
	);
	cmp_ok $exo->complete_cds->seq, 'eq', $exp_complete_cds->seq,
		'got expected complete CDS';

	my $exp_translation = Bio::MUST::Core::Seq->new(
		seq_id => 'PEP',
		seq    => $exp_translation_str,
	);
	cmp_ok $exo->translation->seq, 'eq', $exp_translation->seq,
		'got expected translation';

	return;
}

done_testing;
