package OpenInteract2::Action::User;

# $Id: User.pm,v 1.9 2003/06/10 00:59:43 lachoy Exp $

use strict;
use base qw( OpenInteract2::Action::CommonDisplay
             OpenInteract2::Action::CommonAdd
             OpenInteract2::Action::CommonUpdate
             OpenInteract2::Action::CommonRemove );
use OpenInteract2::Constants qw( :log );
use OpenInteract2::Context   qw( CTX DEBUG LOG );
use SPOPS::Secure qw( :level :scope );
use SPOPS::Utility;

$OpenInteract2::Action::User::VERSION = sprintf("%d.%02d", q$Revision: 1.9 $ =~ /(\d+)\.(\d+)/);

sub search_form {
    my ( $self ) = @_;
    return $self->generate_content(
                    {}, { name => 'base_user::user_search_form' } );
}

sub search {
    my ( $self ) = @_;
    my $fetch_params = $self->_search_params;
    my $iter = eval {
        CTX->lookup_object( 'user' )->fetch_iterator( $fetch_params )
    };
    if ( $@ ) {
        $self->param_add( error_msg => "Error running search: $@" );
    }
    return $self->generate_content(
                    { user_iterator => $iter },
                    { name => 'base_user::user_search_results' } );
}


sub _search_params {
    my ( $self ) = @_;
    my $request = CTX->request;
    my @search_fields = qw( first_name last_name email login_name );
    my %s = map { $_ => $request->param( $_ ) } @search_fields;
    my $user_class = CTX->lookup_object( 'user' );
    if ( $user_class->isa( 'SPOPS::LDAP' ) ) {
        return $self->_search_params_ldap( \%s );
    }
    if ( $user_class->isa( 'SPOPS::DBI' ) ) {
        return $self->_search_params_dbi( \%s );
    }
    return ();
}


sub _search_params_dbi {
    my ( $self, $search ) = @_;
    my @where = ();
    my @value = ();
    foreach my $field ( keys %{ $search } ) {
        next unless ( $search->{ $field } );
        push @where, " $field LIKE ? ";
        push @value, "%$search->{ $field }%";
    }
    return { order => 'login_name',
             where => join( ' AND ', @where ),
             value => \@value };
}


sub _search_params_ldap {
    my ( $self, $search ) = @_;
    my $field_map = CTX->lookup_object( 'user' )
                       ->CONFIG->{field_map} || {};
    my @filter_chunk = ();
    foreach my $field ( keys %{ $search } ) {
        next unless ( $search->{ $field } );
        my $ldap_prop = $field_map->{ $field };
        push @filter_chunk, "($ldap_prop=*$search->{ $field }*)";
    }
    return {} unless ( scalar @filter_chunk );
    return { filter => '(&' . join( '', @filter_chunk ) . ')' };
}


########################################
# COMMON CUSTOMIZATIONS

sub _display_add_customize {
    my ( $self, $template_params ) = @_;
    $template_params->{user}->{theme_id} = CTX->request->theme->id;
    return undef;
}

sub _display_form_customize {
    goto &_display_customize;
}

sub _display_customize {
    my ( $self, $template_params ) = @_;
    $template_params->{groups} = eval {
        $template_params->{user}->group
    };
    if ( $@ ) {
        $self->param_add(
            error_msg => "Failed to fetch user's groups: $@" );
    }
    return undef;
}

sub _add_customize {
    my ( $self, $user, $save_options ) = @_;
    $self->_check_password_change( $user, 'display_add' );
}

sub _update_customize {
    my ( $self, $user, $old_data, $save_options ) = @_;
    $self->_check_password_change( $user, 'display_form' );
}


sub _check_password_change {
    my ( $self, $user, $fail_task ) = @_;
    my $request = CTX->request;
    my $password = $request->param( 'password' );
    unless ( $password ) {
        DEBUG && LOG( LDEBUG, "User DID NOT ask to change password" );
        return;
    }
    my $password_confirm = $request->param( 'password_conf' );
    unless ( $password eq $password_confirm ) {
        $self->param_add(
               error_msg => 'Passwords do not match. Please re-enter.' );
        die $self->execute({ task => $fail_task });
    }
    DEBUG && LOG( LDEBUG, "User asked to change password. Changing." );
    if ( CTX->server_config->{login}{crypt_password} ) {
        $password = SPOPS::Utility->crypt_it( $password );
    }
    $user->{password} = $password;
    $self->param_add( status_msg => 'Password changed' );
    return undef;
}

# If this is a new user, allow the user to edit his/her own record;
# other security settings (WORLD, GROUP for site group) should be done
# in the normal way (via SPOPS configuration entries in
# 'creation_security'; also make the user a member of group 'public'

sub _add_post_action {
    my ( $self ) = @_;
    my $user = $self->param( 'c_object' );
    eval {
        $user->set_security({ scope    => SEC_SCOPE_USER,
                              level    => SEC_LEVEL_WRITE,
                              scope_id => $user->id });
        $user->make_public();
    };
    if ( $@ ) {
        LOG( LERROR, "Error modifying group membership: $@" );
        $self->param_add(
                error_msg => 'Record created but group membership wrong.' );
    }
    $self->param_add( status_msg => "User added to 'public' group" );
    return undef;
}

1;
