package OpenInteract::Handler::SiteSearch;

# $Id: SiteSearch.pm,v 1.14 2003/02/17 21:41:43 lachoy Exp $

use strict;
use Data::Dumper  qw( Dumper );

BEGIN {
    eval {
        require OpenInteract::FullText;
        require OpenInteract::ResultsManage;
    };
}
@OpenInteract::Handler::SiteSearch::ISA               = qw( OpenInteract::Handler::GenericDispatcher );
$OpenInteract::Handler::SiteSearch::VERSION           = sprintf("%d.%02d", q$Revision: 1.14 $ =~ /(\d+)\.(\d+)/);
$OpenInteract::Handler::SiteSearch::author            = 'chris@cwinters.com';
$OpenInteract::Handler::SiteSearch::default_method    = 'search';
@OpenInteract::Handler::SiteSearch::forbidden_methods = ();

use constant DEFAULT_PER_PAGE => 50;
use constant MAIN_SCRIPT      => '/SiteSearch';


sub search {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;

    my $apr = $R->apache;
    my $params = { main_script => MAIN_SCRIPT,
                   error_msg   => $p->{error_msg} };

    $params->{search_id} = $apr->param( 'search_id' );
    $params->{keywords}  = $apr->param( 'keywords' );

    unless ( $params->{keywords} or $params->{search_id} ) {
        my %nk_params = ( title       => 'No Keywords',
                          description => 'You did not give me any keywords to search, ' .
                                         'so I did not run a search. Please try again.'  );
        return $R->template->handler( {}, \%nk_params,
                                      { name => 'full_text::tiny_search' } );
    }

    $params->{current_page} = $apr->param( 'page' ) || 1;
    my $hits_per_page = DEFAULT_PER_PAGE;
    my ( $min, $max ) = OpenInteract::ResultsManage->find_page_boundaries(
                                                  $params->{current_page}, $hits_per_page );

    # Run the search for the first time and save the results

    unless ( $params->{search_id} ) {
        my @terms = split /\s+/, $params->{keywords};
        $params->{search_type} = $apr->param( 'search_type' ) || 'all';
        my @include_classes = $apr->param( 'include_class' );
        $R->DEBUG && $R->scrib( 1, "Searching for ($params->{keywords}) with ($params->{search_type})" );
        my $search_results = OpenInteract::FullText->search_fulltext_index({
                                      search_terms => \@terms,
                                      search_type  => $params->{search_type},
                                      return       => 'raw' }) || [];
        $R->DEBUG && $R->scrib( 2, "List of found object keys and frequencies:",
                                   Dumper( $search_results ) );

        # Persist the raw results and get the ID so we can use them

        my $results = OpenInteract::ResultsManage->new();
        $params->{search_id} = $results->save( [ map { $_->[0] } @{ $search_results } ],
                                               { extra      => [ map { $_->[1] } @{ $search_results } ],
                                                 extra_name => [ 'fulltext_score' ] } );
    }

    # Retrieve the persisted results and pass off to the OpenInteract::FullTextIterator

    my $results = OpenInteract::ResultsManage->new({ search_id => $params->{search_id},
                                                     min => $min, max => $max });
    $results->retrieve;
    $params->{search_iterator} = OpenInteract::FullTextIterator->new({
                                                  saved_results => $results->{result_list} });

    $params->{total_hits}  = $results->{num_records};
    $params->{total_pages} = $results->find_total_page_count( $params->{total_hits} );
    $R->{page}->{title} = 'Search Results';
    return $R->template->handler( {}, $params,
                                  { name => 'full_text::search_results' } );
}

1;

__END__

=head1 NAME

OpenInteract::Handler::SiteSearch - Perform searches using the FullText module.

=head1 SYNOPSIS

 http://www.myoisite.com/SiteSearch/?keywords=this+and+that

=head1 DESCRIPTION

Implement a full-text search of all objects on a website -- or in a
group, or whatever. Most of the real work is done in
L<OpenInteract::FullText> -- you might want to check it out.

=head1 METHODS

B<search>

Runs the search!

Input parameters (GET or POST, does not matter):

=over 4

=item *

B<keywords> ($)

Space-separated list of words to search for in the index.

=item *

B<search_type> ($) (optional -- defaults to 'all')

Type of search to run. Choices are 'any' (OR all the keywords) or
'all' (AND all the keywords).

=item *

B<include_class> (@) (optional -- defaults to all classes)

NOT IMPLEMENTED RIGHT NOW

Restrict your search results to the following classes. This is
typically implemented by a webform with checkboxes or a
multiple-select list where the value the user sees is the object type
(e.g., 'News') and the value passed to the backend is the class for
that object (e.g., 'MySite::News').

=back

=head1 TO DO

Nothing known.

=head1 BUGS

None known.

=head1 COPYRIGHT

Copyright (c) 2001-2002 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters E<lt>chris@cwinters.comE<gt>

=cut
