package FLV::Base;

use warnings;
use strict;

our $VERSION = '0.03';

my $verbose = 0;

=head1 NAME

FLV::Base - Utility methods for other FLV::* classes

=head1 LICENSE

Copyright 2006 Clotho Advanced Media, Inc., <cpan@clotho.com>

This library is free software; you can redistribute it and/or modify it
under the same terms as Perl itself.

=head1 METHODS

=over

=item $pkg->set_verbosity($boolean)

Changes the global verbose flag.  This controls whether debug() emits
messages or not.

=cut

sub set_verbosity
{
   my $pkg = shift;
   $verbose = shift;
   return;
}

=item $pkg->new()

Creates a new, generic instance.

=cut

sub new
{
   my $pkg = shift;
   return bless {}, $pkg;
}

=item $pkg->debug($msg)

=item $self->debug($msg)

Print the message to STDERR if the verbose flag is set to true.

=cut

sub debug
{
   return if (!$verbose);

   my $pkg_or_self = shift;
   my $msg  = shift;

   if ($msg !~ m/\n \z/xms)
   {
      $msg .= "\n";
   }

   print {*STDERR} $msg;
   return;
}

# utility method called by sub class get_info() methods
# Arguments:
#   $prefix is a string that is inserted with an underscore before all outgoing info keys
#   $fields is a hashref
#           the key is a field name for the tag instances
#           the value is undef or a lookup hashref
#               the key is the tag instance field value
#               the value is a human-readable string
#   $tags   is an arrayref of tag instances
# See FLV::Info::get_info() for more discussion

sub _get_info
{
   my $pkg    = shift;
   my $prefix = shift; # string
   my $fields = shift; # hashref
   my $tags   = shift; # arrayref

   my %info = (
      count => scalar @{$tags},
   );
   my %types = map {$_ => {}} keys %{$fields};
   for my $tag (@{$tags})
   {
      for my $field (keys %{$fields})
      {
         if (defined $tag->{$field})
         {
            $types{$field}->{$tag->{$field}}++;
         }
      }
   }
   for my $field (keys %{$fields})
   {
      my $counts = $types{$field};
      my @list = reverse sort {$counts->{$a} <=> $counts->{$b} || $a cmp $b} keys %{$counts};
      my $lookup = $fields->{$field};
      if ($lookup)
      {
         @list = map {$lookup->{$_}} @list;
      }
      $info{$field} = join q{/}, @list;
   }
   return map {$prefix.q{_}.$_ => $info{$_}} keys %info;
}

1;

__END__

=back

=head1 AUTHOR

Clotho Advanced Media Inc., I<cpan@clotho.com>

Primary developer: Chris Dolan

=cut
