package HPCD::SGE::Run;

### INCLUDES ######################################################################################

# safe Perl
use warnings;
use strict;
use Carp;

use Moose;

with 'HPCI::Run' => { theDriver => 'HPCD::SGE' };

my %job_info_mappings = (
    jobname      => 'name',
    jobnumber    => 'number',
    taskid       => 'task_number',
    qname        => 'queue_name',
    hostname     => 'host',
    qsub_time    => 'submission_time',
    ru_wallclock => 'wallclock_time',
    cpu          => 'cpu_time',
    io           => 'io_transfers_made',
    iow          => 'io_wait_time',
    ru_maxrss    => 'peak_real_memory',
    maxvmem      => 'peak_virtual_memory',
    mem          => 'integral_memory_time',
);

my @job_info_fields = (
    qw(
        account  arid      department end_time  failed     granted_pe group
        owner    priority  project    ru_idrss  ru_inblock ru_ismrss  ru_isrss
        ru_ixrss ru_majflt ru_minflt  ru_msgrcv ru_msgsnd  ru_nivcsw  ru_nsignals
        ru_nswap ru_nvcsw  ru_oublock ru_stime  ru_utime   slots      start_time
    )
);


has 'stats_keys' => (
    is => 'ro',
    isa => 'ArrayRef[Str]',
    init_arg => undef,
    default  => sub { [ values %job_info_mappings, @job_info_fields ] },
);

sub _collect_job_stats {
    my $self   = shift;
    my $status = shift;
    my $stath  = $self->stats;

    $stath->{exit_status} = $status;

    my $stage   = $self->stage;
    my $id      = $self->unique_id;
    my %info    = ();
    my $retries = 0;

    $stage->info("collecting job stats ($id)\n");
    GET_STATS:
    while (1) {
        open my $fh, '-|', "qacct -j $id 2>&1"
            or $self->croak("cannot open qacct command: $!");
        my $divcnt = 0;
        %info = ();
        while (<$fh>) {

            # $stage->info("processing qacct line: $_");
            next GET_STATS if /^error:/;
            if (/^===*$/) {
                %info = ();
                $self->warn(
"found duplicate jobs with same id ($id), using last one listed in qacct"
                    ) if ++$divcnt == 2;
                next;
            }
            if (my ( $key, $val ) = /(\w+)\s+(.*)/) {
                $key = $job_info_mappings{$key} // $key;
                $info{$key} = $val;
            }
        }
    }
    continue {
        last GET_STATS if keys %info || ++$retries == 5;
        sleep(10);
    }
    # $stage->info("updating job stats ($id)\n");
    while (my ( $k, $v ) = each %info) {
        $stath->{$k} = $v;
    }
}

around 'soft_timeout' => sub {
    my $orig = shift;
    my $self = shift;
    my $id   = $self->unique_id;
    $self->info( "Running: (qdel $id) to terminate stage job" );
    system( "qdel $id" );
};

around 'hard_timeout' => sub {
    my $orig = shift;
    my $self = shift;
    my $id   = $self->unique_id;
    system( "qdel -f $id" );
    sleep(2);
    $self->$orig(@_);
};

1;
