package Acme::MetaSyntactic::services;
use strict;
use Acme::MetaSyntactic::List;
our @ISA = qw( Acme::MetaSyntactic::List );
our $VERSION = '1.001';
__PACKAGE__->init();

our %Remote = (
    source  => 'http://www.insecure.org/nmap/data/nmap-services',
    extract => sub {
        return grep {/^\D/}
            map { s/[#\s].*//; y!-:.+/*!_!; s/__+/_/g; $_ }
            $_[0] =~ m!^(\S+)!gm;
    },
);

1;

=head1 NAME

Acme::MetaSyntactic::services - The services theme

=head1 DESCRIPTION

The names of the services usually found in F</etc/services>.

The first list came from my Debian system and was extracted with:

    perl -lane '$_=$F[0];y/-/_/;!/#/&&!$s{$_}++&&print' /etc/services

Which was then golfed down to:

    perl -lane '$_=$F[0];y/-/_/;!/#/&&$s{$_}++||print' /etc/services
    perl -lane '$_=$F[0];y/-/_/;/#/||$s{$_}++||print' /etc/services
    perl -ne 's/\s.*//;y/-/_/;/#/||$s{$_}++||print' /etc/services
    perl -pe 's/[#\s].*//;y/-/_/;$s{$_}++&&goto LINE' /etc/services
    perl -ne 's/[#\s].*//;y/-/_/;$s{$_}++||print' /etc/services

For version 0.68, this was golfed a little more, by using the symbol
table as the hash table it is:

    perl -ne 's/[#\s].*//;y/-/_/;$$_++||print' /etc/services

The reference list is given by the IANA, and available at
L<http://www.iana.org/assignments/port-numbers>.

A bigger services lists is used by B<nmap>:
L<http://www.insecure.org/nmap/data/nmap-services>.
This list is used to update the theme.

=head1 CONTRIBUTOR

Philippe "BooK" Bruhat.

=head1 CHANGES

=over 4

=item *

2014-08-18 - v1.001

Changed the source link to use the services list from nmap,
as the I<graffiti.com> site seems to have disappeared.

Updated from the source web site in Acme-MetaSyntactic-Themes version 1.041.

=item *

2012-05-07 - v1.000

Received its own version number in Acme-MetaSyntactic-Themes version 1.000.

=item *

2006-07-24

Updated with a link to the IANA list in Acme-MetaSyntactic version 0.84.

=item *

2006-04-03

Updated and made automatically updatable with the I<graffiti.com> list
in Acme-MetaSyntactic version 0.68.

This increased the theme size by a factor of 14.

=item *

2005-08-01

Introduced in Acme-MetaSyntactic version 0.33, published on August 1, 2005.

I got the idea for this list on IRC, on the day it was published.

=back

=head1 SEE ALSO

L<Acme::MetaSyntactic>, L<Acme::MetaSyntactic::List>.

=cut

__DATA__
# names
a17_an_an
a26_fap_fgw
aairnet_2
aal_lm
aap
abarsd
abatjss
abbaccuray
abcsoftware
about
abr_api
abyss
acap
acas
accelenet
accessbuilder
accessnetwork
acc_raid
accuracer
ace_client
ace_proxy
aci
acmaint_dbd
acmaint_transd
acms
acmsoda
acp
acp_conduit
acp_policy
acp_proto
acr_nema
active_net
activesync
activesync_notify
adap
adapt_sna
admd
admdog
admeng
admind
admins_lms
adobeserver_1
adobeserver_2
adobeserver_3
ads
advocentkvm
aed_512
aegate
aeroflight_ads
af
afesc_mc
affiliate
afp
afrog
afs
afs3_bos
afs3_callback
afs3_errors
afs3_fileserver
afs3_kaserver
afs3_prserver
afs3_rmtsys
afs3_update
afs3_vlserver
afs3_volser
agcat
agentsease_db
agentx
agps_port
ah_esp_encap
aiagent
aibkup
aimpp_port_req
aipn_reg
airport_admin
airs
airshot
ajp12
ajp13
alarm_clock_s
alesquery
alias
allstorcns
alpes
alta_ana_lm
altav_remmgt
altav_tunnel
altbsdp
altcp
altovacentral
amanda
amandaidx
amdsched
amidxtape
amiganetfs
amp
ampify
ampr_info
ampr_inter
ampr_rcmd
amqp
ams
amt_cnf_prot
amt_esd_prot
amt_soap_http
amt_soap_https
amx_icsp
amx_rms
amx_weblinx
analogx
and_lm
anet
anet_l
an_pcp
ansanotify
ansatrader
ans_console
ansoft_lm_1
ansoft_lm_2
ansysli
ansys_lm
ansyslmd
anthony_data
antidotemgrsvr
anynetgateway
aocp
aodv
aol
aol_1
aol_2
aol_3
ap
apani1
apani2
apani5
apc_2160
apc_2161
apc_2260
apc_3052
apc_3506
apc_7846
apc_9950
apc_9952
apc_agent
apc_necmp
apcupsd
apdap
apertus_ldp
apex_edge
apex_mesh
aplx
apm_link
apocd
apple_imap_admin
apple_iphoto
apple_licman
appleqtc
appleqtcsrvr
apple_sasl
apple_vpns_rp
apple_xsrvr_admin
appliance_cfg
applix
applusservice
appserv_http
appss_lm
appworxsrv
apri_lm
apwi_disc
apx500api_1
apx500api_2
arcisdms
arcpd
arcserve
ardt
ardus_cntl
ardus_mtrns
ardusmul
ardusuni
arepa_cas
argis_ds
ariel1
ariel2
ariel3
arns
asa
asa_appl_proto
asam
asap_sctp
asap_sctp_tls
asap_tcp
asci_val
ascomalarm
as_debug
asf_rmcp
asi
asipregistry
asip_webadmin
asnaacceler8db
aspeclmd
aspentec_lm
asprovatalk
asr
as_servermap
assuria_slm
asterix
astromed_main
at_3
at_5
at_7
at_8
atc_lm
at_echo
atex_elmd
atls
atmtcp
atm_zip_office
at_nbp
at_rtmp
ats
attachmate_g32
attachmate_uts
at_zis
audio_activmail
audiojuggler
audionews
audit
auditd
audit_transfer
aurora
aurora_cmgr
aurp
auth
autocuesmi
autocuetime
autodesk_lm
autodesk_nlm
availant_mgr
avantageb2b
avanti_cdp
avenue
avian
avocent_adsap
avocent_proxy
avsecuremgmt
avt_profile_1
avt_profile_2
axis_wimp_port
axon_lm
azeti
b2n
b2_runtime
babel
backburner
BackOrifice
backupedge
backupexec
backup_express
bacnet
bacula_dir
bacula_sd
bakbonenetvault
banyan_net
banyan_rpc
banyan_vip
bb
bbn_mmc
bbn_mmx
bcinameservice
bcslogc
bctp
bdp
bears_02
beorl
BESApi
beserver_msg_q
bex_webadmin
bex_xr
beyond_remote
bfd_control
bfd_multi_ctl
bftp
bgmp
bgp
bgpd
bh611
bhevent
bhfhs
bhmds
biap_mp
biff
bigbrother
biimenu
bim_pem
binderysupport
binkp
bintec_capi
bip
bitcoin
bittorrent_tracker
blackboard
blackice_alerts
blackice_icecap
blackjack
blaze
bl_idm
blp3
blp4
blueberry_lm
bluelance
blwnkl_port
bmap
bmc_ar
bmc_ea
bmc_messaging
bmc_perf_agent
bmpp
bnet
bnetfile
bnetgame
bo2k
board_roar
board_voip
boe_pagesvr
boe_processsvr
boinc
boks
boks_clntd
bootclient
bootserver
borland_dsj
bpjava_msvc
brain
bre
bridgecontrol
brightcore
brlp_0
brlp_2
broker_service
brvread
bsquare_voip
btprjctrl
btrieve
bts_x73
btx
bullant_srap
busboy
bv_agent
bvcontrol
bv_ds
bv_is
bv_queryengine
bv_smcsrv
bvtsonar
bwnfs
bytex
c1222_acse
c3
ca_audit_da
ca_audit_ds
cableport_ax
cab_protocol
cacp
cadis_1
cadis_2
cadkey_licman
cadkey_tablet
cadlock
cadsi_lm
cadview_3d
caerpc
caicci
ca_idms
caids_sensor
cajo_discovery
cal
call_logging
call_sig_trans
candp
canna
can_nds
canocentral0
canon_bjnp1
canon_bjnp2
canon_bjnp3
canon_bjnp4
capioverlan
capwap_control
car
CarbonCopy
cardax
carracho
cart_o_rama
cas
casanswmgmt
casp
caspssl
catchpole
cautcpd
cba8
cbserver
cbt
cce3x
cce4x
ccmail
ccmcomm
ccm_port
ccnx
ccp
ccproxy_ftp
ccproxy_http
ccs_software
cdbroker
cdc
cddbp_alt
cdfunc
cdid
cdn
cedros_fds
celatalk
centra
cernsysmgmtagt
cert_initiator
cert_responder
cfdptkt
cfengine
cfs
cft_0
cft_5
cft_6
cgms
cgn_stat
chargen
checksum
childkey_notif
chimera_hwm
chip_lm
chipper
chmd
chromagrafx
chshell
ci3_software_1
ci3_software_2
cichild_lm
cichlid
cimplex
cimtrak
cinegrfx_lm
ciphire_serv
cisco_avp
cisco_fna
cisco_ipsla
ciscopop
cisco_sccp
cisco_sys
cisco_tdp
cisco_tna
citadel
citrix_ica
citrixima
citriximaclient
citynl
citysearch
cl_1
clariion_evr01
cl_db_attach
clearcase
client_wakeup
cloanto_net_1
clp
cluster_disc
clvm_cfg
cm
cma
cmip_man
cmmdriver
cnrprotocol
coauthor
codaauth2
codasrv
codasrv_se
CodeMeter
cognex_insight
coldfusion_auth
collaborator
com_bardac_dw
comcam_io
commerce
commlinx_avl
commonspace
commplex_link
commtact_http
commtact_https
comotionback
comotionmaster
compaqdiag
compaq_evm
compaq_https
compaq_wcp
composit_server
compressnet
compx_lockview
comscm
con
concert
condor
conf
conference
conferencetalk
config_port
confluent
connection
connect_proxy
connect_server
connlcli
conspiracy
contamac_icm
contclientms
contentserver
controlit
corba_iiop
corba_iiop_ssl
corbaloc
corelccam
corelvideo
corerjd
cosmocall
courier
covia
cp_cluster
cpdi_pidas_cm
cpdlc
cplscrambler_al
cplscrambler_in
cplscrambler_lg
cppdp
cpqrpm_agent
cpq_tasksmart
cpq_wbem
cp_spxdpy
cqg_netlan_1
creativepartnr
creativeserver
crestron_cip
crestron_ctp
cronus
crs
cryptoadmin
crystalenterprise
crystalreports
cs_auth_svr
csbphonemaster
cscp
csdm
csdmbase
csd_mgmt_port
csi_sgwp
cslistener
cs_live
csms
csms2
csnet_ns
cspmlockmgr
cspuni
csrpc
cs_services
cst_port
csvr_proxy
ctf
ctp_state
ctsd
cucme_1
cucme_2
cucme_3
cucme_4
cuelink_disc
cuillamartin
cumulus
cumulus_admin
custix
cvc
cvc_hostd
cvd
cvmmon
cvspserver
cxtp
cxws
cyaserv
cybercash
cybro_a_bus
cycleserv
cycleserv2
cymtec_port
cypress
d2000kernel
daap
dab_sti_c
dandv_tester
danf_ak2
dasp
datasurfsrv
datasurfsrvsec
datex_asn
davsrcs
dayliteserver
dayna
daytime
dbase
dbbrowse
dbcontrol_oms
dbdb
dberegister
dbisamserver1
dbisamserver2
db_lsp
dbm
dbreporter
dbsa_lm
dbstar
dc
dca
dcap
d_cinema_rrp
dcp
dcs
dcs_config
dcsoftware
dctp
dcutility
d_data
d_data_control
dddp
ddi_tcp_2
ddm_dfm
ddm_rdb
ddm_ssl
ddns_v3
ddrepl
ddt
de_cache_query
decap
decauth
decbsrv
decladebug
dec_notes
dectalk
decvms_sysmgt
dei_icda
dell_rm_port
dellwebadmin_2
delta_mcp
denali_server
deos
derby_repli
de_server
deslogin
deslogind
device
device2
deviceshare
dfn
dhcpc
dhcp_failover
dhcp_failover2
dhcps
dhcpv6_client
dhcpv6_server
diagd
diagmond
diagnose_proc
dialpad_voice1
diameter
di_ase
dict
di_drm
dif_port
digiman
digital_vrc
direcpc_video
direct
directplaysrvr
directv_catlg
directv_soft
directv_web
dirgis
dis
discard
disclose
distccd
distinct
distinct32
dist_upgrade
dixie
dj_ilm
dl_agent
dlip
dls
dls_mon
dls_monitor
dlsrap
dlsrpn
dlswpn
dmaf_caster
dmdocbroker
dmidi
dn6_nlm_aud
dn6_smm_red
dna_cml
dnet_keyproxy
dnet_tstproxy
dnox
dnp
dns2go
dnsix
dns_llq
dnx
docstor
documentum
documentum_s
domain
domaintime
doom
dossier
down
downtools_disc
dpm
dproxy
dpsi
DragonIDSConsole
drip
drp
ds_admin
dsatp
dsc
dsETOS
dsf
dsfgw
dsmipv6
d_s_n
dsp
dsp3270
ds_srvr
dtag_ste_sb
dtk
dtn1
dtp
dtpt
dtspc
dtv_chan_req
dvapps
dvbservdsc
dvcprov_port
dvl_activemail
dvr_esm
dvs
dwf
dwmsgserver
dwr
dxadmind
dx_instrument
dxmessagebase1
dxmessagebase2
dxspider
dyna_access
dynamic3d
dynamid
ea1
eapsp
easy_soft_mux
e_builder
echo
ecolor_imager
ecomm
edb_server1
e_design_web
editbench
edm_std_notify
edonkey
e_dpnet
eenet
efcp
efi_lm
efi_mg
ehs_ssl
ehtp
eicon_server
eicon_slp
eicon_x25
eims_admin
eis
eisp
eklogin
ekshell
elan
elatelink
elcn
elcsd
elektron_admin
elfiq_repl
eli
Elite
ellpack
elm_momentum
elpro_tunnel
els
elvin_server
embl_ndt
emb_proj_cmd
emcads
emce
emcrmirccd
emfis_cntl
emfis_data
emperion
empire_empuma
empowerid
emprise_lsc
encore
encrypted_admin
enl
enl_name
enpc
enpp
enrp_sctp
enrp_sctp_tls
ent_engine
entexthigh
entextlow
entextmed
entextnetwk
entextxid
entomb
entrust_aaas
entrust_aams
entrust_ash
entrustmanager
entrust_sps
entrusttime
eoss
epmd
epnsdp
ep_nsp
epp
eppc
ep_pcp
eq_office_4940
eq_office_4941
equationbuilder
erpc
erunbook_agent
esbroker
escp_ip
es_elmd
esip
esl_lm
esp
esp_encap
esp_lm
esri_sde
esro_emsdp
esro_gen
essbase
essp
etebac5
etftp
EtherNetIP_1
EtherNetIP_2
etlservicemgr
ets
eudora_set
evb_elm
event_listener
event_port
evm
ev_services
ewall
e_woa
exasoftport1
exbit_escp
excerpt
excw
exec
exlm_agent
exp1
exp2
extensis
extensisportfolio
eyelink
eyetv
ezmeeting_2
facsys_router
fairview
famdc
farenet
fasttrack
fatserv
fax
faxportwinport
faxstfx_port
fazzt_admin
fc_cli
fc_faultnotify
fcp
fcp_addr_srvr2
fcp_srvr_inst1
fcp_udp
fc_ser
ff_annunc
ff_fms
ff_lr_port
ffserver
ff_sm
fg_fps
fg_sysupdate
fhc
fhsp
filemaker
filenet_powsrm
filenet_rpc
filenet_tms
filesphere
finger
fintrx
fiorano_msgsvc
firepower
firewall1_rdp
firstcall42
fis
fiveacross
fjappmgrbulk
fj_hdnet
fjhpjp
fjicl_tep_a
fjinvmgr
fjippol_cnsl
fjitsuappmgr
fjmpjps
fjmpss
flamenco_proxy
flashfiler
flashmsg
flexlm
flex_lm
flexlm0
flexlm1
flexlm10
flexlm2
flexlm3
flexlm5
flexlm7
flexlm9
fln_spx
florence
fmp
fmpro_fdal
fmpro_v6
fmsas
fmsascon
fmtp
fmwp
fnet_remote_ui
fodms
font_service
foresyte_sec
fotogcad
fpitp
fpo_fns
frc_hp
frc_lp
frc_mp
freeciv
fs_agent
ftp
ftp_agent
ftp_data
ftp_proxy
ftps
ftps_data
ftranhc
ft_role
ftsrv
fujitsu_dev
fujitsu_dtc
fujitsu_dtcns
funk_dialout
funk_logger
funkproxy
fuscript
fw1_mc_fwmodule
fw1_mc_gui
fw1_or_bgmp
fw1_secureremote
fxuptp
gacp
gadgetgate1way
gadgetgate2way
galaxy_network
galileo
galileolog
gamegen1
gandalf_lm
garcon
gat_lmd
gcsp
gdoi
gdomap
gdp_port
gds_adppiw_db
gds_db
geneous
genie
genie_lm
geniuslm
genrad_mux
geognosisman
gf
ggf_ncp
gilatskysurfer
ginad
git
gkrellm
globalcatLDAP
globalcatLDAPssl
global_wlink
globe
glogger
glrpc
gmrupdateserv
gnunet
gnutella
gnutella2
goahead_fldup
goldleaf_licman
go_login
gopher
gotodevice
gpfs
gppitnp
gprs_cube
gprs_sig
graphics
gridgen_elmd
gris
groove
groove_dpp
groupwise
gsidcap
gsiftp
gsigatekeeper
gsmp
gss_http
gss_xlicen
gtegsc_lm
gtp_control
gtrack_server
gvcp
gv_pf
gv_us
gwha
gw_log
h2250_annex_g
h225gatedisc
h248_binary
h2gf_w_2m
h323callsigalt
h323gatedisc
h323gatestat
h323hostcallsc
h323q931
hacl_cfg
hacl_gs
hacl_hb
hacl_local
hacl_probe
hacl_test
ha_cluster
hagel_dump
haipe_otnk
halflife
hao
hap
hart_ip
hassle
hb_engine
hcp_wismar
hdap
hddtemp
hde_lcesrvr_2
hdl_srv
health_trap
hecmtl_db
helix
hello_port
hems
here_lm
heretic2
hermes
hexen2
hfcs
hfcs_manager
hhb_gateway
hiperscan_id
hippad
hiq
hivestor
hks_lm
hmmp_ind
hmmp_op
homeportal_web
hostname
hosts2_ns
hotline
hotu_chat
houdini_lm
hp_3000_telnet
hp_alarm_mgr
hp_clic
hp_collector
hp_dataprotect
hp_hcip
hp_hcip_gwy
hpidsadmin
hp_managed_node
hpnpd
hppronetman
hp_sci
hp_server
hpss_ndapi
hp_status
hpstgmgr
hpvirtgrp
hpvmmagent
hpvmmcontrol
hpvmmdata
hp_webadmin
hri_port
http
http_alt
http_mgmt
http_proxy
http_rpc_epmap
https
https_alt
htuilsrv
husky
hybrid
hybrid_pop
hydap
hylafax
hyper_g
hyperip
hyperwave_isp
i3_sessionmgr
iad1
iad2
iad3
iafdbase
iafserver
ianywhere_dbns
iapp
iascontrol
iascontrol_oms
iasd
ias_neighbor
ias_reg
iatp_highpri
iberiagames
ibm_app
ibm_cics
ibm_db2
ibm_db2_admin
ibm_diradm
ibm_dt_2
ibm_mgr
ibm_mqisdp
ibm_mqseries
ibm_pps
ibm_res
ibm_rsyscon
ibm_ssd
ibm_wrless_lan
icad_el
icap
icb
iccrushmore
ice_location
icg_swp
ici
iclcnet_locate
iclcnet_svinfo
iclpv_dm
iclpv_nlc
iclpv_nls
iclpv_pm
iclpv_sas
iclpv_sc
iclpv_wsm
icl_twobase1
icl_twobase2
icl_twobase4
iconp
icp
icq
icslap
ida_agent
idcp
ideafarm_door
ideafarm_panic
ident
identify
idfp
idmaps
idmgratm
idp
idxp
ieee_mih
ieee_mms
ieee_mms_ssl
ies_lm
ifor_protocol
ifsf_hb_port
igcp
igi_lm
igo_incognito
igrs
iiimsf
iims
iiop
IIS
IISrpc_or_vat
imagequery
imap
imap3
imap4_ssl
imaps
imgames
imoguia_port
imqbrokerd
imsldoc
imsp
imtc_mcs
imyx
incp
index_net
index_pc_wb
indigo_server
indura
indy
i_net_2000_npr
infiniswitchcl
infolibria
infoman
informatik_lm
informer
infoseek
infowave
ingreslock
ingres_net
innosys
innosys_acl
inovaport1
insitu_conf
instantia
instl_bootc
instl_boots
intecom_ps1
intecourier
integra_sme
intellistor_lm
intelsync
interbase
interhdl_elmd
intersan
interserver
intersys_cache
interwise
int_rcv_cntrl
intrinsa
intuitive_edge
intv
invision
invokator
ioc_sea_lm
ionixnetmon
iop
ipcd
ipcd3
ipcore
ipcserver
ipdcesgbs
ipdd
ipether232port
ipfix
ipfixs
iphone_sync
ipp
ip_qsig
ipt_anri_anri
ipulse_ics
ipx
iqobject
iqserver
ique
iRAPP
irc
ircs
ircu
irdmi
irdmi2
irisa
iris_beep
iris_lwz
iris_xpc
iris_xpcs
irtrans
is99c
is99s
isakmp
isbconference1
ischat
iscsi
isdninfo
isg_uda_server
isi_gl
isis
isis_bcast
ismaeasdaqtest
ismc
ismserver
isnetserv
isode_dua
iso_ill
iso_ip
isoipsigport_1
isoipsigport_2
iso_tp0
iso_tsap
iso_tsap_c2
isqlplus
isrp_port
issa
issc
iss_console_mgr
issd
iss_mgmt_ssl
iss_realsec
iss_realsecure
ita_agent
itactionserver2
ita_manager
item
itinternet
itm_mcell_s
ito_e_gui
itu_bicc_stc
itv_control
iua
iuhsctpassoc
ivmanager
ivsd
ivs_video
iwec
iwg1
i_zipqd
izm
jaleosnd
java_or_OTGfileshare
jaxer_manager
jcp
jdmn_port
jediserver
jeol_nsddp_3
jeol_nsddp_4
jetdirect
jetstream
jini_discovery
jmact3
jmact5
jmact6
jmb_cds1
jmb_cds2
joaJewelSuite
joost
jprinter
jstel
jtag_server
kademlia
kauth
kdm
kerberos
kerberos_adm
kerberos_master
kerberos_sec
keyserver
keyshadow
keysrvr
kfserver
kingdomsonline
kingfisher
kink
kiosk
kip
kis
kjtsiteserver
klio
klogin
kme_trap_port
kmscontrol
knet_cmp
knetd
konspire2b
kpasswd
kpasswd5
kpop
krb524
krb5gatekeeper
krb_prop
krbupdate
kryptolan
kshell
kuang2
kvm_via_ip
kx
kyoceranetdev
l2f
L2TP
lam
la_maint
lan900_remote
landesk_cba
landesk_rc
landmarks
lanmessenger
lanrevagent
lanrevserver
lanserver
lansource
lansurveyorxml
laplink
lazy_ptop
lcs_ap
ldap
ldaps
ldapssl
ldgateway
ldp
ldxp
lecroy_vicp
legent_1
legent_2
leoip
lgtomapper
liberty_lm
licensedaemon
link
linuxconf
linx
lipsinc
lisp_cons
lisp_data
lispworks_orb
listcrt_port
listcrt_port_2
listen
litecoin
livelan
ljk_login
lkcmserver
llmnr
llm_pass
llsurfup_http
llsurfup_https
lmdp
lmp
lms
lmsocialserver
lm_sserver
lnvpoller
lnvstatus
loadsrv
localinfosrvr
lockd
locus_con
locus_map
lofr_lm
login
lonewolf_lm
lontalk_norm
lorica_in
lorica_in_sec
lorica_out
lot105_ds_upd
lotusmtap
lotusnotes
lrs_paging
LSA_or_nterm
lsi_raid_mgmt
lsnr
lsp_ping
lstp
ltcudp
ltp_deepspace
lumimgrd
lupa
lutap
lv_ffx
m2mservices
m2pa
m2ua
m3ua
macon
mac_srvr_admin
magenta_logic
magicnotes
mailbox
mailbox_lm
mailprox
mailq
maitrd
man
manage_exec
manyone_http
mao
mapper_mapethd
mapper_nodemgr
mapper_ws_ethd
marcam_lm
masqdialer
matip_type_a
matip_type_b
MaxumSP
maybe_fw1
maybe_veritas
mbg_ctrl
mc_client
mcer_port
mciautoreg
mcidas
mcns_sec
mcns_tel_ret
mcreport
mcs_messaging
mctp
mdap_port
mdbs_daemon
mdc_portmapper
mdns
mdnsresponder
mecomm
med_ci
media_agent
mediabox
mediaspace
medimageportal
med_ovw
meetingmaker
megaco_h248
megardsvr_port
memcachedb
menandmice_dns
menandmice_lpm
menandmice_mon
meregister
mesavistaco
metaconsole
meta_corp
metagram
metasage
metasys
meter
metricadbc
metrics_pas
mevent
mfcobol
mfserver
mftp
mgcp_gateway
mib_streaming
micom_pfs
microcom_sbp
micromuse_lm
micromuse_ncpw
microsan
microsoft_ds
mikey
mimer
minecraft
minger
minilock
mini_sql
miroconnect
mit_dov
miteksys_lm
mit_ml_dev
miva_mqs
mkm_discovery
mlchat_proxy
mloadd
mm_admin
mmcals
mmcc
mmpft
mnotes
mnp_exchange
mobileip_agent
mobilip_mn
mobrien_chat
moldflow_lm
molly
mon
mondex
mongod
monitor
monkeycom
montage_lm
mortgageware
MOS_lower
mosmig
MOS_soap
mount
mountd
movaz_ssc
mpc_lifenet
mpidcagt
mpidcmgr
mpm
mpm_flags
mpm_snd
mpp
mppolicy_mgr
mppolicy_v5
mpshrsv
mps_raft
mptn
mqe_broker
mrm
msantipiracy
ms_cluster_net
msdfsr
msdp
msdtc
msdts1
msexch_routing
msfrs
msfw_control
msg_auth
msg_icp
msgsys
mshvlm
msl_lmd
ms_lsa
msmq
msmq_mgmt
msnp
ms_olap1
ms_olap2
ms_olap3
ms_olap4
msolap_ptp2
msp
msql
ms_rome
msrp
msrpc
msr_plugin_port
ms_shuttle
ms_sna_base
ms_sna_server
ms_sql2000
ms_sql_m
ms_sql_s
ms_streaming
ms_v_worlds
ms_wbt_server
mtl8000_matrix
mtn
mtp
mtport_regist
mtqp
mtsserver
multidropper
multiling_http
multiplex
mumps
munin
mupdate
murray
muse
musiconline
must_backplane
must_p2p
mvs_capacity
mvx_lm
mxi
mxomss
mxxrlogin
myblast
mylex_mapd
mysql
mysql_cluster
mysql_cm_agent
mysql_im
mysql_proxy
mythtv
n1_fwp
n1_rmgmt
n2h2server
n2nremote
naap
nacagent
nacnl
na_er_tip
namemunge
nameserver
namp
napster
nas
nati_logos
nati_svrloc
nati_vi_server
nat_pmp
nat_t_ike
navisphere
nav_port
nbt_pc
nburn_id
nbx_cc
nbx_dir
nbx_ser
ncacn_ip_tcp
ncadg_ip_udp
ncconfig
ncd_conf
ncd_diag
ncd_diag_tcp
ncdmirroring
ncd_pref
ncd_pref_tcp
nced
ncld
ncp
ncpm_hip
ncr_ccl
ncube_lm
ndmp
ndm_requester
ndm_server
ndnp
ndsauth
ndsconnect
ndsp
ndtp
neod1
neod2
nerv
nesh_broker
nessus
nest_protocol
net2display
netarx
netaspi
netassistant
netattachsdmp
netbackup
netbill_auth
netbill_prod
netbios_dgm
netbios_ns
netbios_ssn
netbookmark
netboot_pxe
netbus
netcheque
netconf_beep
netconfsoapbeep
netconfsoaphttp
netconf_ssh
netcp
neteh
netgw
netinfo
netinfo_local
netiq_endpt
netiq_qcheck
netlabs_lm
netmagic
netmap_lm
netml
netmo_http
netmon
netmount
netmpi
netnews
neto_dcs
netopia_vo1
netopia_vo2
netopia_vo3
netop_rc
netop_school
neto_wol_server
netrcs
netrek
netrisk
netrix_sftm
netrjs_1
netrjs_2
netrjs_3
netrjs_4
netsaint
netsc_dev
netsc_prod
netscript
netserialext2
netspeak_cs
netstat
netuitive
netvenuechat
netview_aix_1
netview_aix_10
netview_aix_11
netview_aix_12
netview_aix_2
netview_aix_3
netview_aix_4
netview_aix_5
netview_aix_6
netview_aix_7
netview_aix_8
netview_aix_9
netviewdm1
netviewdm2
netviewdm3
netwall
netware_csp
netware_ip
netwatcher_mon
networklenss
netxms_agent
netxms_mgmt
netxms_sync
newacct
newbay_snc_mc
newgenpay
newlixconfig
newlixengine
newoak
new_rwho
news
newwavesearch
nexgen
nexstorindltd
nextstep
nexus_portal
nfa
nfs
nfsd_keepalive
nfsd_status
NFS_or_IIS
nfsrdma
nhci
nicelink
nicetec_mgmt
ni_ftp
nim
ni_mail
nimaux
nimhub
nimreg
nimsh
niobserver
nip
nitrogen
ni_visa_remote
nkd
nlogin
nmap
nmc_disc
nmea_0183
nm_game_admin
nmmp
nms
nmsd
nms_dpnss
nmsp
nmsserver
nms_topo_serv
nnsp
nntp
noadmin
nokia_ann_ch2
noteit
noteshare
notify
novar_dbase
novastorbakcup
novation
novell_lu6_2
nowcontact
npds_tracker
nping_echo
npmp_gui
npmp_local
npmp_trap
npp
nppmp
nqs
nrcabq_lm
nrpe
ns
nsesrvr
nsiiops
nsjtp_ctrl
nsjtp_data
nsp
nsrexecd
nss
nssagentmgr
nssocketport
nss_routing
nsstp
nst
nsw_fe
ntalk
ntp
nuauth
nucleus
nucleus_sand
nufw
nupaper_ss
nut
nuts_bootp
nuts_dem
nvcnet
nvmsgd
nw_license
oa_system
objcall
objective_dbc
objectmanager
obrpd
ocbinder
oceansoft_lm
ock
oc_lm
ocs_amu
ocs_cmu
ocserver
octopus
odbcpathway
odmr
odsi
oem_agent
office_tools
oftep_rpc
ohimsrv
oi_2000
oidsr
oirtgsvc
olsr
omabcastltkm
omad
oma_dcdocbs
oma_mlp
oma_mlp_s
oma_rlp_s
omasgport
oma_ulp
omfs
omginitialrefs
omhs
omid
omnilink_port
omnivision
omscontact
omserv
onbase_dds
onmux
onscreen
ontime
oob_ws_http
opalis_rdv
opalis_robot
opc_job_start
opc_job_track
opcua_udp
openhpid
openmanage
openmath
opennl_voice
openport
opentable
openvms_sysipc
openvpn
openwebnet
opsec_cvp
opsec_ela
opsec_lea
opsec_sam
opsec_ufp
opsession_prxy
opsession_srvr
opswagent
optech_port1_lm
optilogic
optima_vnet
optiwave_lm
oracle
oracleas_https
oracle_oms
oracle_vp2
ora_lm
orasrv
orbiter
orbix_config
orbix_locator
orbix_loc_ssl
orbplus_iiop
os_licman
osm_appsrvr
ospfd
ospf_lite
osu_nms
osxwebadmin
ottp
otv
ovbus
oveadmgr
overnet
ov_nnm_websrv
ovrimosdbman
ovsam_d_agent
ovsam_mgmt
ovtopmd
owamp_control
p2pcommunity
p2pq
pacerforum
pacmand
padl2sim
paging_port
pago_services1
palace_4
pammratc
pammrpc
pana
panagolin_ident
pando_pub
paradym_31
paragent
park_agent
parsec_master
partimage
passgo
passgo_tivoli
password_chg
patrol_ism
patrol_mq_gm
patrolview
pawserv
pcanywhere
pcanywheredata
pcanywherestat
pcduo
pcduo_old
pciarray
pclemultimedia
pcm
pcmail_srv
pc_mta_addrmap
pcnfs
pctrader
pda_gate
pdap
pdap_np
pda_sys
pdps
pds
pdtp
peerenabler
pegboard
pehelp
pe_mike
peport
perfd
perf_port
personal_link
pftp
ph
pharmasoft
pharos
philips_vc
phonebook
photuris
phrelay
pichat
picknfs
pim_port
pim_rp_disc
pip
pipes
pipe_server
piranha1
piranha2
pirp
pit_vpn
pk
pkagent
pk_electronics
pkix_3_ca_ra
pksd
pktcable_cops
pktcablemmcops
playsta2_app
playsta2_lob
plethora
pmdfmgt
pmsm_webrctl
pn_requester
pn_requester2
pns
polestar
polipo
pop2
pop3
pop3pw
pop3s
popup_reminders
portgate_auth
postgresql
pov_ray
powerburst
powerchute
powerchuteplus
poweronnud
powerschool
ppcontrol
ppp
ppsms
pptconference
pptp
pra_elmd
prat
precise_comm
presence
prgp
primaserver
printer
printer_agent
print_srv
priority_e_com
prismiq_plugin
privateark
privatechat
privatewire
priv_dial
priv_file
priv_mail
privoxy
priv_print
priv_rje
priv_term
priv_term_l
prm_nm
prm_nm_np
prm_sm
prm_sm_np
prnrequest
prnstatus
proaxess
profile
prolink
proofd
propel_msgsys
proshare1
proshare2
proshareaudio
prosharedata
prosharenotify
prosharerequest
prosharevideo
prospero
proxima_lm
proxy_plus
prsvp
ps_ams
pscupd
psi_ptt
pslserver
pssc
pt2_discover
ptcnameservice
ptk_alink
pulseaudio
puparp
purenoise
pvuniwien
pvxpluscs
pwdgen
pwgpsi
pxc_pin
pxc_roid
pxc_splr
pxc_splr_ft
pxc_spvr
pxc_spvr_ft
pyrrho
pythonds
q55_pcc
qaz
qbdb
qb_db_server
qbikgdp
qcp
qft
qip_audup
qip_login
qmqp
qmvideo
qnts_orb
qo_secure
qotd
qpasa_agent
qrh
qsc
qsm_gui
qsm_proxy
qsm_remote
qsnet_assist
qsnet_cond
qsnet_nucl
qsnet_trans
qsnet_workst
qtms_bootstrap
quake
quake2
quake3
quakeworld
quartus_tcl
quasar_server
quest_vista
quickbooksrds
quicktime
quotad
radacct
radan_http
radio_bc
radius
radius_dynauth
radmin
radmind
radsec
raid_ac
raid_am
raid_cc
raid_cd
raid_cs
raid_sf
rap
rapidmq_reg
rapido_ip
rap_ip
rap_listen
rap_service
raqmon_pdu
ratio_adp
ratl
raven_rmp
razor
rbakcup1
rcc_host
rcip_itu
rcp
rcst
rcts
rda
rdrmshc
rds
rds2
reachout
realm_rusd
realserver
rebol
recipe
re_conn_proto
redstone_cpss
reftek
registrar
relief
rellpack
re_mail_ck
remoteanything
remote_as
remote_collab
remotefs
remote_kis
remote_winsock
rendezvous
repcmd
repliweb
repscmd
repsvc
resacommunity
resvc
retrospect
RETS_or_BackupExec
rfa
rfb
rfe
rfio
rfmp
rfx_lm
rgtp
rhp_iibp
rib_slm
ricardo_lm
ridgeway2
rightbrain
rimsl
ripd
ripng
ris
ris_cm
rje
rkinit
rlm
rlm_admin
rlp
rlzdbase
rmc
rmiactivation
rmiaux
rmiregistry
rmonitor
rmonitor_secure
rmpp
rmt
rna_lm
rndc
robcad_lm
robix
roboeda
roboer
rockwell_csp1
rockwell_csp2
roketz
rootd
route
routematch
rpasswd
rpc2portmap
rpcbind
rpi
rplay
rrac
rrh
rrifmm
rrilwm
rrimwm
rrp
rsap
rsf_1
rsftp
rsh_spx
rsip
rsom
rsqlserver
rsvd
rsvp_encap_2
rsvp_tunnel
rsync
rtcm_sc104
rtelnet
rtip
rtmp
rtps_dd_mt
rtps_discovery
rtraceroute
rtsclient
rtsp
rtsps
rtsserv
rusb_sys_port
rushd
rwhois
rww
rxapi
rxe
rxmon
s1_control
sabarsd
sac
sacred
sae_urn
safetynetp
saft
sage_best_com2
sagectlpanel
sah_lm
samba_swat
samsung_unidex
sanavigator
sane_port
sanity
santools
sapcomm
sapeps
saphostctrl
saphostctrls
saposs
saprouter
sapv1
saris
sas_1
sas_2
sas_3
sasg
sasggprs
sasp
satvid_datalnk
savant
sbcap
sbi_agent
sbl
sbook
scan_change
scanstat_1
sccip_media
scc_security
sceanics
scientia_sdb
scientia_ssdb
sco_dtmgr
scohelp
scoi2odialog
sco_inetmgr
scol
scoremgr
sco_sysmgr
scotty_ft
sco_websrvrmg3
sco_websrvrmgr
scp
scp_config
scpi_raw
scrabble
screencast
scriptview
scservp
scte30
sctp_tunneling
scx_proxy
sd
sdadmind
sddp
sde_discovery
sdfunc
sdl_ets
sdlog
sdnskmp
sdo_ssh
sdp_portmapper
sdproxy
sdr
sds
sds_admin
sdsc_lm
sdserv
sdt_lmd
sdxauthd
seagull_ais
seagulllms
search
search_agent
seclayer_tcp
sec_t4net_srv
secure_aux_bus
secure_cfg_svr
secureidprop
securenetpro_sensor
securid
semantix
send
senomix01
sentinel_ent
sentinel_lm
sentinelsrm
seosload
serialgateway
serialnumberd
servergraph
serverview_as
serverview_asn
serverview_icc
servexec
servicemeter
servicetags
servistaitsm
servserv
servstat
set
sflm
sfs_config
sfs_smp_net
sftp
sftsrv
sgcp
sge_qmaster
sgi_dgl
sgi_esphttp
sgi_eventmond
sgmp
sgmp_traps
sgsap
shadowserver
shareapp
shell
shiva_confsrvr
shivadiscovery
shivahose
shivasound
shockwave
shockwave2
shrinkwrap
siam
siebel_ns
sieve
sift_uft
sightline
sigma_port
siipat
silc
silhouette
silkp3
silverpeakcomm
silverplatter
simba_cs
simbaexpress
simbaservices
simco
sim_control
simp_all
simplifymedia
sip
sip_tls
sitaradir
sitaraserver
sitewatch
sixnetudr
sixtrak
sj3
skkserv
skronk
skytelnet
slc_systemlog
slim_devices
slinkysearch
slm_api
slnp
slp
slslavemon
slush
smakynet
smap
smart_lm
smartsdp
smauth_port
smc_http
smc_https
smile
smip_agent
smntubootstrap
sm_pas_3
smpnameres
smpte
smsd
smsp
sms_rcinfo
sms_remctrl
sms_xfer
smtp
smtps
smux
smwan
snagas
snap
snare
s_net
snet_sensor_mgmt
snews
snifferserver
snmp
snmpdtls_trap
snmp_tcp_port
snmptrap
snpp
sns_channels
sns_credit
sns_dispatcher
sntp_heartbeat
soagateway
soap_beep
soap_http
socalia
socks
softaudit
softcm
softdataphone
softpc
softrack_meter
solid_mux
sometimes_rpc1
sometimes_rpc10
sometimes_rpc11
sometimes_rpc12
sometimes_rpc13
sometimes_rpc14
sometimes_rpc15
sometimes_rpc16
sometimes_rpc17
sometimes_rpc18
sometimes_rpc19
sometimes_rpc2
sometimes_rpc20
sometimes_rpc21
sometimes_rpc22
sometimes_rpc23
sometimes_rpc24
sometimes_rpc25
sometimes_rpc26
sometimes_rpc27
sometimes_rpc28
sometimes_rpc3
sometimes_rpc4
sometimes_rpc5
sometimes_rpc6
sometimes_rpc7
sometimes_rpc8
sometimes_rpc9
sonar
sonardata
soniqsync
sophia_lm
sophos
sops
sor_update
spamassassin
spc
spcsdlobby
spdp
spectraport
splitlock
spmp
spramsd
spsc
spss
spt_automation
spw_dialer
spw_dnspreload
sqdr
sqlexec
sqlexec_ssl
sqlnet
sql_net
sqlserv
sqlsrv
squid_htcp
squid_http
squid_ipc
squid_snmp
src
srmp
srp_feedback
srssend
ss7ns
sscan
ssdispatch
sse_app_config
ssh
sshell
ssh_mgmt
sslp
ssmc
sso_control
sso_service
ssp_client
ssserver
ssslic_mgr
ssslog_mgr
sstats
stanag_5066
starfish
starquiz_port
starschool
startron
stat_cc
stat_scanner
statsci1_lm
statsci2_lm
statsrv
statusd
stel
stgxfws
stmf
stm_pproc
stone_design_1
streetperfect
street_stream
stt
stun_p1
stun_p2
stun_p3
stun_port
stx
sua
submission
submit
submitserver
subntbcst_tftp
subseven
suitcase
suitjd
su_mit_tg
sun_answerbook
sun_as_iiops_ca
sun_as_jmxrmi
sun_as_jpda
suncacao_jmxmp
sunclustermgr
sun_dr
sun_manageconsole
sun_sea_port
sun_sr_jmx
sun_user_https
sunwebadmin
supdup
supermon
supfiledbg
supfilesrv
support
surf
surfcontrolcpa
surfpass
sur_meas
surveyinst
svn
svnetworks
svrloc
swdtp_sv
sweetware_apps
swift_rvf
swldy_sias
swx
swxadmin
sxmp
syam_webserver
sybase
sybaseanywhere
sybasedbsynch
sygatefw
symantec_av
symantec_sfdb
symb_sb_port
symplex
synapse
synchronet_db
synel_data
synoptics_trap
synotics_broker
synotics_relay
sype_transport
syscomlan
sysinfo_sp
syslog
syslog_conn
sysopt
systat
t2_brm
tabula
tacacs
tacacs_ds
tacnews
tacticalauth
taep_as_svc
tag_ups_1
talarian_mcast1
talarian_mcast2
talarian_mcast3
taligent_lm
talk
talon_engine
tam
tambora
tams
tapestry
tapeware
targus_getdata
targus_getdata1
targus_getdata2
taurus_wh
tbrpf
tcc_http
tcoflashagent
tcoregagent
tcp_id_port
tcpmux
tcpnethaspsrv
tdaccess
tdmoip
td_postman
tdp_suite
td_replica
td_service
teamcoherence
teamviewer
teedtap
telefinder
telelpathattack
telelpathstart
teleniumdaemon
telesis_licman
tell
telnet
telnets
tempo
tenebris_nts
tenfold
teradataordbms
teredo
terminaldb
texar
tftp
tgp
thrp
ticf_1
ticf_2
tick_port
tig
timbuktu
timbuktu_srv1
timbuktu_srv2
timbuktu_srv3
timbuktu_srv4
time
timed
timeflies
tinc
tinyfw
tip2
tip_app_server
tivoconnect
tl1_raw_ssl
tlisrv
tmi
tmo_icon_sync
tmosms0
tmosms1
tnETOS
tnp
tnp1_port
tns_cml
tn_timing
tn_tl_fd1
tn_tl_fd2
tn_tl_r1
tn_tl_w2
tolteces
topflow
topflow_ssl
topovista_data
topx
tor_control
tor_orport
tor_socks
tor_trans
touchnetplus
tpcsrvr
tpdu
tpip
tpmd
tqdata
tram
trap
trap_daemon
travsoft_ipx_t
tributary
trident_data
trim
Trinoo_Bcast
Trinoo_Master
Trinoo_Register
triomotion
tripwire
trivnet1
trivnet2
troff
tr_rsrb_p1
tr_rsrb_p2
tr_rsrb_p3
tr_rsrb_port
trusted_web
tsa
tscchat
tsdos390
tserver
tsilb
tsp
tsrmagt
ttntspauto
ttyinfo
tungsten_https
tunnel
tvbus
tvpm
twamp_control
twcss
twrpc
uaac
uadtc
uaiact
uarps
ubroker
udrawgraph
udt_os
ufmp
ufsd
uis
ulistserv
ulp
ulpnet
ultraseek_http
ultrex
umeter
unbind_cluster
unet
unicall
unidata_ldm
unieng
unify
unikeypro
unisys_eportal
unisys_lm
unitary
univ_appserver
unix_status
unizensus
unknown
unot
upnotifyp
upnp
ups
ups_engine
ups_onlinet
urm
user_manager
us_gv
us_srv
utcd
utime
utmpcd
utmpsd
uucp
uucp_path
uucp_rlogin
uuidgen
v5ua
vacdsm_sws
valisys_lm
varadero_1
varadero_2
vat
vat_control
vatp
vce
vchat
vcom_tunnel
vdmplay
vemmi
venus
venus_se
veracity
vergencecm
veritas_pbx
veritas_ucl
veritas_vis2
vettcp
vfo
via_ftp
vid
video_activmail
videotex
videte_cipc
vieo_fe
vinainstall
virprot_lm
virtual_places
virtualtape
virtualuser
visionpyramid
visitview
vistium_share
vitalanalysis
vlsi_lm
vmnet
vmodem
vmpwscs
vmrdp
vmsvc
vmsvc_2
vmware_fdm
vnas
vnc
vnc_1
vnc_2
vnc_3
vnc_http
vnc_http_1
vnc_http_2
vnc_http_3
vnetd
vnsstr
vnwk_prapi
vocaltec_admin
vocaltec_hos
vocaltec_wconf
vp2p
vpac
vpad
vpjp
vpnz
vpp
vpps_qua
vpps_via
vpsipport
vpvc
vpvd
vqp
vrace
vrml_multi_use
vrt
vrts_at_port
vrts_ipcserver
vrtstrapserver
vrxpservman
vsamredirector
vscp
vsiadmin
vsinet
vsixml
vslmp
vspread
vs_server
vstat
vtsas
vulture
vvr_data
vxcrnbuport
vytalvaultbrtp
wafs
wag_service
wanscaler
wap_push
wap_push_http
wap_push_https
wap_vcal_s
wap_vcard_s
wap_wsp
wap_wsp_s
warehouse_sss
warmspotMgmt
waste
watchdog_nt
watchme_7272
watcom_sql
watershed_lm
watilapp
wbem_http
wbem_https
wbem_rmi
wdbrpc
web2host
weblogin
webmail_2
webobjects
webster
westec_connect
westell_stats
wfremotertm
wherehoo
whisker
who
whoami
whois
whosells
whosockami
WibuKey
wincim
windb
winddlb
windows_icfw
winjaserver
winpcs
winpharaoh
winpoplanmess
win_rpc
wins
winshadow
wip_port
wizard
wkars
wkstn_mon
wlanauth
wmc_log_svc
wmedistribution
wmereceiving
wmereporting
wms
wnn6
wnn6_DS
work_sol
world_lm
worldscores
worldwire
wormux
wpages
wpgs
writesrv
wrs_registry
wsdapi
ws_discovery
wsman
wsmans
wsmlb
wso2esb_console
wssauthsvc
www_dev
wysdmc
x11
X11
X11_1
X11_2
X11_3
X11_4
X11_5
X11_59
X11_6
X11_7
X11_8
X11_9
x25_svc_port
x2_control
x9_icue
xact_backup
xaudio
xdmcp
xdsxdm
xecp_node
xfer
xfr
xgrid
xic
xiip
xingcsm
xingmpeg
xinuexpansion1
xinuexpansion2
xinuexpansion3
xinuexpansion4
xinupageserver
xlog
xmail_ctrl
xmlink_connect
xmlrpc_beep
xmltec_xmlmail
xmms2
xmpp
xmpp_bosh
xmpp_client
xmpp_server
xmquery
xnm_clear_text
xnmp
xnm_ssl
xns_auth
xns_ch
xns_courier
xns_mail
xns_time
xpanel
xprint_server
xprtld
xribs
xrl
xtell
xtreamx
xtreelic
xtrm
xvttp
xyplex_mux
yak_chat
yo_main
z39_50
zannet
zebra
zebrasrv
zenginkyo_1
zenginkyo_2
zen_pawn
zented
zep
zephyr_clt
zephyr_hm
zephyr_srv
zeroconf
zeus_admin
zfirm_shiprush3
zicom
zigbee_ip
zigbee_ips
zincite_a
zion_lm
zserv
