package Acme::MetaSyntactic::services;
use strict;
use Acme::MetaSyntactic::List;
our @ISA = qw( Acme::MetaSyntactic::List );
__PACKAGE__->init();

our %Remote = (
    source  => 'http://www.graffiti.com/services',
    extract => sub {
        return grep {/^\D/}
            map { s/[#\s].*//; y!-:.+/*!_!; s/__+/_/g; $_ }
            $_[0] =~ m!^(\S+)!gm;
    },
);

1;

=head1 NAME

Acme::MetaSyntactic::services - The services theme

=head1 DESCRIPTION

The names of the services usually found in F</etc/services>.

The first list came from my Debian system and was extracted with:

    perl -lane '$_=$F[0];y/-/_/;!/#/&&!$s{$_}++&&print' /etc/services

Which was then golfed down to:

    perl -lane '$_=$F[0];y/-/_/;!/#/&&$s{$_}++||print' /etc/services
    perl -lane '$_=$F[0];y/-/_/;/#/||$s{$_}++||print' /etc/services
    perl -ne 's/\s.*//;y/-/_/;/#/||$s{$_}++||print' /etc/services
    perl -pe 's/[#\s].*//;y/-/_/;$s{$_}++&&goto LINE' /etc/services
    perl -ne 's/[#\s].*//;y/-/_/;$s{$_}++||print' /etc/services

For version 0.68, this was golfed a little more, by using the symbol
table as the hash table it is:

    perl -ne 's/[#\s].*//;y/-/_/;$$_++||print' /etc/services

A bigger services lists is used by B<nmap>:
L<http://www.insecure.org/nmap/data/nmap-services>.
A huge list of services can also be found at
L<http://www.graffiti.com/services>.

The latest is now used for C<Acme::MetaSyntactic::services>,
which increased the theme size by a factor of 14.

=head1 CONTRIBUTOR

Philippe "BooK" Bruhat.

Introduced in version 0.33, published on August 1, 2005.

Updated and made automatically updatable with the I<graffiti.com> list
in version 0.68, published on April 3, 2006.

=head1 SEE ALSO

L<Acme::MetaSyntactic>, L<Acme::MetaSyntactic::List>.

=cut

__DATA__
# names
a13_an
a14
a15
a1_bs
a1_msc
a3_sdunode
a4_sdunode
aaftp
aairnet_1
aairnet_2
aal_lm
aamp
aap
aas
abacus_remote
abarsd
abatemgr
abatjss
abbaccuray
abbs
abcsoftware
abcvoice_port
about
abr_basic
abr_secure
abscoftware
acap
acas
accelenet
accessbuilder
accessnetwork
ac_cluster
accord_mgc
acc_raid
accu_lmgr
accuracer
accuracer_dbms
ace_client
ace_proxy
ace_server
ace_svr_prop
aci
acl_manager
acmaint_dbd
acmaint_transd
acmsoda
acp
acp_port
acptsys
acr_nema
acs2000_dsp
ac_tech
activememory
active_net
activesync
actnet
adaptecmgr
adapt_sna
admd
admeng
admind
admins_lms
adobeserver_1
adobeserver_2
adobeserver_3
adobeserver_4
adobeserver_5
ads
adtech_test
adtempusclient
advant_lm
aed_512
aeroflight_ads
aeroflight_ret
aes_discovery
af
afesc_mc
afpovertcp
afrog
afs
afs3_bos
afs3_callback
afs3_errors
afs3_fileserver
afs3_kaserver
afs3_prserver
afs3_rmtsys
afs3_update
afs3_vlserver
afs3_volser
aftmux
agcat
agentview
agentx
agpolicy
agps_port
agri_gateway
agriserver
ah_esp_encap
aicc_cmi
aic_np
aic_oncrpc
aimpp_hello
aimpp_port_req
aironetddp
airs
aises
ajp13
aker_cdp
alaris_disc
alarm
alarm_clock_c
alarm_clock_s
alchemy
alias
allstorcns
alpes
alpha_sms
alphatech_lm
alta_ana_lm
altalink
alta_smp
altav_remmgt
altav_tunnel
altserviceboot
amanda
amandaidx
ambit_lm
amdsched
amidxtape
amiganetfs
aminet
amp
ampr_info
ampr_inter
ampr_rcmd
ams
amt
amt_blc_port
amt_cnf_prot
amt_esd_prot
amx_rms
amx_webadmin
amx_weblinx
analogx
and_lm
anet
anet_b
anet_h
anet_l
anet_m
anoto_rendezv
an_pcp
ansanotify
ansatrader
ans_console
ansoft_lm_1
ansoft_lm_2
answersoft_lm
ansys_lm
ansyslmd
anthony_data
anynetgateway
aocp
aodv
aol
aol_1
aol_2
aol_3
ap
apc_2160
apc_2161
apc_2260
apc_3052
apc_3506
apc_5454
apc_5455
apc_5456
apc_6547
apc_6548
apc_6549
apc_7845
apc_7846
apc_9950
apc_9951
apc_9952
apc_necmp
apcupsd
apdap
apertus_ldp
apex_edge
apex_mesh
apogeex_port
apollo_admin
apollo_cc
apollo_data
apollo_gms
apollo_relay
apollo_status
apparenet_as
apparenet_tps
apparenet_ts
apparenet_ui
apple_licman
appleqtc
appleqtcsrvr
apple_sasl
appleugcontrol
appliance_cfg
applianttcp
appliantudp
applix
appman_server
appserv_http
appserv_https
appss_lm
appswitch_emp
appworxsrv
apri_lm
apw_registry
arbortext_lm
archisfcp
arcisdms
arcp
arcpd
ardt
ardus_cntl
ardus_mtrns
ardusmul
ardus_trns
ardusuni
arena_server
arepa_cas
arepa_raft
argis_ds
argis_te
aria
ariel1
ariel2
ariel3
aries_kfinder
ariliamulti
arkivio
armadp
armi_server
armtechdaemon
arns
array_manager
ars_master
ars_vista
artifact_msg
asa
asa_appl_proto
asam
asap_tcp
asap_tcp_tls
asci_val
asc_slmd
as_debug
asf_rmcp
asf_secure_rmcp
asi
asia
asipregistry
asip_webadmin
asnaacceler8db
asoki_sma
aspeclmd
aspen_services
aspentec_lm
asprovatalk
as_servermap
astromed_main
at_3
at_5
at_7
at_8
atc_appserver
atc_lm
at_echo
atex_elmd
athand_mmp
atlinks
atmp
atmtcp
atm_uhas
atm_zip_office
at_nbp
at_rtmp
ats
attachmate_g32
attachmate_s2s
attachmate_uts
att_intercom
at_zis
audio_activmail
audiojuggler
audionews
audit
auditd
auris
aurora_cmgr
aurp
auth
autocueds
autocuelog
autocuesmi
autocuetime
autodesk_lm
autodesk_nlm
availant_mgr
avail_epmap
avantageb2b
avauthsrvprtcl
av_emb_config
avenue
avian
avinstalldisc
aviva_sna
avocent_adsap
avocent_proxy
avsecuremgmt
avt_profile_1
avt_profile_2
awg_proxy
aws_brf
axis_wimp_port
axon_lm
aztec
b2n
backburner
BackOrifice
backroomnet
backupedge
backup_express
bacnet
bacula_dir
bacula_fd
bacula_sd
badm_priv
badm_pub
bandwiz_system
banyan_net
banyan_rpc
banyan_vip
base
battlenet
bb
bbars
bbn_mmc
bbn_mmx
bcinameservice
bcs_broker
bcs_lmserver
bctp
bctp_server
bdir_priv
bdir_pub
bdp
beacon_port
beacon_port_2
bears_01
bears_02
beeyond
beeyond_media
berknet
beserver_msg_q
beyond_remote
bfd
bfd_control
bftp
bgmp
bgp
bgs_nsi
bh611
bhevent
bhfhs
bhmds
biap_mp
biimenu
bim_pem
binderysupport
binkp
bintec_admin
bintec_capi
bintec_tapi
biolink_auth
blackboard
blackice_alerts
blackice_icecap
blackjack
bl_idm
blockade
blockade_bpsp
blocks
blueberry_lm
bluectrlproxy
bluelance
blwnkl_port
bmap
bmc_ar
bmc_data_coll
bmc_ea
bmc_grx
bmc_jmx_port
bmc_messaging
bmc_net_adm
bmc_net_svc
bmc_onekey
bmcpatrolagent
bmc_patroldb
bmcpatrolrnvu
bmc_perf_agent
bmc_perf_mgrd
bmc_perf_sd
bmc_reporting
bmpp
bnet
b_novative_ls
bnt_manager
bo2k
boinc_client
boks
boks_clntd
boks_servc
boks_servm
boldsoft_lm
boomerang
boosterware
bootclient
bootpc
bootps
bootserver
borland_dsj
boscap
bpcd
bpcp_poll
bpcp_trap
bpdbm
bpjava_msvc
bpmd
bprd
brcd
brcm_comm_port
bre
brf_gw
bridgecontrol
broker_service
brp
bruce
brvread
bspne_pcc
bsquare_voip
btpp2audctr1
btpp2sectrans
btprjctrl
btrieve
bts_appserver
bts_x73
btx
buddy_draw
bues_service
bullant_rap
bullant_srap
busboy
business
bvcdaemon_port
bvcontrol
bxp
bytex
bzflag
c3
ca_1
ca_2
cableport_ax
cab_protocol
cabsm_comm
caci_lm
cadabra_lm
cadencecontrol
cadis_1
cadis_2
cadkey_licman
cadkey_tablet
cadlock
cadlock2
cadsi_lm
cadview_3d
caevms
caicci
caiccipc
ca_idms
caids_sensor
CAIlic
Caistoragemgr
cal
caller9
call_logging
call_sig_trans
calltrax
callwaveiam
camac
cambertx_lm
camp
can_dch
canex_watch
can_ferret
can_ferret_ssl
canna
can_nds
can_nds_ssl
canocentral0
canocentral1
canon_bjnp1
canon_bjnp2
canon_bjnp3
canon_bjnp4
canon_capt
cap
capfast_lmd
caps_lm
cardax
cardbox
cardbox_http
cart_o_rama
cas
casanswmgmt
cas_mapi
castorproxy
catalyst
catchpole
caupc_remote
cautcpd
cba8
cbos_ip_port
cbserver
cbt
cce3x
ccmad
ccmail
ccmcomm
ccm_port
ccmrmi
ccowcmr
ccp
ccproxy_http
ccs_software
cctv_port
cd3o_protocol
cdbroker
cdc
cddbp_alt
cddn
cdfunc
cdid
cdl_server
cdn
cecsvc
celatalk
centerline
centra
cera_bcm
cert_initiator
cert_responder
cesdcdman
cesdcdtrn
cesdinv
cfdptkt
cfengine
cfingerd
cft_0
cft_1
cft_2
cft_3
cft_4
cft_5
cft_6
cft_7
cgi_starapi
cgms
chargen
charsetmgr
checkpoint_rtm
checksum
chevinservices
childkey_ctrl
childkey_notif
chimera_hwm
chip_lm
chipper
c_h_it_port
chmd
chromagrafx
chshell
ci3_software_1
ci3_software_2
cichild_lm
cichlid
cifs
cimplex
cimtrak
cindycollab
cinegrfx_elmd
cinegrfx_lm
circle_x
cisco_fna
cisco_net_mgmt
ciscopop
cisco_sccp
cisco_sys
cisco_tdp
cisco_tna
citadel
citrixadmin
citrixima
citriximaclient
citrix_rtmp
citynl
cl_1
clariion_evr01
cleanerliverc
clearcase
clearvisn
client_ctrl
cloanto_lm
cloanto_net_1
close_combat
clp
cluster_disc
clutild
clvm_cfg
cma
cmadmin
cmc_port
cmip_agent
cmip_man
cmmdriver
cnap
cncp
cnhrp
cnip
cnrp
cnrprotocol
cns_srv_port
coauthor
codaauth2
codasrv
codasrv_se
codima_rtp
cogitate
cognex_insight
cognima
cogsys_lm
coldfusion_auth
collaborator
colubris
com_bardac_dw
combox_web_acc
comcam
comcam_io
commandport
commerce
commlinx_avl
commonspace
commplex_link
commplex_main
commtact_http
commtact_https
community
compaqdiag
compaq_evm
compaq_https
compaq_scp
compaq_wcp
composit_server
compressnet
comsat
comscm
con
concert
conclave_cpp
concomp1
concurrent_lm
conf
conference
conferencetalk
config_port
confluent
connect
connect_client
connected
connection
connect_proxy
connect_server
connendp
connlcli
contentserver
continuus
controlit
coord_svr
cops
cops_tls
copysnap
corba_iiop
corba_iiop_ssl
corbaloc
corelccam
corelvideo
corel_vncadmin
corerjd
cosmocall
courier
covia
cp_cluster
cpdi_pidas_cm
cplscrambler_al
cplscrambler_in
cplscrambler_lg
cpqrpm_agent
cpqrpm_server
cpq_tasksmart
cpq_wbem
cpudpencap
cqg_netlan
cqg_netlan_1
creativepartnr
creativeserver
crestron_cip
crestron_ctp
crinis_hb
crip
crmsbits
crs
cruise_config
cruise_diags
cruise_enum
cruise_swroute
cruise_update
cr_websystems
cryptoadmin
cs_auth_svr
csbphonemaster
csccfirewall
csccredir
cscp
csdm
csdmbase
csd_mgmt_port
csd_monitor
csi_lfap
csi_sgwp
cslistener
cs_live
csms
csms2
csnotify
cso
csoft1
csoft_plusclnt
csoft_prev
csoftragent
cspclmulti
cspmlockmgr
cspmulti
cspuni
csregagent
cs_remote_db
cs_services
cst_port
csvr
csvr_proxy
csvr_sslproxy
ctcd
ctdhercules
ctechlicensing
ctf
ctiprogramload
cti_redwood
ctisystemmsg
ctp
ctsd
ctt_broker
ctx_bridge
cucme_1
cucme_2
cucme_3
cucme_4
cuillamartin
cumulus
cumulus_admin
custix
cvc
cvc_hostd
cvd
cvmmon
cvmon
cvspserver
cvsup
cyaserv
cybercash
cyborg_systems
cyc
cycleserv
cycleserv2
cylink_c
cymtec_port
cypress
cypress_stat
cytel_lm
d2000kernel
d2000webserver
d2k_datamover1
d2k_datamover2
d2k_tapestry1
d2k_tapestry2
d3winosfi
daap
dab_sti_c
daishi
dali_port
dandv_tester
danf_ak2
dantz
darcorp_lm
dashpas_port
dasp
datacaptor
data_insurance
data_port
datasurfsrv
datasurfsrvsec
datex_asn
datusorb
davinci
davsrc
davsrcs
dawn
daytime
dbabble
dbase
dbbrowse
dbcontrol_agent
dbcontrol_oms
dbdb
dberegister
dbisamserver1
dbisamserver2
dbm
dbref
dbreporter
dbsa_lm
dbstar
dc
dca
dccm
d_cinema_rrp
dcp
dcs
dcsoftware
dctp
dcutility
ddi_tcp_1
ddi_tcp_2
ddi_tcp_3
ddi_tcp_4
ddi_tcp_5
ddi_tcp_6
ddi_tcp_7
ddi_udp_1
ddi_udp_2
ddi_udp_3
ddi_udp_4
ddi_udp_5
ddi_udp_6
ddi_udp_7
ddm_dfm
ddm_rdb
ddm_ssl
ddns_v3
ddt
de_cache_query
decap
decauth
decbsrv
dec_dlm
decladebug
dec_mbadmin
dec_mbadmin_h
dec_notes
dectalk
decvms_sysmgt
dei_icda
delibo
dellpwrappks
dell_rm_port
dellwebadmin_1
dellwebadmin_2
delos_dms
delta_mcp
denali_server
de_noc
deos
descent3
de_server
designspace_lm
deskshare
desktop_dna
deskview
de_spot
devbasic
device
device2
devshr_nts
d_fence
dfoxserver
dhcp_failover
dhcp_failover2
dhcpv6_client
dhcpv6_server
dhe
diagmond
diagnose_proc
dialogic_elmd
dialog_port
dialpad_voice1
dialpad_voice2
diameter
diamondport
di_ase
dic_aida
dicom_iscl
dicom_tls
dict
dif_port
digiman
digital_notary
digital_vrc
digivote
direcpc_dll
direcpc_si
direcpc_video
direct
directplay
directplay8
directplaysrvr
directv_catlg
directvdata
directv_soft
directv_tick
directv_web
dirgis
discard
disclose
discovery_port
discp_client
discp_server
distcc
distinct
distinct32
dist_upgrade
di_traceware
dixie
dj_ice
dj_ilm
dka
dkmessenger
dl_agent
dlip
dls
dls_mon
dls_monitor
dlsrap
dlsrpn
dlswpn
dmaf_caster
dmaf_server
dmdocbroker
dmidi
dmod_workspace
dn6_nlm_aud
dn6_smm_red
dna
dna_cml
dnap
dnc_port
dnox
dnp
dns2go
dnsix
doc1lm
docent
docstor
domain
domaintime
donnyworld
dossier
down
dpap
dpcp
dpi_proxy
dpkeyserv
dproxy
dpserve
dpserveadmin
dpsi
DragonIDSConsole
drip
driveappserver
drmsfsd
drmsmc
drp
dsatp
dsc
dsdn
dsETOS
dsf
dsfgw
dslremote_mgmt
dsmcc_ccp
dsmcc_config
dsmcc_download
dsmcc_passthru
dsmcc_session
dsnl
dsom_server
dsp
dsp3270
dssiapi
ds_srv
dsx_agent
dtag_ste_sb
dtk
dtn1
dtp
dtp_dia
dts
dtserver_port
dtspcd
dttl
dtv_chan_req
dvbservdscport
dvcprov_port
dvl_activemail
dvr_esm
dvt_data
dvt_system
dwf
dwmsgserver
dwnmshttp
dwr
dxadmind
dx_instrument
dxmessagebase1
dxmessagebase2
dyna_access
dyna_lm
dynamic3d
dynamid
dyniplookup
dyn_site
dzdaemon
dzoglserver
e3consultants
ea1
easy_soft_mux
eba
ebinsite
echo
echonet
ecmport
ecnp
ecolor_imager
ecomm
ecovisiong6_1
ecp
ecsqdmn
edb_server1
edb_server2
editbench
edix
edm_adm_notify
edm_manager
edm_mgr_cntrl
edm_mgr_sync
edm_stager
edm_std_notify
edonkey_cc
edonkey_cm
efcp
efidiningport
efi_lm
eforward
efs
egptlm
egs
ehp_backup
ehtp
eicon_server
eicon_slp
eicon_x25
eims_admin
eisport
elad
elan
elanlm
elatelink
elcsd
eli
Elite
ellpack
elm_momentum
els
elvin_client
elvin_server
ema_sent_lm
embl_ndt
embrace_dp_c
embrace_dp_s
emcads
emce
emc_gateway
emcsymapiport
e_mdu
emfis_cntl
emfis_data
emperion
empire_empuma
emprise_lls
emprise_lsc
emp_server1
emp_server2
emsd_port
emware_epss
emware_moap
emware_oft
encore
e_net
enl
enl_name
enpc
enpp
ent_engine
entexthigh
entextlow
entextmed
entextnetwk
entextxid
entomb
entp
entrust_aaas
entrust_aams
entrust_ash
entrust_kmsh
entrust_sps
entrusttime
eoss
epc
epicon
epl_slp
epmap
epmd
epncdp2
epnsdp
ep_nsp
epp
eppc
ep_pcp
ept_machine
equationbuilder
ergolight
eristwoguns
erpc
esbroker
escp_ip
escvpnet
es_elmd
eserver_pap
esimport
esinstall
esip
esl_lm
esmagent
esmmanager
esnm_zoning
espeech
espeech_rtp
esp_encap
esp_lm
esps_portal
esri_sde
esro_emsdp
esro_gen
essbase
essp
essweb_gw
estamp
etb4j
etc_control
etebac5
etftp
ethercat
etlservicemgr
etp
ets
eudora_set
evb_elm
event_listener
event_port
everydayrc
evtp
evtp_data
ewall
ewcappsrv
ewctsp
ewlm
ewnn
e_woa
exapt_lmgr
exasoftport1
exbit_escp
exce
excerpt
excerpts
excw
exec
exlm_agent
exosee
expresspay
extensis
eye2eye
eyelink
eyetv
ezmeeting
ezmeeting_2
ezproxy
ezproxy_2
ezrelay
f5_globalsite
f5_iquery
facelink
facilityview
facsys_ntp
facsys_router
fagordnc
fastechnologlm
fastlynx
fast_rem_serv
fatpipe
fatserv
fax
faximum
faxportwinport
faxstfx_port
fc_cli
fc_faultnotify
fcip_port
fcmsys
fcopy_server
fcopys_server
fcp
fcp_addr_srvr1
fcp_addr_srvr2
fcp_cics_gw1
fcp_srvr_inst1
fcp_srvr_inst2
fcp_udp
fc_ser
fde
feitianrockey
femis
ferrari_foam
ff_annunc
ff_fms
ff_lr_port
ff_sm
fg_fps
fg_gip
fg_sysupdate
fhc
fhsp
filecast
filenet_cm
filenet_nch
filenet_pa
filenet_re
filenet_rmi
filenet_rpc
filenet_tms
filesphere
filex_lport
findviatv
finger
finle_lm
fintrx
fiorano_msgsvc
fiorano_rtrsvc
firefox
firemonrcc
firepower
firstcall42
fiveacross
fjappmgrbulk
fjcp
fjdmimgr
fjdocdist
fj_hdnet
fjhpjp
fjicl_tep_a
fjicl_tep_b
fjicl_tep_c
fjinvmgr
fjippol_cnsl
fjippol_polsvr
fjippol_port1
fjippol_port2
fjippol_swrly
fjitsuappmgr
fjmpcm
fjmpjps
fjmpss
fjris
fjsv_gssagt
fjsvmpor
fjswapsnp
fksp_audit
flamenco_proxy
flashfiler
flashmsg
flexlm
flex_lm
flexlm10
flirtmitmir
fln_spx
florence
flukeserver
fmpro_fdal
fmpro_internal
fnet_remote_ui
fodms
foliocorp
font_service
foresyte_clear
foresyte_sec
fotogcad
fpitp
fpo_fns
fryeserv
fsc_port
fsportmap
fs_qos
fs_rh_srv
ftp
ftp_agent
ftp_data
ftp_proxy
ftps
ftps_data
ftranhc
ftrapid_1
ftrapid_2
ft_role
ftsrv
fud
fujitsu_dev
fujitsu_dtc
fujitsu_dtcns
fujitsu_mmpdc
fujitsu_neat
funk_dialout
funk_license
funk_logger
funkproxy
futrix
fxaengine_net
fxp
fyre_messanger
g5m
gacp
gadgetgate1way
gadgetgate2way
galaxy7_data
galaxy_server
galileo
galileolog
gamegen1
gamelobby
gamesmith_port
gammafetchsvr
gandalf_lm
garcon
gat_lmd
gbjd816
gbmt_stars
gbs_smp
gbs_stp
gc_config
gcm_app
gcmonitor
gcsp
gdbremote
gdoi
gdomap
gdp_port
gds_db
gemini_lm
geneous
genie
genie_lm
genisar_port
geniuslm
genrad_mux
geolocate
gf
gg
ggf_ncp
ggz
giga_pocket
gilatskysurfer
ginad
giop
giop_ssl
gkrellmd
glbp
glishd
global_cd_port
global_dtserv
global_wlink
globe
globecast_id
globmsgsvc
glogger
glrpc
gmrupdateserv
gnunet
gnutella_rtr
gnutella_svc
goahead_fldup
goldleaf_licman
go_login
gopher
gpfs
gppitnp
gprs_cube
gprs_data
gprs_sig
gpsd
graphics
grf_port
gridgen_elmd
griffin
gris
groove
groove_dpp
groupwise
grubd
gsakmp
gserver
gsi
gsiftp
gsigatekeeper
gsmp
gss_http
gss_xlicen
g_talk
gtegsc_lm
gte_samp
gtp_control
gtp_user
gtrack_ne
gtrack_server
gttp
guibase
gv_us
gw
gw_call_port
gwen_sonya
gwha
gxs_data_port
gxtelmd
h2250_annex_g
h248_binary
h263_video
h2gf_w_2m
h323callsigalt
h323gatedisc
h323gatestat
h323hostcall
h323hostcallsc
hacl_cfg
hacl_gs
hacl_hb
hacl_local
hacl_monitor
hacl_poll
hacl_probe
hacl_qs
hacl_test
ha_cluster
hagel_dump
haipe_discover
haipe_otnk
halflife
hao
hap
harp
hassle
hbci
hb_engine
hcp_wismar
hdap
hde_lcesrvr_1
hde_lcesrvr_2
hdl_srv
healthd
health_polling
health_trap
heartbeat
hecmtl_db
hello
hello_port
hems
here_lm
heretic2
hermes
herodotus_net
hexarc
hexen2
hicp
high_criteria
hippad
hiq
hivep
hkp
hks_lm
hl7
hlibmgr
hlserver
hmmp_ind
hmmp_op
hnmp
homeportal_web
homesteadglory
honyaku
hostname
hosts2_ns
hotu_chat
houdini_lm
houston
hp_3000_telnet
hp_alarm_mgr
hp_clic
hp_collector
hp_dataprotect
hp_device_disc
hp_hcip
hp_hcip_gwy
hpidsadmin
hpidsagent
hp_managed_node
hp_nnm_data
hpnpd
hpoms_ci_lstn
hpoms_dps_lstn
hp_pdl_datastr
hpppssvr
hppronetman
hp_pxpib
hp_san_mgmt
hp_sca
hp_sci
hp_sco
hp_server
hp_sessmon
hpss_ndapi
hp_status
hpstgmgr
hpstgmgr2
hpvmmagent
hpvmmcontrol
hpvmmdata
hp_webadmin
hp_webqosdb
hri_port
hsl_storm
hs_port
hsrp
hsrpv6
htcp
http
http_alt
http_mgmt
http_rpc_epmap
https
htuilsrv
hunt
husky
hybrid
hybrid_pop
hydap
hylafax
hypercube_lm
hyper_g
hyperip
hyperscsi_port
hyperwave_isp
i3_sessionmgr
iad1
iad2
iad3
iafdbase
iafserver
iapp
ias_admind
ias_auth
iascontrol
iascontrol_oms
iasd
ias_reg
iatp_highpri
iatp_normalpri
iax
iberiagames
ibm3494
ibm_abtact
ibm_app
ibm_cics
ibm_db2
ibm_dial_out
ibm_diradm
ibm_diradm_ssl
ibm_dt_2
ibm_mqisdp
ibm_mqseries
ibm_mqseries2
ibm_pps
ibm_res
ibm_ssd
ibm_wrless_lan
ibp
ibprotocol
ibridge_data
ibridge_mgmt
ibus
ica
icabrowser
icad_el
icap
iccrushmore
icg_bridge
icg_iprelay
icg_swp
ici
iclcnet_locate
iclcnet_svinfo
iclpv_dm
iclpv_nlc
iclpv_nls
iclpv_pm
iclpv_sas
iclpv_sc
iclpv_wsm
icl_twobase1
icl_twobase10
icl_twobase2
icl_twobase3
icl_twobase4
icl_twobase5
icl_twobase6
icl_twobase7
icl_twobase8
icl_twobase9
icmpd
icon_discover
icp
icpv2
icslap
idac
ida_discover1
ida_discover2
idcp
ideafarm_catch
ideafarm_chat
ideesrv
iden_ralp
ident
identify
idfp
idmaps
idmgratm
idonix_metanet
idotdist
idp_infotrieve
idps
idrs
idware_router
idxp
iec_104
ieee_mms
ieee_mms_ssl
iee_qfx
ies_lm
ifcp_port
ife_icorp
ifor_protocol
ifsf_hb_port
igateway
igcp
igi_lm
igmpv3lite
igo_incognito
igrid
ii_admin
iims
iiop
iiw_port
ill
ilss
imagepump
imagequery
imap
imap3
imap4_ssl
imaps
imdocsvc
imgames
imip
imip_channels
immedianet_bcn
imoguia_port
impera
imprs
imqbrokerd
imqtunnel
imqtunnels
imsldoc
imsp
imtc_map
imtc_mcs
inbusiness
incognitorv
incp
index_net
index_pc_wb
indi
indigo_server
indigo_vbcp
indigo_vrmi
indura
indx_dds
indy
i_net_2000_npr
infiniswitchcl
influence
info_aps
info_cachesvr
infocrypt
info_eventsvr
infoexch
info_filesvr
infolibria
infoman
infomover
info_pagesvr
info_processvr
informatik_lm
informer
infoseek
infotos
info_was
infowave
ingreslock
ingres_net
iniserve_port
initlsmsad
innosys
innosys_acl
inova_ip_disco
insitu_conf
inspect
instantia
instl_bootc
instl_boots
intecom_ps1
intecom_ps2
intecourier
integral
integra_sme
intellistor_lm
intel_rci
intel_rci_mp
intelsync
interactionweb
interbase
interhdl_elmd
interintelli
interpathpanel
intersan
interserver
intersys_cache
interwise
interworld
intraintra
intrastar
int_rcv_cntrl
intrepid_ssl
intrinsa
intuitive_edge
intv
invision
invision_ag
invokator
ioc_sea_lm
iop
ipass
ipcd
ipcd3
ipcs_command
ipcserver
ipdd
ipdtp_port
ipether232port
iph_policy_adm
iph_policy_cli
ipp
ip_provision
ip_qsig
ipr_dglt
ipsec_nat_t
ipulse_ics
ipx
iqnet_port
iqobject
iqserver
ique
irc
ircs
irc_serv
ircu
irdg_post
irdmi
irdmi2
irisa
iris_beep
ironmail
ironstorm
irtrans
is99c
is99s
isakmp
isbconference1
isbconference2
ischat
iscsi
iscsi_target
isdd
isg_uda_server
isi_gl
isi_irp
isis
isis_am
isis_ambc
isis_bcast
ismaeasdaqlive
ismaeasdaqtest
ismserver
isns
isode_dua
isoft_p2p
iso_ill
iso_ip
isoipsigport_1
isoipsigport_2
isomair
iso_tp0
iso_tp0s
iso_tsap
iso_tsap_c2
ispipes
ispmmgr
isrp_port
issapi
issd
iss_mgmt_ssl
isysg_lm
ita_agent
itactionserver1
itactionserver2
italk
ita_manager
itap_ddtp
itb301
itelserverport
item
itinternet
itm_lm
itm_mccs
itm_mcell_s
itm_mcell_u
ito_e_gui
itose
itscomm_ns
itu_bicc_stc
itv_control
iua
ivcollector
ivecon_port
ivmanager
ivsd
ivs_video
iwb_whiteboard
iwec
iwlistener
iw_mmogame
iwserver
i_zipqd
jabber_client
jacobus_lm
jaleosnd
jamserverport
jargon
jaugsremotec_1
jaugsremotec_2
jaus
jboss_iiop
jboss_iiop_ssl
jbroker
jcp
jdatastore
jdl_dbkitchen
jdmn_port
jediserver
jerand_lm
jetcmeserver
jetform
jetformpreview
jibe_eb
jini_discovery
j_lan_p
jlicelmd
jmact3
jmact5
jmact6
jmb_cds1
jmb_cds2
jmevt2
jmq_daemon_1
jmq_daemon_2
jms
joltid
jomamqmonitor
journee
jpegmpeg
jprinter
jserv
jstel
jt400
jt400_ssl
jtag_server
juxml_port
jvclient
jvserver
jwalkserver
jwclient
jwserver
k3software_cli
k3software_svr
ka0wuc
kali
kamanda
kana
kar2ouche
kastenchasepad
kastenxpipe
kazaa
k_block
kdm
kentrox_prot
kerberos
kerberos_adm
kermit
keyserver
keyshadow
keysrvr
kfxaclicensing
kiosk
kis
kjtsiteserver
klogin
kme_trap_port
kmscontrol
knet_cmp
kofax_svr
konshus_lm
konspire2b
kopek_httphead
kpasswd
kpn_icw
kpop
krb524
krb5gatekeeper
kryptolan
kshell
ksysguard
kti_icad_srvr
kuang2
kv_agent
kv_server
kyoceranetdev
l2f
l3_exprt
l3_hawk
l3_hbmon
l3_ranger
l5nas_parchan
labrat
lam
la_maint
lan900_remote
landesk_cba
lanner_lm
lanserver
lansource
lansurveyor
lansurveyorxml
lanyon_lantern
laplink
launchbird_lm
lavenir_lm
lazy_ptop
lbc_control
lbc_measure
lbc_sync
lbc_watchdog
lbf
lbm
ldap
ldap_admin
ldaps
ldp
lds_distrib
ldxp
lecroy_vicp
legent_1
legent_2
leogic_data
leogic_services
leoip
lhtp
liberty_lm
licensedaemon
LiebDevMgmt_A
LiebDevMgmt_C
LiebDevMgmt_DM
lingwood
link
linkname
linktest
linktest_s
linogridengine
linx
lionhead
lipsinc
lipsinc1
lispworks_orb
listcrt_port
listcrt_port_2
listmgr_port
livelan
livestats
ljk_login
lkcmserver
llm_csv
llmnr
llm_pass
llsurfup_http
llsurfup_https
lmdp
lm_dta
lm_instmgr
lm_mon
lmp
lm_perfworks
lmsocialserver
lm_sserver
lm_webwatcher
lnvalarm
lnvconsole
lnvmailmon
lnvmaps
lnvpoller
lnvstatus
loadav
loaprobe
localinfosrvr
lockd
lockstep
locus_con
locus_map
lofr_lm
login
lonewolf_lm
lontalk_norm
lontalk_urgnt
lonworks
lonworks2
lot105_ds_upd
lotusmtap
lotusnote
lpcp
lpsrecommender
lrp
lrs_paging
ls3
ls3bcast
lsp_ping
lstp
ltctcp
ltcudp
lupa
lv_auth
lv_ffx
lv_frontpanel
lvision_lm
lv_jc
lv_not
lv_pici
lyskom
m2mservices
m2pa
m2ua
m3ua
m4_network_as
macon_tcp
macon_udp
macromedia_fcs
mac_srvr_admin
madcap
madge_ltd
magaya_network
magbind
magenta_logic
magicnotes
magicom
mailbox
mailbox_lm
mailprox
mailq
maincontrol
mainsoft_lm
maitrd
manage_exec
manyone_http
manyone_xml
mao
mapper_mapethd
mapper_nodemgr
mapper_ws_ethd
marcam_lm
masc
masqdialer
matip_type_a
matip_type_b
maxim_asics
maybeveritas
maytagshuffle
mbg_ctrl
mbus
mc2studios
mc3ss
mcagent
mc_appserver
mc_brk_srv
mc_client
mccwebsvr_port
mcer_port
mc_gt_srv
mciautoreg
mcidas
mck_ivpip
mcns_sec
mcns_tel_ret
mcntp
mcp_port
mcs_calypsoicf
mcs_fastmail
mcs_mailsvr
mcs_messaging
mctet_gateway
mctet_jserv
mctet_master
mctp
mdap_port
mdbs_daemon
md_cg_http
mdc_portmapper
mdns
mdnsresponder
mdqs
mdtp
mecomm
med_ci
med_fsp_rx
med_fsp_tx
media_agent
mediabox
mediacntrlnfsd
mediaspace
mediavault_gui
medimageportal
med_ltp
med_net_svc
med_ovw
med_supp
megaco_h248
megardsvr_port
megaregsvrport
menandmice_dns
menandmice_lpm
mentaclient
mentaserver
mercantile
meregister
mesavistaco
messageasap
messageservice
meta5
metaagent
metaconsole
meta_corp
metagram
metasage
metastorm
metasys
metatude_mds
meter
metricadbc
metrics_pas
mfcobol
mfserver
mftp
mgcp_callagent
mgcp_gateway
mgcs_mfp_port
mgxswitch
mice
micom_pfs
microcom_sbp
microgrid
micromuse_lm
micromuse_ncps
micromuse_ncpw
microsan
microsoft_ds
microtalon_com
microtalon_dis
midnight_tech
mikey
mil_2045_47001
miles_apart
mimer
mindfilesys
mindprint
minilock
minipay
mini_sql
minivend
mira
miroconnect
mit_dov
miteksys_lm
mit_ml_dev
miva_mqs
mloadd
mm_admin
mmcal
mmcals
mmcc
mmpft
mngsuite
mni_prot_rout
mnp_exchange
mns_mail
mobile_file_dl
mobileip_agent
mobilip_mn
mobrien_chat
moldflow_lm
molly
mon
mondex
monitor
monkeycom
monp
montage_lm
mortgageware
mosaicsyssvc1
mosaixcc
moshebeeri
movaz_ssc
moy_corp
mpc_lifenet
mpfoncl
mpfwsas
mphlpdmc
mpidcagt
mpidcmgr
mpl_gprs_port
mpm
mpm_flags
mpm_snd
mpnjsc
mpnjsocl
mpnjsomb
mpnjsomg
mpnjsosv
mpp
mppolicy_mgr
mppolicy_v5
mpshrsv
mps_raft
mpsserver
mpsysrmsvr
mptn
mrm
mrssrendezvous
ms_cluster_net
ms_dotnetster
msdp
msdts1
msexch_routing
msft_gc
msft_gc_ssl
msfw_array
msfw_control
msfw_replica
msfw_s_storage
msfw_storage
msg_auth
msgclnt
msg_icp
msgsrvr
msgsys
msiccp
msims
msi_selectplay
ms_la
ms_licensing
msl_lmd
ms_lsa
msmq
msnp
ms_olap1
ms_olap2
ms_olap3
ms_olap4
msolap_ptp2
msp
msql
ms_rome
msr_plugin_port
ms_rule_engine
ms_shuttle
ms_sna_base
ms_sna_server
ms_sql2000
ms_sql_m
ms_sql_s
ms_streaming
ms_theater
ms_v_worlds
ms_wbt_server
msync
mti_tcs_comm
mtl8000_matrix
mtp
mtportmon
mtport_regist
mtqp
mtrgtrans
mt_scaleserver
multiling_http
multiplex
multip_msg
mumps
mupdate
murray
murx
muse
musiconline
must_backplane
must_p2p
mvel_lm
mvs_capacity
mvx_lm
m_wnn
mxxrlogin
myblast
mylex_mapd
mylxamport
mynahautostart
myrtle
mysql
mysql_cluster
mysql_im
mythtv
mzap
n1_fwp
n1_rmgmt
n2h2server
n2nremote
naap
na_er_tip
name
namemunge
namp
nani
napster
nas
nasmanager
nas_metering
natd
natdataservice
nati_dstp
nati_logos
nati_svrloc
nati_vi_server
nat_pmp
nat_stun_port
nattyserver
natuslink
navbuddy
navegaweb_port
navisphere
navisphere_sec
nav_port
nbdb
nbp
nbt_pc
nbt_wol
nburn_id
nbx_au
nbx_cc
nbx_dir
nbx_ser
ncacn_ip_tcp
ncadg_ip_udp
ncconfig
ncd_conf_tcp
ncd_diag
ncd_diag_tcp
ncdloadbalance
ncdmirroring
ncd_pref
ncd_pref_tcp
nced
ncl
ncld
ncp
ncpm_ft
ncpm_hip
ncpm_pm
ncr_ccl
ncu_1
ncu_2
ncube_lm
ndl_aas
ndl_ahp_svc
ndl_als
ndl_aps
ndl_tcp_ois_gw
ndm_agent_port
ndmp
ndm_requester
ndm_server
ndnp
ndsauth
ndsconnect
ndsp
nds_sso
ndtp
neckar
necp
nec_raidplus
nei_management
neod1
neod2
neoiface
neon24x7
nerv
nesh_broker
nessus
nest_protocol
net8_cman
netadmin
netagent
netangel
netarx
netaspi
net_assistant
netattachsdmp
netbill_auth
netbill_cred
netbill_keyrep
netbill_prod
netbill_trans
netbios_dgm
netbios_ns
netbios_ssn
netbookmark
netboot_pxe
NetBus
netcelera
netchat
netcheque
netclip
netcomm1
netcomm2
netcp
netdb_export
net_device
netgw
netinfo_local
netiq
netiq_endpoint
netiq_endpt
netiq_mc
netiq_qcheck
netiq_voipa
netlabs_lm
netlock1
netlock2
netlock3
netlock4
netlock5
netmap_lm
netml
netmo_default
netmo_http
netmon
netmount
netnews
netobjects1
netobjects2
neto_dcs
netopia_vo1
netopia_vo2
netopia_vo3
netopia_vo4
netopia_vo5
netop_rc
netops_broker
netop_school
neto_wol_server
netplan
netplay_port1
netplay_port2
netport_id
netprowler_manager
netprowler_manager2
netprowler_sensor
netrcs
netrek
netrisk
netrix_sftm
netrjs_1
netrjs_2
netrjs_3
netrjs_4
netsc_dev
netsc_prod
netserialext1
netserialext2
netserialext3
netserialext4
netspeak_acd
netspeak_cps
netspeak_cs
netspeak_is
netstat
netsteward
net_steward
netsupport
netsupport2
nettest
nettgain_nms
netuitive
netvenuechat
netview_aix_1
netview_aix_10
netview_aix_11
netview_aix_12
netview_aix_2
netview_aix_3
netview_aix_4
netview_aix_5
netview_aix_6
netview_aix_7
netview_aix_8
netview_aix_9
netviewdm1
netviewdm2
netviewdm3
netwall
netware_csp
netware_ip
netwatcher_db
netwatcher_mon
netwave_ap_mgmt
netwkpathengine
networklens
networklenss
netx_agent
netx_server
neveroffline
newacct
newgenpay
newheights
newlixconfig
newlixengine
newlixreg
newoak
new_rwho
news
newton_dock
newwavesearch
nexstorindltd
nextstep
nexus_portal
nfa
nfsd_keepalive
nfsd_status
ng_umds
nhci
nhserver
nicelink
nicetec_mgmt
nicetec_nmsvc
nicname
ni_ftp
nifty_hmi
nim
ni_mail
nimaux
nimcontroller
nimgtw
nimhub
nimreg
nimrod_agent
nimsh
nimspooler
niobserver
nip
niprobe
nitrogen
ni_visa_remote
njenet_ssl
njfss
nkd
nlogin
nmap
nmasoverip
nm_asses_admin
nm_assessor
nm_game_admin
nm_game_server
nmmp
nms
nmsd
nms_dpnss
nmsigport
nmsp
nmsserver
nms_topo_serv
nnp
nnsp
nntp
nntps
noadmin
noagent
nokia_ann_ch1
nokia_ann_ch2
norton_lambert
notify
notify_srvr
novastorbakcup
novation
novell_ipx_cmd
novell_lu6_2
novell_zen
novell_zfs
npds_tracker
npep_messaging
npmp
npmp_gui
npmp_local
npmp_trap
npp
nppmp
nqs
nrcabq_lm
ns
nsc_ccs
ns_cfg_server
nsc_posa
nsiiops
nsjtp_ctrl
nsjtp_data
nsrmp
nssagentmgr
nssalertmgr
ns_server
nssocketport
nss_routing
nsstp
nsw_fe
nsws
nta_ds
ntalk
nta_us
ntp
nucleus
nucleus_sand
nut
nuts_bootp
nuts_dem
nuxsl
nvd
nvmsgd
nw_license
nxedit
nxlmd
o2server_port
oa_system
obex
objcall
objective_dbc
objectmanager
obrpd
ocbinder
oceansoft_lm
oce_snmp_trap
oc_lm
ocs_amu
ocs_cmu
ocserver
octopus
odette_ftp
odeumservlink
odi_port
odmr
odn_castraq
odsi
oem_agent
officelink2000
ofsd
oftep_rpc
ohimsrv
ohsc
oi_2000
oidocsvc
oidsr
oirtgsvc
olhost
olsr
oma_mlp
oma_mlp_s
omfs
omginitialrefs
omnilink_port
omnisky
omserv
onbase_dds
onesaf
onmux
ontime
ontobroker
opalis_rbt_ipc
opalis_rdv
opalis_robot
opc_job_start
opc_job_track
opcon_xps
opencm
openmail
openmailg
openmailns
openmailpxy
openmanage
openmath
opennl
opennl_voice
openport
openqueue
opentable
opentrac
openvms_sysipc
openvpn
opequus_server
opi_sock
op_probe
opscenter
opsec_cvp
opsec_ela
opsec_lea
opsec_omi
opsec_sam
opsec_uaa
opsec_ufp
opsession_clnt
opsession_prxy
opsession_srvr
opsmgr
optech_port1_lm
optika_emedia
optilogic
optimanet
optima_vnet
optiwave_lm
optocontrol
opus_services
oracle
oracle_em1
oracle_em2
oraclenames
oraclenet8cman
oracle_oms
oracle_vp1
oracle_vp2
ora_lm
orasrv
orbiter
orbix_cfg_ssl
orbix_config
orbixd
orbix_locator
orbix_loc_ssl
orbplus_iiop
ordinox_dbase
ordinox_server
orion
orion_rmi_reg
osdcp
os_licman
osm_appsrvr
osm_oev
osmosis_aeea
osu_nms
otpatch
ott
ottp
outlaws
ovalarmsrv
ovalarmsrv_cmd
ovbus
oveadmgr
ovhpas
ovladmgr
ov_nnm_websrv
ovrimosdbman
ovsam_d_agent
ovsam_mgmt
ovsessionmgr
ovtopmd
ovwdb
p2pgroup
p2plus
p2pq
pacerforum
pacmand
pacom
padl2sim
pafec_lm
paging_port
pago_services1
pago_services2
palace_1
palace_2
palace_3
palace_4
palace_5
palace_6
pammratc
pammrpc
panagolin_ident
panasas
pangolin_laser
panja_axbnet
panja_icsp
paradym_31port
paragent
parallel
park_agent
parsec_game
parsec_master
parsec_peer
partimage
passgo
passgo_tivoli
password_chg
passwrd_policy
patrol
patrol_mq_gm
patrol_mq_nm
patrol_snmp
patrolview
pawserv
pay_per_view
payrouter
pcanywhere
pcanywheredata
pcanywherestat
pcc_image_port
pciarray
pcia_rxp_b
pcihreq
pcle_infex
pclemultimedia
pcmail_srv
pc_mta_addrmap
pconnectmgr
pcs_pcw
pcsync_http
pcsync_https
pc_telecommute
pctrader
pcttunnell
pda_data
pda_gate
pdap
pdap_np
pda_sys
pdb
pdnet
pdp
pdps
pdrncs
pds
pearldoc_xact
peerbook_port
peerwire
pegboard
pehelp
pe_mike
peport
perf_port
periscope
permabit_cs
persona
personal_agent
personal_link
personalos_001
personnel
pftp
pfu_prcallback
ph
pharmasoft
pharos
philips_vc
phoenix_rpc
phone
phonebook
phonex_port
photuris
phrelay
phrelaydbg
piccolo
pichat
picknfs
picodbc
pictrography
pim_rp_disc
ping_pong
pip
pipes
pipe_server
piranha1
piranha2
pirp
pit_vpn
pk
pkagent
pk_electronics
pkix_3_ca_ra
pkix_timestamp
pktcable_cops
pktcablemmcops
pkt_krb_ipsec
plato
plato_lm
playsta2_app
playsta2_lob
plbserve_port
plcy_net_svcs
plethora
plgproxy
pluribus
pmcd
pmcdproxy
pmcp
pmd
pmdfmgt
pmdmgr
pmsm_webrctl
pnaconsult_lm
pnbscada
pnet_conn
pnet_enc
p_net_local
p_net_remote
pn_requester
pn_requester2
pnrp_port
pns
polestar
policyserver
pop2
pop3
pop3s
portgate_auth
postgresql
pov_ray
powerburst
powerclientcsf
powergemplus
powerguardian
poweroncontact
poweronnud
powerschool
powwow_client
powwow_server
ppcontrol
pptconference
pptp
pq_lic_mgmt
pra_elmd
prat
prchat_server
prchat_user
precise_i3
precise_sft
precise_vip
press
primaserver
printer
printer_agent
print_srv
priority_e_com
prism_deploy
prismiq_plugin
private0
private1023
private1024
private60
private87
privateark
privatechat
privatewire
priv_dial
priv_file
privilege
priv_mail
privoxy
priv_print
priv_rje
priv_term
prm_nm
prm_nm_np
prm_sm
prm_sm_np
prnrequest
prnstatus
proactivate
proactivesrvr
procos_lm
prodigy_intrnet
productinfo
pro_ed
profile
profinet_cm
profinet_rt
profinet_rtm
prolink
proofd
propel_msgsys
proshare1
proshare2
proshareaudio
prosharedata
proshare_mc_1
proshare_mc_2
prosharenotify
prosharerequest
prosharevideo
prospero
proxim
proxima_lm
proxy_gateway
proxy_plus
prp
prRegister
prsvp
ps_ams
psbserver
pscupd
psdbserver
pserver
pslserver
psmond
psprserver
pspserver
psrserver
pssc
pt2_discover
ptcnameservice
ptk_alink
ptp
ptp_event
ptp_general
pump
puparp
purenoise
pvsw
pvsw_inet
pvuniwien
pwdgen
pwdis
pwgippfax
pwgpsi
pwrsevent
pxc_epmap
pxc_ntfy
pxc_pin
pxc_roid
pxc_sapxom
pxc_splr
pxc_splr_ft
pxc_spvr
pxc_spvr_ft
pythonds
q55_pcc
qadmifevent
qadmifoper
qaz
qbikgdp
qencp
qft
qftest_lookup
qip_audup
qip_login
qip_msgd
qip_qdhcp
qke_llc_v3
qmqp
qmtp
qnts_orb
qnxnetman
qo_secure
qotd
qotps
qpasa_agent
qrepos
qrh
qsc
qsm_gui
qsm_proxy
qsm_remote
qsnet_assist
qsnet_cond
qsnet_nucl
qsnet_trans
qsnet_workst
qsoft
qtms_bootstrap
qtp
qt_serveradmin
quaddb
quake
quake2
Quake3Server
quakeworld
quartus_tcl
quasar_server
qubes
quest_agent
questdb2_lnchr
quest_launcher
questnotify
quest_vista
quickbooksrds
quicksuite
quotad
raadmin
radan_http
radclientport
radec_corp
radio
radio_bc
radio_sm
radius
radius_acct
radius_dynauth
radix
radmin_port
radwiz_nms_srv
raid_ac
raid_am
raid_cc
raid_cd
raid_cs
raid_sf
ramp
rap
rapidbase
rapidmq_center
rapidmq_reg
rapido_ip
rap_ip
rap_listen
rap_service
rasadv
ratio_adp
ratl
raven1
raven2
raven_rdp
raven_rmp
raw_serial
razor
rbakcup1
rbakcup2
rblcheckd
rbr_debug
rbr_discovery
rcc_host
rcp
rcst
rda
rdb_dbs_disp
rdc_wh_eos
rdlap
rdrmshc
rds
rds2
re101
reachout
realm_rusd
realsecure
rebol
recipe
re_conn_proto
redstone_cpss
redstorm_diag
redstorm_find
redstorm_info
redstorm_join
redwood_broker
redwood_chat
reftek
registrar
relief
rellpack
re_mail_ck
remoteanything
remote_as
remote_collab
remotedeploy
remotefs
remote_kis
remoteware_un
remote_winsock
repcmd
repliweb
repscmd
res
rescap
reserved1
reserved2
reserved3
reserved4
resource_mgr
responselogic
responsenet
res_sap
rets
rets_ssl
rexecj
rfa
rfe
rfile
rfio
rfmp
rfx_lm
rgtp
rhp_iibp
rib_slm
ricardo_lm
rich_cp
ridgeway1
ridgeway2
rimsl
ripng
ris
ris_cm
rje
rkb_oscs
rlp
rlzdbase
rmc
rmiactivation
rmiaux
rmiregistry
rmlnk
rmonitor
rmonitor_secure
rmopagt
rmpp
rmt
rmtserver
rndc
rnm
rnmap
rnrp
robcad_lm
roboeda
roboer
rockwell_csp1
rockwell_csp2
rockwell_csp3
rockwell_encap
roketz
rootd
router
roverlog
rpasswd
rpc2portmap
rprt
rrac
rrdp
rrh
rrifmm
rrilwm
rrimwm
rrirtr
rrisat
rrp
rsap
rsc_robot
rsf_1
rsh_spx
rsip
rsisysaccess
rsmtp
rsom
rs_rmi
rsvd
rsvp_encap_1
rsvp_encap_2
rsvp_tunnel
rsync
rtcm_sc104
rtc_pm_port
rtelnet
rt_event
rt_event_s
rtip
rtmp
rtmp_port
rtnt_1
rtnt_2
rtraceroute
rtsclient
rtsp
rtsp_alt
rtsps
rtss
rtsserv
rusb_sys_port
rushd
rvs_isdn_dcp
rwhois
rxe
rxmon
s8_client_port
sabams
sabarsd
sabp_signal
sacred
saft
sage_best_com1
sage_best_com2
sagectlpanel
sah_lm
sais
saiscm
saiseh
sai_sentlm
saism
salient_dtasrv
salient_mux
salient_usrmgr
samd
sa_msg_port
samsung_unidex
sanavigator
sane_port
sanity
santak_ups
sapv1
saris
sasg
sas_remote_hlp
satvid_datalnk
savant
sbl
sbook
scan_change
scanstat_1
sccip_media
scc_security
scientia_sdb
scientia_ssdb
scipticslsrvr
sco_aip
sco_dtmgr
scohelp
scoi2odialog
sco_inetmgr
scol
sco_peer_tta
scoremgr
sco_sysmgr
sco_websrvrmg3
sco_websrvrmgr
scp
scp_config
scpi_raw
scpi_telnet
scrabble
screencast
scservp
sctp
sctp_tunneling
scx_proxy
sd
sdadmind
sdbproxy
sd_capacity
sdclient
sd_data
sddp
sde_discovery
sd_elmd
sdfunc
sdhelp
sdl_ets
sdlog
sdnskmp
sdo
sdo_ssh
sdo_tls
sdp_id_port
sdp_portmapper
sdproxy
sdreport
sd_request
sds_admin
sdsc_lm
sdserv
sdserver
sdt_lmd
sdxauthd
seagull_ais
seagulllms
seaodbc
search
search_agent
seaview
seclayer_tcp
seclayer_tls
sec_ntb_clnt
secureidprop
securenetpro_sensor
secure_ts
seispoc
semantix
semaphore
send
senip
sentinel
sentinel_ent
sentinel_lm
sent_lm
sentlm_srv2srv
sep
sercomm_scadmin
sercomm_wlink
serialgateway
server_find
servergraph
serverview_as
serverview_asn
serverview_gf
serverview_icc
serverview_rm
servexec
service_ctrl
servicemeter
servserv
servstat
sesi_lm
set
sflm
sf_lm
sflow
sfs_config
sfs_smp_net
sftdst_port
sftp
sftsrv
sftu
sgcp
sgi_dgl
sgi_esphttp
sgi_eventmond
sgi_storman
sg_lm
sgmp
sgmp_traps
shadowserver
shareapp
sharp_server
shell
shilp
shiva_confsrvr
shivadiscovery
shivahose
shivasound
shockwave
shockwave2
shrinkwrap
siam
siebel_ns
siemensgsm
sift_uft
sightline
sigma_port
signal
signet_ctf
siipat
silc
silhouette
silkmeter
silkp1
silkp2
silkp3
silkp4
silverplatter
simba_cs
simbaexpress
simbaservices
simco
sim_control
simctlp
simp_all
simplement_tie
simple_push
simple_push_s
simslink
sip
sip_tls
sis_emt
sitaradir
sitaramgmt
sitaraserver
sitewatch
six_degrees
sixnetudr
sixtrak
sixxsconfig
skip_cert_recv
skip_cert_send
skip_mc_gikreq
skronk
skytelnet
sky_transport
slc_ctrlrloops
slc_systemlog
slim_devices
slingshot
slinkysearch
slinterbase
slm_api
slp
slp_notify
slslavemon
slush
smaclmgr
smakynet
smap
smar_se_port1
smar_se_port2
smartcard_port
smart_diagnose
smart_lm
smartpackets
smartsdp
smauth_port
smc_http
smc_https
smile
smntubootstrap
sm_pas_1
sm_pas_2
sm_pas_3
sm_pas_4
sm_pas_5
smpnameres
smpp
smpppd
smpte
sms_chat
smsd
smsp
smsqp
sms_rcinfo
sms_remctrl
sms_xfer
smtp
smux
snac
sna_cs
snagas
snap
snapd
snapenetio
snapp
snare
snaresecure
s_net
snifferclient
snifferdata
snifferserver
snip
snmp
snmp_tcp_port
snmptrap
snpp
sns_admin
sns_agent
sns_channels
sns_dispatcher
sns_gateway
sns_protocol
sns_query
sns_quote
sntp_heartbeat
soap_beep
soap_http
socks
socorfs
softaudit
softcm
softdataphone
softpc
softrack_meter
solid_e_engine
solid_mux
solve
sometimes_rpc1
sometimes_rpc13
sometimes_rpc14
sometimes_rpc15
sometimes_rpc16
sometimes_rpc17
sometimes_rpc18
sometimes_rpc19
sometimes_rpc2
sometimes_rpc20
sometimes_rpc21
sometimes_rpc22
sometimes_rpc23
sometimes_rpc24
sometimes_rpc25
sometimes_rpc26
sometimes_rpc27
sometimes_rpc28
sonar
sonardata
soniqsync
sonus
sonuscallsig
s_openmail
sophia_lm
sops
sor_update
sos
soundsvirtual
spamd
spamd_cfg
spamtrap
spandataport
spc
spcsdlobby
spearway
spectraport
spice
spiral_admin
splitlock
splitlock_gw
spmp
spock
sprp
spsc
spss_lm
sps_tunnel
spt_automation
spugna
spw_dialer
spw_dnspreload
sqdr
sql_net
sqlserv
sqlsrv
srb
src
srdp
srmp
srp_feedback
srsp
srssend
srvc_registry
ss7ns
ssc_agent
sscan
ssdispatch
ssdp
ssdtp
sse_app_config
ssh
sshell
sslp
ssm_cssps
ssm_cvs
ssm_els
ssmpp
sso_control
sso_service
ssowatch
ssp
ssql
ssrip
ssr_servermgr
ssslic_mgr
ssslog_mgr
sst
sstp_1
sstp_2
sstsys_lm
stanag_5066
starbot
stargatealerts
starquiz_port
starschool
start_network
startron
statsci1_lm
statsci2_lm
statsrv
statusd
stel
ste_smsc
stgxfws
sti_envision
stmf
stm_pproc
stone_design_1
stonefalls
streamcomm_ds
streetperfect
street_stream
streettalk
stryker_com
stt
stun_p1
stun_p2
stun_p3
stun_port
stvp
stx
sua
submission
submit
submitserver
subntbcst_tftp
subseven
sugp
suitcase
suitjd
su_mit_tg
sun_as_nodeagt
sunclustermgr
sun_dr
sunfm_port
sun_lm
sunlps_http
sun_manageconsole
sun_mc_grp
sunrpc
sunscalar_dns
sunscalar_svc
sun_sea_port
supdup
supercell
supermon
supfiledbg
supfilesrv
surebox
surf
surfcontrolcpa
sur_meas
surveyinst
suucp
svn
svnet
svnetworks
svrloc
svs_omagent
swa_1
swa_2
swa_3
swa_4
swc_xds
sweetware_apps
swgps
swiftnet
swift_rvf
swismgr1
swismgr2
swispol
swistrap
swldy_sias
swrmi
swr_port
swtp_port1
swtp_port2
swx
sxmp
sxuptp
syam_agent
syam_smc
syam_webserver
sybaseanywhere
sybasedbsynch
sybase_sqlany
sychrond
sygatefw
symantec_sim
symbios_raid
symb_sb_port
symplex
synapse
synapsis_edge
sync_em7
synchronet_db
synchronet_rtc
synchronet_upd
syncserver
syncserverssl
synel_data
synoptics_trap
synotics_broker
synotics_relay
sype_transport
syscomlan
sysinfo_sp
syslog
syslog_conn
sysopt
sysorb
sysscanner
systat
systemics_sox
system_monitor
t128_gateway
t1distproc
t1distproc60
t1_e1_over_ip
t2_brm
t2_drm
tabula
tacacs
tacacs_ds
tacnews
tacticalauth
tag_ups_1
taiclock
talarian_mcast1
talarian_mcast2
talarian_mcast3
talarian_mcast4
talarian_mcast5
talarian_mqs
talarian_tcp
talarian_udp
taligent_lm
talk
talnet
talon_disc
talon_engine
talon_webserver
tal_pod
tambora
tams
tapestry
tapeware
tappi_boxnet
tarantella
targus_getdata
targus_getdata1
targus_getdata2
targus_getdata3
taserver
taskman_port
taskmaster2000
tasp_net
taurus_wh
tbrpf
tcim_control
tclprodebugger
tcoaddressbook
tcoflashagent
tcoregagent
tcpdataserver
tcp_id_port
tcpmux
tcpnethaspsrv
tdaccess
tdm_over_ip
td_postman
tdp_suite
td_replica
td_service
teamcoherence
techra_server
teedtap
tekpls
telaconsole
telefinder
telelpathattack
telelpathstart
teleniumdaemon
telesis_licman
telindus
tell
tellumat_nms
telnet
telnetcpcd
telnets
telops_lmd
tempest_port
tempo
tenfold
terabase
teradataordbms
teredo
terminaldb
tetrinet
texar
tftp
tftp_mcast
tftps
tgp
theta_lm
tht_treasure
ticf_1
ticf_2
tick_port
tig
timbuktu
timbuktu_srv1
timbuktu_srv2
timbuktu_srv3
timbuktu_srv4
time
timed
timeflies
timelot
timestenbroker
tinc
tinyfw
tip2
tip_app_server
tivoconnect
tivoli_npm
tksocket
tl1
tl1_lv
tl1_raw
tl1_telnet
tlisrv
tmesis_upshot
tmlogin
tnETOS
tnmpv2
tnp1_port
tns_adv
tns_cml
tns_server
tn_timing
tn_tl_fd1
tn_tl_fd2
tn_tl_r1
tn_tl_r2
tn_tl_w1
tn_tl_w2
toad
tomato_springs
topflow
topflow_ssl
topovista_data
topx
tpdu
tpip
tpmd
tqdata
traceroute
track
tragic
tram
translogic_lm
transmit_port
trap
trap_daemon
trap_port
trap_port_mom
travsoft_ipx_t
trc_netpoll
trellisagt
trellissvr
trendchip_dcp
tributary
Trinoo_Bcast
Trinoo_Master
Trinoo_Register
triomotion
trip
tripwire
triquest_lm
trispen_sra
trivnet1
trivnet2
trnsprntproxy
troff
tr_rsrb_p1
tr_rsrb_p2
tr_rsrb_p3
tr_rsrb_port
truecm
trusted_web
trustestablish
tsaf
tsb
tsb2
tscchat
tsdos390
tserver
tsilb
tsp
tsspmap
ttat3lb
ttc
ttc_etap
ttc_etap_ds
ttc_etap_ns
ttcmremotectrl
ttc_ssl
ttg_protocol
ttlpriceproxy
ttl_publisher
ttnrepository
ttntspauto
ttyinfo
tunnel
tunstall_pnc
turbonote_1
turbonote_2
tvbus
tvdumtray_port
tve_announce
tvnetworkvideo
tvpm
tw_auth_key
twcss
twrpc
twsdss
uaac
uacs
uadtc
uaiact
uarps
ua_secureagent
ubxd
ucentric_ds
u_dbap
udpplus
udpradio
udp_sr_port
udrive
udt_os
ufastro_instr
ufsd
uis
ulistproc
ulp
ulpnet
ultimad
ultrabac
ultraseek_http
ultrex
uma
umm_port
ums
umsp
unbind_cluster
unet
unglue
unicall
unicontrol
unidas
unidata_ldm
unieng
unify
unify_adapter
unifyadmin
unihub_server
unimobilectrl
unisnc
unisql
unisql_java
unisys_lm
unite
univ_appserver
univision
unix_status
unot
uohost
uorb
upgrade
upnotifyp
upnotifyps
ups
ups_engine
upsnotifyprot
ups_onlinet
upstriggervsw
urbisnet
urd
urld_port
urm
urn_x_cdchoice
user_manager
us_gv
utcd
utime
utmpcd
utmpsd
utsftp
uucp
uucp_path
uucp_rlogin
uuidgen
v5ua
vacdsm_app
vacdsm_sws
valisys_lm
va_pacbase
vaprtm
varadero_0
varadero_1
varadero_2
vat
vat_control
vatp
vaultbase
vce
vchat
vcnet_link_v10
vcom_tunnel
vcrp
vcs_app
vdmplay
vemmi
venus
venus_se
veracity
vergencecm
verismart
VeritasNetbackup
veritas_pbx
veritas_tcp1
veritas_ucl
veritas_udp1
veritas_vis1
veritas_vis2
veronica
versatalk
versa_tek
vettcp
vfo
vhd
vid
video_activmail
videobeans
videotex
videte_cipc
vids_avtp
vieo_fe
vinainstall
vipremoteagent
virprot_lm
virtual_places
virtualtape
virtual_time
virtualuser
visd
visinet_gui
vision_elmd
visionpyramid
vision_server
visitview
vista_4gl
vistium_share
vitalanalysis
viziblebrowser
vlsi_lm
vmnet
vmodem
vmpwscs
vnas
vnc
vnc_1
vnc_2
vnc_3
vnc_http_2
vnc_http_3
vnetd
vnsstr
vnwk_prapi
vnyx
vocaltec_admin
vocaltec_gold
vocaltec_hos
vocaltec_phone
vocaltec_wconf
vofr_gateway
voispeed_port
volley
v_one_spp
vopied
vp2p
vpac
vpad
vpjp
vpn_encap
vpntpp
vpnz
vpp
vpps_qua
vpps_via
vpsipport
vpvc
vpvd
vqp
vrcommerce
vrml_multi_use
vrpn
vrtl_vmf_ds
vrtl_vmf_sa
vrtp
vrts_at_port
vrts_auth_port
vrts_ipcserver
vrts_registry
vrts_tdd
vrtstrapserver
vsaiport
vsamredirector
vsat_control
vseconnector
vsiadmin
vsinet
vsi_omega
vsixml
vslmp
vsnm_agent
vspread
vs_server
vstat
vtr_emulator
vt_ssl
vulture
vvr_control
vvr_data
vx_auth_port
vxcrnbuport
vytalvaultbrtp
vytalvaultpipe
vytalvaultvsmp
wacp
wago_io_system
wag_service
wap_push
wap_push_http
wap_push_https
wap_pushsecure
wap_vcal
wap_vcal_s
wap_vcard
wap_vcard_s
wap_wsp
wap_wsp_s
wap_wsp_wtp
wap_wsp_wtp_s
watchdoc
watchdognt
watcomdebug
watershed_lm
watilapp
wbem_exp_https
wbem_http
wbem_https
wbem_rmi
web2host
webaccess
webdata
webemshttp
weblogin
webmachine
webmail_2
webmethods_b2b
webobjects
webphone
websm
websphere_snmp
webster
webtie
westell_stats
wfremotertm
wg_netforce
wherehoo
whisker
who
whoami
whois_
whosells
whosockami
wilkenlistener
willy
wimd
wimsic
wincim
windb
winddlb
winddx
windlm
windows_icfw
wininstall_ipc
winjaserver
winpharaoh
winport
winqedit
wins
winshadow
winshadow_hd
wip_port
wizard
wkars
wkstn_mon
wlanauth
wlbs
wmc_log_svc
wms_messenger
wnn4_Cn
wnn4_Kr
wnn4_Tw
wnn6
wnn6_ds
workflow
work_sol
worldfusion1
worldfusion2
world_lm
worldscores
worldwire
wpages
wpgs
writesrv
wrs_registry
ws_discovery
wsicopy
wsmlb
wsm_server
wsm_server_ssl
wsynch
wta_wsp_s
wta_wsp_wtp_s
wusage
wv_csp_sms
wv_csp_sms_cir
wv_csp_udp_cir
wwiotalk
www_dev
www_ldap_gw
wysdma
wysdmc
x11
x25_svc_port
x500ms
xaapi
xact_backup
xap_ha
x_bone_api
x_bone_ctl
xbox
xdmcp
xdsxdm
xdtp
xecp_node
xfer
xfr
xgrid
xiip
xingcsm
xingmpeg
xinuexpansion1
xinuexpansion2
xinuexpansion3
xinuexpansion4
xinupageserver
xiostatus
xmapi
xmlblaster
xmlink_connect
xmlrpc_beep
xmltec_xmlmail
xmpcr_interface
xmpp_client
xmpp_server
xmpv7
xmquery
xmsg
xnm_clear_text
xnmp
xnm_ssl
xns_auth
xns_ch
xns_courier
xns_mail
xns_time
xo_wave
xpilot
xpl
xprint_server
xqosd
xribs
xrl
xrpc_registry
xserveraid
XSIP_network
xs_openstorage
xss_port
xss_srv_port
xsync
xtrm
xtrms
xvttp
xyplex_mux
yak_chat
yestrader
yo_main
z39_50
zannet
zarkov
zenginkyo_1
zenginkyo_2
zephyr_clt
zephyr_hm
zephyr_srv
zfirm_shiprush3
zicom
zion_lm
zip
zmp
zserv
zymed_zpp
