# NAME
Statistics::Covid - Fetch, store in DB, retrieve and analyse Covid-19 statistics from data providers

# VERSION

Version 0.13

# DESCRIPTION

This module fetches, stores in a database, retrieves from a database and analyses
Covid-19 statistics from online or offline data providers, such as
from [the John Hopkins University](https://www.arcgis.com/apps/opsdashboard/index.html#/bda7594740fd40299423467b48e9ecf6)
which I hope I am not obstructing (please send an email to the author if that is the case).

After specifying one or more data providers (as a url and a header for data and
optionally for metadata), this module will attempt to fetch the latest data
and store it in a database (SQLite and MySQL, only SQLite was tested so far).
Each batch of data should ideally contain information about one or more locations
and at a given point in time. All items in this batch are extracted and stored
in DB each with its location name and time (it was published, not fetched) as primary keys.
Each such data item (Datum) is described in [Statistics::Covid::Datum::Table](https://metacpan.org/pod/Statistics::Covid::Datum::Table)
and the relevant class is [Statistics::Covid::Datum](https://metacpan.org/pod/Statistics::Covid::Datum). It contains
fields such as: `population`, `confirmed`, `unconfirmed`, `terminal`, `recovered`.

Focus was on creating very high-level which distances as much as possible
the user from the nitty-gritty details of fetching data using [LWP::UserAgent](https://metacpan.org/pod/LWP::UserAgent)
and dealing with the database using [DBI](https://metacpan.org/pod/DBI) and [DBIx::Class](https://metacpan.org/pod/DBIx::Class).

This is an early release until the functionality and the table schemata
solidify.

# SYNOPSIS

        use Statistics::Covid;
        use Statistics::Covid::Datum;
        
        $covid = Statistics::Covid->new({   
                'config-file' => 't/example-config.json',
                'providers' => ['UK::BBC', 'UK::GOVUK', 'World::JHU'],
                'save-to-file' => 1,
                'save-to-db' => 1,
                'debug' => 2,
        }) or die "Statistics::Covid->new() failed";
        # fetch all the data available (posibly json), process it,
        # create Datum objects, store it in DB and return an array 
        # of the Datum objects just fetched  (and not what is already in DB).
        my $newobjs = $covid->fetch_and_store();
        
        print $_->toString() for (@$newobjs);
        
        print "Confirmed cases for ".$_->name()
                ." on ".$_->date()
                ." are: ".$_->confirmed()
                ."\n"
        for (@$newobjs);
        
        my $someObjs = $covid->select_datums_from_db({
                'conditions' => {
                        belongsto=>'UK',
                        name=>'Hackney'
                }
        });
        
        print "Confirmed cases for ".$_->name()
                ." on ".$_->date()
                ." are: ".$_->confirmed()
                ."\n"
        for (@$someObjs);
        
        # or for a single place (this sub sorts results wrt publication time)
        my $timelineObjs = $covid->select_datums_from_db_for_location('Hackney');
        for my $anobj (@$timelineObjs){
                print $anobj->toString()."\n";
        }

        print "datum rows in DB: ".$covid->db_count_datums()."\n"

# EXAMPLE SCRIPT

`script/statistics-covid-fetch-data-and-store.pl` is
a script which accompanies this distribution. It can be
used to fetch any data from specified providers using a
specified configuration file.

For a quick start:

    cp t/example-config.json config.json
    # optionally modify config.json to change the destination data dirs
    # now fetch data from some default data providers:
    script/statistics-covid-fetch-data-and-store.pl --config-file config.json

The above will fetch the latest data and insert it into an SQLite
database in `data/db/covid19.sqlite` directory.
When this script is called again, it will fetch the data again
and will be saved into a file timestamped with publication date.
So, if data was already fetched it will be simply overwritten by
this same data. 

As far as updating the database is concerned, only newer, up-to-date data
will be inserted. So, calling this script, say once or twice will
make sure you have the latest data without accummulating it
redundantly.

**But please call this script AT MAXIMUM one or two times per day so as not to
obstruct public resources. Please, Please.**

When the database is up-to-date, analysis of data is the next step.

In the synopis, it is shown how to select records from the database,
as an array of [Statistics::Covid::Datum](https://metacpan.org/pod/Statistics::Covid::Datum) objects. Feel free to
share any modules you create on analysing this data, either
under this namespace (for example Statistics::Covid::Analysis::XYZ)
or any other you see appropriate.

# CONFIGURATION FILE

Below is an example configuration file which is essentially JSON with comments.
It can be found in `t/example-config.json` relative to the root directory 
of this distribution.

        # comments are allowed, otherwise it is json
        # this file does not get eval'ed, it is parsed
        # only double quotes! and no excess commas
        {
                # fileparams options
                "fileparams" : {
                        # dir to store datafiles, each DataProvider class
                        # then has its own path to append
                        "datafiles-dir" : "datazz/files"
                },
                # database IO options
                "dbparams" : {
                        # which DB to use: SQLite, MySQL (case sensitive)
                        "dbtype" : "SQLite",
                        # the name of DB
                        # in the case of SQLite, this is a filepath
                        # all non-existing dirs will be created (by module, not by DBI)
                        "dbdir" : "datazz/db",
                        "dbname" : "covid.sqlite",
                        # how to handle duplicates in DB? (duplicate=have same PrimaryKey)
                        # only-better : replace records in DB if outdated (meaning number of markers is less, e.g. terminal or confirmed)
                        # replace     : force replace irrespective of markers
                        # ignore      : if there is a duplicate in DB DONT REPLACE/DONT INSERT
                        # (see also Statistics::Covid::Datum for up-to-date info)
                        "replace-existing-db-record" : "only-better",
                        # username and password if needed
                        # unfortunately this is in plain text
                        # BE WARNED: do not store your main DB password here!!!!
                        # perhaps create a new user or use SQLite
                        # there is no need for these when using SQLite
                        "hostname" : "", # must be a string (MySQL-related)
                        "port"     : "", # must be a string (MySQL-related)
                        "username" : "", # must be a string
                        "password" : "", # must be a string
                        # options to pass to DBI::connect
                        # see https://metacpan.org/pod/DBI for all options
                        "dbi-connect-params" : {
                                "RaiseError" : 1, # die on error
                                "PrintError" : 0  # do not print errors or warnings
                        }
                }
        }

# DATABASE SUPPORT

SQLite and MySQL database types are supported through the
abstraction offered by [DBI](https://metacpan.org/pod/DBI) and [DBIx::Class](https://metacpan.org/pod/DBIx::Class).

**However**, only the SQLite support has been tested.

**Support for MySQL is totally untested**.

# AUTHOR

Andreas Hadjiprocopis, `<bliako at cpan.org>`, `<andreashad2 at gmail.com>`

# BENCHMARKS

There are some benchmark tests to time database insertion and retrieval
performance. These are
optional and will not be run unless explicitly stated via
`make bench`

These tests do not hit the online data providers at all. And they
should not. They only time the creation of objects and insertion
to the database.

# CAVEATS

This module has been put together very quickly and under pressure.
There are must exist quite a few bugs. In addition, the database
schema, the class functionality and attributes are bound to change.
A migration database script may accompany new versions in order
to use the data previously collected and stored.

**Support for MySQL is totally untested**. Please use SQLite for now
or test the MySQL interface.

# BUGS

This module has been put together very quickly and under pressure.
There are must exist quite a few bugs.

Please report any bugs or feature requests to `bug-statistics-Covid at rt.cpan.org`, or through
the web interface at [http://rt.cpan.org/NoAuth/ReportBug.html?Queue=Statistics-Covid](http://rt.cpan.org/NoAuth/ReportBug.html?Queue=Statistics-Covid).  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.

# SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Statistics::Covid

You can also look for information at:

- github [repository](https://github.com/hadjiprocopis/statistics-covid)  which will host data and alpha releases
- RT: CPAN's request tracker (report bugs here)

    [http://rt.cpan.org/NoAuth/Bugs.html?Dist=Statistics-Covid](http://rt.cpan.org/NoAuth/Bugs.html?Dist=Statistics-Covid)

- AnnoCPAN: Annotated CPAN documentation

    [http://annocpan.org/dist/Statistics-Covid](http://annocpan.org/dist/Statistics-Covid)

- CPAN Ratings

    [http://cpanratings.perl.org/d/Statistics-Covid](http://cpanratings.perl.org/d/Statistics-Covid)

- Search CPAN

    [http://search.cpan.org/dist/Statistics-Covid/](http://search.cpan.org/dist/Statistics-Covid/)

- Information about the basis module DBIx::Class

    [http://search.cpan.org/dist/DBIx-Class/](http://search.cpan.org/dist/DBIx-Class/)

# DEDICATIONS

Almaz

# ACKNOWLEDGEMENTS

- [Perlmonks](https://www.perlmonks.org) for supporting the world with answers and programming enlightment
- [DBIx::Class](https://metacpan.org/pod/DBIx::Class)
- the data providers:
    - [John Hopkins University](https://www.arcgis.com/apps/opsdashboard/index.html#/bda7594740fd40299423467b48e9ecf6),
    - [UK government](https://www.gov.uk/government/publications/covid-19-track-coronavirus-cases),
    - [https://www.bbc.co.uk](https://www.bbc.co.uk) (for disseminating official results)

# LICENSE AND COPYRIGHT

Copyright 2020 Andreas Hadjiprocopis.

This program is free software; you can redistribute it and/or modify it
under the terms of the the Artistic License (2.0). You may obtain a
copy of the full license at:

[http://www.perlfoundation.org/artistic\_license\_2\_0](http://www.perlfoundation.org/artistic_license_2_0)

Any use, modification, and distribution of the Standard or Modified
Versions is governed by this Artistic License. By using, modifying or
distributing the Package, you accept this license. Do not use, modify,
or distribute the Package, if you do not accept this license.

If your Modified Version has been derived from a Modified Version made
by someone other than you, you are nevertheless required to ensure that
your Modified Version complies with the requirements of this license.

This license does not grant you the right to use any trademark, service
mark, tradename, or logo of the Copyright Holder.

This license includes the non-exclusive, worldwide, free-of-charge
patent license to make, have made, use, offer to sell, sell, import and
otherwise transfer the Package with respect to any patent claims
licensable by the Copyright Holder that are necessarily infringed by the
Package. If you institute patent litigation (including a cross-claim or
counterclaim) against any party alleging that the Package constitutes
direct or contributory patent infringement, then this Artistic License
to you shall terminate on the date that such litigation is filed.

Disclaimer of Warranty: THE PACKAGE IS PROVIDED BY THE COPYRIGHT HOLDER
AND CONTRIBUTORS "AS IS' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.
THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
PURPOSE, OR NON-INFRINGEMENT ARE DISCLAIMED TO THE EXTENT PERMITTED BY
YOUR LOCAL LAW. UNLESS REQUIRED BY LAW, NO COPYRIGHT HOLDER OR
CONTRIBUTOR WILL BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, OR
CONSEQUENTIAL DAMAGES ARISING IN ANY WAY OUT OF THE USE OF THE PACKAGE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
