## Domain Registry Interface, Registration Type Extension Mapping for EPP
##
## Copyright (c) 2016,2018-2019 Patrick Mevzek <netdri@dotandco.com>. All rights reserved.
##
## This file is part of Net::DRI
##
## Net::DRI is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## See the LICENSE file that comes with this distribution for more details.
####################################################################################################

package Net::DRI::Protocol::EPP::Extensions::CentralNic::RegType;

use strict;
use warnings;
use feature 'state';

use Net::DRI::Util;
use Net::DRI::Exception;

####################################################################################################

sub register_commands
{
 my ($class,$version)=@_;
 state $rcmds = { 'domain' => {
                                'check'       => [ \&check,  undef ],
                                'check_multi' => [ \&check,  undef ],
                                'info'        => [ undef,    \&info_parse ],
                                'create'      => [ \&create, undef ],
                                'update'      => [ \&update, undef ],
                              },
                };
 return $rcmds;
}

sub setup
{
 my ($class,$po,$version)=@_;
 state $rns = { 'regType' => 'urn:ietf:params:xml:ns:regtype-0.2' };
 $po->ns($rns);
 return;
}

sub capabilities_add { return ('domain_update', 'registration_type', [qw/set/]); }

sub implements { return 'https://gitlab.centralnic.com/centralnic/epp-registration-type-extension/blob/e210c27730c8926324e502efff4d0ad77190d7c2/draft-brown-regtype.txt'; }

####################################################################################################

sub _regtype
{
 my ($rd) = @_;
 return unless Net::DRI::Util::has_key($rd, 'registration_type');
 Net::DRI::Exception::usererr_invalid_parameters('registration_type must be an XML token with at least one character') unless Net::DRI::Util::xml_is_token($rd->{registration_type},1);
 return $rd->{registration_type};
}

sub _build
{
 my ($epp,$domain,$rd,$command)=@_;
 my $regtype = _regtype($rd);
 $epp->message()->command_extension('regType', [$command, ['type', $regtype]]) if $regtype;

 return;
}

sub check
{
 my ($epp,$domain,$rd)=@_;
 return _build($epp,$domain,$rd,'check');
}

sub info_parse
{
 my ($po,$otype,$oaction,$oname,$rinfo)=@_;
 my $mes=$po->message();
 return unless $mes->is_success();

 my $ns=$mes->ns('regType');
 my $data=$mes->get_extension('regType', 'infData');
 return unless defined $data;

 $rinfo->{domain}->{$oname}->{registration_type}=Net::DRI::Util::xml_child_content($data, $ns, 'type');

 return;
}

sub create
{
 my ($epp,$domain,$rd)=@_;
 return _build($epp,$domain,$rd,'create');
}

sub update
{
 my ($epp,$domain,$todo)=@_;

 my $regtype = _regtype({ registration_type => $todo->set('registration_type') });
 $epp->message()->command_extension('regType', ['update', ['chg', ['type', $regtype]]]) if $regtype;

 return;
}

####################################################################################################
1;

__END__

=pod

=head1 NAME

Net::DRI::Protocol::EPP::Extensions::CentralNic::RegType - EPP Registration Type Extension mapping (draft-brown-epp-regtype-00) for Net::DRI

=head1 DESCRIPTION

Please see the README file for details.

=head1 SUPPORT

For now, support questions should be sent to:

E<lt>netdri@dotandco.comE<gt>

Please also see the SUPPORT file in the distribution.

=head1 SEE ALSO

E<lt>http://www.dotandco.com/services/software/Net-DRI/E<gt>

=head1 AUTHOR

Patrick Mevzek, E<lt>netdri@dotandco.comE<gt>

=head1 COPYRIGHT

Copyright (c) 2016,2018-2019 Patrick Mevzek <netdri@dotandco.com>.
All rights reserved.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

See the LICENSE file that comes with this distribution for more details.

=cut
