package WARC::Index;						# -*- CPerl -*-

use strict;
use warnings;

use Carp;

our @ISA = qw();

require WARC; *WARC::Index::VERSION = \$WARC::VERSION;

=head1 NAME

WARC::Index - base class for WARC index classes

=head1 SYNOPSIS

  use WARC::Index::File::CDX;	# or ...
  use WARC::Index::File::SDBM;
  # or some other WARC::Index::File::* implementation

  $index = attach WARC::Index::File::CDX (...);	# or ...
  $index = attach WARC::Index::File::SDBM (...);

  $record = $index->search(url => $url, time => $when);
  @results = $index->search(url => $url, time => $when);

  build WARC::Index::File::CDX (...);	# or ...
  build WARC::Index::File::SDBM (...);

=head1 DESCRIPTION

C<WARC::Index> is an abstract base class for indexes on WARC files and
WARC-alike files.  This class establishes the expected interface and
provides a simple interface for building indexes.

=head2 Methods

=over

=item $index = attach WARC::Index::File::* (...)

Construct an index object using the indicated technology and whatever
parameters the index implementation needs.

Typically, indexes are file-based and a single parameter is the name of an
index file which in turn contains the names of the indexed WARC files.

=cut

sub attach {
  die __PACKAGE__." is an abstract base class and "
    .(shift)." must override the 'attach' method"
}

=item $record = $collection-E<gt>search( ... )

=item @records = $collection-E<gt>search( ... )

Search an index for records matching parameters.  The C<WARC::Collection>
class uses this method to search each index in a collection.

The details of the parameters for this method are documented in the
L<"Search Keys" section|WARC::Collection/"Search Keys"> of the
C<WARC::Collection> page.

=cut

sub search {
  die __PACKAGE__." is an abstract base class and "
    .(ref shift)." must override the 'search' method"
}

=item build WARC::Index::File::* (into =E<gt> $dest, from =E<gt> ...)

=item build WARC::Index::File::* (from =E<gt> [...], into =E<gt> $dest)

The C<WARC::Index> base class B<does> provide this method, however.  The
C<build> method works by loading the corresponding index builder class and
driving the process or simply returning the newly-constructed object.

The C<build> method itself handles the C<from> key for specifying the files
to index.  The C<from> key can be given an array reference, after which
more key =E<gt> value pairs may follow, or can simply use the rest of the
argument list as its value.

If the C<from> key is given, the C<build> method will read the indicated
files, construct an index, and return nothing.  If the C<from> key is not
given, the C<build> method will construct and return an index builder.

All index builders accept at least the C<into> key for specifying where to
store the index.  See the documentation for WARC::Index::File::*::Builder
for more information.

=cut

sub build {
}

=back

=head2 Index system registration

The C<WARC::Index> package also maintains a registry of loaded index
support.  The C<register> function adds the calling package to the list.

=over

=item WARC::Index::register( filename =E<gt> $filename_re )

Add the calling package to an internal list of available index handlers.
The calling package must be a subclass of C<WARC::Index> or this function
will croak().

The C<filename> key indicates that the calling package expects to handle
index files with names matching the provided regex.

=cut

sub register {
}

=item WARC::Index::find_handler( $filename )

Return the registered handler for $filename or undef if none match.

=cut

sub find_handler {
}

=back

=cut

1;
__END__

=head1 AUTHOR

Jacob Bachmeyer, E<lt>jcb@cpan.orgE<gt>

=head1 SEE ALSO

L<WARC>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2019 by Jacob Bachmeyer

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=cut
