/*
 * dproc.c - SCO Unix process access functions for lsof
 */


/*
 * Copyright 1995 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1995 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dproc.c,v 1.13 95/09/19 10:22:15 abe Exp $";
#endif

#include "lsof.h"


/*
 * Local static values
 */

static KA_T Kp;				/* kernel process table address */
static caddr_t *Nc = NULL;		/* node cache */
static int Nn = 0;			/* number of Nc[] entries allocated */

#if	_SCOV<500
static int Npp = 0;			/* number of pregions per process */
static struct pregion *Pr = NULL;	/* pregion buffer */
static int Prsz = 0;			/* size of Pr */
#endif	/* _SCOV<500 */


static struct var Var;			/* kernel variables */



_PROTOTYPE(static void get_cdevsw,(void));
_PROTOTYPE(static void get_kernel_access,(void));
_PROTOTYPE(static void readfsinfo,(void));
_PROTOTYPE(static void process_text,(struct pregion *prp));


/*
 * gather_proc_info() -- gather process information
 */

void
gather_proc_info()
{
	int i, j, nf, pbc, px;
	struct proc *p;
	static char *pb = NULL;
	int pid, pgrp;
	short pss, sf;
	static struct user *u;
	static char *ua = NULL;
	unsigned int uid;
/*
 * Allocate user structure buffer.
 */
	if (!ua) {
	    if ((ua = (char *)malloc(MAXUSIZE * NBPC)) == (char *)NULL) {
		(void) fprintf(stderr,
		    "%s: no space for %d byte user structure buffer\n",
		    Pn, MAXUSIZE * NBPC);
		exit(1);
	    }
	    u = (struct user *)ua;
	}
/*
 * Allocate proc structure buffer.
 */
	if (!pb) {
	    if ((pb = (char *)malloc(sizeof(struct proc) * PROCBFRD)) == NULL) {
		(void) fprintf(stderr, "%s: no space for %d proc structures\n",
		    Pn, PROCBFRD);
		exit(1);
	    }
	}
/*
 * Examine proc structures and their associated information.
 */
	for (pbc = px = 0; px < Var.v_proc; px++)
	{
		if (px >= pbc) {

		/*
		 * Refill proc buffer.
		 */
			i = Var.v_proc - px;
			if (i > PROCBFRD)
				i = PROCBFRD;
			j = kread((KA_T)(Kp + (px * sizeof(struct proc))), pb,
				sizeof(struct proc) * i);
			pbc = px + i;
			p = (struct proc *)pb;
			if (j) {
				px += i;
				continue;
			}
		} else
			p++;
		if (p->p_stat == 0 || p->p_stat == SZOMB)
			continue;
	/*
	 * Get Process ID, Process group ID, and User ID.
	 */
		pid = (int)p->p_pid;
		pgrp = (int)p->p_pgrp;
		uid = (unsigned int)p->p_uid;
		if (is_proc_excl(pid, pgrp, (UID_ARG)uid, &pss, &sf))
			continue;
	/*
	 * Get the user area associated with the process.
	 */
		if (sysi86(RDUBLK, pid, ua, MAXUSIZE * NBPC) == -1)
			continue;
	/*
	 * Allocate a local process structure.
	 */
		if (is_cmd_excl(u->u_comm, &pss, &sf))
			continue;
		alloc_lproc(pid, pgrp, (UID_ARG)uid, u->u_comm,
			(int)pss, (int)sf);
		Plf = NULL;
	/*
	 * Save current working directory information.
	 */
		if (u->u_cdir) {
			alloc_lfile(CWD, -1);
			process_node((caddr_t)u->u_cdir);
			if (Lf->sf)
				link_lfile();
		}
	/*
	 * Save root directory information.
	 */
		if (u->u_rdir) {
			alloc_lfile(RTD, -1);
			process_node((caddr_t)u->u_rdir);
			if (Lf->sf)
				link_lfile();
		}
	/*
	 * Print information on the text file.
	 */
		if (p->p_region)
			process_text(p->p_region);
	/*
	 * Save information on file descriptors.
	 */

#if	_SCOV<42
		nf = Var.v_nofiles;
#else	/* _SCOV>=42 */
		nf = u->u_nofiles ? u->u_nofiles : Var.v_nofiles;
#endif	/* _SCOV<42 */

		for (i = 0; i < nf; i++) {
			if (u->u_ofile[i]) {
				alloc_lfile(NULL, i);
				process_file(u->u_ofile[i]);
				if (Lf->sf)
					link_lfile();
			}
		}
	/*
	 * Examine results.
	 */
		if (examine_lproc())
			return;
	}
}


/*
 * get_cdevsw() - get cdevsw[] names and record clone major device number
 */

void
get_cdevsw()
{
	char buf[16];
	struct cdevsw *c, *tmp;
	int i, j;
/*
 * Check cdevsw[] kernel addresses.
 * Read cdevsw[] count from kernel's cdevcnt.
 */
	if (Nl[X_CDEVSW].n_value == (long)NULL
	||  Nl[X_CDEVCNT].n_value == (long)NULL
	||  kread((KA_T)Nl[X_CDEVCNT].n_value, (char *)&Cdevcnt,
		sizeof(Cdevcnt))
	||  Cdevcnt < 1)
		return;
/*
 * Allocate cache space.
 */
	if ((Cdevsw = (char **)malloc(Cdevcnt * sizeof(char *))) == NULL) {
		(void) fprintf(stderr, "%s: no space for %d cdevsw[] names\n",
			Pn, Cdevcnt);
		exit(1);
	}
/*
 * Allocate temporary space for a copy of cdevsw[] and read it.
 */
	i = Cdevcnt * sizeof(struct cdevsw);
	if ((tmp = (struct cdevsw *)malloc(i)) == NULL) {
		(void) fprintf(stderr, "%s: no space for %d cdevsw[] entries\n",
			Pn, Cdevcnt);
		exit(1);
	}
	if (kread((KA_T)Nl[X_CDEVSW].n_value, (char *)tmp, i)) {
		(void) free((FREE_P *)Cdevsw);
		Cdevsw = NULL;
		Cdevcnt = 0;
		(void) free((FREE_P *)tmp);
		return;
	}
/*
 * Cache the names from cdevsw[].d_name.
 * Record the number of the "clone" device.
 */
	j = sizeof(buf) - 1;
	buf[j] = '\0';
	for (c = tmp, i = 0; i < Cdevcnt; c++, i++) {
		Cdevsw[i] = NULL;
		if (c->d_name == (char *)NULL)
		    continue;
		if (kread((KA_T)c->d_name, buf, j)) {
		    (void) fprintf(stderr,
			"%s: WARNING: can't read name for cdevsw[%d]: %#x\n",
			Pn, i, c->d_name);
		    continue;
		}
		if ( ! buf[0])
		    continue;
		if ((Cdevsw[i] = (char *)malloc(strlen(buf) + 1)) == NULL) {
		    (void) fprintf(stderr,
			"%s: no space for cdevsw[%d] name: %s\n",
			Pn, i, buf);
		    exit(1);
		}
		(void) strcpy(Cdevsw[i], buf);
		if ( ! HaveCloneMajor && strcmp(buf, "clone") == 0) {
		    CloneMajor = i;
		    HaveCloneMajor = 1;
		}
	}
	(void) free((FREE_P *)tmp);
}


/*
 * get_kernel_access() - get access to kernel memory
 */

static void
get_kernel_access()
{
	time_t lbolt;
	MALLOC_S len;

#if	defined(WILLDROPGID)
/*
 * If kernel memory isn't coming from KMEM, drop setgid permission
 * before attempting to open the (Memory) file.
 */
	if (Memory)
		(void) dropgid();
#else	/* !defined(WILLDROPGID) */
/*
 * See if the non-KMEM memory file is readable.
 */
	if (Memory && !is_readable(Memory, 1))
		exit(1);
#endif	/* defined(WILLDROPGID) */

/*
 * Open kernel memory access.
 */
	if ((Kmem = open(Memory ? Memory : KMEM, O_RDONLY, 0)) < 0) {
	    (void) fprintf(stderr, "%s: can't open %s: %s\n", Pn,
		Memory ? Memory : KMEM, strerror(errno));
	    exit(1);
	}

#if	defined(WILLDROPGID)
/*
 * Drop setgid permission, if necessary.
 */
	if (!Memory)
		(void) dropgid();
#else	/* !defined(WILLDROPGID) */
/*
 * See if the name list file is readable.
 */
	if (Nmlst && !is_readable(Nmlst, 1))
		exit(1);
#endif	/* defined(WILLDROPGID) */

/*
 * Access kernel symbols and values.
 */
	if (nlist(Nmlst ? Nmlst : N_UNIX, Nl) < 0) {
	    (void) fprintf(stderr,
		"%s: can't read kernel name list from %s\n",
		Pn, Nmlst ? Nmlst : N_UNIX);
	    exit(1);
	}
	if ((Kp = Nl[X_PROC].n_value) == (long)NULL) {
	    (void) fprintf(stderr, "%s: no proc table pointer\n", Pn);
	    exit(1);
	}

#if	_SCOV<500
/*
 * Read pregion information.
 */
	if (Nl[X_PREGPP].n_value == (long)NULL
	||  kread((KA_T)Nl[X_PREGPP].n_value, (char *)&Npp, sizeof(Npp))
	||  Npp < 1) {
	    (void) fprintf(stderr,
		"%s: can't read pregion count (%d) from 0x%#x\n",
		Pn, Npp, Nl[X_PREGPP].n_value); 
	    exit(1);
	}
	Prsz = (MALLOC_S)(Npp * sizeof(struct pregion));
	if ((Pr = (struct pregion *)malloc(Prsz)) == NULL) {
	    (void) fprintf(stderr,
		"%s: can't allocate space for %d pregions\n",
		Pn, Npp);
	    exit(1);
	}
#endif	/* _SCOV< 500 */

/*
 * Read system configuration information.
 */
	if (Nl[X_VAR].n_value == (long)NULL
	||  kread((KA_T)Nl[X_VAR].n_value, (char *)&Var, sizeof(Var)))
	{
	    (void) fprintf(stderr,
		"%s: can't read system configuration info\n", Pn);
	    exit(1);
	}
/*
 * Read system clock values -- Hz and lightning bolt timer.
 */
	if (Nl[X_HZ].n_value == (long)NULL
	||  kread((KA_T)Nl[X_HZ].n_value, (char *)&Hz, sizeof(Hz)))
	{
	    if (!Fwarn)
		(void) fprintf(stderr, "%s: WARNING: can't read Hz from %#lx\n",
		    Pn, Nl[X_HZ].n_value);
	    Hz = -1;
	}
	if (!Fwarn
	&&  (Nl[X_LBOLT].n_value == (long)NULL
	||   kread((KA_T)Nl[X_LBOLT].n_value, (char *)&lbolt, sizeof(lbolt))))
	{
	    (void) fprintf(stderr,
		"%s: WARNING: can't read lightning bolt timer from %#lx\n",
		Pn, Nl[X_LBOLT].n_value);
	}
/*
 * Get socket device number and socket table address.
 */
	if (Nl[X_SOCKDEV].n_value == (long)NULL
	||  kread((KA_T)Nl[X_SOCKDEV].n_value, (char *)&Sockdev,
		sizeof(Sockdev)))
	{
	    (void) fprintf(stderr,
		"%s: WARNING: can't identify socket device.\n", Pn);
	    (void) fprintf(stderr,
		"      Socket output may be incomplete.\n");
	    return;
	}
	if ((Socktab = (KA_T)Nl[X_SOCKTAB].n_value) == (long)NULL) {
	    (void) fprintf(stderr,
		"%s: WARNING: socket table address is NULL.\n", Pn);
	    (void) fprintf(stderr,
		"      Socket output may be incomplete.\n");
	    return;
	}

#if	_SCOV>=40
/*
 * Get extended device table parameters.  These are needed by the
 * kernel versions of the major() and minor() device numbers; they
 * identify socket devices and assist in the conversion of socket
 * device numbers to socket table addresses.
 */
	if (Nl[X_NXD].n_value == (long)NULL
	||  kread((KA_T)Nl[X_NXD].n_value,(char *)&nxdevmaps,sizeof(nxdevmaps))
	||  nxdevmaps < 0)
	{
	    (void) fprintf(stderr,
		"%s: WARNING: bad extended device table size (%d) at 0x%#x.\n",
		Pn, nxdevmaps, Nl[X_NXD].n_value);
	    (void) fprintf(stderr,
		"      Socket output may be incomplete.\n");
	    return;
	}
	len = (MALLOC_S)((nxdevmaps + 1) * sizeof(struct XDEVMAP));
	if ((Xdevmap = (struct XDEVMAP *)malloc(len)) == (struct XDEVMAP *)NULL)
	{
	    (void) fprintf(stderr, "%s: no space for %d byte xdevmap table\n",
		Pn, len);
	    exit(1);
	}
	if (Nl[X_XDEV].n_value == (long)NULL
	||  kread((KA_T)Nl[X_XDEV].n_value, (char *)Xdevmap, len))
	{
	    (void) fprintf(stderr,
		"%s: WARNING: can't read %d byte xdevmap table at 0x%#x.\n",
		Pn, len, Nl[X_XDEV].n_value);
	    (void) fprintf(stderr,
		"      Socket output may be incomplete.\n");
	    return;
	}
#endif	/* _SCOV>=40 */

	HaveSockdev = 1;
}


/*
 * initialize() - perform all initialization
 */

void
initialize()
{
	get_kernel_access();
	get_cdevsw();
	iuidcache(Var.v_proc);
	readfsinfo();
}


/*
 * kread() - read from kernel memory
 */

int
kread(addr, buf, len)
	KA_T addr;			/* kernel memory address */
	char *buf;			/* buffer to receive data */
	READLEN_T len;			/* length to read */
{
	int br;

	if (lseek(Kmem, (off_t)addr, SEEK_SET) == (off_t)-1L)
		return(1);
	if ((br = read(Kmem, buf, len)) < 0)
		return(1);
	return(((READLEN_T)br == len) ? 0 : 1);
}


/*
 * process_text() - process text access information
 */

static void
process_text(prp)
	struct pregion *prp;		/* process region pointer */
{
	int i, j, k;
	struct pregion *p;
	struct region r;
	caddr_t na;
	char *ty, tyb[8];

#if	_SCOV>=500
	struct pregion *pc, ps;
#endif	/* _SCOV>=500 */

/*
 * Read and process the pregions.
 */

#if	_SCOV<500
	if (kread((KA_T)prp, (char *)Pr, Prsz))
		return;
	for (i = j = 0, p = Pr; i < Npp; i++, p++)
#else	/* _SCOV>=500 */
	for (i = j = 0, p = &ps, pc = prp; pc; pc = p->p_next, i++)
#endif	/* _SCOV<500 */

	{

#if	_SCOV>=500
	/*
	 * Avoid infinite loop.
	 */
		if (i > 1000) {
		    if (!Fwarn)
			(void) fprintf(stderr,
			    "%s: too many virtual address regions for PID %d\n",
			    Pn, Lp->pid);
		    return;
		}
		if ((i && pc == prp)
		||  kread((KA_T)pc, (char *)p, sizeof(ps)))
			return;
#endif	/* _SCOV>=500 */

		if (p->p_reg == NULL)
			continue;
	/*
	 * Read the region.
	 * Skip entries with no node pointers and duplicate node addresses.
	 */
		if (kread((KA_T)p->p_reg, (char *)&r, sizeof(r)))
			continue;
		if ((na = (caddr_t)r.r_iptr) == NULL)
			continue;
		for (k = 0; k < j; k++) {
			if (Nc[k] == na)
				break;
			}
		if (k < j)
			continue;
	/*
	 * Cache the node address for duplicate checking.
	 */
		if (Nc == NULL) {
			if ((Nc = (caddr_t *)malloc((MALLOC_S)
				  (sizeof(KA_T) * 10)))
			== NULL) {
				(void) fprintf(stderr,
					"%s: no txt ptr space, PID %d\n",
					Pn, Lp->pid);
				exit(1);
			}
			Nn = 10;
		} else if (j >= Nn) {
			Nn += 10;
			if ((Nc = (caddr_t *)realloc((MALLOC_P *)Nc,
				   (MALLOC_S)(Nn * sizeof(KA_T))))
			== NULL) {
				(void) fprintf(stderr,
					"%s: no more txt ptr space, PID %d\n",
					Pn, Lp->pid);
				exit(1);
			}
		}
		Nc[j++] = na;
	/*
	 * Save text node and mapped region information.
	 */
		switch (p->p_type) {
 			case PT_DATA:		/* data and text of */
 			case PT_TEXT:		/* executing binaries */
 				ty = " txt";
 				break;
  			case PT_LIBDAT:		/* shared library data and */
  			case PT_LIBTXT:		/* COFF format text */
  				ty = " ltx";
  				break;
 			case PT_SHFIL:		/* memory mapped file */
 				ty = " mem";
 				break;
 			case PT_V86:		/* virtual 8086 mode */
 				ty = " v86";
				break;
 			case PT_VM86:		/* MERGE386 vm86 region */
 				ty = " m86";
 				break;
  			default:		/* all others as a hex number */
				(void) sprintf(tyb, " M%02x", p->p_type & 0xff);
 				ty = tyb;
		}
		if (ty) {
			alloc_lfile(ty, -1);
			process_node(na);
			if (Lf->sf)
				link_lfile();
		}
	}
}


/*
 * readfsinfo() - read file system information
 */

static void
readfsinfo()
{
	char buf[FSTYPSZ+1];
	int i;

	if ((Fsinfomax = sysfs(GETNFSTYP)) == -1) {
		(void) fprintf(stderr, "%s: sysfs(GETNFSTYP) error: %s\n",
			Pn, strerror(errno));
		exit(1);
	} 
	if (Fsinfomax == 0)
		return;
	if ((Fsinfo = (char **)malloc((MALLOC_S)(Fsinfomax * sizeof(char *))))
	== NULL) {
		(void) fprintf(stderr, "%s: no space for sysfs info\n", Pn);
		exit(1);
	}
	for (i = 1; i <= Fsinfomax; i++) {
		if (sysfs(GETFSTYP, i, buf) == -1) {
			(void) fprintf(stderr,
				"%s: sysfs(GETFSTYP) error: %s\n",
				Pn, strerror(errno));
			exit(1);
		}
		buf[FSTYPSZ] = '\0';
		if ((Fsinfo[i-1] = (char *)malloc((MALLOC_S)(strlen(buf) + 1)))
		== NULL) {
			(void) fprintf(stderr,
				"%s: no space for file system entry %s\n",
				Pn, buf);
			exit(1);
		}
		(void) strcpy(Fsinfo[i-1], buf);
	}
}
