/*
 * dfile.c - SCO Unix file processing functions for lsof
 */


/*
 * Copyright 1995 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1995 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dfile.c,v 1.8 95/09/12 07:51:46 abe Exp $";
#endif


#include "lsof.h"


/*
 * is_file_named() - is this file named?
 */

int
is_file_named(p, ty)
	char *p;			/* path name; NULL = search by device
					 * and inode (from *Lf) */
	int ty;				/* inode type */
{
	int f;
	struct sfile *s;
/*
 * Search the file chain.
 */
	for (f = 0, s = Sfile; s; s = s->next) {

	/*
	 * Search by path name, as requested.
	 */
		if (p != NULL) {
			if (strcmp(p, s->aname) == 0
			||  strcmp(p, s->name)  == 0)
			{
				f = 2;
				break;
			}
			continue;
		}
	/*
	 * If this is a stream, check for a clone device match.
	 */
		if (Lf->is_stream) {
			if (HaveCloneMajor && major(s->dev) == CloneMajor
			&&  Lf->dev_def && major(Lf->dev) == minor(s->dev)) {
				f = 3;
				break;
			}
			continue;
		}
	/*
	 * Check for a regular file or directory -- the device and
	 * indoe numbers must match.
	 */
		if (s->type && Lf->dev_def) {
			if (Lf->dev == s->dev && (ino_t)Lf->inode == s->i) {
				f = 1;
				break;
			}
			continue;
		}
	/*
	 * Check for a file system match.
	 * Try to avoid matching character files to non-character devices.
	 */
		if (Lf->dev_def && (Lf->dev == s->dev)) {
			if ( ! (ty == IFCHR && s->mode != S_IFCHR)) {
				f = 1;
				break;
			}
		}
	}
/*
 * Convert the name if a match occurred.
 */
	switch (f) {
	case 0:
		return(0);
	case 1:
		(void) strcpy(Namech, s->name);
		if (s->devnm)
			(void) sprintf(endnm(), " (%s)", s->devnm);
		break;
	case 2:
		(void) strcpy(Namech, p);
		break;
	/* case 3:		do nothing for stream clone matches */
	}
	s->f = 1;
	return(1);
}


/*
 * print_file() - print file
 */

void
print_file()
{
	char buf[16];

	if (Hdr == 0) {
	    (void) printf("COMMAND     PID%s     USER   FD   TYPE     DEVICE ",
		Fpgrp ? "   PGRP" : "");
	    (void) printf("%10s", Foffset ? "OFFSET"
					  : Fsize ? "SIZE"
						  : "SIZE/OFF");
	    (void) puts("  INODE NAME");
	    Hdr++;
	}
	(void) printf("%-9.9s%6d", Lp->cmd, Lp->pid);
	if (Fpgrp)
	    printf(" %6d", Lp->pgrp);
	(void) printf(" %8.8s %4s%c%c %4.4s ",
			printuid((UID_ARG)Lp->uid, NULL),
			Lf->fd,
			Lf->access,
			Lf->lock,
			Lf->type
		);
	if (Lf->dev_def)
		(void) printf(" %4d,%4d ",
			Lf->is_nfs ? ((~(Lf->dev >> 8)) & 0xff)
				   : emajor(Lf->dev),
			eminor(Lf->dev));
	else {
		if (Lf->dev_ch) {
			(void) fputs(Lf->dev_ch, stdout);
			putchar(' ');
		} else
			(void) fputs("           ", stdout);
	}
	if (Lf->sz_def)
		(void) printf("%10u", Lf->sz);
	else if (Lf->off_def) {
		if ((unsigned int)Lf->off < 100000000) {
			(void) sprintf(buf, "0t%d", Lf->off);
			(void) printf("%10.10s", buf);
		} else
			(void) printf("%#10x", (Lf->off & 0xffffffff));
	} else
		(void) fputs("          ", stdout);
	switch (Lf->inp_ty) {
	case 1:
		if (Lf->inode < 1000000l)
			(void) printf(" %6lu ", Lf->inode);
		else
			(void) printf(" *%05lu ", Lf->inode % 100000l);
		break;
	case 2:
		if (Lf->iproto[0] == '\0')
			(void) fputs("        ", stdout);
		else
			(void) printf(" %6.6s ", Lf->iproto);
		break;
	default:
		(void) fputs("        ", stdout);
	}
	printname(1);
}


/*
 * process_file() - process file
 */

void
process_file(fp)
	struct file *fp;		/* kernel file structure address */
{
	struct file f;
	int flag;

	if (kread((KA_T)fp, (char *)&f, sizeof(f))) {
		(void) sprintf(Namech, "can't read file struct from %#x", fp);
		enter_nm(Namech);
		return;
	}
	Lf->off = f.f_offset;

	if (f.f_count) {

	/*
	 * Construct access code.
	 */
		if ((flag = (f.f_flag & (FREAD | FWRITE))) == FREAD)
			Lf->access = 'r';
		else if (flag == FWRITE)
			Lf->access = 'w';
		else if (flag == (FREAD | FWRITE))
			Lf->access = 'u';
	/*
	 * Process structure.
	 */
		process_node((caddr_t)f.f_inode);
		return;

	}
	enter_nm("no more information");
}


/*
 * The ck_file_arg() function is obtained from ../common/ckfa.frag.
 */

/* CKFA_EXPDEV is not defined. */
/* CKFA_MPXCHAN is not defined. */
/* CKFA_XDEVTST is not defined. */
