/*
 * dproc.c - Motorola V/88 process access functions for lsof
 */


/*
 * Copyright 1994 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1994 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dproc.c,v 1.13 95/09/19 10:50:49 abe Exp $";
#endif

#include "lsof.h"

#if     defined(HASNCACHE)
#include <sys/dnlc.h>
#endif  /* defined(HASNCACHE) */


/*
 * Local static values
 */

#if	V88<40401
static KA_T Kp;				/* kernel process table address */
static caddr_t *Nc = NULL;		/* node cache */
static MALLOC_S Nn = 0;			/* number of Nc[] entries allocated */
static int Npp = 0;			/* number of pregions per process */
static struct pregion *Pr = NULL;	/* pregion buffer */
static int Prsz = 0;			/* size of Pr */
#endif	/* V88<40401 */

static struct var Var;			/* kernel variables */



_PROTOTYPE(static void get_kernel_access,(void));
_PROTOTYPE(static void readfsinfo,(void));

#if	V88<40401
_PROTOTYPE(static void process_text,(struct pregion *prp));
#else   /* V88>=40401 */
_PROTOTYPE(static void process_text,(struct as *pa));
#endif	/* V88<40401 */

#if     defined(HASNCACHE)
_PROTOTYPE(static void ncache_load,(void));
#endif  /* defined(HASNCACHE) */


/*
 * gather_proc_info() -- gather process information
 */

void
gather_proc_info()
{
	struct file *f;
	int i, j, nf;
	struct proc *p, ps;
	int pgrp, pid;
	short pss, sf;
	static struct user *u = NULL;
	uid_t uid;

#if	V88<40401
	KA_T pa;
	int px;
	static char *uaf;
	static MALLOC_S uafl;
#else	/* V88>=40401 */
	struct dirent *dp;
	DIR *pdfd;
	int  pfd;
	struct pid pids;
	char pn[MAXPATHLEN+1];
	size_t psz;
#endif	/* V88<40401 */

#if	V88<40401
/*
 * Allocate space for user area and file pointers for V/88 version
 * below R40V42.
 */
	if (!u) {
	    uafl = (MALLOC_S)(sizeof(struct user)
		 + (Var.v_nofiles * sizeof(struct file *)));
	    if ((uaf = (char *)malloc(uafl)) == NULL) {
		(void) fprintf(stderr,
		    "%s: no space for uarea and file pointers\n", Pn);
		exit(1);
	    }
	    u = (struct user *)uaf;
	}
#else	/* V88>=40401 */
/*
 * Allocate space for user area for V/88 versions R40V42 and above.
 */
	if (!u) {
	    psz = sysconf(_SC_PAGESIZE);
	    if ((u = (struct user *)memalign(psz, (sizeof(struct user) + psz)))
	    == NULL) {
		(void) fprintf(stderr,
			"%s: can't memalign user structure buffer\n", Pn);
		exit(1);
	    }
	}
#endif	/* V88<40401 */

#if     defined(HASNCACHE)
/*
 * Read kernel name cache.
 */
        ncache_load();
#endif  /* defined(HASNCACHE) */

/*
 * Examine proc structures and their associated information.
 */

#if	V88<40401
	for (p = &ps, pa = Kp, px = 0;
	     px < Var.v_proc;
	     pa += sizeof(struct proc), px++)
#else	/* V88>=40401 */
	(void) sprintf(pn, "/%s", HASPROCFS);
	if ((pdfd = opendir(pn)) == NULL) {
		(void) fprintf(stderr, "%s: can't open %s\n", Pn, pn);
		exit(1);
	}
	(void) rewinddir(pdfd);
	for (p = &ps, pfd = -1; (dp = readdir(pdfd)) != NULL; )
#endif	/* V88<40401 */

	{

#if	V88<40401
		if (kread(pa, (char *)&ps, sizeof(ps)))
			continue;
#else	/* V88>=40401 */
		if (pfd >= 0) {
			(void) close(pfd);
			pfd = -1;
		}
		if (dp->d_name[0] == '.')
			continue;
		(void) sprintf(pn, "/%s/%s", HASPROCFS, dp->d_name);
		if ((pfd = open(pn, O_RDONLY, 0)) < 0)
			continue;
		if (ioctl(pfd, PIOCGETPR, p) < 0)
			continue;
#endif	/* V88<40401 */

		if (p->p_stat == 0 || p->p_stat == SZOMB)
			continue;
	/*
	 * Get Process ID, Process group ID, and User ID.
	 */

#if	V88<40401
		pid = (int)p->p_pid;
		pgrp = p->p_pgrp;
#else	/* V88>=40401 */
		if (p->p_pidp == NULL
		||  kread((KA_T)p->p_pidp, (char *)&pids, sizeof(pids)))
			continue;
		if ((pid = (int)pids.pid_id) != atoi(dp->d_name))
			continue;
		if (p->p_pgidp == NULL
		||  kread((KA_T)p->p_pgidp, (char *)&pids, sizeof(pids)))
			continue;
		pgrp = (int)pids.pid_id;
#endif	/* V88<40401 */

		uid = (uid_t)p->p_uid;
		if (is_proc_excl(pid, pgrp, (UID_ARG)uid, &pss, &sf))
			continue;
	/*
	 * Get the user area associated with the process.
	 */

#if	V88<40401
		if (sysmot(RDUBLK, pid, uaf, uafl) != uafl)
			continue;
#else	/* V88>=40401 */
		if (ioctl(pfd, PIOCGETU, u) < 0)
			continue;
#endif	/* V88<40401 */

	/*
	 * Allocate a local process structure.
	 */
		if (is_cmd_excl(u->u_comm, &pss, &sf))
			continue;
		alloc_lproc(pid, pgrp, (UID_ARG)uid, u->u_comm,
			(int)pss, (int)sf);
		Plf = NULL;
	/*
	 * Save current working directory information.
	 */
		if (u->u_cdir) {
			alloc_lfile(CWD, -1);
			process_node((caddr_t)u->u_cdir);
			if (Lf->sf)
				link_lfile();
		}
	/*
	 * Save root directory information.
	 */
		if (u->u_rdir) {
			alloc_lfile(RTD, -1);
			process_node((caddr_t)u->u_rdir);
			if (Lf->sf)
				link_lfile();
		}
	/*
	 * Print information on the text file.
	 */

#if	V88<40401
		if (p->p_region)
			process_text(p->p_region);
#else	/* V88>=40401 */
		if (p->p_as && Nl[X_SEGVN].n_value)
			process_text(p->p_as);
#endif	/* V88<40401 */

	/*
	 * Save information on file descriptors.
	 */

#if	V88<40401
	/*
	 * The file pointers for V/88 versions below R40V42 are stored in an
	 * array of size Var.v_nofiles that follows the user area segment.
	 */
		for (i = 0, nf = Var.v_nofiles; i < nf; i++)
#else	/* V88>=40401 */
	/*
	 * The file pointers for V/88 versions R40V42 and above are stored
	 * in dynamically linked ufchunk structures, each containing
	 * NFPCHUNK file pointers.  The first ufchunk structure is in
	 * the user area.
	 */
		for (i = 0, j = 0; i < u->u_nofiles; i++)
#endif	/* V88<40401 */

		{

#if	V88<40401
			f = u->u_ofile[i];
#else	/* V88>=40401 */
			if (++j > NFPCHUNK) {
			    if ( ! u->u_flist.uf_next)
				break;
			    if (kread((KA_T)u->u_flist.uf_next,
				(char *)&u->u_flist, sizeof(struct ufchunk)))
				    break;
			    j = 1;
			}
			f = u->u_flist.uf_ofile[j-1];
#endif	/* V88<40401 */

			if (f) {
				alloc_lfile(NULL, i);
				process_file(f);
				if (Lf->sf)
					link_lfile();
			}
		}
	/*
	 * Examine results.
	 */
		if (examine_lproc())
			return;
	}

#if	V88>=40401
/*
 * Close proc file system access controls.
 */
	if (pfd >= 0)
		(void) close(pfd);
	(void) closedir(pdfd);
#endif	/* V88>=40401 */

}


/*
 * get_kernel_access() - get access to kernel memory
 */

static void
get_kernel_access()
{

#if	V88<40401 && defined(WILLDROPGID)
/*
 * If kernel memory isn't coming from KMEM, drop setgid permission
 * before attempting to open the (Memory) file.
 */
	if (Memory)
		(void) dropgid();
#else	/* V88>=40401 || !defined(WILLDROPGID) */
/*
 * See if the non-KMEM memory file is readable.
 */
	if (Memory && !is_readable(Memory, 1))
		exit(1);
#endif	/* V88<40401 && defined(WILLDROPGID) */

/*
 * Open kernel memory access.
 */
	if ((Kmem = open(Memory ? Memory : KMEM, O_RDONLY, 0)) < 0) {
		(void) fprintf(stderr, "%s: can't open %s: %s\n", Pn,
			Memory ? Memory : KMEM, strerror(errno));
		exit(1);
	}

#if	V88<40401 && defined(WILLDROPGID)
/*
 * Drop setgid permission, if necessary.
 */
	if (!Memory)
		(void) dropgid();
#else	/* V88>=40401 || !defined(WILLDROPGID) */
/*
 * See if the name list file is readable.
 */
	if (Nmlst && !is_readable(Nmlst, 1))
		exit(1);
#endif	/* V88<40401 && defined(WILLDROPGID) */

/*
 * Access kernel symbols and values.
 */
        if (nlist(Nmlst ? Nmlst : N_UNIX, Nl) < 0) {
		(void) fprintf(stderr,
			"%s: can't read kernel name list from %s\n",
			Pn, Nmlst ? Nmlst : N_UNIX);
		exit(1);
	}

#if	V88<40401
	if ((Kp = Nl[X_PROC].n_value) == NULL) {
		(void) fprintf(stderr, "%s: no proc table pointer\n", Pn);
		exit(1);
	}
/*
 * Read pregion information for V/88 versions below R40V42.
 */
	if (Nl[X_PREGPP].n_value == NULL
	||  kread((KA_T)Nl[X_PREGPP].n_value, (char *)&Npp, sizeof(Npp))
	||  Npp < 1) {
		(void) fprintf(stderr,
			"%s: can't read pregion count (%d) from %s\n",
			Pn, Npp, Nl[X_PREGPP].n_value); 
		exit(1);
	}
	Prsz = (MALLOC_S)(Npp * sizeof(struct pregion));
	if ((Pr = (struct pregion *)malloc(Prsz)) == NULL) {
		(void) fprintf(stderr,
			"%s: can't allocate space for %d pregions\n",
			Pn, Npp);
		exit(1);
	}
#endif	/* V88<40401 */

	if (Nl[X_VAR].n_value == NULL
	||  kread((KA_T)Nl[X_VAR].n_value, (char *)&Var, sizeof(Var)))
	{
		(void) fprintf(stderr,
			"%s: can't read system configuration info\n", Pn);
		exit(1);
	}

#if	V88>=40403
/*
 * Read clone major device number, if possible.
 */
	if (Nl[X_CLONEMAJ].n_value
	&&  kread((KA_T)Nl[X_CLONEMAJ].n_value, (char *)&CloneMajor,
		sizeof(CloneMajor)) == 0)
	{
	    HaveCloneMajor = 1;
	    findclones();
	} else
	    HaveCloneMajor = 0;
#endif	/* V88>=40403 */

}


/*
 * initialize() - perform all initialization
 */

void
initialize()
{
	get_kernel_access();
	iuidcache(Var.v_proc);
	readfsinfo();
}


/*
 * kread() - read from kernel memory
 */

int
kread(addr, buf, len)
	KA_T addr;			/* kernel memory address */
	char *buf;			/* buffer to receive data */
	READLEN_T len;			/* length to read */
{
	int br;

	if (lseek(Kmem, (long)addr, L_SET) == (long)-1L)
		return(-1);
	br = read(Kmem, buf, len);
	return((br == len) ? 0 : 1);
}
#if	V88<40401


/*
 * process_text() - process R32V3 text access information
 */

static void
process_text(prp)
	struct pregion *prp;		/* process region pointer */
{
	int i, j, k;
	struct pregion *p;
	struct region r;
	caddr_t na;
/*
 * Read and process the pregions.
 */
	if (kread((KA_T)prp, (char *)Pr, Prsz))
		return;
	for (i = j = 0, p = Pr; i < Npp; i++, p++) {
		if (p->p_reg == NULL)
			continue;
	/*
	 * Read the region.
	 * Skip entries with no node pointers and duplicate node addresses.
	 */
		if (kread((KA_T)p->p_reg, (char *)&r, sizeof(r)))
			continue;
		if ((na = (caddr_t)r.r_iptr) == NULL)
			continue;
		for (k = 0; k < i; k++) {
			if (Nc[k] == na)
				break;
			}
		if (k < j)
			continue;
	/*
	 * Cache the node address for duplicate checking.
	 */
		if (Nc == NULL) {
			if ((Nc = (caddr_t *)malloc((MALLOC_S)
				  (sizeof(KA_T) * 10)))
			== NULL) {
				(void) fprintf(stderr,
					"%s: no txt ptr space, PID %d\n",
					Pn, Lp->pid);
				exit(1);
			}
			Nn = 10;
		} else if (j >= Nn) {
			Nn += 10;
			if ((Nc = (caddr_t *)realloc((MALLOC_P *)Nc,
				   (MALLOC_S)(Nn * sizeof(KA_T))))
			== NULL) {
				(void) fprintf(stderr,
					"%s: no more txt ptr space, PID %d\n",
					Pn, Lp->pid);
				exit(1);
			}
		}
		Nc[j++] = na;
	/*
	 * Save text node information.
	 */
		switch (p->p_type) {
			case PT_LIBTXT:
			case PT_LIBDAT:
				alloc_lfile(" ltx", -1);
				break;
			case PT_DATA:
			case PT_TEXT:
			default:
				alloc_lfile(" txt", -1);
		}
		process_node(na);
		if (Lf->sf)
			link_lfile();
	}
}
#else	/* V88>=40401 */


/*
 * process_text() - process V/88 R40V42 text access information
 */

static void
process_text(pa)
	struct as *pa;			/* address space description pointer */
{
	struct as as;
	int i, j, k, l;
	struct seg s;
	union segdata {
		struct segvn_data vn;
		struct segdev_data dev;
		struct segmap_data map;
	} sd;
	struct vnode *v[MAXSEGS];
/*
 * Get address space description.
 */
	if (kread((KA_T)pa, (char *)&as, sizeof(as))) {
		alloc_lfile(" txt", -1);
		(void) sprintf(Namech, "can't read text segment list (%#x)",
			pa);
		enter_nm(Namech);
		link_lfile();
		return;
	}
/*
 * Loop through the segments.  The loop should stop when the segment
 * pointer returns to its starting point, but just in case, it's stopped
 * when MAXSEGS have been recorded or 2*MAXSEGS have been examined.
 */
	s.s_next = as.a_segs;
	for (i = j = k = 0; i < MAXSEGS && j < 2*MAXSEGS; j++) {
		if (s.s_next == NULL
		||  kread((KA_T)s.s_next, (char *)&s, sizeof(s)))
			break;
		if ((u_long)s.s_ops == Nl[X_SEGVN].n_value
		&&  s.s_data != NULL) {
			if (kread((KA_T)s.s_data, (char *)&sd, sizeof(sd)))
				break;
			if (sd.vn.vp) {
			
			/*
			 * This is a virtual node segment.
			 *
			 * If its vnode pointer has not been seen already,
			 * print its information.
			 */
				for (l = 0; l < k; l++) {
					if (v[l] == sd.vn.vp)
						break;
				}
				if (l >= k) {
					alloc_lfile(" txt", -1);
					process_node((caddr_t)sd.vn.vp);
					if (Lf->sf) {
						link_lfile();
						i++;
					}
					v[k++] = sd.vn.vp;
				}
			}
		}
	/*
	 * Follow the segment link to the starting point in the address
	 * space description.  (The i and j counters place an absolute
	 * limit on the loop.)
	 */
		if (s.s_next == as.a_segs)
			break;
	}
}
#endif	/* V88<40401 */


/*
 * readfsinfo() - read file system information
 */

static void
readfsinfo()
{

#if	V88>=40401
	char buf[FSTYPSZ+1];
	int i;

	if ((Fsinfomax = sysfs(GETNFSTYP)) == -1) {
		(void) fprintf(stderr, "%s: sysfs(GETNFSTYP) error: %s\n",
			Pn, strerror(errno));
		exit(1);
	} 
	if (Fsinfomax == 0)
		return;
	if ((Fsinfo = (char **)malloc((MALLOC_S)(Fsinfomax * sizeof(char *))))
	== NULL) {
		(void) fprintf(stderr, "%s: no space for sysfs info\n", Pn);
		exit(1);
	}
	for (i = 1; i <= Fsinfomax; i++) {
		if (sysfs(GETFSTYP, i, buf) == -1) {
			(void) fprintf(stderr,
				"%s: sysfs(GETFSTYP) error: %s\n",
				Pn, strerror(errno));
			exit(1);
		}
		buf[FSTYPSZ] = '\0';
		if ((Fsinfo[i-1] = (char *)malloc((MALLOC_S)(strlen(buf) + 1)))
		== NULL) {
			(void) fprintf(stderr,
				"%s: no space for file system entry %s\n",
				Pn, buf);
			exit(1);
		}
		(void) strcpy(Fsinfo[i-1], buf);
	}

#else	/* V88<40401 */

	char *cp;
	int i;
	MALLOC_S len;
/*
 * Access the file system information table in the kernel.
 */
	if (Nl[X_NFSTYP].n_value == NULL
	||  kread((KA_T)Nl[X_NFSTYP].n_value, (char *)&Nfstyp, sizeof(Nfstyp))
	||  Nfstyp < 1) {
		(void) fprintf(stderr,
			"%s: nfstyp value (%d) error, kernel = %#x\n",
			Pn, Nfstyp, Nl[X_NFSTYP].n_value);
		exit(1);
	}
	len = (MALLOC_S)(Nfstyp * sizeof(struct fsinfo));
	if (Nl[X_FSINFO].n_value == NULL
	||  (Fsinfo = (struct fsinfo *)malloc(len)) == NULL
	||  kread((KA_T)Nl[X_FSINFO].n_value, (char *)Fsinfo, len)) {
		(void) fprintf(stderr,
			"%s: fstyp read error, length = %d, kernel = %#x\n",
			Pn, len, Nl[X_FSINFO].n_value);
		exit(1);
	}
/*
 * Get the file system names.
 */
	for (i = 0; i < Nfstyp; i++) {
	    if (Fsinfo[i].fs_name) {
		if ((cp = malloc(FSNAMEL)) == NULL) {
		    (void) fprintf(stderr,
			"%s: no space for file system name[%d]\n", Pn, i);
		    exit(1);
		}
		if (kread((KA_T)Fsinfo[i].fs_name, cp, FSNAMEL)) {
		    (void) fprintf(stderr,
			"%s: can't read file system name[i] from %#x\n",
			Pn, i, Fsinfo[i].fs_name);
		    exit(1);
		}
		cp[FSNAMEL - 1] = '\0';
		Fsinfo[i].fs_name = cp;
	    }
	}
#endif	/* V88>=40401 */

}


/*
 * The ncache_addr(), ncache_load(), and ncache_lookup() functions are
 * obtained from ../common/rnch.frag.
 */


