/*
 * dfile.c - Linux file processing functions for lsof
 */


/*
 * Copyright 1994 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1994 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dfile.c,v 1.3 95/06/30 08:45:55 abe Exp $";
#endif


#include "lsof.h"


/*
 * ck_file_arg() - check file arguments
 */

int
ck_file_arg(i, ac, av)
	int i;			/* first file argument index */
	int ac;			/* argument count */
	char *av[];		/* argument vector */
{
	unsigned char ad, an;
	short err = 0;
	char *fnm, *fsnm, *path;
	int ftype, j;
	struct mounts *mp;
	struct stat sb;
	struct sfile *sfp;

#if	defined(HASPROCFS)
	struct procfsid *pfi;
	short pfsnl = -1;
	pid_t pid;
	char *pr;
#endif

	for (; i < ac; i++) {
		if ((path = Readlink(av[i])) == NULL) {
			err = 1;
			continue;
		}
	/*
	 * Remove extra terminating `/'.
	 *
	 * Check for file system argument.
	 */
		if ((j = strlen(path)) > 1 && path[j-1] == '/')
			path[j-1] = '\0';
		for (ftype = 1, mp = Mtab; mp; mp = mp->next) {
			if (strcmp(mp->dir, path) == 0) {
				ftype = 0;
				fnm = path;
				fsnm = mp->fsname;
				break;
			}
			if (strcmp(mp->fsname, path) == 0) {
				ftype = 0;
				fnm = mp->dir;
				fsnm = path;
				break;
			}
		}
		if (ftype) {
			fnm = path;
			fsnm = NULL;
		} else {

#if	defined(HASPROCFS)
			if (mp == Mtprocfs) {
				Procsrch = 1;
				continue;
			}
#endif

			sb.st_dev = mp->dev;
			sb.st_rdev = mp->rdev;
			sb.st_ino = mp->inode;
			sb.st_mode = mp->mode;
		}
	/*
	 * Stat the argument to obtain its mode and device.
	 */
		if (ftype && statsafely(fnm, &sb) != 0) {
			(void) fprintf(stderr, "%s: status error on %s: %s\n",
				Pn, fnm, strerror(errno));
			err = 1;
			continue;
		}
	/*
	 * Allocate an sfile structure and fill in the type, inode,
	 * find-flag and linkages.
	 */
		if ((sfp = (struct sfile *)malloc(sizeof(struct sfile)))
		== NULL) {
			(void) fprintf(stderr, "%s: no space for files\n", Pn);
			exit(1);
		}
		sfp->next = Sfile;
		Sfile = sfp;
		sfp->type = ftype;
		sfp->i = sb.st_ino;
		sfp->f = 0;
	/*
	 * Store the file name and file system name pointers in the sfile
	 * structure, allocating space as necessary.
	 */
		if (fnm == NULL || fnm == path) {
			sfp->name = fnm;
			an = 0;
		} else {
		    if ((sfp->name = (char *)malloc((MALLOC_S)(strlen(fnm)+1)))
		    == NULL) {
			(void) fprintf(stderr,
			    "%s: no space for file name %s\n", Pn, fnm);
			exit(1);
		    }
		    (void) strcpy(sfp->name, fnm);
		    an = 1;
		}
		if (fsnm == NULL || fsnm == path) {
			sfp->devnm = fsnm;
			ad = 0;
		} else {
		    if ((sfp->devnm=(char *)malloc((MALLOC_S)(strlen(fsnm)+1)))
		    == NULL) {
			(void) fprintf(stderr,
			    "%s: no space for file system name %s\n", Pn, fsnm);
			exit(1);
		    }
		    (void) strcpy(sfp->devnm, fsnm);
		    ad = 1;
		}
		if ((sfp->aname = (char *)malloc((MALLOC_S)(strlen(av[i]) + 1)))
		== NULL) {
			(void) fprintf(stderr,
			    "%s: no space for argument file name %s\n",
			    Pn, av[i]);
			exit(1);
		}
		(void) strcpy(sfp->aname, av[i]);
	/*
	 * Save the stat() buffer mode value in the sfile structure.
	 * Use st_rdev if the mode value is S_IFBLK or S_IFCHR; otherwise
	 * use st_dev.
	 */
		sfp->mode = sb.st_mode & S_IFMT;
		if (sfp->mode == S_IFBLK || sfp->mode == S_IFCHR)
			sfp->dev = sb.st_rdev;
		else
			sfp->dev = sb.st_dev;

#if	defined(HASPROCFS)
	/*
	 * See if this is an individual member of a proc file system.
	 */
		if (Mtprocfs == NULL || Procsrch)
			continue;
		if (pfsnl == -1)
			pfsnl = strlen(Mtprocfs->dir);
		if (! pfsnl)
			continue;
		if (strncmp(Mtprocfs->dir, path, pfsnl) != 0)
			continue;
		if (path[pfsnl] != '/')
			continue;
		for (j = pfsnl+1; path[j]; j++) {
			if ( ! isdigit(path[j]))
				break;
		}
		if (path[j])
			continue;
		if ((pfi = (struct procfsid *)malloc((MALLOC_S)
			    sizeof(struct procfsid)))
		== NULL) {
			(void) fprintf(stderr,
				"%s: no space for %s ID: %s\n",
				Pn, Mtprocfs->dir,path);
			exit(1);
		}
		pfi->pid = atoi(&path[pfsnl+1]);
		pfi->next = Procfsid;
		Procfsid = pfi;
	/*
	 * Abandon the Sfile entry, lest it be used in is_file_named().
	 */
		Sfile = sfp->next;
		(void) free((FREE_P *)sfp->aname);
		if (ad)
			(void) free((FREE_P *)sfp->devnm);
		if (an)
			(void) free((FREE_P *)sfp->name);
		(void) free((FREE_P *)sfp);
#endif	/* HASPROCFS */

	}
	return((int)err);
}


/*
 * is_file_named() - is this file named?
 */

int
is_file_named(p, ty)
	char *p;			/* path name; NULL = search by device
					 * and inode (from *Lf) */
	umode_t ty;			/* inode mode */
{
	int f;
	struct sfile *s;
/*
 * Search the file chain.
 */
	for (f = 0, s = Sfile; s; s = s->next) {

	/*
	 * Search by path name, as requested.
	 */
		if (p != NULL) {
			if (strcmp(p, s->aname) == 0
			||  strcmp(p, s->name)  == 0)
			{
				f = 2;
				break;
			}
			continue;
		}
	/*
	 * Check for a regular file or directory -- the device and
	 * indoe numbers must match.
	 */
		if (s->type) {
			if (Lf->dev == s->dev && (ino_t)Lf->inode == s->i) {
				f = 1;
				break;
			}
			continue;
		}
	/*
	 * Check for a file system match.
	 * Try to avoid matching character files to non-character devices.
	 */
		if (Lf->dev == s->dev) {
			if ( ! (ty == S_IFCHR && s->mode != S_IFCHR)) {
				f = 1;
				break;
			}
		}
	}
/*
 * Convert the name if a match occurred.
 */
	if (f) {
		if (f == 2)
			(void) strcpy(Namech, p);
		else {
			(void) strcpy(Namech, s->name);
			if (s->devnm)
				(void) sprintf(endnm(), " (%s)", s->devnm);
		}
		s->f = 1;
		return(1);
	}
	return(0);
}


/*
 * process_file() - process file
 */

void
process_file(fp)
	struct file *fp;		/* kernel file structure address */
{
	int flag;

	Lf->off = fp->f_pos;
	if (fp->f_count) {

	/*
	 * Construct access code.
	 */
		if ((flag = (fp->f_mode & 3)) == 1)
			Lf->access = 'r';
		else if (flag == 2)
			Lf->access = 'w';
		else if (flag == 3)
			Lf->access = 'u';
	/*
	 * Process inode.
	 */
		process_inode(fp->f_inode, fp->f_op);
	}
}


/*
 * print_file() - print file
 */

void
print_file()
{
	char buf[16];

	if (Hdr == 0) {
	    (void) printf("COMMAND     PID%s     USER   FD   TYPE     DEVICE ",
		Fpgrp ? "   PGRP" : "");
	    (void) printf("%10s", Foffset ? "OFFSET"
					  : Fsize ? "SIZE"
						  : "SIZE/OFF");
	    (void) puts("     INODE NAME");
	    Hdr++;
	}
	(void) printf("%-9.9s%6d", Lp->cmd, Lp->pid);
	if (Fpgrp)
	    printf(" %6d", Lp->pgrp);
	(void) printf(" %8.8s %4s%c%c %4.4s ",
			printuid((UID_ARG)Lp->uid, NULL),
			Lf->fd,
			Lf->access,
			Lf->lock,
			Lf->type
		);
	if (Lf->dev_def)
		(void) printf(" %4d,%4d ", major(Lf->dev), minor(Lf->dev));
	else {
		if (Lf->dev_ch) {
			(void) fputs(Lf->dev_ch, stdout);
			putchar(' ');
		} else
			(void) fputs("           ", stdout);
	}
	if (Lf->sz_def)
		(void) printf("%10u", Lf->sz);
	else if (Lf->off_def) {
		if ((unsigned int)Lf->off < 100000000) {
			(void) sprintf(buf, "0t%d", Lf->off);
			(void) printf("%10.10s", buf);
		} else
			(void) printf("%#10x", (Lf->off & 0xffffffff));
	} else
		(void) fputs("          ", stdout);
	switch (Lf->inp_ty) {
	case 1:
		if (Lf->inode < 1000000000l)
			(void) printf(" %9lu ", Lf->inode);
		else
			(void) printf(" *%08lu ", Lf->inode % 100000000l);
		break;
	case 2:
		if (Lf->iproto[0] == '\0')
			(void) fputs("           ", stdout);
		else
			(void) printf(" %9.9s ", Lf->iproto);
		break;
	default:
		(void) fputs("           ", stdout);
	}
	printname(1);
}
