/*
 * dproc.c - DYNIX process access functions for lsof
 */


/*
 * Copyright 1994 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1994 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dproc.c,v 1.10 95/09/19 10:12:51 abe Exp $";
#endif

#include "lsof.h"

#if     defined(HASNCACHE)
#include <sys/dnlc.h>
#endif  /* defined(HASNCACHE) */


/*
 * Local static values
 */

static long Kp;				/* kernel's process table address */
static int Np = 0;			/* number of processes */


_PROTOTYPE(static void get_kernel_access,(void));

#if     defined(HASNCACHE)
_PROTOTYPE(static void ncache_load,(void));
#endif  /* defined(HASNCACHE) */

_PROTOTYPE(static off_t vtophys,(off_t vaddr));


/*
 * gather_proc_info() -- gather process information
 */

void
gather_proc_info()
{
	static ds = 0;
	int i, nf;
	static struct ofile *ofp = NULL;
	struct proc *p;
	KA_T pa;
	struct proc ps;
	short pss, sf;
	int px;
	off_t sp;
	static struct user *u = NULL;
	struct ofile *uf;
	MALLOC_S ui;
	static int xnf = 0;

#if	defined(DYNIX31)
	struct ofile_tab oft;
#else
	struct ofile_ext ofx;
#endif

#if     defined(HASNCACHE)
/*
 * Read kernel name cache.
 */
        ncache_load();
#endif  /* defined(HASNCACHE) */

/*
 * Allocate space for the user structure.
 */
	if (!u) {
		if ((u = (struct user *)valloc(ctob(UPAGES))) == NULL) {
			(void) fprintf(stderr,
				"%s: can't allocate space for user struct\n",
				Pn);
			exit(1);
		}
	}
/*
 * Examine proc structures and their associated information.
 */
	if (!ds)
		ds = getdtablesize();
	for (p = &ps, px = 0; px < Np; px++) {

	/*
	 * Read the proc structure.
	 */
		pa = (KA_T)(Kp + (long)(px * sizeof(struct proc)));
		if (kread(pa, (char *)&ps, sizeof(ps)))
			continue;
		if (p->p_stat == 0 || p->p_stat == SZOMB)
			continue;
		if (is_proc_excl(p->p_pid, (int)p->p_pgrp, (UID_ARG)p->p_uid,
		    &pss, &sf))
			continue;
	/*
	 * Read the user area.
	 */
		if ((p->p_flag & SLOAD) == 0) {
			if (Swap < 0)
				continue;
			sp = (off_t) dtob(p->p_swaddr);
			if (lseek(Swap, sp, L_SET) != sp
			||  read(Swap, (char*)u, ctob(UPAGES)) != ctob(UPAGES))
				continue;
		} else {
			if (kread((KA_T)p->p_uarea, (char *)u, U_SIZE))
				continue;
		}
	/*
	 * Allocate a local process structure.
	 */
		if (is_cmd_excl(u->u_comm, &pss, &sf))
			continue;
		alloc_lproc(p->p_pid, (int)p->p_pgrp, (UID_ARG)p->p_uid,
			u->u_comm, (int)pss, (int)sf);
		Plf = NULL;
	/*
	 * Save current working directory information.
	 */
		if (u->u_cdir) {
			alloc_lfile(CWD, -1);
			process_node((caddr_t)u->u_cdir);
			if (Lf->sf)
				link_lfile();
		}
	/*
	 * Save root directory information.
	 */
		if (u->u_rdir) {
			alloc_lfile(RTD, -1);
			process_node((caddr_t)u->u_rdir);
			if (Lf->sf)
				link_lfile();
		}
	/*
	 * Read the open file structures.
	 */

#if	defined(DYNIX31)
	/*
	 * Under DYNIX 3.1, the file pointers should be located in an extension
	 * to the user structure.  Determine if is the case and read the
	 * pointers.
	 */
		if (u->u_ofile_tab) {
		    if (kread((KA_T)u->u_ofile_tab, (char *)&oft, sizeof(oft)))
			nf = 0;
		    else {
			nf = oft.oft_nofile;
			ui = (MALLOC_S)(oft.oft_nofile * sizeof(struct ofile));
			if (ofp == NULL) {
			    xnf = nf;
			    if ((ofp = (struct ofile *)malloc(ui))
			    == NULL) {
				(void) fprintf(stderr,
				    "%s: PID %d, no malloc ofile space\n",
				    Pn, p->p_pid);
				exit(1);
			    }
			} else if (nf > xnf) {
			    xnf = nf;
			    if ((ofp = (struct ofile *)realloc(
					(MALLOC_P *)ofp, ui))
			    == NULL) {
				(void) fprintf(stderr,
				    "%s: PID %d, no realloc ofile space\n",
				    Pn, p->p_pid);
				exit(1);
			    }
			}
			if (kread((KA_T)oft.oft_ofile, (char *)ofp, (int)ui))
			    nf = 0;
			else {
			    nf = oft.oft_nofile;
			    uf = ofp;
			}
		    }
		} else {
		    nf = ds;
		    uf = u->u_lofileXXX;
		}
#else	/* DYNIX31 */

	/*
	 * Under DYNIX, the file pointers may be located in the user
	 * structure or in an extension to it.  Determine which is the
	 * case and read the pointers.
	 */

		if (u->u_ofile_ext) {
		    if (kread((KA_T)u->u_ofile_ext, (char *)&ofx, sizeof(ofx)))
			nf = 0;
		    else {
			nf = ofx.oe_nofile;
			ui = (MALLOC_S)(ofx.oe_nofile * sizeof(struct ofile));
			if (ofp == NULL) {
			    xnf = nf;
			    if ((ofp = (struct ofile *)malloc(ui))
			    == NULL) {
				(void) fprintf(stderr,
				    "%s: PID %d, no malloc ofile space\n",
				    Pn, p->p_pid);
				exit(1);
			    }
			} else if (nf > xnf) {
			    xnf = nf;
			    if ((ofp = (struct ofile *)realloc(
					(MALLOC_P *)ofp, ui))
			    == NULL) {
				(void) fprintf(stderr,
				    "%s: PID %d, no realloc ofile space\n",
				    Pn, p->p_pid);
				exit(1);
			    }
			}
			if (kread((KA_T)ofx.oe_ofile, (char *)ofp, (int)ui))
			    nf = 0;
			else {
			    nf = ofx.oe_nofile;
			    uf = ofp;
			}
		    }
		} else {
			nf = ds;
			uf = u->u_lofile;
		}
# endif	/* ! DYNIX31 */

	/*
	 * Save information on file descriptors.
	 */
		for (i = 0; i < nf; i++) {
			if (uf[i].of_file) {
				alloc_lfile(NULL, i);
				process_file(uf[i].of_file);
				if (Lf->sf)
					link_lfile();
			}
		}
	/*
	 * Examine results.
	 */
		if (examine_lproc())
			return;
	}
}


/*
 * get_kernel_access() - access the required information in the kernel
 */

static void
get_kernel_access()
{

/*
 * Open swap access.
 */
	if (Memory == NULL || strcmp(Memory, KMEM) == 0) {
		if ((Swap = open(SWAP, O_RDONLY, 0)) < 0) {
			(void) fprintf(stderr, "%s: %s: %s\n",
				Pn, SWAP, strerror(errno));
			exit(1);
		}
	}

#if	defined(WILLDROPGID)
/*
 * If kernel memory isn't coming from KMEM, drop setgid permission
 * before attempting to open the (Memory) file.
 */
	if (Memory)
		(void) dropgid();
#else	/* !defined(WILLDROPGID) */
/*
 * See if the non-KMEM memory file is readable.
 */
	if (Memory && !is_readable(Memory, 1))
		exit(1);
#endif	/* defined(WILLDROPGID) */

/*
 * Open kernel memory access.
 */
	if ((Kmem = open(Memory ? Memory : KMEM, O_RDONLY, 0)) < 0) {
		(void) fprintf(stderr, "%s: can't open %s: %s\n", Pn,
			Memory ? Memory : KMEM, strerror(errno));
		exit(1);
	}

#if	defined(WILLDROPGID)
/*
 * Drop setgid permission, if necessary.
 */
	if (!Memory)
		(void) dropgid();
#else	/* !defined(WILLDROPGID) */
/*
 * See if the name list file is readable.
 */
	if (Nmlst && !is_readable(Nmlst, 1))
		exit(1);
#endif	/* defined(WILLDROPGID) */

/*
 * Access kernel symbols.
 */
	if (nlist(Nmlst ? Nmlst : N_UNIX, Nl) < 0) {
		(void) fprintf(stderr, "%s: can't read namelist from %s\n",
			Pn, Nmlst ? Nmlst : N_UNIX);
		exit(1);
	}
	if (Nl[X_PROC].n_value == NULL
	||  kread((KA_T)Nl[X_PROC].n_value, (char *)&Kp, sizeof(Kp))
	||  Nl[X_NPROC].n_value == NULL
	||  kread((KA_T)Nl[X_NPROC].n_value, (char *)&Np, sizeof(Np))
	||  Kp == NULL || Np == 0) {
		(void) fprintf(stderr, "%s: can't read proc table info\n",
		Pn);
		exit(1);
	}
/*
 * If the Memory file isn't /dev/kmem, prepare for kernel virtual
 * to physical address mapping in the core file.
 */
	if (Memory != NULL && strcmp(Memory, KMEM) != 0) {
		if (Nl[X_SYSMAP].n_value == NULL
		||  kread((KA_T)Nl[X_SYSMAP].n_value, (char *)&Sysmap,
			sizeof(Sysmap))
		||  Sysmap == NULL)
		{
		    (void) fprintf(stderr,
			"%s: no _Sysmap for virtual to physical mapping\n", Pn);
		    exit(1);
		}
		Notkmem = 1;
	}
}


/*
 * initialize() - perform all initialization
 */

void
initialize()
{
	get_kernel_access();
	(void) iuidcache(Np);
}


/*
 * kread() - read from kernel memory
 */

int
kread(addr, buf, len)
	KA_T addr;			/* kernel memory address */
	char *buf;			/* buffer to receive data */
	READLEN_T len;			/* length to read */
{
	int br;

	if (Notkmem) {
		if ((addr = vtophys(addr)) == (off_t)-1)
			return(-1);
	}
	if (lseek(Kmem, addr, L_SET) == (off_t)-1L)
		return(-1);
	br = read(Kmem, buf, len);
	return((br == len) ? 0 : 1);
}


/*
 * vtophys() - map kernel virtual address to physical address
 */

static off_t
vtophys(vaddr)
	off_t vaddr;
{
	off_t paddr;
	
# if	defined(i386)
	if (vaddr < (off_t)8192)
		return(vaddr);
# endif

	paddr = (off_t)((int *)Sysmap + (vaddr / NBPG));
	if (lseek(Kmem, (off_t)paddr, L_SET) == (off_t)-1)
		return((off_t)-1);
	if (read(Kmem, (char *)&paddr, sizeof(paddr)) != sizeof(paddr))
		return((off_t)-1);
	return ((paddr & ~PTBITS) | (vaddr & PTBITS));
}


/*
 * The ncache_addr(), ncache_load(), and ncache_lookup() functions are
 * obtained from ../common/rnch.frag.
 */

#if     defined(HASNCACHE)
#define ADDR_NCACHE     1
#define FIXED_NCSIZE    NC_SIZE
#endif  /* defined(HASNCACHE) */


