/*
 * dnode.c - DYNIX node functions for lsof
 */


/*
 * Copyright 1994 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1994 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dnode.c,v 1.8 95/06/26 09:29:05 abe Exp $";
#endif


#include "lsof.h"


/*
 * process_node() - process vnode
 */

void
process_node(va)
	caddr_t va;			/* vnode kernel space address */
{
	dev_t dev;
	struct inode i;
	struct rnode r;
	char *ty;
	enum vtype type;
	struct vnode v;
	struct l_vfs *vfs;
/*
 * Read the vnode.
 */
	if ( ! va) {
		enter_nm("no vnode address");
		return;
	}
	if (readvnode((caddr_t)va, &v)) {
                enter_nm(Namech);
                return;
        }

# if	defined(HASNCACHE)
	Lf->na = (unsigned long)va;
# endif	/* defined(HASNCACHE) */

/*
 * Determine the vnode type.
 */
	if (Nl[X_NFS_OPS].n_value
	&& (unsigned long)v.v_op == Nl[X_NFS_OPS].n_value)
		Ntype = N_NFS;
	else if (v.v_type == VFIFO)
		Ntype = N_FIFO;
/*
 * Determine the lock type.
 */
	if (v.v_shlockc || v.v_exlockc) {
		if (v.v_shlockc && v.v_exlockc)
			Lf->lock = 'u';
		else if (v.v_shlockc)
			Lf->lock = 'R';
		else
			Lf->lock = 'W';
	}
/*
 * Establish the local virtual file system structure.
 */
	if ((vfs = readvfs(v.v_vfsp)) == NULL) {
                (void) sprintf(Namech, "can't read vfs for %#x at %#x", va,
			v.v_vfsp);
                enter_nm(Namech);
		return;
	}
/*
 * Read the inode or rnode.
 */
	switch (Ntype) {
	case N_NFS:
		if (!v.v_data || readrnode((caddr_t)v.v_data, &r)) {
			(void) sprintf(Namech,
				"vnode at %#x: can't read rnode (%#x)",
				va, v.v_data);
			enter_nm(Namech);
			return;
		}
		break;
	case N_REGLR:
	default:
		if (!v.v_data || readinode((struct inode *)v.v_data, &i)) {
			(void) sprintf(Namech,
				"vnode at %#x: can't read inode (%#x)",
				va, v.v_data);
			enter_nm(Namech);
			return;
		}
	}
/*
 * Get device and type for printing.
 */
	switch(Ntype) {
	case N_NFS:

	/*
	 * Neither the rnode nor the vnode under DYNIX 3.0.12 contains the
	 * correct device number, so it's reclaimed (if possible) from the
	 * local, virtual file system table entry (put there by completevfs()
	 * from information placed in Mtab by readmnt()).
	 */
		dev = vfs ? vfs->dev : 0;
		break;
	default:
		dev = (v.v_type == VCHR) ? v.v_rdev : i.i_dev;
	}
	type = v.v_type;
	if (vfs && vfs->dir == NULL)
		(void) completevfs(vfs, &dev);
/*
 * Obtain the inode number.
 */
	switch (Ntype) {
	case N_NFS:
		Lf->inode = (unsigned long)r.r_nfsattr.na_nodeid;
		Lf->inp_ty = 1;
		break;
	case N_FIFO:
	case N_REGLR:
		Lf->inode = (unsigned long)i.i_number;
		Lf->inp_ty = 1;
	}
/*
 * Obtain the file size.
 */
	if (Foffset)
		Lf->off_def = 1;
	else {
		switch (Ntype) {
		case N_NFS:
			Lf->sz = r.r_nfsattr.na_size;
			Lf->sz_def = 1;
			break;
		case N_REGLR:
			if (type == VREG || type == VDIR) {
				Lf->sz = i.i_size;
				Lf->sz_def = 1;
			} else if (type == VCHR && !Fsize)
				Lf->off_def = 1;
			break;
		}
	}
/*
 * Record an NFS file selection.
 */
	if (Ntype == N_NFS && Fnfs)
		Lf->sf |= SELNFS;
/*
 * Save the file system names.
 */
	if (vfs) {
		Lf->fsdir = vfs->dir;
		Lf->fsdev = vfs->fsname;
	}
/*
 * Format the vnode type, and possibly the device name.
 */
	switch (type) {

	case VNON:
		ty ="VNON";
		break;
	case VREG:
	case VDIR:
		ty = (type == VREG) ? "VREG" : "VDIR";
		Lf->dev_def = 1;
		Lf->dev = dev;
		break;
	case VBLK:
		ty = "VBLK";
		break;
	case VCHR:
		Lf->dev = dev;
		Lf->dev_def = Lf->is_chr_dev = 1;
		ty = "VCHR";
		break;
	case VLNK:
		ty = "VLNK";
		break;

#if	defined(VSOCK)
	case VSOCK:
		ty = "SOCK";
		break;
#endif

	case VBAD:
		ty = "VBAD";
		break;
	case VFIFO:
		Lf->dev = dev;
		Lf->dev_def = 1;
		ty = "FIFO";
		break;
	default:
		if (type > 9999)
			(void) sprintf(Lf->type, "*%03d", type % 1000);
		else
			(void) sprintf(Lf->type, "%4d", type);
		(void) strcpy(Namech, "unknown type");
		ty = NULL;
	}
	if (ty)
		(void) strcpy(Lf->type, ty);
/*
 * If this is a VCHR file and it's missing an inode number, try to
 * supply one.
 */
	if (Lf->inp_ty == 0 && type == VCHR && Lf->dev_def)
		find_ch_ino();
/*
 * Test for specified file.
 */
	if (Sfile && is_file_named(NULL, type))
		Lf->sf |= SELNM;
/*
 * Enter name characters.
 */
	if (Namech[0])
		enter_nm(Namech);
}
