/*
 * dnode.c - Novell UnixWare node functions for lsof
 */


/*
 * Copyright 1994 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1994 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dnode.c,v 1.7 95/06/28 13:25:36 abe Exp $";
#endif


#include "lsof.h"
#include <sys/fs/namenode.h>
#include <sys/fs/sfs_inode.h>
#include <sys/fs/xnamnode.h>


_PROTOTYPE(static int examine_stream,(struct stdata *vs, struct queue *q, char *mn));
_PROTOTYPE(static char isvlocked,(struct vnode *va));
_PROTOTYPE(static int readlino,(int fx, struct vnode *v, struct l_ino *i));


static struct protos {
	char *module;			/* stream module name */
	char *proto;			/* TCP/IP protocol name */
} Protos[] = {
	{ "tcp",	"TCP"	},
	{ "udp",	"UDP"	},
	{ "ip",		"IP"	},
	{ "icmp",	"ICMP"	}
};
#define	NPROTOS	(sizeof(Protos)/sizeof(struct protos))


/*
 * examine_stream() - examine stream
 */

static int
examine_stream(vs, q, mn)
	struct stdata *vs;
	struct queue *q;
	char *mn;
{
	struct module_info mi;
	struct queue *qp;
	struct qinit qi;
	struct stdata sd;
/*
 * Read stream's head.
 */
	if (vs == NULL || readstdata(vs, &sd)) {
		(void) sprintf(Namech, "can't read stream head from %x", vs);
		enter_nm(Namech);
		return(1);
	}
	if (sd.sd_wrq == NULL) {
		enter_nm("no stream write queue");
		return(1);
	}
/*
 * Examine the write queue.
 */
	for (qp = sd.sd_wrq, *mn = '\0'; qp; qp = q->q_next)
	{

	/*
	 * Read stream queue entry.
	 */
		if (kread((KA_T)qp, (char *)q, sizeof(struct queue))) {
			(void) sprintf(Namech,
				"can't read stream queue from %x", qp);
			enter_nm(Namech);
			return(1);
		}
	/*
	 * Read queue's information structure.
	 */
		if (q->q_qinfo == NULL || readstqinit(q->q_qinfo, &qi)) {
			(void) sprintf(Namech,
				"can't read qinit for %x from %x",
				qp, q->q_qinfo);
			enter_nm(Namech);
			return(1);
		}
	/*
	 * Read module information structure.
	 */
		if (qi.qi_minfo == NULL || readstmin(qi.qi_minfo, &mi)) {
			(void) sprintf(Namech,
				"can't read module info for %x from %x",
				q->q_qinfo, qi.qi_minfo);
			enter_nm(Namech);
			return(1);
		}
	/*
	 * Read module name.
	 */
		if (mi.mi_idname == NULL ||
		    kread((KA_T)mi.mi_idname, mn, STRNML-1))
		{
			(void) sprintf(Namech,
				"can't read module name for %x from %x",
				qi.qi_minfo, mi.mi_idname);
			enter_nm(Namech);
			return(1);
		}
		*(mn + STRNML - 1) = '\0';
	}
	return(0);
}


/*
 * isvlocked() - is a vnode locked
 */

static char
isvlocked(va)
	struct vnode *va;		/* local vnode address */
{
	struct filock f;
	off_t l;

	if (va->v_filocks == NULL)
		return(' ');
	if (kread((KA_T)va->v_filocks, (char *)&f, sizeof(f)))
		return(' ');
	if (f.set.l_whence == 0 && f.set.l_start == 0 && f.set.l_len == 0)
		l = 1;
	else
		l = 0;
	switch (f.set.l_type & (F_RDLCK | F_WRLCK)) {

	case F_RDLCK:
		return((l) ? 'R' : 'r');
	case F_WRLCK:
		return((l) ? 'W' : 'w');
	case (F_RDLCK + F_WRLCK):
		return('u');
	}
	return(' ');
}


/*
 * process_node() - process node
 */

void
process_node(na)
	caddr_t na;			/* vnode kernel space address */
{
	dev_t dev;
	char dev_ch[32];
	struct fifonode f;
	int fx;
	struct l_ino i;
	struct vfs kv;
	struct vfs *la = NULL;
	struct module_info mi;
	char mn[STRNML];
	int ni = 0;
	int px;
	struct queue q;
	struct rnode r;
	struct snode rs;
	struct vnode rv, v;
	struct snode s;
	unsigned char sd = 1;
	char *ty;
	enum vtype type;
	struct l_vfs *vfs;

#if	defined(HASPROCFS)
	struct as as;
	struct proc p;
	struct procfsid *pfi;
	struct pid pids;
	struct prnode pr;
#endif	/* HASPROCFS */
/*
 * Read the vnode.
 */
	if ( ! na) {
		enter_nm("no vnode address");
		return;
	}
	if (readvnode((caddr_t)na, &v)) {
                enter_nm(Namech);
                return;
        }
/*
 * Determine the vnode type.
 */
	if (v.v_vfsp
	&&  kread((KA_T)v.v_vfsp, (char *)&kv, sizeof(kv)) == 0) {

	/*
	* Check the file system type.
	*/
		fx = kv.vfs_fstype;
		if (fx > 0 && fx <= Fsinfomax) {
			if (strcmp(Fsinfo[fx-1], "fifofs") == 0)
				Ntype = N_FIFO;
			else if (strcmp(Fsinfo[fx-1], "nfs") == 0)
				Ntype = N_NFS;

#if	defined(HASPROCFS)
			else if (strcmp(Fsinfo[fx-1], "proc") == 0)
				Ntype = N_PROC;
#endif	/* HASPROCFS */

		}
	} else
		fx = Fsinfomax;
	if (Ntype == N_REGLR) {
		if (v.v_type == VFIFO)
			Ntype = N_FIFO;
		else if (v.v_stream) {
			Ntype = N_STREAM;
			Lf->is_stream = 1;
		}
	}
/*
 * Determine the lock state.
 */
	Lf->lock = isvlocked(&v);
/*
 * Establish the local virtual file system structure.
 */
	if (v.v_vfsp == NULL)
		vfs = NULL;
	else {
	    if ((vfs = readvfs(v.v_vfsp, &kv)) == NULL) {
		(void) sprintf(Namech, "bad vfs for %#x at %#x", na, v.v_vfsp);
		enter_nm(Namech);
		return;
	    }
	}
/*
 * Read the fifonode, inode, pipenode, prnode, rnode, or snode.
 */
	switch (Ntype) {
	case N_NFS:
		if (!v.v_data || readrnode((caddr_t)v.v_data, &r)) {
			(void) sprintf(Namech,
				"vnode at %#x: can't read rnode (%#x)",
				na, v.v_data);
			enter_nm(Namech);
			return;
		}
		break;
	case N_FIFO:
		if (!v.v_data || readfifonode(v.v_data, &f)) {
			(void) sprintf(Namech,
				"vnode at %#x: can't read fifonode (%#x)",
				na, v.v_data);
			enter_nm(Namech);
			return;
		}
		if (f.fn_realvp) {
			if (readvnode((caddr_t)f.fn_realvp, &rv)) {
			    (void) sprintf(Namech,
				"fifonode at %#x: can't read real vnode (%#x)",
				v.v_data, f.fn_realvp);
			    enter_nm(Namech);
			    return;
			}
			if (!rv.v_data
			||  readlino(fx, &rv, &i)) {
				(void) sprintf(Namech,
				    "fifonode at %#x: can't read inode (%#x)",
				    v.v_data, rv.v_data);
				enter_nm(Namech);
				return;
			}
		}
		break;

#if	defined(HASPROCFS)
	case N_PROC:
		ni = 1;
		if (v.v_data == NULL
		||  kread((KA_T)v.v_data, (char *)&pr, sizeof(pr))) {
			(void) sprintf(Namech,
				"vnode at %#x: can't read prnode (%#x)",
				na, v.v_data);
			enter_nm(Namech);
			return;
		}
		if (pr.pr_proc == NULL) {
			sd = 0;
			if (v.v_type == VDIR) {
				(void) sprintf(Namech, "/%s", HASPROCFS);
				enter_nm(Namech);
				i.number = PR_ROOTINO;
			} else {
				(void) sprintf(Namech, "/%s/???", HASPROCFS);
				enter_nm(Namech);
				i.number = -1L;
			}
			break;
		}
		if (kread((KA_T)pr.pr_proc, (char *)&p, sizeof(p))) {
			(void) sprintf(Namech,
				"prnode at %#x: can't read proc (%#x)",
				v.v_data, pr.pr_proc);
			enter_nm(Namech);
			return;
		}
		if (p.p_pidp == NULL
		||  kread((KA_T)p.p_pidp, (char *)&pids, sizeof(pids))) {
			(void) sprintf(Namech,
				"proc struct at %#x: can't read pid (%#x)",
				pr.pr_proc, p.p_pidp);
			enter_nm(Namech);
			return;
		}
		if (p.p_as == NULL
		||  kread((KA_T)p.p_as, (char *)&as, sizeof(as)))
			sd = 0;
		else
			i.size = as.a_size;
		(void) sprintf(Namech, "/%s/%0*d", HASPROCFS, PNSIZ,
			pids.pid_id);
		i.number = (long)pids.pid_id + PR_INOBIAS;
		enter_nm(Namech);
		break;
#endif	/* HASPROCFS */

	case N_STREAM:
		(void) strcpy(Lf->iproto, "STR");
		if (v.v_stream) {

		/*
		 * Get the queue pointer and module name at the
		 * end of the stream.  The module name indentifies
		 * socket streams.
		 */
			if (examine_stream(v.v_stream, &q, mn))
				return;
			for (px = 0; px < NPROTOS; px++) {
				if (strcmp(mn, Protos[px].module) == 0) {
					process_socket(Protos[px].proto, &q);
					return;
				}
			}
			if (mn[0])
				(void) strcpy(Namech, mn);
			else
				(void) strcpy(Namech, "(no module name)");
		}
		break;
	case N_REGLR:
	default:

	/*
	 * Follow a VCHR vnode to its snode, then to its real vnode, finally
	 * to its inode.
	 */
		if (v.v_type == VCHR) {
			if (!v.v_data || readsnode(v.v_data, &s)) {
				(void) sprintf(Namech,
					"vnode at %#x: can't read snode(%#x)",
					na, v.v_data);
				enter_nm(Namech);
				return;
			}
			if (s.s_realvp) {
			    if (readvnode((caddr_t)s.s_realvp, &rv)) {
				(void) sprintf(Namech,
				    "snode at %#x: can't read real vnode (%#x)",
				    v.v_data, s.s_realvp);
				enter_nm(Namech);
				return;
			    }
			    if (!rv.v_data
			    ||  readlino(fx, &rv, &i)) {
				(void) sprintf(Namech,
					"snode at %#x: can't read inode (%#x)",
					v.v_data, rv.v_data);
				enter_nm(Namech);
				return;
			    }
			}
		/*
		 * If there's no real vnode, look for a common vnode and a
		 * common snode.
		 */
			else if (s.s_commonvp) {
			    if (readvnode((caddr_t)s.s_commonvp, &rv)) {
				(void) sprintf(Namech,
				    "snode at %#x: can't read real vnode (%#x)",
				    v.v_data, s.s_commonvp);
				enter_nm(Namech);
				return;
			    }
			    if (!rv.v_data || readsnode(rv.v_data, &rs)) {
				(void) sprintf(Namech,
					"vnode at %#x: can't read snode(%#x)",
					s.s_commonvp, rv.v_data);
				enter_nm(Namech);
				return;
			    }
			    ni = 1;
			} else
				ni = 1;
			break;
		}
		if (v.v_data == NULL) {
			(void) sprintf(Namech,
				"vnode at %#x: no further information", na);
			enter_nm(Namech);
			return;
		}
	/*
	 * Read inode information.
	 */
		if (readlino(fx, &v, &i)) {
			(void) sprintf(Namech,
				"snode at %#x: can't read inode (%#x)",
				v.v_data, rv.v_data);
			enter_nm(Namech);
			return;
		}
	}
/*
 * Get device and type for printing.
 */
	switch (Ntype) {
	case N_NFS:

#if	defined(HASPROCFS)
	case N_PROC:
		dev = kv.vfs_dev;
		break;
#endif	/* HASPROCFS */

	default:
		dev = (v.v_type == VCHR || ni) ? v.v_rdev : i.dev;
	}
	type = v.v_type;
	if (vfs && vfs->dir == NULL)
		(void) completevfs(vfs, &dev);
/*
 * Obtain the inode number.
 */
	switch (Ntype) {
	case N_NFS:
		Lf->inode = (unsigned long)r.r_nfsattr.na_nodeid;
		Lf->inp_ty = 1;
		break;

#if	defined(HASPROCFS)
	case N_PROC:
		Lf->inode = (unsigned long)i.number;
		Lf->inp_ty = 1;
		break;
#endif	/* HASPROCFS */

	case N_FIFO:
		if ( ! f.fn_realvp) {
			Lf->inode = (unsigned long)f.fn_ino;
			Lf->inp_ty = 1;
			(void) sprintf(dev_ch, "0x%08x", v.v_data);
			enter_dev_ch(dev_ch);
			if (f.fn_flag & ISPIPE)
				(void) strcpy(Namech, "PIPE");
			if (f.fn_mate)
				(void) sprintf(endnm(), "->0x%08x", f.fn_mate);
			break;
		}
		/* fall through */
	case N_REGLR:
		if (!ni) {
			Lf->inode = (unsigned long)i.number;
			Lf->inp_ty = 1;
		}
		break;
	}
/*
 * Obtain the file size.
 */
	if (Foffset)
		Lf->off_def = 1;
	else {
		switch (Ntype) {
		case N_FIFO:
		case N_STREAM:
			if (!Fsize)
				Lf->off_def = 1;
			break;
		case N_NFS:
			Lf->sz = r.r_nfsattr.na_size;
			Lf->sz_def = sd;
			break;

#if	defined(HASPROCFS)
		case N_PROC:
			Lf->sz = i.size;
			Lf->sz_def = sd;
			break;
#endif	/* HASPROCFS */

		case N_REGLR:
			if (type == VREG || type == VDIR) {
				if (!ni) {
					Lf->sz = i.size;
					Lf->sz_def = sd;
				}
			} else if (type == VCHR && !Fsize)
				Lf->off_def = 1;
			break;
		}
	}
/*
 * Record an NFS file selection.
 */
	if (Ntype == N_NFS && Fnfs)
		Lf->sf |= SELNFS;
/*
 * Save the file system names.
 */
	if (vfs) {
		Lf->fsdir = vfs->dir;
		Lf->fsdev = vfs->fsname;
	}
/*
 * Format the vnode type, and possibly the device name.
 */
	switch (type) {
	case VNON:
		ty ="VNON";
		break;
	case VREG:
	case VDIR:
		ty = (type == VREG) ? "VREG" : "VDIR";
		Lf->dev_def = 1;
		Lf->dev = dev;
		break;
	case VBLK:
		ty = "VBLK";
		break;
	case VCHR:
		Lf->dev = dev;
		Lf->dev_def = 1;
		ty = "VCHR";
		if (Lf->is_stream == 0)
			Lf->is_chr_dev = 1;
		break;
	case VLNK:
		ty = "VLNK";
		break;

#if	defined(VSOCK)
	case VSOCK:
		ty = "SOCK";
		break;
#endif	/* VSOCK */

	case VBAD:
		ty = "VBAD";
		break;
	case VFIFO:
		if (!Lf->dev_ch || Lf->dev_ch[0] == '\0') {
			Lf->dev = dev;
			Lf->dev_def = 1;
		}
		ty = "FIFO";
		break;
	default:
		if (type > 9999)
			(void) sprintf(Lf->type, "*%03d", type % 1000);
		else
			(void) sprintf(Lf->type, "%4d", type);
		(void) strcpy(Namech, "unknown type");
		ty = NULL;
	}
	if (ty)
		(void) strcpy(Lf->type, ty);
/*
 * If this is a VCHR file and it's missing an inode number, try to
 * supply one.
 */
	if (Lf->inp_ty == 0 && type == VCHR && Lf->dev_def)
		find_ch_ino();
/*
 * Test for specified file.
 */

#if	defined(HASPROCFS)
	if (Ntype == N_PROC) {
		if (Procsrch)
			Lf->sf |= SELNM;
		else {
			for (pfi = Procfsid; pfi; pfi = pfi->next) {
				if (pfi->pid == pids.pid_id) {
					Lf->sf |= SELNM;
					break;
				}
			}
		}
	} else
#endif	/* HASPROCFS */

	{
		if (Sfile && is_file_named(NULL, type))
			 Lf->sf |= SELNM;
	}
/*
 * Enter name characters.
 */
	if (Namech[0])
		enter_nm(Namech);
}


/*
 * readlino() - read local inode information
 */

static int
readlino(fx, v, i)
	int fx;				/* file system index */
	struct vnode *v;		/* vnode pointing to inode */
	struct l_ino *i;		/* local inode */
{
	struct namenode nm;
	struct inode sn;
	struct xnamnode xn;

	if (fx < 1 || fx > Fsinfomax || v->v_data == NULL)
		return(1);
	if (strcmp(Fsinfo[fx-1], "vxfs") == 0)
		return(readvxfslino(v, i));
	else if (strcmp(Fsinfo[fx-1], "fifofs") == 0
	     ||  strcmp(Fsinfo[fx-1], "sfs") == 0
	     ||  strcmp(Fsinfo[fx-1], "ufs") == 0)
	{
		if (kread((KA_T)v->v_data, (char *)&sn, sizeof(sn)))
			return(1);
		i->dev = sn.i_dev;
		i->rdev = v->v_rdev;
		i->number = sn.i_number;
		i->size = sn.i_size;
		return(0);
	} else if (strcmp(Fsinfo[fx-1], "cdfs") == 0)
		return(readcdfslino(v, i));
	else if (strcmp(Fsinfo[fx-1], "s5") == 0)
		return(reads5lino(v, i));
	else if (strcmp(Fsinfo[fx-1], "namefs") == 0) {
		if (kread((KA_T)v->v_data, (char *)&nm, sizeof(nm)))
			return(1);
		i->dev = nm.nm_vattr.va_fsid;
		i->rdev = nm.nm_vattr.va_rdev;
		i->number = nm.nm_vattr.va_nodeid;
		i->size = nm.nm_vattr.va_size;
		return(0);
	} else if (strcmp(Fsinfo[fx-1], "xnamfs") == 0
	       ||  strcmp(Fsinfo[fx-1], "XENIX") == 0) {
		if (kread((KA_T)v->v_data, (char *)&xn, sizeof(xn)))
			return(1);
		i->dev = xn.x_dev;
		i->rdev = xn.x_fsid;
		i->size = xn.x_size;
		return(0);
	}
	return(1);
}
