

#if	defined(HASDCACHE)
/*
 * dvch.frag - fragment that contains common device cache functions
 *
 * The caller may define the following:
 *
 *	DCACHE_CLONE	is the name of the function that reads and writes
 *			the clone section of the device cache file.  The
 *			clone section follows the device section.
 *
 *	DCACHE_CLR	is the name of the function that clears the clone
 *			and pseudo caches when reading the device cache
 *			fails.
 *
 *	DCACHE_PSEUDO	is the name of the function that reads and writes
 *			the pseudo section of the device cache file.  The
 *			pseudo section follows the device section and the
 *			clone section, if there is one.
 *
 *	DVCH_DEVPATH	if the path to the device directory isn't "/dev".
 *
 *	DVCH_EXPDEV	if st_rdev must be expanded with the expdev()
 *			macro before use.  (This is an EP/IX artifact.)
 *
 *	DVCH_FCHOWN	use fchown() to change the ownerships of the device
 *			cache file after its  creation to the real UID and
 *			GID.  Setuid(root) lsof's need to do this.
 *
 *	DVCH_ST_CTIME	is the dialect's special element name for ctime in
 *			the stat structure (default = st_ctime).
 *
 *	DVCH_ST_MTIME	is the dialect's special element name for mtime in
 *			the stat structure (default = st_mtime).
 *
 *	These functions are supplied a single integer argument, where:
 *
 *		1 = read
 *		2 = write
 *
 *	The functions must return 0 on success and 1 on failure.  The
 *	failure return must be accompanied by the issuance of stderr
 *	messages under the discretion of Fwarn.
 */


/*
 * Local definitions
 */

# if	!defined(DVCH_DEVPATH)
#define	DVCH_DEVPATH	"/dev"
# endif	/* !defined(DVCH_DEVPATH) */

# if	!defined(DVCH_ST_CTIME)
#define	DVCH_ST_CTIME	st_ctime
# endif	/* !defined(DVCH_ST_CTIME) */

# if	!defined(DVCH_ST_MTIME)
#define	DVCH_ST_MTIME	st_mtime
# endif	/* !defined(DVCH_ST_MTIME) */

/*
 * Local storage
 */

static int crctbl[CRC_TBLL];		/* crc partial results table */


/*
 * alloc_dcache() - allocate device cache
 */

void
alloc_dcache()
{
	if ((Devtp = (struct l_dev *)calloc((MALLOC_S)Ndev,
				     sizeof(struct l_dev)))
	== NULL) {
	    (void) fprintf(stderr, "%s: no space for devices\n", Pn);
	    exit(1);
	}
	if ((Sdev = (struct l_dev **)malloc((MALLOC_S)(sizeof(struct l_dev *)
	    * Ndev)))
	== NULL) {
	    (void) fprintf(stderr, "%s: no space for device pointers\n",
		Pn);
	    exit(1);
	}
}


/*
 * crc(b, l, s) - compute a crc for a block of bytes
 */

void
crc(b, l, s)
	char *b;			/* block address */
	int  l;				/* length */
	unsigned *s;			/* sum */
{
	char *cp;			/* character pointer */
	char *lm;			/* character limit pointer */
	unsigned sum;			/* check sum */

	cp = b;
	lm = cp + l;
	sum = *s;
	do {
		sum ^= ((int) *cp++) & 0xff;
		sum = (sum >> 8) ^ crctbl[sum & 0xff];
	} while (cp < lm);
	*s = sum;
}


/*
 * crcbld - build the CRC-16 partial results table
 */

void
crcbld()
{
	int bit;			/* temporary bit value */
	unsigned entry;			/* entry under construction */
	int i;				/* polynomial table index */
	int j;				/* bit shift count */

	for(i = 0; i < CRC_TBLL; i++) {
		entry = i;
		for (j = 1; j <= CRC_BITS; j++) {
			bit = entry & 1;
			entry >>= 1;
			if (bit)
				entry ^= CRC_POLY;
		}
		crctbl[i] = entry;
	}
}


/*
 * open_dcache() - open device cache file
 */

int
open_dcache(m, s)
	int m;			/* mode: 1 = read; s = write */
	struct stat *s;		/* stat() receiver */
{
	int err;
	char *w = NULL;

/*
 * Create the device cache file path, if necessary.
 */
	if (!DCpath) {

	    char buf[MAXPATHLEN+1], *cp, hn[MAXPATHLEN+1];
	    struct passwd *p;

	    buf[0] = '\0';
	    if ((p = getpwuid(getuid())) != NULL) {
		if (gethostname(hn, sizeof(hn) - 1) < 0)
		    hn[0] = '\0';
		else {
		    hn[sizeof(hn) - 1] = '\0';
		    if ((cp = strchr(hn, '.')) != NULL && cp > hn)
			*cp = '\0';
		}
		if (p->pw_dir[0] == '/'
		&&  (strlen(p->pw_dir) + strlen(HASDCACHE) + strlen(hn) + 3)
		< sizeof(buf)) {
		    cp = strrchr(p->pw_dir, '/');
		    (void) sprintf(buf, "%s%s%s_%s", p->pw_dir,
			(*(cp + 1) == '\0') ? "" : "/",
			HASDCACHE, hn);
		}
	    }
	    if (!buf[0]) {
		if (!Fwarn)
		    (void) fprintf(stderr,
			"%s: WARNING: can't make path for %s\n",
			Pn, HASDCACHE);
		return(1);
	    }
	    if ((DCpath = (char *)malloc((MALLOC_S)(strlen(buf) + 1))) == NULL)
	    {
		(void) fprintf(stderr,
		    "%s: no space for device cache file path: %s\n", Pn, buf);
		exit(1);
	    }
	    (void) strcpy(DCpath, buf);
	}
/*
 * Switch to the requested open() action.
 */
	switch (m) {
	case 1:

	/*
	 * Open for reading.  First check for the right to access.
	 */
	    if (!is_readable(DCpath)) 
		return(1);
	    if ((DCfd = open(DCpath, O_RDONLY, 0)) < 0) {
		if (DCstate == 3 && errno == ENOENT)
		    return(1);

cant_open:
		    (void) fprintf(stderr,
			"%s: WARNING: can't open %s: %s\n",
			Pn, DCpath, strerror(errno));
		return(1);
	    }
	    if (stat(DCpath, s) != 0) {

cant_stat:
		if (!Fwarn)
		    (void) fprintf(stderr,
			"%s: WARNING: can't stat(%s): %s\n",
			Pn, DCpath, strerror(errno));
close_exit:
		(void) close(DCfd);
		DCfd = -1;
		return(1);
	    }
	    if ((s->st_mode & 0777) != 0600)
		w = "not mode 0600";
	    else if ((s->st_mode & S_IFMT) != S_IFREG)
		w = "not a regular file";
	    else if (!s->st_size)
		w = "empty";
	    if (w) {
		if (!Fwarn)
		    (void) fprintf(stderr,
			"%s: WARNING: %s is %s.\n", Pn, DCpath, w);
		goto close_exit;
	    }
	    return(0);
	case 2:

	/*
	 * Open for writing: first unlink any previous version; then
	 * open exclusively, specifying it's an error if the file exists.
	 */
	    if (unlink(DCpath) < 0) {
		if (errno != ENOENT) {
		    if (!Fwarn)
			(void) fprintf(stderr,
			    "%s: WARNING: can't unlink %s: %s\n",
			    Pn, DCpath, strerror(errno));
		    return(1);
		}
	    }
	    if ((DCfd = open(DCpath, O_RDWR|O_CREAT|O_EXCL, 0600)) < 0)
		goto cant_open;

#if	defined(DVCH_FCHOWN)
	    if (fchown(DCfd, getuid(), getgid()) < 0) {
		if (!Fwarn)
		    (void) fprintf(stderr,
			"%s: WARNING: can't change ownerships of %s: %s\n",
			Pn, DCpath, strerror(errno));
		(void) unlink(DCpath);
		goto close_exit;
	    }
#endif	/* defined(DVCH_FCHOWN) */

	    if (!Fwarn)
		(void) fprintf(stderr,
		    "%s: WARNING: created device cache file: %s\n", Pn, DCpath);
	    if (stat(DCpath, s) != 0) {
		err = errno;
		(void) unlink(DCpath);
		errno = err;
		goto cant_stat;
	    }
	    return(0);
	default:

	/*
	 * Oops!
	 */
	    (void) fprintf(stderr, "%s: internal error: open_dcache=%d\n",
		Pn, m);
	    exit(1);
	}
	return(1);		/* to make IRIX 6.0's C compiler happy */
}


/*
 * read_dcache() - read device cache file
 */

int
read_dcache()
{
	char buf[MAXPATHLEN*2], cbuf[32], *cp;
	int i, len;
	struct stat sb, devsb;
/*
 * Open the device cache file.  If it's last mtime isn't greater than
 * DVCH_DEVPATH's mtime, ignore it.
 */
	if (open_dcache(1, &sb))
		return(1);
	if (stat(DVCH_DEVPATH, &devsb) != 0) {
	    if (!Fwarn)
		(void) fprintf(stderr,
		    "%s: WARNING: can't stat(%s): %s\n",
		    Pn, DVCH_DEVPATH, strerror(errno));
	} else {
	    if (sb.DVCH_ST_MTIME <= devsb.DVCH_ST_MTIME
	    ||  sb.DVCH_ST_MTIME <= devsb.DVCH_ST_CTIME) {
		if (!Fwarn)
		    (void) fprintf(stderr,
			"%s: WARNING: %s needs to be rebuilt; %s is newer.\n",
			Pn, DCpath, DVCH_DEVPATH);
		if (DCstate != 2) {
		    (void) close(DCfd);
		    DCfd = -1;
		    return(1);
		}
	    }
	}
	if ((DCfs = fdopen(DCfd, "r")) == NULL) {
	    if (!Fwarn)
		(void) fprintf(stderr,
		    "%s: WARNING: can't fdopen(%s)\n", Pn, DCpath);
	    (void) close(DCfd);
	    DCfd = -1;
	    return(1);
	}
/*
 * Read the section count line; initialize the CRC table;
 * validate the section count line.
 */
 
	if (fgets(buf, sizeof(buf), DCfs) == NULL) {

cant_read:
	    if (!Fwarn)
		(void) fprintf(stderr,
		    "%s: WARNING: can't fread %s: %s\n", Pn, DCpath,
		    strerror(errno));
read_close:
		(void) fclose(DCfs);
		DCfd = -1;
		DCfs = NULL;
		if (Devtp) {
			for (i = 0; i < Ndev; i++) {
				if (Devtp[i].name) {
					(void) free((FREE_P *)Devtp[i].name);
					Devtp[i].name = NULL;
				}
			}
			(void) free((FREE_P *)Devtp);
			Devtp = NULL;
		}
		if (Sdev) {
			(void) free((FREE_P *)Sdev);
			Sdev = NULL;
		}
		Ndev = 0;

# if	defined(DCACHE_CLR)
		(void) DCACHE_CLR();
# endif	/* defined(DCACHE_CLR) */

		return(1);
	}
	(void) crcbld();
	DCcksum = 0;
	(void) crc(buf, strlen(buf), &DCcksum);
	i = 1;
	cp = "";

# if	defined(DCACHE_CLONE)
	i++;
	cp = "s";
# endif	/* defined(DCACHE_CLONE) */

# if	defined(DCACHE_PSEUDO)
	i++;
	cp = "s";
# endif	/* defined(DCACHE_PSEUDO) */

	(void) sprintf(cbuf, "%d section%s\n", i, cp);
	if (strcmp(buf, cbuf) != 0) {
	    if (!Fwarn)
		(void) fprintf(stderr,
		    "%s: WARNING: bad section count line in %s: %s\n",
		    Pn, DCpath, buf);
	    goto read_close;
	}
/*
 * Read device section header and validate it.
 */
	if (fgets(buf, sizeof(buf), DCfs) == NULL)
	    goto cant_read;
	(void) crc(buf, strlen(buf), &DCcksum);
	len = strlen("device section: ");
	if (strncmp(buf, "device section: ", len) != 0) {

read_dhdr:
	    if (!Fwarn)
		(void) fprintf(stderr,
		    "%s: WARNING: bad device section header in %s: %s\n",
		    Pn, DCpath, buf);
	    goto read_close;
	}
/*
 * Compute the device count; allocate Sdev[] and Devtp[] space.
 */
	if ((Ndev = atoi(&buf[len])) < 1)
	    goto read_dhdr;
	alloc_dcache();
/*
 * Read the device lines and store their information in Devtp[].
 * Construct the Sdev[] pointers to Devtp[].
 */
	for (i = 0; i < Ndev; i++) {
	    if (fgets(buf, sizeof(buf), DCfs) == NULL) {
		if (!Fwarn)
		    (void) fprintf(stderr,
			"%s: WARNING: can't read device %d from %s\n",
			Pn, i + 1, DCpath);
		goto read_close;
	    }
	    (void) crc(buf, strlen(buf), &DCcksum);
	/*
	 * Convert hexadecimal device number.
	 */
	    if ((cp = x2dev(buf, &Devtp[i].rdev)) == NULL || *cp != ' ') {
		if (!Fwarn)
		    (void) fprintf(stderr,
			"%s: device %d: bad device in %s: %s",
			Pn, i + 1, DCpath, buf);
		goto read_close;
	    }
	/*
	 * Convert inode number.
	 */
	    for (cp++, Devtp[i].inode = 0; *cp != ' '; cp++) {
		if (*cp < '0' || *cp > '9') {
		if (!Fwarn)
		    (void) fprintf(stderr,
			"%s: WARNING: device %d: bad inode # in %s: %s",
			Pn, i + 1, DCpath, buf);
		    goto read_close;
		}
		Devtp[i].inode = (Devtp[i].inode * 10) + (int)(*cp - '0');
	    }
	/*
	 * Get path name; allocate space for it; copy it; store the
	 * pointer in Devtp[]; construct the Sdev[] pointer to Devtp[].
	 */
	    if ((len = strlen(++cp)) < 2 || *(cp + len - 1) != '\n') {
		if (!Fwarn)
		    (void) fprintf(stderr,
			"%s: WARNING: device %d: bad path in %s: %s",
			Pn, i + 1, DCpath, buf);
		    goto read_close;
	    }
	    if ((Devtp[i].name = (char *)malloc(len)) == NULL) {
		(void) fprintf(stderr,
		    "%s: no space for %s", Pn, buf);
		exit(1);
	    }
	    *(cp + len - 1) = '\0';
	    (void) strcpy(Devtp[i].name, cp);
	    Sdev[i] = &Devtp[i];
	}

# if	defined(DCACHE_CLONE)
/*
 * Read the clone section.
 */
	if (DCACHE_CLONE(1))
	    goto read_close;
# endif	/* defined(DCACHE_CLONE) */

# if	defined(DCACHE_PSEUDO)
/*
 * Read the pseudo section.
 */
	if (DCACHE_PSEUDO(1))
	    goto read_close;
# endif	/* defined(DCACHE_PSEUDO) */

/*
 * Read and check the CRC section; it must be the last thing in the file.
 */
	(void) sprintf(cbuf, "CRC section: %x\n", DCcksum);
	if (fgets(buf, sizeof(buf), DCfs) == NULL || strcmp(buf, cbuf) != 0) {
	    if (!Fwarn)
		(void) fprintf(stderr,
		    "%s: WARNING: bad CRC section in %s: %s", Pn, DCpath, buf);
	    goto read_close;
	}
	if (fgets(buf, sizeof(buf), DCfs) != NULL) {
	    if (!Fwarn)
		(void) fprintf(stderr,
		    "%s: WARNING: data follows CRC section in %s: %s",
		    Pn, DCpath, buf);
	    goto read_close;
	}
/*
 * Check one device entry at random -- the randomness based on our
 * PID.
 */
	i = Mypid % Ndev;
	if (stat(Devtp[i].name, &sb) != 0

# if	defined(DVCH_EXPDEV)
	||  expdev(sb.st_rdev) != Devtp[i].rdev
# else	/* !defined(DVCH_EXPDEV) */
	||  sb.st_rdev != Devtp[i].rdev
# endif	/* defined(DVCH_EXPDEV) */

	|| sb.st_ino != Devtp[i].inode) {
	    if (!Fwarn)
		(void) fprintf(stderr,
			"%s: WARNING: device cache mismatch: %s\n",
			Pn, Devtp[i].name);
	    goto read_close;
	}
/*
 * Close the device cache file and return OK.
 */
	(void) fclose(DCfs);
	DCfd = -1;
	DCfs = NULL;
	return(0);
}


/*
 * write_dcache() - write device cache file
 */

void
write_dcache()
{
	char buf[MAXPATHLEN*2], *cp;
	struct l_dev *dp;
	int i;
	struct stat sb;

/*
 * Open the cache file; set up the CRC table; write the section count.
 */
	if (open_dcache(2, &sb))
		return;
	i = 1;
	cp = "";

# if	defined(DCACHE_CLONE)
	i++;
	cp = "s";
# endif	/* defined(DCACHE_CLONE) */

# if	defined(DCACHE_PSEUDO)
	i++;
	cp = "s";
# endif	/* defined(DCACHE_PSEUDO) */

	(void) sprintf(buf, "%d section%s\n", i, cp);
	(void) crcbld();
	DCcksum = 0;
	if (wr2DCfd(buf, &DCcksum))
		return;
/*
 * Write the device section from the contents of Sdev[] and Devtp[].
 */
	(void) sprintf(buf, "device section: %d\n", Ndev);
	if (wr2DCfd(buf, &DCcksum))
	    return;
	for (i = 0; i < Ndev; i++) {
	    dp = Sdev[i];
	    (void) sprintf(buf, "%x %ld %s\n", dp->rdev, dp->inode, dp->name);
	    if (wr2DCfd(buf, &DCcksum))
		return;
	}

# if	defined(DCACHE_CLONE)
/*
 * Write the clone section.
 */
	if (DCACHE_CLONE(2))
	    return;
# endif	/* defined(DCACHE_CLONE) */

# if	defined(DCACHE_PSEUDO)
/*
 * Write the pseudo section.
 */
	if (DCACHE_PSEUDO(2))
	    return;
# endif	/* defined(DCACHE_PSEUDO) */

/*
 * Write the CRC section and close the file.
 */
	(void) sprintf(buf, "CRC section: %x\n", DCcksum);
	if (wr2DCfd(buf, NULL))
		return;
	if (close(DCfd) != 0) {
	    if (!Fwarn)
		(void) fprintf(stderr,
		    "%s: WARNING: can't close %s: %s\n",
		    Pn, DCpath, strerror(errno));
	    (void) unlink(DCpath);
	    DCfd = -1;
	}
	DCfd = -1;
}


/*
 * wr2DCfd() - write to the DCfd file descriptor
 */

int
wr2DCfd(b, c)
	char *b;			/* buffer */
	unsigned *c;			/* checksum receiver */
{
	int bl, bw;

	bl = strlen(b);
	if (c)
	    (void) crc(b, bl, c);
	while (bl > 0) {
	    if ((bw = write(DCfd, b, bl)) < 0) {
		if (!Fwarn)
		    (void) fprintf(stderr,
			"%s: WARNING: can't write to %s: %s\n",
			Pn, DCpath, strerror(errno));
		(void) unlink(DCpath);
		(void) close(DCfd);
		DCfd = -1;
		return(1);
	    }
	    b += bw;
	    bl -= bw;
	}
	return(0);
}
#endif	/* defined(HASDCACHE) */
