;;; jde-gen.el -- Integrated Development Environment for Java.
;; $Revision: 1.59 $ 

;; Author: Paul Kinnucan <paulk@mathworks.com>
;; Maintainer: Paul Kinnucan
;; Keywords: java, tools

;; Copyright (C) 1997, 1998, 2000, 2001, 2002 Paul Kinnucan.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; Gnu Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.
;;

(require 'tempo)

(defgroup jde-gen nil
  "JDE Autocoder"
  :group 'jde
  :prefix "jde-gen-")

(defcustom jde-gen-k&r t
  "*If non-nil, use braces in Original Kernighan & Ritchie Style.
The Creators of C started using brace placement style:

    class Some {

    }

But there is also the alternative line-up style:

    class Some
    {

    }

Setting this variable to t, uses K&R style in skeletons and tempaltes.

Note: According to the Java Code Convention [section 6.4], this value should
      be non-nil."
  :group 'jde-gen
  :type  'boolean)

; There must be some cleverer way to do this ...
(defun jde-gen-delete-preceding-whitespace ()
  "Delete any syntactical whitespace (including newlines)
before point."
  (while (and (not (bobp)) 
	      (or (bolp) 
		  (re-search-backward "\\s-\\=" nil t)))
    (delete-char -1)))

(defun jde-gen-extract-ids-from-params (params)
  "Given a parameter lsit \"Type1 id1, Type2, id2, ...\" extract the
ids and return as \"id1, id2, ...\"."
  (mapconcat 
   (lambda (arg) 
     (nth 1 (split-string 
	     (replace-in-string 
	      (replace-in-string arg "^[ \t\n\f\l]+" "")
	      "[ \t\n\f\l]+$" ""))))
   (split-string params "[,]") ", "))

(defun jde-gen-lookup-named (name)
  "Lookup some saved data under the name NAME.
Returns the data if NAME was found, and nil otherwise."
  (cdr (assq name tempo-named-insertions)))

(defun jde-gen-read-template (strings)
  "Converts an autocode template represented as a list
of strings to a list of Lisp objects as required by
tempo."
  (let ((template-string "")
	(n (length strings))
	(i 0))
    (while (< i n)
      (setq template-string
	    (concat template-string " " (nth i strings)))
      (setq i (1+ i)))
    (setq template-string
	  (concat "'(" template-string ")"))
    (eval (car (read-from-string template-string)))))

(defcustom jde-gen-buffer-boilerplate nil
  "*Lines of boilerplate text to put at the head of a buffer template."
  :group 'jde-gen
  :type '(repeat (string :tag "Line")))

(defcustom jde-gen-boilerplate-function 'jde-gen-create-buffer-boilerplate
  "*Specifes buffer boilerplate text function.
This variable specifies a function to create boilerplate text for
insertion at the head of Java source buffers generated by JDE
templates. The specified function should take no arguments and should
return a text string.  The default value is
`jde-gen-create-buffer-boilerplate', which returns the lines of text
specified by `jde-gen-buffer-boilerplate'."
  :group 'jde-gen
  :type 'function)

(defun jde-gen-create-buffer-boilerplate ()
  "This function creates buffer boilerplate from the
variable `jde-gen-buffer-boilerplate."
  (if jde-gen-buffer-boilerplate
      (let ((bp "")
	    (n (length jde-gen-buffer-boilerplate))
	    (i 0))
	(while (< i n)
	  (setq bp
		(concat bp (elt jde-gen-buffer-boilerplate i) "\n"))
	  (setq i (1+ i)))
	bp)))

(defun jde-gen-get-extend-class ()
  (let ((super-class (read-from-minibuffer "extends: ")))
    (if (not (string= super-class ""))
	(progn 
	  (jde-import-find-and-import super-class)
	  (concat "extends " super-class " ")))))

; In order to avoid problems with recursive tempo-template invocations
; interface generation has been split in two parts. 
; jde-gen-get-interface-implementation prompts for the
; interface and stores the point and interface name in buffer local
; variables.  jde-gen-insert-interface-implementation is invoked after
; the template for class or inner class and generates the interface 
; implementation.

(defvar jde-gen-interface-to-gen nil
  "Insertion point and name (POINT . NAME) of interface
implemented by the most recently generated class template.
Used by `jde-gen-get-interface-implementation' to
store name and location of interface to be inserted
by `jde-gen-insert-interface-implementation'.")

(defun jde-gen-get-interface-implementation ()
  "Prompts the user to enter the name of an interface
implemented by a class being generated by a tempo template.
If the user enters a name, stores the current point in
the buffer and the interface name as a cons (POINT . NAME)
in the global variable `jde-gen-interface-to-gen'. Otherwise
it sets this variable to nil."
  (let ((interface (read-from-minibuffer "implements: ")))
    (if (not (string= interface ""))
	(setq jde-gen-interface-to-gen
		(cons (point) interface))
      (setq jde-gen-interface-to-gen nil)))
  nil) ;; must return nil to prevent template insertion.

(defun jde-gen-insert-interface-implementation ()
  "Generates the interface specified by the cdr of `jde-gen-interface-to-gen'
and inserts it in the current buffer at the location specified
by the car of `jde-gen-interface-to-gen'."
  (if jde-gen-interface-to-gen
      (let ((ins-pos (car jde-gen-interface-to-gen))
	    (interface (cdr jde-gen-interface-to-gen)))
	(save-excursion
	  (goto-char ins-pos)
	  (jde-wiz-implement-interface-internal interface)))))

(defun jde-gen-get-package-statement ()
  (require 'jde-package)
  (let* ((package-dir (jde-package-get-package-directory))
	 (suggested-package-name
	  (if (or
	       (string= package-dir jde-package-unknown-package-name)
	       (string= package-dir ""))
	      nil
	    (jde-package-convert-directory-to-package package-dir)))
	 (package-name
	  (read-from-minibuffer "Package: " suggested-package-name)))
    (if (and
	 package-name
	 (not (string= package-name "")))
	(format "package %s;\n\n" package-name))))


(defcustom jde-gen-method-signature-padding-1 ""
  "String that comes just after the function name and just before
the opening parenthesis of the argument list for a method call or definition.
For example, if set to a single space [\" \"], then a generated method might
look like:

    void setXxxx () {
                ^
If not set, the same generated method would look like:

    void setXxxx() {
                ^
Note: According to the Java Code Convention [section 6.4], this value should
      be the empty string."
  :group 'jde-gen
  :type 'string)


(defcustom jde-gen-method-signature-padding-2 ""
  "String which comes just after the opening parenthesis and just before
the closing parenthesis of an argument list for a method call or definition.
For example, if set to a single space [\" \"], then a generated method might
look like:

    void setXxxx( boolean newValue ) {
                 ^                ^
If not set, the same generated method would look like:

    void setXxxx(boolean newValue) {
                 ^              ^
Note: According to the Java Code Convention [section 6.4], this value should
      be the empty string."
  :group 'jde-gen
  :type 'string)


(defcustom jde-gen-method-signature-padding-3 " "
  "String which comes just after the closing parenthesis of an
argument list for a method call or definition.  For example, if set to
a single space [\" \"], then a generated method might look like:

    void setXxxx(boolean newValue) {
                                  ^
If not set, the same generated method would look like:

    void setXxxx(boolean newValue){
                                  ^
Note: According to the Java Code Convention [section 6.4], this value should
      be a single space."
  :group 'jde-gen
  :type 'string)


(defcustom jde-gen-conditional-padding-1 " "
  "The string to be inserted between a conditional keyword (if, while, etc.)
and the opening parenthesis for its conditional expression:

    <keyword><1>(<2><expression><2>)<3>{

also:  <3>else<3>{

where <1> is jde-gen-conditional-padding-1
  and <2> is jde-gen-conditional-padding-2
  and <3> is jde-gen-conditional-padding-3.

For example, if <1> is a space, <2> is nil and <3> is a space, then a while
clause might look like:

    while (true) {

Note: According to the Java Code Convention [section 7.4], this value should
      be a single space."
  :group 'jde-gen
  :type 'string)


(defcustom jde-gen-conditional-padding-2 ""
  "The string to be inserted before and after a conditional expression
between the parentheses.  See `jde-gen-conditional-padding-1' for details.

Note: According to the Java Code Convention [section 7.4], this value should
      be the empty string."
  :group 'jde-gen
  :type 'string)


(defcustom jde-gen-conditional-padding-3 " "
  "The string to be inserted after the closing parenthesis of the
conditional expression and before the opening brace.  See
`jde-gen-conditional-padding-1' for details.

Note: According to the Java Code Convention [section 7.4], this value should
      be a single space."
  :group 'jde-gen
  :type 'string)


(defun jde-gen-method-signature (access type name arglist &optional throwslist)
  "This function generates method signatures with a consistent format.
All jde-gen functions and/or templates should use this function when
creating Java methods and constructors.

The signature is of the format:

  <access> <type> <name><1>(<2><arglist><2>) throws <throwslist><3>

where <1> is the value of jde-gen-method-signature-padding-1
  and <2> is the value of jde-gen-method-signature-padding-2
  and <3> is the value of jde-gen-method-signature-padding-3.

<3> will not be used if `jde-gen-k&r' is nil.

If a parameter to this function is empty or nil, then it is omitted
(as well as the corresponding padding, whitespace and/or Java
keywords)."

  (let ((sig
	 (concat

	  (if (> (length access) 0)
	      (concat access " ")
	    ());; if no access (e.g. "public static"), then nothing

	  (if (> (length type) 0)
	      (concat type " ")
	    ());; if no type (e.g. "boolean" or "void"), then nothing

	  name
	  jde-gen-method-signature-padding-1
	  "("
	  (if (> (length arglist) 0)
	      (concat jde-gen-method-signature-padding-2 arglist
		      jde-gen-method-signature-padding-2 )
	    ())
	  ")"
	  (if (> (length throwslist) 0)
	      (concat " throws " throwslist)
	    ())
	  (if jde-gen-k&r
	      jde-gen-method-signature-padding-3
	    ()))))
    sig))


;;(makunbound 'jde-gen-class-buffer-template)
(defcustom jde-gen-class-buffer-template
  (list
   "(funcall jde-gen-boilerplate-function)"
   "(jde-gen-get-package-statement)"
   "(progn (require 'jde-javadoc) (jde-javadoc-insert-start-block))"
   "\" * \""
   "(file-name-nondirectory buffer-file-name) '>'n"
   "\" \" (jde-javadoc-insert-empty-line)"
   "\" \" (jde-javadoc-insert-empty-line)"
   "\" * Created: \" (current-time-string) '>'n"
   "\" \" (jde-javadoc-insert-empty-line)"
   "\" \" (jde-javadoc-insert 'tempo-template-jde-javadoc-author-tag)"
   "\" \" (jde-javadoc-insert 'tempo-template-jde-javadoc-version-tag)"
   "\" \" (jde-javadoc-insert 'tempo-template-jde-javadoc-end-block \"*/\")"
   "\"public class \""
   "(file-name-sans-extension (file-name-nondirectory buffer-file-name))"
   "\" \" (jde-gen-get-extend-class)"

   "(if jde-gen-k&r "
   " ()"
   " '>'n)"
   "\"{\"'>'n"

   "(jde-gen-method-signature"
   "\"public\""
   "\"\""
   "(file-name-sans-extension (file-name-nondirectory buffer-file-name))"
   "\"\""
   ")"

   "(if jde-gen-k&r "
   "()"
   "'>'n)"
   "\"{\"'>'n"

   "'>'p'n"
   "\"}\">"
   "\" // \""
   "(file-name-sans-extension (file-name-nondirectory buffer-file-name)) \" constructor\""
   "'>'n"
   "(jde-gen-get-interface-implementation)"
   "'>'n"
   "\"}\">"
   "\" // \""
   "(file-name-sans-extension (file-name-nondirectory buffer-file-name))"
   "'>'n")
  "*Template for new Java class.
Setting this variable defines a template instantiation
command `jde-gen-class', as a side-effect."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (tempo-define-template "java-class-buffer-template"
				 (jde-gen-read-template val)
				 nil
				 "Insert a generic Java class buffer skeleton.")
	  (defalias 'jde-gen-class
	    (list 'lambda (list)
		  (list 'interactive)
		  (list 'tempo-template-java-class-buffer-template)
		  (list 'jde-gen-insert-interface-implementation)))
	  (set-default sym val)))

;;;###autoload
(defun jde-gen-class-buffer (file)
  "Create a new Java buffer containing a class of the same name.
This command inserts the class template generated by `jde-gen-class'.
It then moves the point to the location to the constructor."
  (interactive "F")
  (find-file file)
  (jde-gen-class)
  (beginning-of-buffer)
  (search-forward "{")
  (backward-char 1)
  (c-indent-exp)
  (tempo-forward-mark))


;;(makunbound 'jde-gen-interface-buffer-template)
(defcustom jde-gen-interface-buffer-template
  (list
   "(funcall jde-gen-boilerplate-function)"
   "(jde-gen-get-package-statement)"
   "(progn (require 'jde-javadoc) (jde-javadoc-insert-start-block))"
   "\" * \""
   "(file-name-nondirectory buffer-file-name) '>'n"
   "\" \" (jde-javadoc-insert-empty-line)"
   "\" \" (jde-javadoc-insert-empty-line)"
   "\" * Created: \" (current-time-string) '>'n"
   "\" \" (jde-javadoc-insert-empty-line)"
   "\" \" (jde-javadoc-insert 'tempo-template-jde-javadoc-author-tag)"
   "\" \" (jde-javadoc-insert 'tempo-template-jde-javadoc-version-tag)"
   "\" \" (jde-javadoc-insert 'tempo-template-jde-javadoc-end-block \"*/\")"
   "'>'n"
   "\"public interface \""
   "(file-name-sans-extension (file-name-nondirectory buffer-file-name))"
   "\" \" (jde-gen-get-extend-class)"

   "(if jde-gen-k&r "
   " ()"
   " '>'n)"
   "\"{\"'>'n"
   "'>'p'n"
   "\"}\">"
   "\"// \""
   "(file-name-sans-extension (file-name-nondirectory buffer-file-name))"
   "'>'n")
  "*Template for new Java interface.
Setting this variable defines a template instantiation
command `jde-gen-interface', as a side-effect."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (tempo-define-template "java-interface-buffer-template"
				 (jde-gen-read-template val)
				 nil
				 "Insert a generic Java interface buffer skeleton.")
	  (defalias 'jde-gen-interface
	    (list 'lambda (list)
		  (list 'interactive)
		  (list 'tempo-template-java-interface-buffer-template)))
	  (set-default sym val)))

;;;###autoload
(defun jde-gen-interface-buffer (file)
  "Create a new Java buffer containing an interface of the same name.
This command inserts the interface template generated by `jde-gen-interface'.
It then moves the point to the location of the first method."
  (interactive "F")
  (find-file file)
  (jde-gen-interface)
  (beginning-of-buffer)
  (search-forward "{")
  (backward-char 1)
  (c-indent-exp)
  (tempo-forward-mark))


;;(makunbound 'jde-gen-console-buffer-template)
(defcustom jde-gen-console-buffer-template
  (list
   "(funcall jde-gen-boilerplate-function)"
   "(jde-gen-get-package-statement)"
   "(progn (require 'jde-javadoc) (jde-javadoc-insert-start-block))"
   "\" * \""
   "(file-name-nondirectory buffer-file-name) '>'n"
   "\" \" (jde-javadoc-insert-empty-line)"
   "\" \" (jde-javadoc-insert-empty-line)"
   "\" * Created: \" (current-time-string) '>'n"
   "\" \" (jde-javadoc-insert-empty-line)"
   "\" \" (jde-javadoc-insert 'tempo-template-jde-javadoc-author-tag)"
   "\" \" (jde-javadoc-insert 'tempo-template-jde-javadoc-version-tag)"
   "\" \" (jde-javadoc-insert 'tempo-template-jde-javadoc-end-block \"*/\")"
   "'>'n"
   "\"public class \""
   "(file-name-sans-extension (file-name-nondirectory buffer-file-name))"

   "(if jde-gen-k&r "
   " ()"
   " '>'n)"
   "\"{\"'>'n"

   "(jde-gen-method-signature"
   "  \"public\""
   "  \"\""
   "  (file-name-sans-extension (file-name-nondirectory buffer-file-name))"
   "  \"\""
   " )"
   "'>"

   "(if jde-gen-k&r "
   " ()"
   " 'n)"
   "\"{\"'>'n"

   "'>'n"
   "\"}\"'>'n"
   "'>'n"
   "(jde-gen-method-signature"
   "   \"public static\""
   "   \"void\""
   "   \"main\""
   "   \"String[] args\""
   " )"
   "'>"

   "(if jde-gen-k&r "
   " ()"
   " 'n)"
   "\"{\"'>'n"

   "'>'p'n"
   "\"}\"'>'n"
   "\"} // \"'>"
   "(file-name-sans-extension (file-name-nondirectory buffer-file-name))"
   "'>'n")
  "*Template for new Java console app main class buffer.
Setting this variable defines a template instantiation
command `jde-gen-console', as a side-effect."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (defalias 'jde-gen-console
	    (tempo-define-template "java-console-buffer-template"
				   (jde-gen-read-template val)
				   nil
				   "Insert skeleton for a new Java console buffer"))
	  (set-default sym val)))

;;;###autoload
(defun jde-gen-console-buffer (file)
  "Create a new Java buffer containing a console class of the same name.
This command inserts the class template generated by `jde-gen-class'.
It then moves the point to the location to the constructor."
  (interactive "F")
  (find-file file)
  (jde-gen-console)
  (beginning-of-buffer)
  (search-forward "{")
  (backward-char 1)
  (c-indent-exp)
  (tempo-forward-mark))


(defcustom jde-gen-jfc-app-buffer-template
  (list
   "(funcall jde-gen-boilerplate-function)"
   "(jde-gen-get-package-statement)"
   "\"import java.awt.Dimension;\" '>'n"
   "\"import java.awt.Graphics;\" '>'n"
   "\"import java.awt.Graphics2D;\" '>'n"
   "\"import java.awt.Color;\" '>'n"
   "\"import java.awt.geom.Ellipse2D;\" '>'n"
   "\"import java.awt.event.WindowAdapter;\" '>'n"
   "\"import java.awt.event.WindowEvent;\" '>'n"
   "\"import javax.swing.JFrame;\" '>'n"
   "\"import javax.swing.JPanel;\" '>'n"
   "\"import javax.swing.JScrollPane;\" '>'n"
   "\"import javax.swing.JMenuBar;\" '>'n"
   "\"import javax.swing.JMenu;\" '>'n"
   "\"import java.awt.event.ActionEvent;\" '>'n"
   "\"import javax.swing.AbstractAction;\" '>'n '>'n"
   "(progn (require 'jde-javadoc) (jde-javadoc-insert-start-block))"
   "\" * \""
   "(file-name-nondirectory buffer-file-name) '>'n"
   "\" \" (jde-javadoc-insert-empty-line)"
   "\" \" (jde-javadoc-insert-empty-line)"
   "\" * Created: \" (current-time-string) '>'n"
   "\" \" (jde-javadoc-insert-empty-line)"
   "\" \" (jde-javadoc-insert 'tempo-template-jde-javadoc-author-tag)"
   "\" \" (jde-javadoc-insert 'tempo-template-jde-javadoc-version-tag)"
   "\" \" (jde-javadoc-insert 'tempo-template-jde-javadoc-end-block \"*/\")"
   "'>'n"
   "\"public class \""
   "(file-name-sans-extension (file-name-nondirectory buffer-file-name))"
   "\" extends JFrame\""

   "(if jde-gen-k&r "
   "()"
   "'>'n)"
   "\"{\"'>'n"

   "\"class Canvas extends JPanel\""

   "(if jde-gen-k&r "
   "()"
   "'>'n)"
   "\"{\"'>'n"

   "(jde-gen-method-signature"
   "  \"public\""
   "  \"\""
   "  \"Canvas\""
   "  \"\""
   " )"
   "'>"

   "(if jde-gen-k&r "
   " ()"
   " 'n)"
   "\"{\"'>'n"

   "\"setSize(getPreferredSize());\" '>'n"
   "\"Canvas.this.setBackground(Color.white);\" '>'n"
   "\"}\"'>'n '>'n"

   "(jde-gen-method-signature"
   "  \"public\""
   "  \"Dimension\""
   "  \"getPreferredSize\""
   "  \"\""
   " )"
   "'>"

   "(if jde-gen-k&r "
   " ()"
   " 'n)"
   "\"{\"'>'n"

   "\"return new Dimension(600, 600);\" '>'n"
   "\"}\"'>'n '>'n"

   "(jde-gen-method-signature"
   "  \"public\""
   "  \"void\""
   "  \"paintComponent\""
   "  \"Graphics g\""
   " )"
   "'>"

   "(if jde-gen-k&r "
   " ()"
   " 'n)"
   "\"{\"'>'n"

   "\"super.paintComponent(g);\" '>'n"
   "\"Graphics2D g2d = (Graphics2D) g;\" '>'n"
   "\"Ellipse2D circle = new Ellipse2D.Double(0d, 0d, 100d, 100d);\" '>'n"
   "\"g2d.setColor(Color.red);\" '>'n"
   "\"g2d.translate(10, 10);\" '>'n"
   "\"g2d.draw(circle);\" '>'n"
   "\"g2d.fill(circle);\" '>'n"
   "\"}\"'>'n "

   "\"}\"'>'n '>'n"


   ;; Constructor
   "(jde-gen-method-signature"
   "  \"public\""
   "  \"\""
   "  (file-name-sans-extension (file-name-nondirectory buffer-file-name))"
   "  \"\""
   " )"
   "'>"

   "(if jde-gen-k&r "
   " ()"
   " 'n)"
   "\"{\"'>'n"


   "\"super(\\\"\" (P \"Enter app title: \") \"\\\");\" '>'n"
   "\"setSize(300, 300);\" '>'n"
   "\"addWindowListener(new WindowAdapter() \""

   "(if jde-gen-k&r "
   "()"
   "'>'n)"
   "\"{\"'>'n"


   "\"public void windowClosing(WindowEvent e) {System.exit(0);}\" '>'n"
   "\"public void windowOpened(WindowEvent e) {}\" '>'n"
   "\"});\"'>'n"


   "\"setJMenuBar(createMenu());\" '>'n"
   "\"getContentPane().add(new JScrollPane(new Canvas()));\" '>'n"
   "\"}\"'>'n"
   "'>'n"

   ;; Main method
   "(jde-gen-method-signature"
   "   \"public static\""
   "   \"void\""
   "   \"main\""
   "   \"String[] args\""
   " )"

   "(if jde-gen-k&r "
   "()"
   "'>'n)"
   "\"{\"'>'n"

   "'>'n"
   "(file-name-sans-extension (file-name-nondirectory buffer-file-name))"
   "\" f = new \""
   "(file-name-sans-extension (file-name-nondirectory buffer-file-name))"
   "\"();\" '>'n"
   "\"f.show();\" '>'n"
   "\"}\"'>'n '>'n"
   ;; createMenu method
   "\"protected JMenuBar createMenu() \""


   "(if jde-gen-k&r "
   "()"
   "'>'n)"
   "\"{\"'>'n"


   "\"JMenuBar mb = new JMenuBar();\" '>'n"
   "\"JMenu menu = new JMenu(\\\"File\\\");\" '>'n"
   "\"menu.add(new AbstractAction(\\\"Exit\\\") \""

   "(if jde-gen-k&r "
   "()"
   "'>'n)"
   "\"{\"'>'n"


   "(jde-gen-method-signature"
   "  \"public\""
   "  \"void\""
   "  \"actionPerformed\""
   "  \"ActionEvent e\""
   " )"
   "'>"

   "(if jde-gen-k&r "
   " ()"
   " 'n)"
   "\"{\"'>'n"


   "\"System.exit(0);\" '>'n"
   "\"}\" '>'n"
   "\"});\" '>'n"
   "\"mb.add(menu);\" '>'n"
   "\"return mb;\" '>'n"
   "\"}\"'>'n "
   "\"} // \"'>"
   "(file-name-sans-extension (file-name-nondirectory buffer-file-name))"
   "'>'n")
  "*Template for JFC (Swing) application buffer.
Setting this variable defines a template instantiation
command `jde-gen-jfc-app', as a side-effect."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (defalias 'jde-gen-jfc-app
	    (tempo-define-template "java-jfc-app-buffer-template"
				   (jde-gen-read-template val)
				   nil
				   "Insert skeleton for a JFC app buffer"))
	  (set-default sym val)))

;;;###autoload
(defun jde-gen-jfc-app-buffer (file)
  "Create a new Java buffer containing a JFC application class.
This command inserts the class template generated by `jde-gen-jfc-app'.
It then moves the point to the location to the constructor."
  (interactive "F")
  (find-file file)
  (jde-gen-jfc-app)
  (beginning-of-buffer)
  (search-forward "{")
  (backward-char 1)
  (c-indent-exp)
  (tempo-forward-mark))


(defcustom jde-gen-buffer-templates
  (list (cons "Class" 'jde-gen-class)
	(cons "Interface" 'jde-gen-interface)
	(cons "Console" 'jde-gen-console)
	(cons "Swing App" 'jde-gen-jfc-app)
	(cons "Unit Test" 'jde-gen-junit-test-class))
  "*Specifies available autocode templates for creating buffers.
The value of this variable is an association list. The car of
each element specifies the template's title. The cdr specifies
a command that inserts the template into a buffer. See the function
`tempo-define-template' for any easy way to create a template
insertion command."
  :group 'jde-gen
  :type '(repeat
	  (cons :tag "Template"
		(string :tag "Title")
		(function :tag "Command")))
  :set '(lambda (sym val)
	  (let ((n (length val))
		(i 0))
	    (setq jde-gen-buffer-template-names (list))
	    (while (< i n)
	      (setq jde-gen-buffer-template-names
		    (append
		     jde-gen-buffer-template-names
		     (list (cons (car (nth i val)) (1+ i)))))
	      (setq i (1+ i))))
	  (set-default sym val)))

;;;###autoload
(defun jde-gen-buffer (template file)
  "Create a new Java buffer containing a code template.
This command inserts the specified template at the beginning
of the buffer."
  (interactive
   (list (completing-read "Template: " jde-gen-buffer-template-names)
	 (read-file-name "File: ")))
  (find-file file)
  (funcall (cdr (assoc template jde-gen-buffer-templates)))
  (beginning-of-buffer)
  (search-forward "{")
  (backward-char 1)
  (c-indent-exp))


(defun jde-gen-lookup-and-capitalize (val)
  "If the given value (val) is the name of saved data, the data is
stripped of its prefix/suffix (see `jde-wiz-get-name') and it is
capitalized.  Otherwise, the given value is stripped and capitalized."
  (if (jde-gen-lookup-named val)
      (upcase-initials (jde-wiz-get-name (jde-gen-lookup-named val)))
    (upcase-initials (jde-wiz-get-name val))))


(defcustom jde-gen-section-comment-template
  '(
    "(p \"Comment: \" comment-line 'noinsert)"
    "(end-of-line) '&"
    "'n"
    "\"// \" (s comment-line) '>'n'n'>"
    )
  "*Template for generating a section comment. Used as an introductory
comment to a source code section by code generating functions."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (defalias 'jde-gen-section-comment
	    (tempo-define-template
	     "section comment"
	     (jde-gen-read-template val)
	     nil
	     "Insert section comment."))
	  (set-default sym val)))


(defcustom jde-gen-get-set-var-template
  '(
    "(end-of-line) '&"
    "(P \"Variable type: \" type) \" \""
    "(P \"Variable name: \" name) \";\" '>'n 'n"
    ;;we begin by the getter
    "\"/**\" '>'n"
    "\"* Get the \" (jde-gen-lookup-and-capitalize 'name) \" value.\" '>'n"
    "\"* @return the \" (jde-gen-lookup-and-capitalize 'name) \" value.\" '>'n"
    "\"*/\" '>'n"
    "'>'"
    "(jde-gen-method-signature"
    "  \"public\""
    "  (jde-gen-lookup-named 'type)"
    "  (if (string= \"boolean\" (jde-gen-lookup-named 'type) ) "
    "    (concat \"is\" (jde-gen-lookup-and-capitalize 'name))"
    "   (concat \"get\" (jde-gen-lookup-and-capitalize 'name)))"
    "  nil"
    " )"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " '>'n)"
    "\"{\" '>'n"
    "\"return \" (s name) \";\" '>'n \"}\"'>'n"
    ;; leave a blank line with no indentation
    "'n"

    ;;we continue with the setter
    "\"/**\" '>'n"
    "\"* Set the \" (jde-gen-lookup-and-capitalize 'name) \" value.\" '>'n"
    "\"* @param new\" (jde-gen-lookup-and-capitalize 'name) \""
    "The new \" (jde-gen-lookup-and-capitalize 'name) \" value.\" '>'n"
    "\"*/\" '>'n"

    ;; name the method
    "'>'"
    "(jde-gen-method-signature "
    "  \"public\""
    "  \"void\""
    "  (concat \"set\" (jde-gen-lookup-and-capitalize 'name))"
    "  (concat (jde-gen-lookup-named 'type) \" new\" "
    "          (jde-gen-lookup-and-capitalize 'name))"
    " )"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " '>'n)"
    "\"{\" '>'n"

    "'>'\"this.\" (s name) \" = new\" (jde-gen-lookup-and-capitalize 'name) "
    "\";\" '>'n \"}\" '>'n'n'>"
    )
  "*Template for creating a get/set method pair.
Setting this variable defines a template instantiation
command `jde-gen-get-set', as a side-effect."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (defalias 'jde-gen-get-set
	    (tempo-define-template
	     "java-get-set-pair"
	     (jde-gen-read-template val)
	     nil
	     "Insert variable get-set method pair."))
	  (set-default sym val)))

(defcustom jde-gen-inner-class-template
  '(
    "(end-of-line) '& \"class \" (P \"Class name: \" class) '>"
    "\" \" (jde-gen-get-extend-class)"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " '>'n)"
    "\"{\" '>'n"

    "(jde-gen-method-signature"
    "  \"public\""
    "  nil"
    "  (jde-gen-lookup-named 'class)"
    "  nil"
    " )"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " '>'n)"
    "\"{\"'>'n"
    "\"}\"'>'n"

    "'>'n"
    "(jde-gen-get-interface-implementation)"
    "'>'n"

    "\"}\" '>'n'>"
    )
  "*Template that creates an empty private class at point."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (tempo-define-template "java-inner-class"
				 (jde-gen-read-template val)
				 nil
				 "Insert inner class.")
	  (defalias 'jde-gen-inner-class-internal
	    (list 'lambda (list)
		  (list 'tempo-template-java-inner-class)
		  (list 'jde-gen-insert-interface-implementation)))
	  (set-default sym val)))

(defun jde-gen-inner-class ()
  (interactive)
  (jde-gen-inner-class-internal)
  (goto-char (scan-lists (point) -1 0))
  (c-indent-exp))

(defcustom jde-gen-action-listener-template
  '(
    "'& (P \"Component name: \")"
    "\".addActionListener(\" jde-gen-method-signature-padding-2 "
    "\"new ActionListener\" jde-gen-method-signature-padding-1 \"()\" '>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " jde-gen-method-signature-padding-3"
    " 'n)"

    "\"{\"'>'n"

    "(jde-gen-method-signature"
    "  \"public\""
    "  \"void\""
    "  \"actionPerformed\""
    "  \"ActionEvent e\""
    " )"
    "'>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"
    "\"}\"'>'n"

    "\"}\" jde-gen-method-signature-padding-2 \");\"'>'n'>'n'>"
    )
  "*Template for generating an action listener.
Setting this variable defines a template instantiation
command, `jde-gen-action-listener', as a side-effect."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (defalias 'jde-gen-action-listener
	    (tempo-define-template
	     "java-action-listener"
	     (jde-gen-read-template val)
	     nil
	     "Insert skeleton action listener."))
	  (set-default sym val)))

(defcustom jde-gen-window-listener-template
  '(
    "(end-of-line) '& (P \"Window name: \")"
    "\".addWindowListener(\" jde-gen-method-signature-padding-2 "
    "\"new WindowAdapter\" jde-gen-method-signature-padding-1 \"()\"'>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " jde-gen-method-signature-padding-3"
    " 'n)"
    "\"{\"'>'n"

    "(jde-gen-method-signature"
    "  \"public\""
    "  \"void\""
    "  \"windowActivated\""
    "  \"WindowEvent e\""
    " )"
    "'>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"
    "\"}\"'>'n"

    "(jde-gen-method-signature"
    "  \"public\""
    "  \"void\""
    "  \"windowClosed\""
    "  \"WindowEvent e\""
    " )"
    "'>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"
    "\"}\"'>'n"

    "(jde-gen-method-signature"
    "  \"public\""
    "  \"void\""
    "  \"windowClosing\""
    "  \"WindowEvent e\""
    " )"
    "'>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"
    " \"System.exit(0);\" '>'n \"}\""
    "'>'n"

    "(jde-gen-method-signature"
    "  \"public\""
    "  \"void\""
    "  \"windowDeactivated\""
    "  \"WindowEvent e\""
    " )"
    "'>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"
    "\"}\"'>'n"

    "(jde-gen-method-signature"
    "  \"public\""
    "  \"void\""
    "  \"windowDeiconified\""
    "  \"WindowEvent e\""
    " )"
    "'>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"
    "\"}\"'>'n"

    "(jde-gen-method-signature"
    "  \"public\""
    "  \"void\""
    "  \"windowIconified\""
    "  \"WindowEvent e\""
    " )"
    "'>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"
    "\"}\"'>'n"

    "(jde-gen-method-signature"
    "  \"public\""
    "  \"void\""
    "  \"windowOpened\""
    "  \"WindowEvent e\""
    " )"
    "'>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"
    "\"}\"'>'n"

    "\"}\" jde-gen-method-signature-padding-2 \");\" '>'n'>"
    )
  "*Template for generating a window listener.
Setting this variable defines a template instantiation
command, `jde-gen-window-listener', as a side-effect."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (defalias 'jde-gen-window-listener
	    (tempo-define-template
	     "java-window-listener"
	     (jde-gen-read-template val)
	     nil
	     "Insert skeleton window listener."))
	  (set-default sym val)))



(defcustom jde-gen-mouse-listener-template
  '(
    "(end-of-line) '& (P \"Component name: \")"
    "\".addMouseListener(\" jde-gen-method-signature-padding-2 "
    "\"new MouseAdapter\" jde-gen-method-signature-padding-1 \"()\" '>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " jde-gen-method-signature-padding-3"
    " 'n)"
    "\"{\"'>'n "

    "(jde-gen-method-signature"
    "  \"public\""
    "  \"void\""
    "  \"mouseClicked\""
    "  \"MouseEvent e\""
    " )"
    "'>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"
    "\"}\"'>'n"

    "(jde-gen-method-signature"
    "  \"public\""
    "  \"void\""
    "  \"mouseEntered\""
    "  \"MouseEvent e\""
    " )"
    "'>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"
    "\"}\"'>'n"

    "(jde-gen-method-signature"
    "  \"public\""
    "  \"void\""
    "  \"mouseExited\""
    "  \"MouseEvent e\""
    " )"
    "'>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"
    "\"}\"'>'n"

    "(jde-gen-method-signature"
    "  \"public\""
    "  \"void\""
    "  \"mousePressed\""
    "  \"MouseEvent e\""
    " )"
    "'>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"
    "\"}\"'>'n"

    "(jde-gen-method-signature"
    "  \"public\""
    "  \"void\""
    "  \"mouseReleased\""
    "  \"MouseEvent e\""
    " )"
    "'>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"
    "\"}\"'>'n"

    "\"}\" jde-gen-method-signature-padding-2 \");\"'>'n'>"
    )
  "*Template for generating a mouse listener.
Setting this variable defines a template instantiation
command, `jde-gen-mouse-listener', as a side-effect."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (defalias 'jde-gen-mouse-listener
	    (tempo-define-template
	     "java-mouse-listener"
	     (jde-gen-read-template val)
	     nil
	     "Insert skeleton mouse listener."))
	  (set-default sym val)))

(defcustom jde-gen-mouse-motion-listener-template
  '(
    "(end-of-line) '& (P \"Component name: \")"
    "\".addMouseMotionListener(\" jde-gen-method-signature-padding-2 "
    "\"new MouseMotionAdapter\" jde-gen-method-signature-padding-1 \"()\" '>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " jde-gen-method-signature-padding-3"
    " 'n)"
    "\"{\"'>'n"

    "(jde-gen-method-signature"
    "  \"public\""
    "  \"void\""
    "  \"mouseDragged\""
    "  \"MouseEvent e\""
    " )"
    "'>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"
    "\"}\"'>'n"

    "(jde-gen-method-signature"
    "  \"public\""
    "  \"void\""
    "  \"mouseMoved\""
    "  \"MouseEvent e\""
    " )"
    "'>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"
    "\"}\"'>'n"

    "\"}\" jde-gen-method-signature-padding-2 \");\"'>'n'>"
    )
  "*Template for generating a mouse listener.
Setting this variable defines a template instantiation
command, `jde-gen-mouse-motion-listener', as a side-effect."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (defalias 'jde-gen-mouse-motion-listener
	    (tempo-define-template
	     "java-mouse-motion-listener"
	     (jde-gen-read-template val)
	     nil
	     "Insert skeleton mouse motion listener."))
	  (set-default sym val)))



(defcustom jde-gen-to-string-method-template
  '(
    "(end-of-line) '&"
    "\"/**\" '>'n"
    "\"* Retrieve a String representation of this object\" '>'n"
    "\"* \" '>'n"
    "\"* @return a <code>String</code> representation of this object.\" '>'n"
    "\"* @see Object#toString()\" '>'n"   ;; unnecessary, but is a good reminder
    "\"*/\" '>'n"
    "(jde-gen-method-signature"
    "  \"public\""
    "  \"String\""
    "  \"toString\""
    "  nil"
    " )"
    "'>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"
    "'>'r'n"
    "\"}\"'>'n'>"
    )
  "*Template for generating a toString method.
Setting this variable defines a template instantiation
command, `jde-gen-to-string-method', as a side-effect."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (defalias 'jde-gen-to-string-method
	    (tempo-define-template
	     "to-string-method"
	     (jde-gen-read-template val)
	     nil
	     "Insert skeleton toString method."))
	  (set-default sym val)))

(defcustom  jde-gen-println
  '(
    "(end-of-line) '&"
    "\"System.out.println(\" (P \"Print out: \") \");\" '>'n'>"
    )
  "*Template for generating a System.out.println statement."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (defalias 'jde-gen-println
	    (tempo-define-template
	     "println"
	     (jde-gen-read-template val)
	     nil
	     "Insert println statement."))
	  (set-default sym val)))

(defcustom  jde-gen-beep
  '(
    "(end-of-line) '&"
    "\"Toolkit.getDefaultToolkit().beep();\"'>'n'>"
    )
  "*Template for generating a Toolkit.getDefaultToolkit().beep() statement."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (defalias 'jde-gen-beep
	    (tempo-define-template
	     "beep statement"
	     (jde-gen-read-template val)
	     nil
	     "Insert beep statement."))
	  (set-default sym val)))

(defcustom  jde-gen-property-change-support
  '(
    "(end-of-line) '&"

    "\"protected PropertyChangeSupport pcs =  new PropertyChangeSupport(this);\" '>'n '>'n"


    "\"/**\" '>'n"
    "\"* Adds a PropertyChangeListener to the listener list.\" '>'n"
    "\"* The listener is registered for all properties.\" '>'n"
    "\"*\" '>'n"
    "\"* @param listener The PropertyChangeListener to be added\" '>'n"
    "\"*/\" '>'n"

    "(jde-gen-method-signature"
    "  \"public\""
    "  \"void\""
    "  \"addPropertyChangeListener\""
    "  \"PropertyChangeListener listener\""
    " )"
    "'>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"

    "\"pcs.addPropertyChangeListener(listener);\" '>'n \"}\" '>'n '>'n"

    "\"/**\" '>'n"
    "\"* Removes a PropertyChangeListener from the listener list.\" '>'n"
    "\"* This removes a PropertyChangeListener that was registered\" '>'n"
    "\"* for all properties.\" '>'n"
    "\"*\" '>'n "
    "\"* @param listener The PropertyChangeListener to be removed\" '>'n"
    "\"*/\" '>'n"

    "(jde-gen-method-signature"
    "  \"public\""
    "  \"void\""
    "  \"removePropertyChangeListener\""
    "  \"PropertyChangeListener listener\""
    " )"
    "'>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"

    "'>\"pcs.removePropertyChangeListener(listener);\" '>'n \"}\" '>'n '>'n"

    "\"/**\" '>'n"
    "\"* Adds a PropertyChangeListener for a specific property.\" '>'n"
    "\"* The listener will be invoked only when a call on firePropertyChange\" '>'n"
    "\"* names that specific property.\" '>'n"
    "\"*\" '>'n"
    "\"* @param propertyName The name of the property to listen on\" '>'n"
    "\"* @param listener The PropertyChangeListener to be added\" '>'n"
    "\"*/\" '>'n"

    "(jde-gen-method-signature"
    "  \"public\""
    "  \"void\""
    "  \"addPropertyChangeListener\""
    "  \"String propertyName, PropertyChangeListener listener\""
    " )"
    "'>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"

    "'> \"pcs.addPropertyChangeListener(propertyName, listener);\" '>'n"
    "\"}\" '>'n '>'n"

    "\"/**\" '>'n"
    "\"* Removes a PropertyChangeListener for a specific property.\" '>'n"
    "\"*\" '>'n"
    "\"* @param propertyName The name of the property that was listened on\" '>'n"
    "\"* @param listener The PropertyChangeListener to be removed\"'>'n"
    "\"*/\" '>'n"

    "(jde-gen-method-signature"
    "  \"public\""
    "  \"void\""
    "  \"removePropertyChangeListener\""
    "  \"String propertyName, PropertyChangeListener listener\""
    " )"
    "'>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"

    "'> \"pcs.removePropertyChangeListener(propertyName, listener);\" '>'n"
    "\"}\" '>'n '>'n"

    "\"/**\" '>'n"
    "\"* Reports a bound property update to any registered listeners. \" '>'n"
    "\"* No event is fired if old and new are equal and non-null.\" '>'n"
    "\"*\" '>'n"
    "\"* @param propertyName The programmatic name of the property\" '>'n"
    "\"*                     that was changed\" '>'n"
    "\"* @param oldValue The old value of the property\" '>'n"
    "\"* @param newValue The new value of the property.\" '>'n \"*/\" '>'n"

    "(jde-gen-method-signature"
    "  \"public\""
    "  \"void\""
    "  \"firePropertyChange\""
    "  \"String propertyName, Object oldValue, Object newValue\""
    " )"
    "'>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"

    "'> \"pcs.firePropertyChange(propertyName, oldValue, newValue);\" '>'n"
    "\"}\" '>'n '>'n"

    "\"/**\" '>'n"
    "\"* Reports a bound property update to any registered listeners. \" '>'n"
    "\"* No event is fired if old and new are equal and non-null.\" '>'n"
    "\"* This is merely a convenience wrapper around the more general\" '>'n"
    "\"* firePropertyChange method that takes Object values.\" '>'n"
    "\"* No event is fired if old and new are equal and non-null.\" '>'n"
    "\"*\" '>'n"
    "\"* @param propertyName The programmatic name of the property\" '>'n"
    "\"*                     that was changed\" '>'n"
    "\"* @param oldValue The old value of the property\" '>'n"
    "\"* @param newValue The new value of the property.\" '>'n \"*/\" '>'n"

    "(jde-gen-method-signature"
    "  \"public\""
    "  \"void\""
    "  \"firePropertyChange\""
    "  \"String propertyName, int oldValue, int newValue\""
    " )"
    "'>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"

    "'> \"pcs.firePropertyChange(propertyName, oldValue, newValue);\" '>'n"
    "\"}\" '>'n '>'n"

    "\"/**\" '>'n"
    "\"* Reports a bound property update to any registered listeners. \" '>'n"
    "\"* No event is fired if old and new are equal and non-null.\" '>'n"
    "\"* This is merely a convenience wrapper around the more general\" '>'n"
    "\"* firePropertyChange method that takes Object values.\" '>'n"
    "\"* No event is fired if old and new are equal and non-null.\" '>'n"
    "\"*\" '>'n"
    "\"* @param propertyName The programmatic name of the property\" '>'n"
    "\"*                     that was changed\" '>'n"
    "\"* @param oldValue The old value of the property\" '>'n"
    "\"* @param newValue The new value of the property.\" '>'n \"*/\" '>'n"

    "(jde-gen-method-signature"
    "  \"public\""
    "  \"void\""
    "  \"firePropertyChange\""
    "  \"String propertyName, boolean oldValue, boolean newValue\""
    " )"
    "'>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"

    "'> \"pcs.firePropertyChange(propertyName, oldValue, newValue);\" '>'n"
    "\"}\" '>'n '>'n"

    "\"/**\" '>'n"
    "\"* Fires an existing PropertyChangeEvent to any registered listeners.\" '>'n"
    "\"* No event is fired if the given event's old and new values are\"'>'n"
    "\"* equal and non-null. \" '>'n"
    "\"*\" '>'n"
    "\"* @param evt The PropertyChangeEvent object.\" '>'n\"*/\" '>'n"

    "(jde-gen-method-signature"
    "  \"public\""
    "  \"void\""
    "  \"firePropertyChange\""
    "  \"PropertyChangeEvent evt\""
    " )"
    "'>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"

    "'> \"pcs.firePropertyChange(evt);\" '>'n \"}\" '>'n '>'n"

    "\"/**\" '>'n"
    "\"* Checks if there are any listeners for a specific property.\" '>'n"
    "\"*\" '>'n"
    "\"* @param evt The PropertyChangeEvent object.\" '>'n"
    "\"* @return <code>true</code>if there are one or more listeners\"'>'n"
    "\"*             for the given property\" '>'n"
    "\"*/\" '>'n"

    "(jde-gen-method-signature"
    "  \"public\""
    "  \"boolean\""
    "  \"hasListeners\""
    "  \"String propertyName\""
    " )"
    "'>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"

    "'> \"return pcs.hasListeners(propertyName);\" '>'n \"}\" '>'n '>'n'>"
    )
  "*Template for adding property change support to a class."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (defalias 'jde-gen-property-change-support
	    (tempo-define-template
	     "property change support template"
	     (jde-gen-read-template val)
	     nil
	     "Insert property change support template."))
	  (set-default sym val)))

(defcustom  jde-gen-listener-registry
  '(
    "(p \"Listener class (fully qualified): \" listenerFQN 'noinsert)"
    "(tempo-save-named 'listener-class "
    " (replace-in-string (tempo-lookup-named 'listenerFQN)"
    "                    \"[^\\\\.]+\\\\.\" \"\"))"
    "(tempo-save-named 'listener-vector "
    " (concat (jde-wiz-downcase-initials (tempo-lookup-named 'listener-class))"
    "         \"s\"))"

    "(end-of-line) '&"
    "\"protected Vector \" (s listener-vector) \" = new Vector();\" '>'n '>'n"

    "\"/**\" '>'n"
    "\"* The method <CODE>add\" (s listener-class)"
    "\"</CODE> allows to \" '>'n"
    "\"* add a new <CODE>\" (s listener-class) \"</CODE>\" '>'n"
    "\"* that will be notified of incoming events.\" '>'n"
    "\"*\" '>'n"
    "\"* @see \" (s listenerFQN) '>'n"
    "\"*/\" '>'n"

    "(jde-gen-method-signature"
    "  \"public\""
    "  \"void\""
    "  (concat \"add\" (tempo-lookup-named 'listener-class))"
    "  (concat (tempo-lookup-named 'listener-class) \" l\")"
    " )"
    "'>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"

    "(s listener-vector) \".addElement(l);\" '> 'n"

    "\"}\" '>'n '>'n"

    "\"/**\" '>'n"
    "\"* The method <CODE>remove\" (s listener-class)"
    "\"</CODE> allows to \" '>'n"
    "\"* remove a <CODE>\" (s listener-class) \"</CODE> that was\" '>'n"
    "\"* previously registered to be notified of incoming events.\" '>'n"
    "\"*\" '>'n"
    "\"* @see \" (s listenerFQN) '>'n"
    "\"*/\" '>'n"

    "(jde-gen-method-signature"
    "  \"public\""
    "  \"void\""
    "  (concat \"remove\" (tempo-lookup-named 'listener-class))"
    "  (concat (tempo-lookup-named 'listener-class) \" l\")"
    " )"
    "'>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"

    "(s listener-vector) \".removeElement(l);\" '> 'n"

    "\"}\" '>'n '>'n"
    )
  "*Template for adding a registry for a class of listeners."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (defalias 'jde-gen-listener-registry
	    (tempo-define-template
	     "listener registry template"
	     (jde-gen-read-template val)
	     nil
	     "Insert listener registry template."))
	  (set-default sym val)))

(defcustom  jde-gen-event-source-fire-method-template
  '(
    "(p \"Listener class (fully qualified): \" listenerFQN 'noinsert)"
    "(p \"Listener method name: \" method-name 'noinsert)"
    "(p \"Method name: \" return-type 'noinsert)"
    "(p \"Method name: \" params 'noinsert)"
    "(tempo-save-named 'listener-class "
    " (replace-in-string (tempo-lookup-named 'listenerFQN)"
    "                    \"[^\\\\.]+\\\\.\" \"\"))"
    "(tempo-save-named 'listener-vector "
    " (concat (jde-wiz-downcase-initials (tempo-lookup-named 'listener-class))"
    "         \"s\"))"
    "(tempo-save-named 'fire-method "
    " (concat \"fire\" (upcase-initials (tempo-lookup-named 'method-name))))"
    "(tempo-save-named 'param-ids "
    " (jde-gen-extract-ids-from-params (tempo-lookup-named 'params)))"

    "(end-of-line) '&"
    "\"/**\" '>'n"
    "\"* The method <CODE>\" (s fire-method)"
    "\"</CODE> is used \" '>'n"
    "\"* to call the <CODE>\" (s method-name) \"</CODE> method of\" '>'n"
    "\"* every previously registered <CODE>\" (s listener-class) \"</CODE>.\" '>'n"
    "\"*\" '>'n"
    "\"* @see \" (s listenerFQN) '>'n"
    "\"*/\" '>'n"

    "(jde-gen-method-signature"
    "  \"public\""
    "  (tempo-lookup-named 'return-type)"
    "  (tempo-lookup-named 'fire-method)"
    "  (tempo-lookup-named 'params)"
    " )"
    "'>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"

    " \"for(int i = 0; i < \" (s listener-vector) \".size(); i++)\" '>"
    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"
    "\"((\" (s listener-class) \")\" (s listener-vector)"
    "\".elementAt (i)).\" (s method-name) \" (\" (s param-ids) \");\" '> 'n"
    "\"}\" '>'n"

    "\"}\" '>'n '>'n"
    )
  "*Template for adding a registry for a class of listeners."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (defalias 'jde-gen-event-source-fire-method
	    (tempo-define-template
	     "event source fire method template"
	     (jde-gen-read-template val)
	     nil
	     "Insert event source fire method template."))
	  (set-default sym val)))

;; (makunbound 'jde-gen-enity-bean-template)
(defcustom jde-gen-entity-bean-template
  '(
    "(jde-import-insert-imports-into-buffer "
    "  (list \"javax.ejb.*\""
    "        \"java.rmi.RemoteException\"))"
    "'>"

    "(jde-gen-method-signature"
    "   \"public\""
    "  \"void\""
    "  \"ejbActivate\""
    "  nil"
    "  \"RemoteException\""
    " )"
    "'>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"
    "\"}\"'>'n 'n"

    "'>"
    "(jde-gen-method-signature"
    "  \"public\""
    "  \"void\""
    "  \"ejbPassivate\""
    "  nil"
    "  \"RemoteException\""
    " )"

    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"
    "\"}\"'>'n 'n"

    "'>"
    "(jde-gen-method-signature"
    "  \"public\""
    "  \"void\""
    "  \"ejbLoad\""
    "  nil"
    "  \"RemoteException\""
    " )"

    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"
    "\"}\"'>'n 'n"

    "'>"
    "(jde-gen-method-signature"
    "  \"public\""
    "  \"void\""
    "  \"ejbStore\""
    "  nil"
    "  \"RemoteException\""
    " )"

    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"
    "\"}\"'>'n 'n"

    "'>"
    "(jde-gen-method-signature"
    "  \"public\""
    "  \"void\""
    "  \"ejbRemove\""
    "  nil"
    "  \"RemoteException\""
    " )"

    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"
    "\"}\"'>'n 'n"

    "'>"
    "(jde-gen-method-signature"
    "  \"public\""
    "  \"void\""
    "  \"setEntityContext\""
    "  \"EntityContext ctx\""
    "  \"RemoteException\""
    " )"

    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"
    "\"}\"'>'n 'n"

    "'>"
    "(jde-gen-method-signature"
    "  \"public\""
    "  \"void\""
    "  \"unsetEntityContext\""
    "  nil"
    "  \"RemoteException\""
    " )"

    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"
    "\"}\"'>'n 'n '>"
    )
  "*Template that creates an empty implementation of an EJB Entity Bean."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (defalias 'jde-gen-entity-bean
	    (tempo-define-template
	     "ejb-entity-bean"
	     (jde-gen-read-template val)
	     nil
	     "Adds an implementation of the EJB Entity Bean interface to the 
class in the current buffer at the current point in the buffer. Before invoking
this command,  position point at the point in the buffer where you want the first 
Entity Bean method to appear. Use `jde-ejb-entity-bean-buffer' to create a complete
skeleton entity bean implementation from scratch."))
	  (set-default sym val)))


;; (makunbound 'jde-gen-session-bean-template)
(defcustom jde-gen-session-bean-template
  '(
    "(jde-import-insert-imports-into-buffer "
    "  (list \"javax.ejb.*\""
    "        \"java.rmi.RemoteException\"))"
    "'>"

    "(jde-wiz-update-implements-clause \"SessionBean\")"
    "'>"

    "(jde-gen-method-signature"
    "  \"public\""
    "  \"void\""
    "  \"ejbActivate\""
    "  nil"
    "  \"RemoteException\""
    " )"
    "'>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"
    "\"}\"'>'n 'n"

    "(jde-gen-method-signature"
    "  \"public\""
    "  \"void\""
    "  \"ejbPassivate\""
    "  nil"
    "  \"RemoteException\""
    " )"
    "'>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"
    "\"}\"'>'n 'n"

    "(jde-gen-method-signature"
    "  \"public\""
    "  \"void\""
    "  \"ejbRemove\""
    "  nil"
    "  \"RemoteException\""
    " )"
    "'>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"
    "\"}\"'>'n 'n"

    "(jde-gen-method-signature"
    "  \"public\""
    "  \"void\""
    "  \"setSessionContext\""
    "  \"SessionContext ctx\""
    "  \"RemoteException\""
    " )"
    "'>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"
    "\"}\"'>'n 'n"

    "(jde-gen-method-signature"
    "  \"public\""
    "  \"void\""
    "  \"unsetSessionContext\""
    "  nil"
    "  \"RemoteException\""
    " )"
    "'>"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"
    "\"}\"'>'n 'n"
    "'>"
    )
  "*Template that creates an empty implementation of an EJB Session Bean."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (defalias 'jde-gen-session-bean
	    (tempo-define-template
	     "ejb-session-bean"
	     (jde-gen-read-template val)
	     nil
	     "Adds an implementation of the EJB Session Bean interface to the 
class in the current buffer at the current point in the buffer. Before invoking
this command,  position point at the point in the buffer where you want the first 
Session Bean method to appear. Use `jde-ejb-session-bean-buffer' to create a complete
skeleton session bean implementation from scratch."))
	  (set-default sym val)))


(defcustom jde-gen-method-template
  '(
    "(p \"Method modifiers: \" modifiers 'noinsert)"
    "(p \"Method return type: \" return-type 'noinsert)"
    "(p \"Method name: \" name 'noinsert)"
    "(p \"Method parameters: \" parameters 'noinsert)"
    "(p \"Method exceptions: \" exceptions 'noinsert)"
    "(p \"Method body: \" default-body 'noinsert)"
    "(jde-gen-delete-preceding-whitespace) 'n 'n '> 'p"
    "(jde-gen-method-signature"
    "  (tempo-lookup-named 'modifiers)"
    "  (tempo-lookup-named 'return-type)"
    "  (tempo-lookup-named 'name)"
    "  (tempo-lookup-named 'parameters)"
    "  (tempo-lookup-named 'exceptions)"
    " )"
    "'> 'p"

    ;;we open the bracket according to k&r style or not
    "(if jde-gen-k&r "
    " ()"
    " 'n)"
    "\"{\"'>'n"
    "(s default-body) '>'r'n"
    "\"}\"'>'n'>"
    "(progn (tempo-backward-mark) (beginning-of-line)"
    "(jde-javadoc-autodoc-at-line) nil)"
    )
  "*Template for generating a method.
Setting this variable defines a template instantiation
command, `jde-gen-method', as a side-effect."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (defalias 'jde-gen-method
	    (tempo-define-template
	     "method"
	     (jde-gen-read-template val)
	     nil
	     "Insert skeleton method."))
	  (set-default sym val)))

(defcustom jde-gen-code-templates
  (list (cons "Get Set Pair" 'jde-gen-get-set)
	(cons "toString method" 'jde-gen-to-string-method)
	(cons "Action Listener" 'jde-gen-action-listener)
	(cons "Window Listener" 'jde-gen-window-listener)
	(cons "Mouse Listener" 'jde-gen-mouse-listener)
	(cons "Mouse Motion Listener" 'jde-gen-mouse-motion-listener)
	(cons "Inner Class" 'jde-gen-inner-class)
	(cons "println" 'jde-gen-println)
        (cons "beep" 'jde-gen-beep)
	(cons "property change support" 'jde-gen-property-change-support)
	(cons "EJB Entity Bean" 'jde-gen-entity-bean)
	(cons "EJB Session Bean" 'jde-gen-session-bean)
	)
  "*Specifies available autocode templates.
The value of this variable is an association list. The car of
each element specifies a template name. The cdr specifies
a command that inserts the template into a buffer. See the function
`tempo-define-template' for any easy way to create a template
insertion command."
  :group 'jde-gen
  :type '(repeat
	  (cons :tag "Template"
		(string :tag "Name")
		(function :tag "Command")))
  :set '(lambda (sym val)
	  (let ((n (length val))
		(i 0))
	    (setq jde-gen-template-names (list))
	    (while (< i n)
	      (setq jde-gen-template-names
		    (append
		     jde-gen-template-names
		     (list (cons (car (nth i val)) (1+ i)))))
	      (setq i (1+ i))))
	  (set-default sym val)))

(defun jde-gen-code (name)
  "Insert the code template specified by NAME at point.
The template must be one of those specified by the
variable `jde-gen-code-templates'."
  (interactive
   (list
    (completing-read "Template name: " jde-gen-template-names)))
  (funcall (cdr (assoc name jde-gen-code-templates))))



;;; Control Flow Templates
;;; Contributed by Eric D. Friedman <friedman@lmi.net>

(defvar jde-gen-abbrev-templates nil
  "List of abbreviation templates defined by
`jde-gen-define-abbrev-template'.")

(defun jde-gen-define-abbrev-template (abbrev template)
  "Defines a TEMPLATE that replaces ABBREV when you type ABBREV
in a JDE source buffer. TEMPLATE is a list of tempo template
elements. See `tempo-define-template' for information on
template elements. The resulting template is added to the
list bound to `jde-gen-abbrev-templates'. "
  (let ((template-name (concat "jde-gen-" abbrev)))
    (defalias (intern template-name)
      (tempo-define-template
       template-name
       template 
       abbrev 
       (format "JDE template for %s control flow abbreviation." abbrev)
       'jde-gen-abbrev-templates))))

(defcustom jde-gen-cflow-enable t
  "Enables abbreviations for Java control flow constructs."
  :group 'jde-gen
  :type 'boolean)


(defcustom jde-gen-comments t
  "*If no-nil, use comments, else do not use comments.
with comments:


      try
      {



      } catch (Exception e)
      {

      } // end of try-catch


witout comments:


      try
      {



      } catch (Exception e)
      {

      }


Setting this variable to t, uses comments in skeletons and templates."
  :group 'jde-gen
  :type 'boolean)

;; (makunbound 'jde-gen-cflow-if)
(defcustom jde-gen-cflow-if
  '(
    "(if (jde-parse-comment-or-quoted-p)"
    " '(l \"if\")"
    " '(l '> \"if\" jde-gen-conditional-padding-1 "
    "  \"(\" jde-gen-conditional-padding-2 (p \"if-clause: \" clause)"
    "        jde-gen-conditional-padding-2 \")\""
    " (if jde-gen-k&r "
    "   jde-gen-conditional-padding-3 "
    "  '>'n)"
    " \"{\"'>'n'>'r'n"
    " \"}\""
    " (if jde-gen-comments "
    " '(l \" // end of if (\" (s clause) \")\"))"
    " '>'n'> (c-indent-defun))"
    " )"
    )
  "Skeleton if statement. To insert the if statement at point, type if
and then space. Note that abbrev mode must be enabled. See
`jde-enable-abbrev-mode'"
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (jde-gen-define-abbrev-template
	   "if"
	   (jde-gen-read-template val))
	  (set-default sym val)))

(defcustom jde-gen-cflow-else
  '(
    "(if (jde-parse-comment-or-quoted-p)"
    " '(l \"else\")"
    " '(l '> \"else\""
    " (if jde-gen-k&r "
    "   jde-gen-conditional-padding-3 "
    "  '>'n)"
    " \"{\"'>'n'>'r'n"
    " \"}\""
    " (if jde-gen-comments "
    "  '(l \" // end of else\"))"
    " '>'n'>)"
    " )"
    )
  "Skeleton else statement. To insert the statement at point, type else
and then space. Note that abbrev mode must be enabled. See
`jde-enable-abbrev-mode' for more information."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (jde-gen-define-abbrev-template
	   "else"
	   (jde-gen-read-template val))
	  (set-default sym val)))

(defcustom jde-gen-cflow-if-else
  '(
    "(if (jde-parse-comment-or-quoted-p)"
    " '(l \"ife\")"
    " '(l '> \"if\" jde-gen-conditional-padding-1 "
    " \"(\" jde-gen-conditional-padding-2 (p \"if-clause: \" clause)"
    "       jde-gen-conditional-padding-2 \")\""
    " (if jde-gen-k&r "
    "   jde-gen-conditional-padding-3 "
    "  '>'n)"
    " \"{\"'>'n'>'r'n"
    " \"}\" '>"
    " (if jde-gen-comments "
    "  '(l \" // end of if (\" (s clause) \")\" '>'n)"
    "  (if jde-gen-k&r "
    "   jde-gen-conditional-padding-3 "
    "   'n))"
    " '> \"else\""
    " (if jde-gen-k&r "
    "   jde-gen-conditional-padding-3"
    "  '>'n)"
    " \"{\"'>'n'>'r'n"
    " \"}\""
    " (if jde-gen-comments "
    "  '(l \" // end of if (\" (s clause) \") else\"))"
    " '>'n'>)"
    " )"
    )
  "Skeleton if-else statement. To insert the statement at point, type ife
and then space. Note that abbrev mode must be enabled. See
`jde-enable-abbrev-mode' for more information."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (jde-gen-define-abbrev-template
	   "ife"
	   (jde-gen-read-template val))
	  (set-default sym val)))

(defcustom jde-gen-cflow-else-if
  '(
    "(if (jde-parse-comment-or-quoted-p)"
    " '(l \"eif\")"
    " '(l '> \"else if\" jde-gen-conditional-padding-1 "
    " \"(\" jde-gen-conditional-padding-2 (p \"else-if-clause: \" clause) "
    "       jde-gen-conditional-padding-2 \")\""
    " (if jde-gen-k&r "
    "  jde-gen-conditional-padding-3 "
    " '>'n)"
    " \"{\"'>'n'>'r'n"
    " \"}\""
    " (if jde-gen-comments "
    "  '(l \" // end of else if (\" (s clause) \")\"))"
    " '>'n'>)"
    " )"
    )
  "Skeleton else-if statement. To insert the statement at point, type eif
and then space. Note that abbrev mode must be enabled. See
`jde-enable-abbrev-mode' for more information."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (jde-gen-define-abbrev-template
	   "eif"
	   (jde-gen-read-template val))
	  (set-default sym val)))

;; (makunbound 'jde-gen-cflow-while)
(defcustom jde-gen-cflow-while
  '(
    "(if (jde-parse-comment-or-quoted-p)"
    " '(l \"while\")"
    " '(l '> \"while\" jde-gen-conditional-padding-1 "
    " \"(\" jde-gen-conditional-padding-2 (p \"while-clause: \" clause) "
    "       jde-gen-conditional-padding-2 \")\""
    " (if jde-gen-k&r "
    "   jde-gen-conditional-padding-3 "
    "  '>'n)"
    " \"{\"'>'n'>'r'n"
    " \"}\""
    " (if jde-gen-comments "
    "  '(l \" // end of while (\" (s clause) \")\"))"
    " '>'n'> (c-indent-defun))"
    " )"
    )
  "Skeleton while statement. To insert the statement at point, type while
and then space. Note that abbrev mode must be enabled. See
`jde-enable-abbrev-mode' for more information."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (jde-gen-define-abbrev-template
	   "while"
	   (jde-gen-read-template val))
	  (set-default sym val)))

(defcustom jde-gen-cflow-for
  '(
    "(if (jde-parse-comment-or-quoted-p)"
    " '(l \"for\")"
    " '(l '> \"for\" jde-gen-conditional-padding-1 "
    " \"(\" jde-gen-conditional-padding-2 (p \"for-clause: \" clause) "
    "       jde-gen-conditional-padding-2 \")\""
    " (if jde-gen-k&r "
    "   jde-gen-conditional-padding-3 "
    "  'n)"
    " \"{\"'>'n'>'r'n"
    " \"}\""
    " (if jde-gen-comments "
    "  '(l \" // end of for (\" (s clause) \")\"))"
    " '>'n'>)"
    " )"
    )
  "Skeleton for statement. To insert the statement at point, type for
and then space. Note that abbrev mode must be enabled. See
`jde-enable-abbrev-mode' for more information."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (jde-gen-define-abbrev-template
	   "for"
	   (jde-gen-read-template val))
	  (set-default sym val)))

(defcustom jde-gen-cflow-for-i
  '(
    "(if (jde-parse-comment-or-quoted-p)"
    " '(l \"fori\")"
    " '(l '> \"for\" jde-gen-conditional-padding-1 "
    " \"(\" jde-gen-conditional-padding-2 \"int \" (p \"variable: \" var) "
    " \" = 0; \" (s var) \" < \" (p \"upper bound: \" ub) \"; \" (s var) \"++\""
    "       jde-gen-conditional-padding-2 \")\""
    " (if jde-gen-k&r "
    "   jde-gen-conditional-padding-3 "
    "  'n)"
    " \"{\"'>'n'>'r'n"
    " \"}\""
    " (if jde-gen-comments "
    "  '(l \" // end of for (int \" (s var) \" = 0; \""
    " (s var) \" < \" (s ub) \"; \" (s var) \"++)\"))"
    " '>'n'>)"
    " )"
    )
  "Skeleton for i statement. To insert the statement at point, type fori
and then space. Note that abbrev mode must be enabled. See
`jde-enable-abbrev-mode' for more information.

Note: `tempo-interactive' must be set to a non-nil value to be prompted
      for variable name and upper-bounds information."

  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (jde-gen-define-abbrev-template
	   "fori"
	   (jde-gen-read-template val))
	  (set-default sym val)))

(defcustom jde-gen-cflow-for-iter
  '(
    "(if (jde-parse-comment-or-quoted-p)"
    " '(l \"foriter\")"
    " '(l '> \"for\" jde-gen-conditional-padding-1 "
    " \"(\" jde-gen-conditional-padding-2 \"Iterator \" (p \"variable: \" var) "
    " \" = \" (p \"collection: \" coll) \".iterator(); \""
    " (s var) \".hasNext();\""
    "       jde-gen-conditional-padding-2 \")\""
    " (if jde-gen-k&r "
    "   jde-gen-conditional-padding-3 "
    "  'n)"
    " \"{\"'>'n'>'r"
    " (s var) \".next();\" '>'n'>"
    " \"}\""
    " (if jde-gen-comments "
    "  '(l \" // end of for (Iterator \" (s var) \" = \" (s coll)"
    "  \".iterator(); \" (s var) \".hasNext();)\"))"
    " '>'n'>)"
    " )"
    )
  "Skeleton for iterator statement. To insert the statement at point,
type foriter and then space.  Note that abbrev mode must be
enabled. See `jde-enable-abbrev-mode' for more information.

Note: `tempo-interactive' must be set to a non-nil value to be prompted
      for variable name and collection name information."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
          (jde-gen-define-abbrev-template
	   "foriter"
	   (jde-gen-read-template val))
          (set-default sym val)))

(defcustom jde-gen-cflow-switch
  '(
    "(if (jde-parse-comment-or-quoted-p)"
    " '(l \"switch\")"
    " '(l '> \"switch\" jde-gen-conditional-padding-1 "
    "  \"(\" jde-gen-conditional-padding-2 (p \"switch-condition: \" clause) "
    "        jde-gen-conditional-padding-2 \")\""
    " '>"
    " (if jde-gen-k&r "
    "   jde-gen-conditional-padding-3 "
    "  'n)"
    " \"{\"'>'n'"
    " \"case \" (p \"first value: \") \":\"'>'n'"
    " >'p'n"            ;; point will end up here
    " \"break;\"'>'n'n"
    " \"default:\"'>'n'>"
    " \"break;\"'>'n"
    " \"}\""
    " (if jde-gen-comments "
    "  '(l \" // end of switch (\" (s clause) \")\"))"
    " '>'n'>)"
    " )"
    )
  "Skeleton switch statement. To insert the statement at point, type switch
and then space. Note that abbrev mode must be enabled. See
`jde-enable-abbrev-mode' for more information."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (jde-gen-define-abbrev-template
	   "switch"
	   (jde-gen-read-template val))
	  (set-default sym val)))

(defcustom jde-gen-cflow-case
  '(
    "(if (jde-parse-comment-or-quoted-p)"
    " '(l \"case\")"
    " '(l 'n \"case \" (p \"value: \") \":\"'>'n"
    " '>'p'n"           ;; point will end up here
    " \"break;\"'>'n)"
    " )"
    )
  "Skeleton case statement. To insert the statement at point, type case
and then space. Note that abbrev mode must be enabled. See
`jde-enable-abbrev-mode' for more information."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (jde-gen-define-abbrev-template
	   "case"
	   (jde-gen-read-template val))
	  (set-default sym val)))

;; (makunbound 'jde-gen-cflow-try-catch)
(defcustom jde-gen-cflow-try-catch
  '(
    "(if (jde-parse-comment-or-quoted-p)"
    " '(l \"try\")"
    " '(l '> \"try \""
    " (if jde-gen-k&r "
    "  ()"
    "  'n)"
    " \"{\"'>'n'>'r'n"
    " \"}\" '>"
    " (if jde-gen-k&r "
    "   jde-gen-conditional-padding-3 "
    "  'n)"
    " \"catch\" jde-gen-conditional-padding-1 "
    " \"(\" jde-gen-conditional-padding-2 (p \"catch what: \" clause) \" e\""
    "       jde-gen-conditional-padding-2 \")\" '>"
    " (if jde-gen-k&r "
    "   jde-gen-conditional-padding-3 "
    "  'n)"
    " \"{\"'>'n'>'p'n"
    " \"}\""
    "  (if jde-gen-comments "
    " 	'(l \" // end of try-catch\"))"
    " '>'n'> (c-indent-defun))"
    " )"
    )
  "Skeleton try-catch statement. To insert the statement at point, type try
and then space. Note that abbrev mode must be enabled. See
`jde-enable-abbrev-mode' for more information."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (jde-gen-define-abbrev-template
	   "try"
	   (jde-gen-read-template val))
	  (set-default sym val)))

(defcustom jde-gen-cflow-catch
  '(
    "(if (jde-parse-comment-or-quoted-p)"
    " '(l \"catch\")"
    " '(l '> \"catch\" jde-gen-conditional-padding-1  "
    " \"(\" jde-gen-conditional-padding-2 (p \"catch what: \" clause) \" e\""
    "       jde-gen-conditional-padding-2 \")\""
    " (if jde-gen-k&r "
    "   jde-gen-conditional-padding-3 "
    "  'n)"
    " \"{\"'>'n'>'r'n"
    " \"}\""
    " (if jde-gen-comments "
    "  '(l \" // end of catch\"))"
    " '>'n'>)"
    " )"
    )
  "Skeleton catch statement. To insert the statement at point, type catch
and then space. Note that abbrev mode must be enabled. See
`jde-enable-abbrev-mode' for more information."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (jde-gen-define-abbrev-template
	   "catch"
	   (jde-gen-read-template val))
	  (set-default sym val)))

(defcustom jde-gen-cflow-try-finally
  '(
    "(if (jde-parse-comment-or-quoted-p)"
    " '(l \"try\")"
    " '(l '> \"try \""
    " (if jde-gen-k&r "
    "  ()"
    "  'n)"
    " \"{\"'>'n'>'r'n"
    " \"}\" '>"
    " (if jde-gen-k&r "
    "   jde-gen-conditional-padding-3 "
    "  'n)"
    " \"catch\" jde-gen-conditional-padding-1 "
    " \"(\" jde-gen-conditional-padding-2 (p \"catch what: \" clause) \" e\""
    "       jde-gen-conditional-padding-2 \")\" '>"
    " (if jde-gen-k&r "
    "   jde-gen-conditional-padding-3 "
    "  'n)"
    " \"{\"'>'n'>'r'n"
    " \"}\" '> "
    " (if jde-gen-k&r "
    "   jde-gen-conditional-padding-3 "
    "  'n)"
    " \"finally\" '>"
    " (if jde-gen-k&r "
    "   jde-gen-conditional-padding-3 "
    "  'n)"
    " \"{\"'>'n'>'r'n"
    " \"}\""
    " (if jde-gen-comments "
    "  '(l \" // end of try-finally\"))"
    " '>'n'>)"
    " )"
    )
  "Skeleton try-catch-finally statement. To insert the statement at point, type
tryf and then space. Note that abbrev mode must be enabled. See
`jde-enable-abbrev-mode' for more information."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (jde-gen-define-abbrev-template
	   "tryf"
	   (jde-gen-read-template val))
	  (set-default sym val)))

(defcustom jde-gen-cflow-finally
  '(
    "(if (jde-parse-comment-or-quoted-p)"
    " '(l \"finally\")"
    " '(l '> \"finally\""
    " (if jde-gen-k&r "
    "   jde-gen-conditional-padding-3 "
    "  'n)"
    " \"{\"'>'n'>'r'n"
    " \"}\""
    " (if jde-gen-comments "
    "  '(l \" // end of finally\"))"
    " '>'n'>)"
    " )"
    )
  "Skeleton finally statement. To insert the statement at point, type finally
and then space. Note that abbrev mode must be enabled. See
`jde-enable-abbrev-mode' for more information."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (jde-gen-define-abbrev-template
	   "finally"
	   (jde-gen-read-template val))
	  (set-default sym val)))


(defcustom jde-gen-cflow-main
  '(
    "(if (jde-parse-comment-or-quoted-p)"
    " '(l \"main\")"
    " '(l '> "
    " (jde-gen-method-signature"
    "   \"public static\""
    "   \"void\""
    "   \"main\""
    "   \"String[] args\""
    "  )"

    " (if jde-gen-k&r "
    "  ()"
    "  'n)"
    " \"{\"'>'n'>'r'n'>"
    " \"}\""
    " (if jde-gen-comments "
    "  '(l \" // end of main()\"))"
    " '>'n'>)"
    " )"
    )
  "Skeleton main method. To insert the method at point, type main
and then space. Note that abbrev mode must be enabled. See
`jde-enable-abbrev-mode' for more information."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (jde-gen-define-abbrev-template
	   "main"
	   (jde-gen-read-template val))
	  (set-default sym val)))

;;JUnit templates
(defvar jde-gen-junit-class-regexp 
  "\\(.*\\)Test.java"
  "Regular expression to determine the class a unit test tests")

(defun jde-gen-junit-class (class-name)
  (let* ((start (string-match jde-gen-junit-class-regexp class-name))
         (end (match-end 1)))
    (if start
        (substring class-name start end)
      class-name)))

(defcustom jde-gen-junit-test-class-buffer-template
  (list
   "(funcall jde-gen-boilerplate-function)"
   "(jde-gen-get-package-statement)"
   "\"import junit.framework.Test;\" '>'n"
   "\"import junit.framework.TestCase;\" '>'n"
   "\"import junit.framework.TestSuite;\" '>'n"
   "'n"
   "(progn (require 'jde-javadoc) (jde-javadoc-insert-start-block))"
   "\" * \""
   "\" Unit Test for class \""
   "(jde-gen-junit-class (file-name-nondirectory buffer-file-name)) '>'n"
   "\" \" (jde-javadoc-insert-empty-line)"
   "\" \" (jde-javadoc-insert-empty-line)"
   "\" * Created: \" (current-time-string) '>'n"
   "\" \" (jde-javadoc-insert-empty-line)"
   "\" \" (jde-javadoc-insert 'tempo-template-jde-javadoc-author-tag)"
   "\" \" (jde-javadoc-insert 'tempo-template-jde-javadoc-version-tag)"
   "\" \" (jde-javadoc-insert 'tempo-template-jde-javadoc-end-block \"*/\")"
   "\"public class \""
   "(file-name-sans-extension (file-name-nondirectory buffer-file-name))"
   "\" extends TestCase \" "
    
   "(if jde-gen-k&r "
   "()"
   "'>'n)"
   "\"{\"'>'n"
   "'n" 
   
   " \" /** \" '>'n"
   " \"* Creates a new <code>\""
   "(file-name-sans-extension (file-name-nondirectory buffer-file-name))"
   "\"</code> instance.\" '>'n"
   " \"*\" '>'n" 
   " \"* @param name test name\" '>'n"
   " \"*/\"'>'n"

   "\"public \""
   "(file-name-sans-extension (file-name-nondirectory buffer-file-name))"
   "\" (String name)\""

   "(if jde-gen-k&r "
   "()"
   "'>'n)"
   "\"{\"'>'n"
   "\"super(name);\"'>'n"

   "\"}\"'>"
   "'>'n"
   "'n"
    
   "\"/**\" '>'n"
   "\"* @return a <code>TestSuite</code>\" '>'n"
   "\"*/\" '>'n"
   "\"public static TestSuite suite()\" '>" 
    
   "(if jde-gen-k&r "
   "()"
   "'>'n)"
   "\"{\"'>'n"

   "\"TestSuite suite = new TestSuite ();\" '>'n"
   "'>'n"
   "\"return suite;\" '>'n"
   "\"}\"'>'n'n"
   
   "\"/** \" '>'n"
   "\"* Entry point \" '>'n"
   "\"*/ \" '>'n"
   "\"public static void main(String[] args) \""
   "(if jde-gen-k&r "
   "()"
   "'>'n)"
   "\"{\"'>'n"
   "\"junit.textui.TestRunner.run(suite());\"'>'n"
   "\"}\"'>'n"
   
   "\"}\">"
   "\"// \""
   "(file-name-sans-extension (file-name-nondirectory buffer-file-name))"
   "'>'n")
  "*Template for new Java class.
Setting this variable defines a template instantiation
command `jde-gen-junit-test-class', as a side-effect."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (defalias 'jde-gen-junit-test-class
	    (tempo-define-template
             "java-junit-test-class-buffer-template"
             (jde-gen-read-template val)
             nil
             "Insert a generic JUnit test class buffer skeleton."))
	  (set-default sym val)))

;;;###autoload
(defun jde-gen-junit-test-class-buffer (file)
  "Create a buffer containing a skeleton unit test class having the same name as the
root name of the buffer. This command prompts you to enter the file name
of the test class. It assumes that the file name has the form CLASSTest.java
where CLASS is the name of the class to be tested, e.g., MyAppTest.java. Use 
`jde-gen-junit-add-test-to-suite' to add tests to the test suite. Use of
tests generated with this template requires the JUnit test framework. For
more information, see http://www.junit.org."
  (interactive "F")
  (find-file file)
  (jde-gen-junit-test-class)
  (beginning-of-buffer)
  (search-forward "{")
  (backward-char 1)
  (c-indent-exp)
  (tempo-forward-mark))

(defcustom jde-gen-add-test-to-suite
  '(
    "\"suite.addTest(new \""
    "(file-name-sans-extension (file-name-nondirectory buffer-file-name))"
    "\"(\\\"\" (P \"Test Name: \") \"\\\") \""
    
    "(if jde-gen-k&r "
    "()"
    "'>'n)"
    "\"{\"'>'n"
    
    "\"public void runTest()\""
    
    "(if jde-gen-k&r "
    "()"
    "'>'n)"
    "\"{\"'>'n"
    
    "(P \"Method to call: \") \"();\"'>'n"
    "\"}\"'>'n"
    "\"});\"'>'n"
   )
  "*Template for generating a test case for suite."
  :group 'jde-gen
  :type '(repeat string)
  :set '(lambda (sym val)
	  (defalias 'jde-gen-junit-add-test-to-suite
	    (tempo-define-template
	     "Adding JUnit test to suit"
	     (jde-gen-read-template val)
	     nil
	     "Insert JUnit test to suite."))
	  (set-default sym val)))

(defun jde-gen-load-abbrev-templates ()
  "Defines jde-mode abbrevs for the control flow templates."
  (loop for template in jde-gen-abbrev-templates do
	(define-abbrev
	  local-abbrev-table
	  (car template) "" (cdr template) 0)))

(defun jde-gen-try-catch-wrapper (beg end)
  "Wrap the region from BEG to END into a try/catch block.
BEG and END are modified so the region only contains complete lines."
  (interactive "r")
  (jde-gen-generic-try-wrapper beg end "catch"))

(defun jde-gen-try-finally-wrapper (beg end)
  "Wrap the region from BEG to END into a try/finally block.
BEG and END are modified so the region only contains complete lines."
  (interactive "r")
  (jde-gen-generic-try-wrapper beg end "finally"))

;;This code is a modified version of the method qflib-make-try-wrapper
(defun jde-gen-generic-try-wrapper (beg end expr)
  "Wrap the region from BEG to END into a try/EXPR block.
BEG and END are modified so the region only contains complete lines."
  (let ((to (make-marker))
	indent-region-function)
    (set-marker to
		(save-excursion
		  (goto-char end)
		  (if (and (bolp)
			   (not (= beg end)))
		      (point)
		    (end-of-line)
		    (1+ (point)))))
    (goto-char beg)
    (beginning-of-line)
    (insert "try")
    (if jde-gen-k&r
        (insert " ")
      (insert "\n"))
    (insert "{\n")
    (if jde-gen-k&r
        (forward-char -1)
      (forward-char -4))
    (indent-for-tab-command)
    (indent-region (point) to nil)
    (goto-char to)
    (insert "}")
    (if jde-gen-k&r
        (insert jde-gen-conditional-padding-3)
      (insert "\n"))
    (if (string= expr "catch")
          (insert (concat "catch" jde-gen-conditional-padding-1
                          "(" jde-gen-conditional-padding-2 " e"
                          jde-gen-conditional-padding-2 ")")) 
      (insert "finally"))
    (if jde-gen-k&r 
        (insert jde-gen-conditional-padding-2)
      (insert "\n"))
    (insert "{\n}")
    (if jde-gen-comments
          (insert " // end of try-"
                  (if (string= expr "catch")
                      expr
                    "finally")))
    (insert "\n")
    (indent-region (marker-position to) (point) nil)
    (goto-char to)
    (if (string= expr "catch")
        (search-forward "("))))

(provide 'jde-gen)

;; Unit Test Table for JDE Gen Methods
;; -----------------------------------

;;         Comm = jde-gen-comments|Comm t|Comm nil|Comm t  |Comm nil
;;          K&R = jde-gen-k&r     |K&R  t|K&R  t  |K&R  nil|K&R  nil
;; -------------------------------+------+--------+--------+--------
;; jde-gen-get-set	          |      |        |        |        
;; jde-gen-inner-class            |      |        |        |        
;; jde-gen-action-listener        |      |        |        |        
;; jde-gen-window-listener        |      |        |        |        
;; jde-gen-mouse-listener         |      |        |        |        
;; jde-gen-mouse-motion-listener  |      |        |        |        
;; jde-gen-to-string-method       |      |        |        |        
;; jde-gen-property-change-support|      |        |        |        
;; jde-gen-entity-bean            |      |        |        |        
;; jde-gen-session-bean           |      |        |        |        
;; jde-gen-cflow-if               |      |        |        |        
;; jde-gen-cflow-else             |      |        |        |        
;; jde-gen-cflow-if-else          |      |        |        |        
;; jde-gen-cflow-else-if          |      |        |        |        
;; jde-gen-cflow-while            |      |        |        |        
;; jde-gen-cflow-for              |      |        |        |        
;; jde-gen-cflow-for-I            |      |        |        |        
;; jde-gen-cflow-for-iter         |      |        |        |        
;; jde-gen-cflow-main             |      |        |        |        
;; jde-gen-cflow-switch           |      |        |        |        
;; jde-gen-cflow-case             |      |        |        |        
;; jde-gen-cflow-try-catch        |      |        |        |        
;; jde-gen-cflow-catch            |      |        |        |        
;; jde-gen-cflow-try-finally      |      |        |        |        
;; jde-gen-cflow-finally          |      |        |        |        

;; $Log: jde-gen.el,v $
;; Revision 1.59  2002/12/19 06:36:00  paulk
;; Changed to permit autoloading of jde-package.el file.
;;
;; Revision 1.58  2002/12/13 08:33:23  paulk
;; Enhance the following code templates
;;
;;   - jde-gen-interface-buffer-template
;;   - jde-gen-console-buffer-template
;;   - jde-gen-jfc-app-buffer-template
;;   - jde-gen-junit-test-class-buffer-template
;;
;; to use the following javadoc templates instead of
;; literal text:
;;
;;   - jde-javadoc-author-tag-template
;;   - jde-javadoc-version-tag-template
;;   - jde-javadoc-end-block-template
;;
;; Thanks to Peter Dobratz <dobratzp@ele.uri.edu>.
;;
;; Revision 1.57  2002/11/30 03:59:27  paulk
;; Changes the default value for jde-gen-class-buffer-template to use the
;; relevant tempo-template-jde-javadoc instead of literal text. This way
;; jde-gen-class-buffer reflects, for example, a change to
;; tempo-template-jde-javadoc-author-tag. Thanks to Peter Dobratz <dobratzp@ele.uri.edu>.
;;
;; Revision 1.56  2002/09/30 04:22:45  paulk
;; Expanded doc strings for jde-gen-entity/session-bean template commands to
;; explain their usage and to mention the existence of the more complete
;; jde-ejb-session/entity-buffer commands.
;;
;; Revision 1.55  2002/08/30 14:52:24  jslopez
;; Adds new method jde-gen-try-catch-wrapper and jde-gen-try-finally-wrapper.
;; You can select a region and call this method to wrap the area withing a
;; try/wrap.
;;
;; Revision 1.54  2002/08/11 14:03:53  paulk
;; Expanded doc for jde-gen-junit-test-class-buffer.
;;
;; Revision 1.53  2002/08/11 13:34:18  paulk
;; Add jde-gen-junit-test-class to list of class buffer templates.
;;
;; Revision 1.52  2002/06/21 06:42:20  paulk
;; Fixed bug in interface implementation code for class templates. Thanks to
;; Phillip Lord for reporting the bug.
;;
;; Revision 1.51  2002/05/31 19:02:27  mnl
;; Added new template to generate a new interface from scratch.
;;
;; Revision 1.50  2002/05/14 06:29:56  paulk
;; Enhances code generation wizards for implementing interfaces, abstract
;; classes, etc., to use customizable templates to generate skeleton methods
;; instead of hard-wired skeletons. Thanks to "Dr. Michael Lipp" <lipp@danet.de>
;; for proposing and implementing this improvement.
;;
;; Revision 1.49  2002/04/10 04:18:28  paulk
;; The JDEE now imports the superclasses of classes that it creates. Thanks
;; to "Timothy Babin"<tbabin@nortelnetworks.com>.
;;
;; Revision 1.48  2002/02/03 07:51:16  paulk
;; Fixes spacing bug in jde-gen-class-buffer-template.
;; Thanks to Petter Mahlen.
;;
;; Revision 1.47  2002/01/06 05:59:32  paulk
;; Changed the name of the println and toString templates to println
;; and to-string-method, respectively, thereby eliminating the
;; spaces that were in the previous names.
;;
;; Revision 1.46  2001/11/21 06:48:09  paulk
;; Fixed typo jde-gen-signature-padding-3. Thanks to David A. Ventimiglia.
;;
;; Revision 1.45  2001/11/05 05:01:51  paulk
;; Removed debugging form.
;;
;; Revision 1.44  2001/11/03 06:32:11  paulk
;; Restore JUnit templates.
;;
;; Revision 1.43  2001/11/02 06:59:58  paulk
;; Revamps code generation templates to make them more
;; consistent among themselves and with Java coding
;; conventions. Specific changes include:
;;
;;   - Generates consistent formats for method signatures.
;;   - Strips prefix and suffix of variable names (if configured)
;;   - Adds jde-gen-conditional-padding-1 to -3 to allow for
;;     configurability of the jde-gen-cflow-* templates
;;   - Changes many default templates to conform to Java code
;;     conventions
;;   - Fixes many subtle inconsistencies in templates when used
;;     with different values for jde-gen-comments and jde-gen-k&r
;;   - All templates now have consistent coding structure
;;     and indentation, as well as reduced insertion of white-space.
;;   - Modifies Javadoc comments to be less descriptive of implementation
;;     (e.g. instead of "Get the value of _flag", now is
;;     "Get the Flag value")
;;
;;     Thanks to Greg Fenton.
;;
;; Revision 1.42  2001/10/25 15:08:30  jslopez
;; Fixing hard coded strings in the javadoc
;; for the junit test class template.
;;
;; Revision 1.41  2001/10/25 02:58:34  jslopez
;; Fixing bug in JUnit template caused by a class
;; whose name does not contain the word Test.
;;
;; Revision 1.40  2001/10/25 02:14:45  jslopez
;; Adds JUnit supports
;; Add templates jde-gen-junit-test-class
;; and jde-gen-junit-add-test-to-suite
;;
;; Revision 1.39  2001/10/21 06:10:40  paulk
;; Extends the jde-gen-comments flag to all cflow templates.
;; Thanks to Robert Mecklenburg <mecklen@cimsoft.com>.
;;
;; Revision 1.38  2001/08/26 02:14:37  paulk
;; Fixed catch and tryf templates. Thanks to Javier Lopes.
;;
;; Revision 1.37  2001/06/30 12:35:28  paulk
;; Adds jde-gen-define-abbrev-template function.
;;
;; Revision 1.36  2001/06/10 04:07:05  paulk
;; Adds a control flow template for a for iterator loop. Thanks to Robert Mecklenburg <mecklen@cimsoft.com>.
;;
;; Revision 1.35  2001/06/05 05:33:10  paulk
;; Added else-f control flow template. Thanks to "Javier Lopez" <jlopez@cellexchange.com>.
;;
;; Revision 1.34  2001/06/05 04:54:49  paulk
;; Minor updates.
;;
;; Revision 1.33  2001/05/21 06:48:16  paulk
;; The class buffer template now generates skeletal implementations of interfaces that the class implements. Thanks to Javier Lopez for this enhancement.
;;
;; The inner class template now generates skeletal implementations of interfaces implemented by the class.
;;
;; Revision 1.32  2001/03/16 03:57:37  paulk
;; Fixed author line in javadoc comments. Thanks to Karel.Sprenger@compaq.com.
;;
;; Revision 1.31  2001/02/22 05:05:29  paulk
;; The class, console app, and Swing app templates now prompt you to enter a package name. The prompt includes a suggested package name based on the location of the current directory in the classpath.
;;
;; Revision 1.30  2001/01/19 04:28:09  paulk
;; Adds cflow expansions for try, catch, and finally. Thanks to Venkatesh Prasad Ranganath <rvprasad@ksu.edu> for these expansions.
;;
;; Revision 1.29  2000/12/18 05:22:45  paulk
;; *** empty log message ***
;;
;; Revision 1.28  2000/09/30 16:50:20  paulk
;; Correct type in jde-gen-cflow-enable.
;;
;; Revision 1.27  2000/09/07 04:37:28  paulk
;; Tweaked get-set pair template to indent correctly. Thanks to Lou Aloia <xlxa@rims.com> for this fix.
;;
;; Revision 1.26  2000/08/19 05:10:05  paulk
;; Adds jde-gen-cflow-enable variable.
;;
;; Revision 1.25  2000/07/23 02:44:44  paulk
;; Templates now indent correctly when inserted in a buffer. Thanks to el mono <mono@utp.edu.co> for this enhancement.
;;
;; Revision 1.24  2000/07/20 06:08:59  paulk
;; Extended K&R coding style to all templates. Thanks to Stephane Nicolas <s.nicolas@videotron.ca> for doing this.
;;
;; Revision 1.23  2000/06/28 02:46:48  paulk
;; Get/set pair template now generates correct method name for getting the value of boolean variables. Thanks to Stephane <s.nicolas@videotron.ca> for contributing this fix.
;;
;; Revision 1.22  2000/06/01 06:43:01  paulk
;; Added control flow templates contributed by Eric D. Friedman <friedman@lmi.net>.
;;
;; Revision 1.21  2000/02/01 05:22:51  paulk
;; Provide choice of coding styles for code generated by templates. Thanks to Jari Aalto for this enhancement.
;;
;; Revision 1.20  1999/09/23 03:23:41  paulk
;; Added code templates implementing EJB EntityBean and SessionBean
;; interfaces. Thanks to Brendan.Burns@tfsma-ims.tfn.com for contributing
;; the templates.
;;
;; Revision 1.19  1999/08/29 04:29:18  paulk
;; Patches provided by Michael Ernst <mernst@alum.mit.edu>
;;
;; Revision 1.18  1999/02/11 17:03:00  paulk
;; Updated the Swing application template to the JDK 1.2 Swing package
;; scheme and expanded the template to provide a menu and scrollable
;; canvas.
;;
;; Revision 1.17  1998/09/16 22:55:51  paulk
;; Added template for Java bean property change support.
;;
;; Revision 1.16  1998/09/13 00:34:28  paulk
;; Added a template for generating a System.out.println statement.
;;
;; Revision 1.15  1998/07/22 00:28:04  paulk
;; Modified class buffer creation templates to use tempo-marks
;; to mark initial position for user to insert code. Thanks
;; to David Hull <david.hull@trw.com> for suggesting this.
;;
;; Revision 1.14  1998/07/06 06:39:42  paulk
;; class buffer template now prompts for super class and
;; interface
;;
;; Revision 1.13  1998/07/06 05:06:13  paulk
;; Added boilerlate to other buffer generation templates.
;;
;; Revision 1.12  1998/07/01 03:54:40  paulk
;; Added source file boilerplate support.
;;
;; Revision 1.11  1998/06/27 03:04:46  paulk
;; Fixed capitalization on get-set method pair. Thanks to Jere_McDevitt@HomeDepot.COM
;;
;; Revision 1.10  1998/06/17 03:49:21  paulk
;; Fixed bug that caused jfc-app to be generated instead of console app.
;; Added a mouse motion listener template.
;; Added a toString method template.
;;
;; Revision 1.9  1998/05/27 05:55:20  paulk
;; Added autoload comments.
;;
;; Revision 1.8  1998/05/27 05:51:20  paulk
;; *** empty log message ***
;;
;; Revision 1.7  1998/05/17 06:20:37  paulk
;; Added templates for a Swing application and an inner class.
;;
;; Fixed a bug in jde-gen-buffer
;;
;; Revision 1.6  1998/04/18 14:08:55  kinnucan
;; Fixes some bugs in the generated code.
;;
;; Revision 1.5  1998/04/10 02:55:00  kinnucan
;; * Updated some of the doc strings.
;;
;; Revision 1.4  1998/04/09 04:51:09  kinnucan
;; * Added the capability to define your own custom autocode templates.
;;   The facility consists of the following items:
;;
;;   - jde-gen-code-templates
;;
;;     Defines a list of templates for code inserted at point. The
;;     list by default contains the templates defined by the JDE.
;;     You can define your own templates and add them to the list,
;;     using the Emacs customization feature. See tempo.el for
;;     information on creating templates.
;;
;;   - jde-gen-buffer-templates
;;
;;     Defines a list of templates for code to be inserted in a
;;     newly created Java buffer.
;;
;;   - jde-gen-code (JDE->Generate->Custom)
;;
;;     This command inserts a specified code template at point.
;;
;;   - jde-gen-buffer (Files->JDE New->Custom)
;;
;;     This command creates the specified buffer and inserts
;;     a specified template at the beginning of the buffer.
;;
;; Revision 1.3  1998/04/08 04:38:16  kinnucan
;; * Provided each template variable with a set function that regenerates
;;   the corresponding template command whenever the template is changed.
;;
;; Revision 1.2  1998/04/06 03:47:20  kinnucan
;; * Added jde-gen-class-buffer and jde-gen-console-buffer functions.
;;
;; Revision 1.1  1998/04/01 05:33:43  kinnucan
;; Initial revision
;;

;; End of jde-gen.el
