// Copyright (C) 2005 Open Source Telecom Corp.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include <cc++/process.h>
#include <cc++/slog.h>
#include <cstdarg>
#include "server.h"

#if defined(_MSC_VER) && _MSC_VER >= 1300
#if defined(_WIN64_) || defined(__WIN64__)     
#define RLL_SUFFIX "x64"
#elif defined(_M_IX86)
#define RLL_SUFFIX "x86"
#else
#define RLL_SUFFIX "xlo"
#endif
#endif 

#if defined(__MINGW32__) | defined(__CYGWIN32__)
#define RLL_SUFFIX "dso"
#endif

#ifdef  W32
#ifndef RLL_SUFFIX
#define RLL_SUFFIX "rll"
#endif
#endif 

#ifndef RLL_SUFFIX
#define RLL_SUFFIX "dso"
#endif 

namespace server {
using namespace ost;
using namespace std;

Runtime::Runtime() :
ScriptRuntime()
{
	static ScriptInterp::Define interp[] = {
		{"bgm", false, (Method)&Methods::scrBGM,
			(Check)&Checks::chkBGM},
		{"config", true, (Method)&ScriptMethods::scrNop,
			(Check)&Checks::chkConfig},
		{"dialtone", false, (Method)&Methods::scrTone,
			(Check)&Checks::chkTone},
                {"busytone", false, (Method)&Methods::scrTone,
                        (Check)&Checks::chkTone},
                {"ringback", false, (Method)&Methods::scrTone,
                        (Check)&Checks::chkTone},
                {"reorder", false, (Method)&Methods::scrTone,
                        (Check)&Checks::chkTone},
                {"beep", false, (Method)&Methods::scrTone,
                        (Check)&Checks::chkTone},
                {"sit", false, (Method)&Methods::scrTone,
                        (Check)&Checks::chkTone},
		{"tone", false, (Method)&Methods::scrTonegen,
			(Check)&Checks::chkTonegen},
                {"callwait", false, (Method)&Methods::scrTone,
                        (Check)&Checks::chkTone},
                {"callback", false, (Method)&Methods::scrTone,
                        (Check)&Checks::chkTone},
		{"lang", true, (Method)&ScriptMethods::scrNop,
			(Check)&Checks::chkLang},
		{"languages", true, (Method)&ScriptMethods::scrNop,
			(Check)&Checks::chkLang},
		{"load", true, (Method)&ScriptMethods::scrNop,
			(Check)&Checks::chkLoad},
		{"dir", true, (Method)&ScriptMethods::scrNop,
			(Check)&Checks::chkDir},
		{"list", true, (Method)&Methods::scrList,
			(Check)&Checks::chkList},
		{"pathname", false, (Method)&Methods::scrPathname,
			(Check)&Checks::chkPathname},
		{"readpath", true, (Method)&Methods::scrReadpath,
			(Check)&Checks::chkPaths},
		{"writepath", true, (Method)&Methods::scrWritepath,
			(Check)&Checks::chkPaths},
		{"path", true, (Method)&Methods::scrPath,
			(Check)&Checks::chkPaths},
		{"voicelib", false, (Method)&Methods::scrVoicelib,
			(Check)&Checks::chkVoicelib},
		{"timeslot", true, (Method)&Methods::scrTimeslot,
			(Check)&ScriptChecks::chkType},
		{"position", true, (Method)&Methods::scrPosition,
			(Check)&ScriptChecks::chkType},
        	{"echo", false, (Method)&Methods::scrEcho,
                	(Check)&ScriptChecks::chkHasArgs},
		{"pickup", false, (Method)&Methods::scrPickup,
			(Check)&ScriptChecks::chkOnlyCommand},
                {"hangup", false, (Method)&Methods::scrHangup, 
                        (Check)&ScriptChecks::chkOnlyCommand},
		{"collect", false, (Method)&Methods::scrCollect,
			(Check)&Checks::chkCollect},
		{"keyinput", false, (Method)&Methods::scrKeyinput,
			(Check)&Checks::chkKeyinput},
		{"read", false, (Method)&Methods::scrRead,
			(Check)&Checks::chkInput},
		{"input", false, (Method)&Methods::scrInput,
			(Check)&Checks::chkInput},
		{"keywait", false, (Method)&Methods::scrWaitkey,
			(Check)&Checks::chkSleep},
		{"sync", false, (Method)&Methods::scrSync,
			(Check)&Checks::chkSleep},
		{"sleep", false, (Method)&Methods::scrSleep,
			(Check)&Checks::chkSleep},
		{"route", false, (Method)&Methods::scrRoute,
			(Check)&Checks::chkRoute},
		{"copy", false, (Method)&Methods::scrCopy,
			(Check)&Checks::chkMove},
		{"move", false, (Method)&Methods::scrMove,
			(Check)&Checks::chkMove},
		{"erase", false, (Method)&Methods::scrErase,
			(Check)&Checks::chkErase},
		{"build", false, (Method)&Methods::scrBuild,
			(Check)&Checks::chkBuild},
		{"record", false, (Method)&Methods::scrRecord,
			(Check)&Checks::chkRecord},
		{"append", false, (Method)&Methods::scrAppend,
			(Check)&Checks::chkAppend},
		{"mf", false, (Method)&Methods::scrMF,
			(Check)&Checks::chkDialer},
		{"dtmf", false, (Method)&Methods::scrDTMF,
			(Check)&Checks::chkDialer},
		{"transfer", false, (Method)&Methods::scrTransfer,
			(Check)&Checks::chkTransfer},
		{"libexec", false, (Method)&Methods::scrLibexec,
			(Check)&Checks::chkLibexec},
		{"exec", false, (Method)&Methods::scrLibexec,
			(Check)&Checks::chkLibexec},
		{"cleardigits", false, (Method)&Methods::scrCleardigits,
			(Check)&Checks::chkCleardigits},
		{"replay", false, (Method)&Methods::scrReplay,
			(Check)&Checks::chkReplay},
		{"prompt", false, (Method)&Methods::scrPrompt,
			(Check)&Checks::chkPathname},
		{"speak", false, (Method)&Methods::scrPrompt,
			(Check)&Checks::chkPathname},
		{"play", false, (Method)&Methods::scrPlay,
			(Check)&Checks::chkPathname},
		{"xmlbind", false, (Method)&Methods::scrBind,
			(Check)&Checks::chkBind},
                {"xml", false, (Method)&Methods::scrBind,
                        (Check)&Checks::chkBind}, 
        	{NULL, false, NULL, NULL}};

        trap("timeout", false);
        trap("dtmf");

        trap("0");      /* 0x10 */
        trap("1");
        trap("2");
        trap("3");

        trap("4");      /* 0x100 */
        trap("5");
        trap("6");
        trap("7");

        trap("8");      /* 0x1000 */
        trap("9");
        trap("star");
        trap("pound");

        trap("a");      /* 0x00010000 */
        trap("b");
        trap("c");
        trap("d");

	trap("ring");	/* 0x00100000 */
	trap("tone");
	trap("event");
	trap("wink");

	trap("child");	/* 0x01000000 */
	trap("fail");
	trap("pickup");
	trap("part");

	trap("invalid");
	trap("cancel");
	trap("join");

	load(interp);

	ScriptInterp::addConditional("file", &testFile);
	ScriptInterp::addConditional("dir", &testDir);
	ScriptInterp::addConditional("span", &testSpan);
	ScriptInterp::addConditional("driver", &testDriver);
	ScriptInterp::addConditional("timeslot", &testTimeslot);
	ScriptInterp::addConditional("voice", &testVoice);
	ScriptInterp::addConditional("lang", &testLang);
	ScriptInterp::addConditional("language", &testLang);
}

unsigned long Runtime::getTrapMask(const char *trapname)
{
        unsigned long mask;

        if(!stricmp(trapname, "hangup"))
                return 0x00000001;

        if(!strcmp(trapname, "!"))
                return 0x0000fff8;

        if(!stricmp(trapname, "override"))
                return 0x00010000;

        if(!stricmp(trapname, "flash"))
                return 0x00020000;

        if(!stricmp(trapname, "immediate"))
                return 0x00040000;

        if(!stricmp(trapname, "priority"))
                return 0x00080000;

        mask = ScriptRuntime::getTrapMask(trapname);
        if(mask == 0x00000008)
                return 0x0000fff8;

        if(mask & 0x0000fff0)
                mask |= 0x00000008;

        return mask;
}

const char *Runtime::getExternal(const char *opt)
{
	if(strnicmp(opt, "server.", 7))
		return NULL;

	opt += 7;
	if(!stricmp(opt, "software"))
		return "bayonne2";
	else if(!stricmp(opt, "version"))
		return VERSION;
	else if(!stricmp(opt, "node") || !stricmp(opt, "name"))
		return keyserver.getLast("node");
	else if(!stricmp(opt, "platform"))
		return keyserver.getLast("platform");
	else if(!stricmp(opt, "state"))
		return keyserver.getLast("state");
	else if(!stricmp(opt, "timeslots"))
		return keyserver.getLast("timeslots");
	else if(!stricmp(opt, "location"))
		return keyserver.getLast("location");
	else if(!stricmp(opt, "voice"))
		return keyserver.getLast("voice");
	return NULL;
}

void Runtime::errlog(const char *level, const char *msg)
{
        const char *path = keypaths.getLast("errlog");
	char buffer[256];
	char *m;
        struct tm *dt, tbuf;
        time_t now;
        int year;

        static char *months[12] = {"Jan", "Feb", "Mar", "Apr", "May", "Jun",
                "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"};

	if(!path)
		return;

        time(&now); 
        dt = localtime_r(&now, &tbuf);  

        year = dt->tm_year;
        if(year < 1000)
                year += 1900;

        m = months[dt->tm_mon];
        snprintf(buffer, sizeof(buffer), "[%s %02d %02d:%02d:%02d %d] [%s] %s\n",
        	m, dt->tm_mday, dt->tm_hour, dt->tm_min, dt->tm_sec, year, level, msg);

	m = strchr(buffer, '\n');
	if(m)
		*(++m) = 0;

#ifdef  WIN32
        DWORD count;
        enter();
        HANDLE fd = CreateFile(path, GENERIC_WRITE, 0, NULL, OPEN_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
        if(fd != INVALID_HANDLE_VALUE)
        {
                SetFilePointer(fd, 0, NULL, FILE_END);
                WriteFile(fd, buffer, (DWORD)strlen(buffer), &count, NULL);
                CloseHandle(fd);
        }
        leave();

#else
        int fd = ::open(path, O_WRONLY | O_APPEND | O_CREAT, 0660);
        if(fd > -1)
        {
                ::write(fd, buffer, strlen(buffer));
                ::close(fd);
        }
#endif
}

bool Runtime::isInput(Line *line)
{
	if(line->scr.method == (Method)&Methods::scrKeyinput)
		return true;

	if(line->scr.method == (Method)&Methods::scrInput)
		return true;

        if(line->scr.method == (Method)&Methods::scrRead)  
                return true;
                               
	if(line->scr.method == (Method)&Methods::scrCleardigits)
		return true;

	if(line->scr.method == (Method)&Methods::scrCollect)
		return true;

	return false;
}

bool Runtime::testBinding(ScriptInterp *interp, const char *v)
{
	const char *cp = keyserver.getLast("binding");

	if(!v || !cp)
		return false;

	if(!stricmp(cp, v))
		return true;

	return false;
}

bool Runtime::testDriver(ScriptInterp *interp, const char *v)
{
	if(BayonneDriver::get(v))
		return true;
	
	return false;
}

bool Runtime::testFile(ScriptInterp *interp, const char *v)
{
	BayonneSession *s = (BayonneSession *)(interp);
	char buf[MAX_PATHNAME];
	char *ext;

	if(!v || !*v)
		return false;

	// lets accept any valid path, including those from read/writepath
	if(*v == '/' || v[1] == ':')
		goto cont;

	ext = strrchr(v, '/');
	if(ext)
		ext = strrchr(ext, '.');
	else
		ext = strrchr(v, '.');

	if(ext)
		ext = "";
	else
		ext = ".au";

	if(!strnicmp(buf, "tmp:", 4))
	{
		snprintf(buf, sizeof(buf), "%s/%s%s", path_tmp, v + 4, ext);
		v = buf;
		goto cont;
	}

        if(!strnicmp(buf, "ram:", 4) || !strnicmp(buf, "mem:", 4))
        {
                snprintf(buf, sizeof(buf), "%s/%s%s", path_tmpfs, v + 4, ext);
                v = buf;
                goto cont;
        }

	if(!isalnum(*v))
		return false;

	if(strchr(v, ':'))
		return false;

	if(strchr(v, '/'))
		snprintf(buf, sizeof(buf), "%s/%s%s", 
			keypaths.getLast("datafiles"), v, ext);
	else
		snprintf(buf, sizeof(buf), "%s/%s/%s%s",
			path_prompts, s->defVoicelib(), v, ext);

	v = buf;

cont:
	if(strstr(v, ".."))
		return false;

	if(strstr(v, "/."))
		return false;

	return isFile(v);
}

bool Runtime::testDir(ScriptInterp *interp, const char *v)
{
	char buf[MAX_PATHNAME];

	if(!v || !*v)
		return false;

	if(*v == '/' || v[1] == ':')
		goto cont;

	if(!strnicmp(buf, "tmp:", 4))
	{
		snprintf(buf, sizeof(buf), "%s/%s", path_tmp, v + 4);
		v = buf;
		goto cont;
	}

        if(!strnicmp(buf, "ram:", 4) || !strnicmp(buf, "mem:", 4))
        {
                snprintf(buf, sizeof(buf), "%s/%s", path_tmpfs, v + 4);
                v = buf;
                goto cont;
        }

	if(!isalnum(*v))
		return false;

	if(strchr(v, ':'))
		return false;

	snprintf(buf, sizeof(buf), "%s/%s", keypaths.getLast("datafiles"), v);
	v = buf;

cont:
	if(strstr(v, ".."))
		return false;

	if(strstr(v, "/."))
		return false;

	return isDir(v);
}
	

bool Runtime::testSpan(ScriptInterp *interp, const char *v)
{
	if(BayonneSpan::get(atoi(v)))
		return true;

	return false;
}

bool Runtime::testTimeslot(ScriptInterp *interp, const char *v)
{
	if(getSid(v))
		return true;

	return false;
}

bool Runtime::testVoice(ScriptInterp *interp, const char *v)
{
	BayonneSession *s = (BayonneSession *)(interp);
	return (s->audio.getVoicelib(v) != NULL);
}

bool Runtime::testLang(ScriptInterp *interp, const char *v)
{
	BayonneSession *s = (BayonneSession *)(interp);
	BayonneTranslator *t = s->getTranslator();
	const char *id = t->getId();

	if(!id[2] || id[2] == '_')
	{
		if(!strnicmp(id, v, 2))
			return true;
	}
	else if(!stricmp(id, v))
		return true;

	return false;
}	

#ifdef	WIN32

void Runtime::process(void)
{
	Thread::sleep(TIMEOUT_INF);
}

#else

static int fifo = -1;

static void readline(char *buf, unsigned max)
{
        unsigned count = 0;
        *buf = 0;

        --max;

        while(count < max)
        {
                if(read(fifo, buf + count, 1) < 1)
                        break;

                if(buf[count] == '\n')
                        break;

                ++count;
        }
        buf[count] = 0;
}

bool Runtime::startSelect(const char *name, const char *dial, const char *caller, const char *display)
{
	BayonneDriver *trunk = BayonneDriver::getTrunking();
	BayonneSpan *span;
	BayonneSession *session;
	ScriptImage *img = useImage();
	Name *scr = img->getScript(name);
	Event event;
	Line *sel;
	unsigned idx = 0, count;
	const char *cp;
	timeslot_t pos;

	if(!scr || !scr->select || scr->access != scrPUBLIC)
	{
		endImage(img);
		if(trunk && scr)
			return startDriver(trunk, name, dial, caller, display);
		return false;
	}

	sel = scr->select;
	
	while(sel)
	{
                idx = 0;
                cp = strchr(cp, '.');
                if(cp && !stricmp(cp, ".span"))
                     while(NULL != (cp = sel->args[idx++]))
                {
                        span = BayonneSpan::get(atoi(cp));
                        if(!span)   
                                continue;

                        pos = span->getFirst();
                        count = span->getCount();     
                        while(count--)
                        {
                                session = getSession(pos++);
                                if(!session)
                                        continue;

                                session->enter();
                                if(session->isIdle())
                                        goto start;
                                session->leave();
                        }
		}
                else while(NULL != (cp = sel->args[idx++]))
                {
                        session = getSid(cp);
                        if(!session)
                                continue;

                        session->enter();
                        if(session->isIdle())
                                goto start;
                        session->leave();
                }
		sel = sel->next;

	}
	endImage(img);
	return false;

start:
        memset(&event, 0, sizeof(event));
        event.id = START_OUTGOING;
        event.start.img = img; 
        event.start.scr = scr;
        event.start.dialing = dial;

        session->setConst("session.caller", caller);
        session->setConst("session.display", display);

        if(!session->postEvent(&event))
        {
                session->leave();
                endImage(img);
                return false;
        }

	session->leave();
	return true;
}

bool Runtime::startDriver(BayonneDriver *d, const char *name, const char *dial, const char *caller, const char *display)
{
	BayonneSession *session;
	ScriptImage *img = useImage();
	Name *scr = img->getScript(name);
	Event event;

	if(!d || !scr || scr->access != scrPUBLIC)
	{
		endImage(img);
		return false;
	}

	if(!caller)
	{
		display = "bayonne";
		caller = "none";
	}

	if(!display)
		display = caller;

	session = d->getIdle();
	if(!session)
	{
		endImage(img);
		return false;
	}

	memset(&event, 0, sizeof(event));
	event.id = START_OUTGOING;
	event.start.img = img;
	event.start.scr = scr;
	event.start.dialing = dial;

	session->enter();
	session->setConst("session.caller", caller);
	session->setConst("session.display", display);

	if(!session->postEvent(&event))
	{
		session->leave();
		endImage(img);
		return false;
	}

	session->leave();
	return true;
}

bool Runtime::command(char *cmd)
{
	Event event;
	char *tok;
	static fstream *logout = NULL;
	BayonneSession *s;
	const char *cp;
	bool rtn;
	BayonneDriver *d;
	const char *dial, *caller, *display;

	cmd = strtok_r(cmd, " \t\r\n", &tok);
	if(!cmd || ispunct(*cmd) || !*cmd)
		return true;
	
	if(!stricmp(cmd, "down"))
	{
		Bayonne::down();
		return true;
	}

	if(!stricmp(cmd, "start"))
	{
		cmd = strtok_r(NULL, " \t\r\n", &tok);
		if(!cmd)
			return false;
		dial = strtok_r(NULL, " \t\r\n", &tok);
		caller = strtok_r(NULL, " \t\r\n", &tok);
		display = strtok_r(NULL, " \t\r\n", &tok);
		return startSelect(cmd, dial, caller, display);
	}			

	if(!strnicmp(cmd, "lang", 4))
	{
		cmd = strtok_r(NULL, " \t\r\n", &tok);
		while(cmd)
		{
			BayonneTranslator::loadTranslator(cmd);
			cmd = strtok_r(NULL, " \t\r\n", &tok);
		}
		return true;
	}
		
	if(!stricmp(cmd, "load"))
	{
		cmd = strtok_r(NULL, " \t\r\n", &tok);
		if(!cmd)
			return false;
		rtn = Bayonne::loadPlugin(cmd);
		if(rtn)
			BayonneService::start();
		return rtn;
	}

	if(!strnicmp(cmd, "log", 3))
	{
		cmd = strtok_r(NULL, " \t\r\n", &tok);
		if(!cmd)
			return false;
		
		memset(&event, 0, sizeof(event));
		if(!stricmp(cmd, "off") || !stricmp(cmd, "none"))
			event.id = DISABLE_LOGGING;
		else
		{
			event.id = ENABLE_LOGGING;
			event.debug.logstate = cmd;
#ifndef	WIN32
			if(getppid() > 1)
			{
				event.debug.output = &cerr;
				goto skipcerr1;
			}
#endif
			if(!logout)
			{
				cp = runtime->getLast("evtlog");
				logout = new fstream(cp, fstream::app);
			}

			event.debug.output = logout;
skipcerr1:
			while(NULL !=(cmd = strtok_r(NULL, " \t\r\n", &tok)))
			{
				s = getSid(cmd);
				if(s)
					s->postEvent(&event);	
			}
		}
	}

	if(!stricmp(cmd, "service"))
	{
		cmd = strtok_r(NULL, " \t\r\n", &tok);
		if(!cmd)
			return false;
		return service(cmd);
	}
	
	if(!stricmp(cmd, "reload"))
	{
		Bayonne::reload();
		return true;
	}
	
	d = BayonneDriver::get(cmd);
	if(d)
	{
		cmd = strtok_r(NULL, " \t\r\n", &tok);
		if(!cmd)
			return false;
		dial = strtok_r(NULL, " \t\r\n", &tok);
		caller = strtok_r(NULL, " \t\r\n", &tok);
		display = strtok_r(NULL, " \t\r\n", &tok);
		return startDriver(d, cmd, dial, caller, display);
	}

	slog.error("fifo: unknown command %s", cmd);
	return false;
}
	
void Runtime::process(void)
{
	const char *argv0 = keyserver.getLast("argv0");
	FILE *fp;
	char buf[256];
	char *cp;
	int pid;

	snprintf(buf, sizeof(buf), "%s/provision.conf", keypaths.getLast("config"));
	if(!isFile(buf))
		snprintf(buf, sizeof(buf), "%s/startup.conf", keypaths.getLast("config"));
	
	fp = fopen(buf, "r");
	if(fp)
	{
		for(;;)
		{
			fgets(buf, sizeof(buf), fp);
			if(feof(fp))
				break;
			cp = buf;
                	while(isspace(*cp))
                        	++cp;

			if(*cp == '[')
				break;

			if(!isalpha(*cp))
				continue;

			command(cp);                               
		}
		fclose(fp);
	}

	snprintf(buf, sizeof(buf), "%s/bayonne.ctrl", keypaths.getLast("runfiles"));
	slog.debug("fifo=%s", buf);
	remove(buf);
	mkfifo(buf, 0770);
	keypaths.setValue("runfifo", buf);
	fifo = ::open(buf, O_RDWR);
	for(;;)
	{
		readline(buf, sizeof(buf));
		cp = buf;
		while(isspace(*cp))
			++cp;

		pid = 0;
		if(isdigit(*cp))
		{
			pid = atoi(cp);
			while(isdigit(*cp))
				++cp;
		}
		if(!pid)
		{
			command(cp);
			continue;
		}

		if(command(cp))
			kill(pid, SIGUSR1);
		else
			kill(pid, SIGUSR2);
	}
}

#endif

bool Runtime::loadBinder(const char *path)
{
        char pathbuf[256];
        const char *cp, *kv;
        const char *prefix = NULL;
        DSO *dso;  

#ifdef  WIN32
#ifdef  HAVE_TESTING
        if(!prefix && Bayonne::provision_test)
#ifdef  _DEBUG
                prefix = "DEBUG";
#else    
                prefix = "RELEASE";
#endif
#endif   
        if(!prefix)
   	 		prefix = "C:\\Program Files\\Common Files\\GNU Telephony\\Bayonne Binders";
#else

#ifdef  HAVE_TESTING
        char prefixbuf[256];

        if(!prefix && Bayonne::provision_test && !strchr(path, '/'))
        {
                snprintf(prefixbuf, sizeof(prefixbuf),
                        "%s/binders/%s/.libs", SOURCE_FILES, path);
                if(!isDir(prefixbuf))
                        snprintf(prefixbuf, sizeof(prefixbuf),
                                "%s/binders/%s", SOURCE_FILES, path);
                prefix = prefixbuf;
        }
#endif
        if(!prefix)
                prefix = LIBDIR_FILES;
#endif 

#ifdef  WIN32
        snprintf(pathbuf, sizeof(pathbuf), "%s/%s.%s", prefix, path, RLL_SUFFIX);
#else
        snprintf(pathbuf, sizeof(pathbuf), "%s/%s.bin", prefix, path);
#endif  

        cp = path;
        path = pathbuf;

        kv = getLast(path); 

        if(kv)
        {
                if(!stricmp(kv, "loaded"))
                        return true;
                return false;
        }

        if(!canAccess(path))
        {
                Bayonne::errlog("access", "cannot load binder %s", path);
                return false;
        }  

        dso = new DSO(path);
        if(!dso->isValid())
        {
                kv = dso->getError();
                setValue(path, kv);
                Bayonne::errlog("error", "cannot initialize %s", path);
                return false;
        }
        setValue(path, "loaded");
	ripple = Bayonne::provision_ripple;
        return true;
}                     
		
Runtime runtime;

} // namespace
