/* savedir.c -- save the list of files in a directory in a string
   Copyright (C) 1990, 1992 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* Written by David MacKenzie <djm@ai.mit.edu>. */
/* Adapted to GNU Finger by Jan Brittenson <bson@gnu.ai.mit.edu>
   Based on the files savedir.c and stpcpy.c by djm */

#include <config.h>


#ifdef VOID_CLOSEDIR
/* Fake a return value. */
#define CLOSEDIR(d) (closedir (d), 0)
#else
#define CLOSEDIR(d) closedir (d)
#endif


#ifdef STDC_HEADERS

#include <stdlib.h>
#include <string.h>

#else

char *malloc ();
char *realloc ();

#ifndef __GCC__
int strlen ();
#endif

#ifndef NULL
#define NULL (void *) 0
#endif

#endif

char *stpcpy ();


/* Return a freshly allocated string containing the filenames in
   directory DIR, separated by '\0' characters; the end is marked by two
   '\0' characters in a row.  NAME_SIZE is the number of bytes to
   initially allocate for the string; it will be enlarged as needed.
   Return NULL if DIR cannot be opened or if out of memory. NPROC
   is set to the number of file names returned. */

char *
savedir (dir, name_size, nfiles)
  char *dir;
  unsigned name_size;
  int *nfiles;
{
  DIR *dirp;
  struct dirent *dp;
  char *name_space;
  char *namep;
  int n = 0;

  dirp = opendir (dir);
  if (dirp == NULL)
    return NULL;

  name_space = (char *) malloc (name_size);
  if (name_space == NULL)
    {
      closedir (dirp);
      return NULL;
    }
  namep = name_space;

  while ((dp = readdir (dirp)) != NULL)
    {
      /* Skip "." and ".." (some NFS filesystems' directories lack them). */
      if (dp->d_name[0] != '.'
	  || (dp->d_name[1] != '\0'
	      && (dp->d_name[1] != '.' || dp->d_name[2] != '\0')))
	{
	  unsigned size_needed = (namep - name_space) + NLENGTH (dp) + 2;

	  if (size_needed > name_size)
	    {
	      char *new_name_space;

	      while (size_needed > name_size)
		name_size += 1024;

	      new_name_space = realloc (name_space, name_size);
	      if (new_name_space == NULL)
		{
		  closedir (dirp);
		  *nfiles = 0;
		  return NULL;
		}
	      namep += new_name_space - name_space;
	      name_space = new_name_space;
	    }
	  namep = stpcpy (namep, dp->d_name) + 1;
	  n++;
	}
    }
  *namep = '\0';
  if (CLOSEDIR (dirp))
    {
      free (name_space);
      *nfiles = 0;
      return NULL;
    }
  *nfiles = n;
  return name_space;
}

/* Copy SOURCE into DEST, stopping after copying the first '\0', and
   return a pointer to the '\0' at the end of DEST; in other words,
   return DEST + strlen (SOURCE). */

char *
stpcpy (dest, source)
     char *dest;
     char *source;
{
  while ((*dest++ = *source++) != '\0')
    /* Do nothing. */ ;
  return dest - 1;
}
