/*
   This file is part of TALER
   Copyright (C) 2025 Taler Systems SA

   TALER is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 3, or (at your option) any later version.

   TALER is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with
   TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 * @file backenddb/pg_lookup_mfa_challenge.c
 * @brief Implementation of the lookup_mfa_challenge function for Postgres
 * @author Christian Grothoff
 */
#include "platform.h"
#include <taler/taler_error_codes.h>
#include <taler/taler_dbevents.h>
#include <taler/taler_pq_lib.h>
#include "pg_lookup_mfa_challenge.h"
#include "pg_helper.h"


enum GNUNET_DB_QueryStatus
TMH_PG_lookup_mfa_challenge (
  void *cls,
  uint64_t challenge_id,
  const struct TALER_MERCHANT_MFA_BodyHash *h_body,
  struct TALER_MERCHANT_MFA_BodySalt *salt,
  char **required_address,
  enum TALER_MERCHANT_MFA_CriticalOperation *op,
  struct GNUNET_TIME_Absolute *confirmation_date,
  struct GNUNET_TIME_Absolute *retransmission_date,
  uint32_t *retry_counter,
  enum TALER_MERCHANT_MFA_Channel *tan_channel)
{
  struct PostgresClosure *pg = cls;
  struct GNUNET_TIME_Absolute now = GNUNET_TIME_absolute_get ();
  struct GNUNET_PQ_QueryParam params[] = {
    GNUNET_PQ_query_param_uint64 (&challenge_id),
    GNUNET_PQ_query_param_auto_from_type (h_body),
    GNUNET_PQ_query_param_absolute_time (&now),
    GNUNET_PQ_query_param_end
  };
  char *op_str;
  char *chan_str;
  bool no_conf;
  struct GNUNET_PQ_ResultSpec rs[] = {
    GNUNET_PQ_result_spec_string ("op",
                                  &op_str),
    GNUNET_PQ_result_spec_auto_from_type ("salt",
                                          salt),
    GNUNET_PQ_result_spec_allow_null (
      GNUNET_PQ_result_spec_absolute_time ("confirmation_date",
                                           confirmation_date),
      &no_conf),
    GNUNET_PQ_result_spec_absolute_time ("retransmission_date",
                                         retransmission_date),
    GNUNET_PQ_result_spec_uint32 ("retry_counter",
                                  retry_counter),
    GNUNET_PQ_result_spec_string ("tan_channel",
                                  &chan_str),
    GNUNET_PQ_result_spec_string ("required_address",
                                  required_address),
    GNUNET_PQ_result_spec_end
  };
  enum GNUNET_DB_QueryStatus qs;

  PREPARE (pg,
           "lookup_mfa_challenge",
           "SELECT "
           "  op::TEXT"
           " ,salt"
           " ,confirmation_date"
           " ,retransmission_date"
           " ,retry_counter"
           " ,required_address"
           " ,tan_channel::TEXT"
           "  FROM tan_challenges"
           " WHERE (challenge_id = $1)"
           "   AND (h_body = $2)"
           "   AND (expiration_date > $3)");
  /* Initialize to conservative values in case qs ends up <= 0 */
  *tan_channel = TALER_MERCHANT_MFA_CHANNEL_NONE;
  *op = TALER_MERCHANT_MFA_CO_NONE;
  *retry_counter = 0;
  qs = GNUNET_PQ_eval_prepared_singleton_select (pg->conn,
                                                 "lookup_mfa_challenge",
                                                 params,
                                                 rs);
  if (qs <= 0)
    return qs;
  if (no_conf)
    *confirmation_date = GNUNET_TIME_UNIT_FOREVER_ABS;
  *tan_channel = TALER_MERCHANT_MFA_channel_from_string (chan_str);
  *op = TALER_MERCHANT_MFA_co_from_string (op_str);
  GNUNET_free (chan_str);
  GNUNET_free (op_str);
  return qs;
}
