/*  file: window.c
 *
 *  kehpager, Charset aware pager, Kari E. Hurtta
 *
 *  Copyright (c) 1993, 1994 Kari E. Hurtta
 *
 *  Redistribution and use in source and binary forms are permitted
 *  provided that the above copyright notice and this paragraph are
 *  duplicated in all such forms. This software is provided 'as is'
 *  and without any warranty. 
 */

#include <stdio.h>

#include "kehpager.h"
#include "memory.h"
#include "charset.h"

#include "window.h"
#include "control.h"
#include "terminal.h"

#define WINDOW_COUNT 6

static int prompt_win = -1;
static int prompt_X = -1;
static int prompt_Y = -1;

static int hl_win = -1;
static int hl_flags = 0;
static int hl_Y = -1;
static int hl_X1 = -1;
static int hl_X2 = -1;

static struct window {
  /* booleans */
  unsigned enable_tab :1;
  unsigned enable_wrapping :1;
  unsigned showspc :1;
  unsigned pagemode :1;

  /* values */
  int priority;
  int tab_step;
  int indent_step;

  int curpage;

  int first; /* base line (screen Y cordinate) */
  int first_line; /* first logical line in buffer (first = 0) (mapping_line) */
  int mapped;
  int buffer_len;
  int mapping_len;
  struct oneline {
    short indent;   /* chapter indent */
    short rindent;
    short submargin;
    short justify;
    short excerpt_count;
    short excerpt_flag;
    short force_wrap;
    int wrappos;
    int len;    
    struct onechar {
      CHAR_IDX mapped_ch;
      unsigned short flags;
    } *chars;
  } *buffer_lines;
  struct wrappos {
    int indent;
    int fill;
    int line;
    int offset;
    int pagenum;
  } *line_mapping;
} windows[WINDOW_COUNT];
  
void init_windows(void) {
  int i;
  for (i = 0; i < WINDOW_COUNT; i++) {
    windows[i].enable_tab = 1;
    windows[i].enable_wrapping = 1;
    windows[i].showspc = 0;
    windows[i].tab_step = 8;
    windows[i].indent_step = 4;
    windows[i].priority = 0;
    windows[i].pagemode = 0;

    windows[i].first = 1;
    windows[i].first_line = 0;
    windows[i].mapped = 0;
    windows[i].buffer_len = 0;
    windows[i].buffer_lines = NULL;
    windows[i].mapping_len = 0;
    windows[i].line_mapping = NULL;
    windows[i].curpage = 0;
  }
}


static int visible_len(int win) {
  int next_Y = windows[win].first +
    windows[win].mapping_len - windows[win].first_line,i;
  if (!windows[win].mapped) return 0;

  if (next_Y < windows[win].first)
    next_Y = windows[win].first;

#if defined(MAX_LINES)
  if (next_Y > MAX_LINES)
    next_Y = MAX_LINES;
#endif

  for (i = win+1; i < WINDOW_COUNT; i++)
    if (windows[i].mapped) {
      next_Y = windows[i].first;
      break;
    }

  if (next_Y < windows[win].first) {
    reset_terminal_state();
    print_notify("(%s) Software error/visible_len: next_Y=%d < windows[win=%d].first=%d",
		 prog,next_Y,win,windows[win].first);
    close_files(); close_terminal();
    exit(8);
  }

  return next_Y - windows[win].first;
}

static int was_last(int win) {
  int i;
  for (i = win+1; i < WINDOW_COUNT; i++)
    if (windows[i].mapped) {
      return 0;
    }
  return 1;
}

static void reset_line_flags(int win, int Y) {
  windows[win].buffer_lines[Y-1].indent = 0;
  windows[win].buffer_lines[Y-1].rindent = 0;
  windows[win].buffer_lines[Y-1].submargin = 0;
  windows[win].buffer_lines[Y-1].justify = JUST_left;
  windows[win].buffer_lines[Y-1].excerpt_count = 0;
  windows[win].buffer_lines[Y-1].excerpt_flag = 0;
  windows[win].buffer_lines[Y-1].force_wrap = 0;
}

static void reserve_line(int win, int Y) {
  int pg = 1;
  if (windows[win].mapping_len > 0)
    pg = windows[win].line_mapping[windows[win].mapping_len-1].pagenum;
  
  if (!windows[win].pagemode) pg = 0;

  if (windows[win].buffer_len < Y) {
    int y, ADD = Y - windows[win].buffer_len,WP = windows[win].mapping_len;
    windows[win].buffer_lines =
      (struct oneline *)REALLOC((void *)windows[win].buffer_lines,
				Y*sizeof(struct oneline));
    
    windows[win].line_mapping =
      (struct wrappos *)REALLOC((void *)windows[win].line_mapping,
 				(WP+ADD)*sizeof(struct wrappos));

    windows[win].mapping_len = WP+ADD;
    for (y = windows[win].buffer_len; y < Y; y++) {
      windows[win].buffer_lines[y].len = 0;
      windows[win].buffer_lines[y].chars = NULL;
      windows[win].buffer_lines[y].wrappos = WP;
      reset_line_flags(win,y+1);
      windows[win].line_mapping[WP].line    = y;
      windows[win].line_mapping[WP].offset  = 0;
      windows[win].line_mapping[WP].indent  = 0;
      windows[win].line_mapping[WP].fill    = 0;
      windows[win].line_mapping[WP].pagenum = pg;
      WP++;
    }
    windows[win].buffer_len = Y;
  }
}

static void add_char(int win, int X,int Y, int flags, CHAR_IDX mapped_ch) {

  reserve_line(win,Y);

  if (windows[win].buffer_lines[Y-1].len < X) {
    int x;
    windows[win].buffer_lines[Y-1].chars = (struct onechar *)
      REALLOC((void *)windows[win].buffer_lines[Y-1].chars,
	      X*sizeof(struct onechar));

    for (x = windows[win].buffer_lines[Y-1].len; x < X; x++) {
      windows[win].buffer_lines[Y-1].chars[x].mapped_ch = CH_SPC;
      windows[win].buffer_lines[Y-1].chars[x].flags = 0;
    }

    windows[win].buffer_lines[Y-1].len = X;
  }

  windows[win].buffer_lines[Y-1].chars[X-1].mapped_ch = mapped_ch;
  windows[win].buffer_lines[Y-1].chars[X-1].flags = flags;
}

static int is_special(int win,CHAR_IDX mapped, int spfill) {
  if (windows[win].enable_tab && mapped == CH_TAB) return 1;
  if (windows[win].enable_tab && mapped == CH_SHY) return 1;
  if (windows[win].enable_tab && mapped == CH_SHY2) return 1;
  if (windows[win].enable_tab && mapped == CH_NBSP) return 1;
  if (windows[win].enable_tab && mapped == CH_NBHY) return 1;
  if ((windows[win].enable_tab || windows[win].pagemode) 
      && mapped == CH_LSEP) return 1;
  if ((windows[win].enable_tab || windows[win].pagemode)
      && mapped == CH_PSEP) return 1;
  if (mapped == CH_SPC_C) return 1;
  if (mapped == CH_IND_B) return 1;
  if (mapped == CH_IND_E) return 1;
  if (mapped == CH_MAR_B) return 1;
  if (mapped == CH_MAR_E) return 1;
  if (spfill > 0 && mapped == CH_SPC) return 1;
  return 0;
}

static int is_break(int win,CHAR_IDX mapped) {
  if (!windows[win].pagemode && !windows[win].enable_tab) return 0;
  if (mapped == CH_LSEP) return 1;
  if (mapped == CH_PSEP) return 1;
  return 0;
}

static int is_wrapping(int win,CHAR_IDX mapped) {
  if (windows[win].enable_tab && mapped == CH_TAB) return 1;
  if (windows[win].enable_tab && mapped == CH_SHY) return 1;
  if (windows[win].enable_tab && mapped == CH_SHY2) return 1;
  if (mapped == CH_HY) return 1;
  if (mapped == CH_HY2) return 1;
  if (mapped == CH_SPC) return 1;
  if (mapped == CH_SPC_C) return 1;
  return 0;
}

static int is_hyphen(int win,CHAR_IDX mapped) {
  if (windows[win].enable_tab && mapped == CH_NBHY) return 1;
  if (mapped == CH_HY) return 1;
  if (mapped == CH_HY2) return 1;
  if (mapped == CH_SHY) return 1;
  if (mapped == CH_SHY2) return 1;
  return 0;
}

static int is_space(int win,CHAR_IDX mapped) {
  if (windows[win].enable_tab && mapped == CH_TAB) return 1;
  if (windows[win].enable_tab && mapped == CH_NBSP) return 1;
  if (mapped == CH_SPC) return 1;
  if (mapped == CH_SPC_C) return 1;
  return 0;
}

static int is_pagebreak(int win,CHAR_IDX mapped) {
  if (!windows[win].pagemode) return 0;
  if (mapped == CH_FF) return 1;
  if (mapped == CH_NP) return 1;
  return 0;
}

static int nflags(int win,int line, int x) {
  if (win != hl_win) return 0;
  if (line != hl_Y-1) return 0;
  if (x < hl_X1-1 || x > hl_X2-1) return 0;
  return hl_flags;
}

static void print_win_line(int win,int rline,int Y) {
  if (rline >= 0 && rline < windows[win].mapping_len &&
      (!windows[win].pagemode || 
       windows[win].line_mapping[rline].pagenum == windows[win].curpage))  {
    int line = windows[win].line_mapping[rline].line;
    int X1 = windows[win].line_mapping[rline].offset;
    int X2 = rline+1 < windows[win].mapping_len &&
	         windows[win].line_mapping[rline+1].line ==
		 windows[win].line_mapping[rline].line ?
               windows[win].line_mapping[rline+1].offset :
               windows[win].buffer_lines[line].len;
    int fill = windows[win].line_mapping[rline].fill;
    int x,x1,need = MAX(MAX(MAX(MAX(MAX(X2 - X1, 10),
			windows[win].line_mapping[rline].indent),fill),
			1+windows[win].buffer_lines[line].excerpt_count),
			windows[win].tab_step+1);
    CHAR_IDX *mapped = (CHAR_IDX *) MALLOC(need * sizeof(int));
    int pos = 0;
    int last = CH_SPC;
    int flags = 0;
    int spfill = 0;
    int nfill = 0;

    if (fill < 0) { 
      spfill = -fill; 
      fill = 0; 
    }
    /* clear_line(Y); */

    if (windows[win].line_mapping[rline].indent > 0) {
      for (x = 0; x < windows[win].line_mapping[rline].indent; x++) 
	mapped[x] = CH_SPC;
      
      pos += print_XY(1+pos,Y,0,
		      windows[win].line_mapping[rline].indent,mapped,
		      windows[win].showspc,windows[win].priority);

    }

    if (windows[win].buffer_lines[line].excerpt_count > 0) {
      for (x = 0; x < windows[win].buffer_lines[line].excerpt_count; x++) 
	mapped[x] = CH_GT;

      mapped[windows[win].buffer_lines[line].excerpt_count] = CH_SPC;

      pos += print_XY(1+pos,Y,windows[win].buffer_lines[line].excerpt_flag,
		      1+windows[win].buffer_lines[line].excerpt_count,
		      mapped,windows[win].showspc,windows[win].priority);
    }

    if (fill > 0) {
      for (x = 0; x < fill; x++) 
	mapped[x] = CH_SPC;
      
      nfill=print_XY(1+pos,Y,0,fill,mapped,windows[win].showspc,
		     windows[win].priority);

    }

    if (X2 > 0) last = windows[win].buffer_lines[line].chars[X2-1].mapped_ch;
    /* don't print last char if space */
    if (X2 > 0 && !windows[win].showspc && is_space(win,last)) X2--;

    /* don't print first char is FF */
    if (windows[win].buffer_lines[line].len > X1 &&
	is_pagebreak(win,
		     windows[win].buffer_lines[line].chars[X1].mapped_ch))
      X1++;

    for (x = X1; x < X2; x = x1) {
      flags = (windows[win].buffer_lines[line].chars[x].flags
	       | nflags(win,line,x));
	
      for (x1 = x;
	   x1 < X2 &&
	   flags == (windows[win].buffer_lines[line].chars[x1].flags 
		     | nflags(win,line,x1)) &&
	   !is_special(win,windows[win].buffer_lines[line].chars[x1].mapped_ch,
		       spfill);
	   x1++) {	   	
	mapped[x1-x] = windows[win].buffer_lines[line].chars[x1].mapped_ch;
	   }

      if (x1 > x) pos += print_XY(1+pos+nfill,Y,flags,x1-x,mapped,
				  windows[win].showspc,windows[win].priority);


      for (;
	   x1 < X2 &&
	   flags == (windows[win].buffer_lines[line].chars[x1].flags 
		     | nflags (win,line,x1)) &&
	   is_special(win,windows[win].buffer_lines[line].chars[x1].mapped_ch,
		      spfill);
	   x1++) {
	int slen = 0;
	int CH = windows[win].buffer_lines[line].chars[x1].mapped_ch;
	if (CH == CH_TAB) {
	  int i;

	  if (windows[win].tab_step > 0)
	    for (i=0; 
		 i < (windows[win].tab_step-pos%windows[win].tab_step); 
		 i++)
	      mapped[slen++] = CH_SPC;
	}

	/* if (CH == CH_IND_B)  ignore!  ; */
	/* if (CH == CH_IND_E)  ignore!  ; */
	/* if (CH == CH_MAR_B)  ignore!  ; */
	/* if (CH == CH_MAR_E)  ignore!  ; */

	if (CH == CH_SPC_C && windows[win].showspc)
	    mapped[slen++] = CH_SPC;

	if (CH == CH_SPC) {
	  mapped[slen++] = CH_SPC;
	  if (spfill > 0 && x1 < X2-1) {
	    mapped[slen++] = CH_SPC;
	    spfill--;
	  }
	}

	if (CH == CH_SHY)
	   if (x1 == X2-1) mapped[slen++] = CH_HY;
	if (CH == CH_SHY2)
	   if (x1 == X2-1) mapped[slen++] = CH_HY;
	if (CH == CH_LSEP)
	   if (x1 < X2-1 && x1 > X1) mapped[slen++] = CH_LSEP;
	if (CH == CH_PSEP)
	   if (x1 < X2-1 && x1 > X1) mapped[slen++] = CH_PSEP;

	if (CH == CH_NBSP)
	   mapped[slen++] = CH_SPC;
	if (CH == CH_NBHY)
	   mapped[slen++] = CH_HY;

	pos += print_XY(1+pos+nfill,Y,flags,slen,mapped,windows[win].showspc,
			windows[win].priority);

      }

    }

    if (rline+1 < windows[win].mapping_len &&
	         windows[win].line_mapping[rline+1].line ==
		 windows[win].line_mapping[rline].line &&
	!is_wrapping(win,last) && !is_break(win,last))
	   pos += print_XY(1+pos+nfill,Y,flags,1,&CH_WRA,windows[win].showspc,
			   windows[win].priority);
  
    FREE(mapped);

    clear_rest_of_line(1+pos+nfill,Y);

  } else {
    clear_line(Y);
  }
}

static int shoft_line_len(int win,int line,int X1, int X2,
			  int *lenmap,int indent, int fill) {
#define STATLEN 120
  CHAR_IDX smapped[STATLEN];
  int slenlist[STATLEN];
  int pos = 0,x,x1,X2P=X2;
  int need = MAX(MAX(MAX(X2-X1,indent),
		     1+windows[win].buffer_lines[line].excerpt_count),fill);
  CHAR_IDX *mapped = X2-X1 <= STATLEN ? smapped :
    (CHAR_IDX *) MALLOC(need * sizeof(CHAR_IDX));
  int *lenlist = X2-X1 <= STATLEN ? slenlist :
    (int *) MALLOC(need * sizeof(int));
  int spfill = 0;

  if (fill < 0) { 
    spfill = -fill; 
    fill = 0; 
  }

  if (indent > 0) {
    for (x = 0; x < indent; x++) 
      mapped[x] = CH_SPC;
    
    pos += printable_len(indent,mapped,lenlist,windows[win].showspc);
    
  }



  if (fill > 0) {
    for (x = 0; x < fill; x++) 
      mapped[x] = CH_SPC;
    
    pos += printable_len(fill,mapped,lenlist,windows[win].showspc);
    
  }


  
  if (windows[win].buffer_lines[line].excerpt_count > 0) {
      for (x = 0; x < windows[win].buffer_lines[line].excerpt_count; x++) 
	mapped[x] = CH_GT;

      mapped[windows[win].buffer_lines[line].excerpt_count] = CH_SPC;

      pos += printable_len(1+windows[win].buffer_lines[line].excerpt_count,
			   mapped,lenlist,windows[win].showspc);
  }

  /* don't print last char if space */
  if (X2 > 0 && !windows[win].showspc &&
      is_space(win,windows[win].buffer_lines[line].chars[X2-1].mapped_ch)) 
    X2P--;

    /* don't print first char is FF */
    if (windows[win].buffer_lines[line].len > X1 &&
	is_pagebreak(win,
		     windows[win].buffer_lines[line].chars[X1].mapped_ch)) {
      if (lenmap)
	lenmap[X1] = pos;
      X1++;
    }

      
  for (x = X1; x < X2P; x = x1) {
	
      for (x1 = x;
	   x1 < X2P &&
	   !is_special(win,windows[win].buffer_lines[line].chars[x1].mapped_ch,
		       spfill);
	   x1++)
	mapped[x1-x] = windows[win].buffer_lines[line].chars[x1].mapped_ch;

      if (x1 > x) {
	int n = printable_len(x1-x,mapped,lenlist,windows[win].showspc);
	int i;
	if (lenmap)
	  for (i = x; i < x1; i++)
	    lenmap[i] = pos + lenlist[i-x];

	pos += n;
      }

      for (;
	   x1 < X2P &&
	   is_special(win,windows[win].buffer_lines[line].chars[x1].mapped_ch,
		      spfill);
	   x1++) {
	int extra = 0;

	/* if (windows[win].buffer_lines[line].chars[x1].mapped_ch == CH_IND_B)
	     ignore! ; */
	/* if (windows[win].buffer_lines[line].chars[x1].mapped_ch == CH_IND_E)
	     ignore!  ; */
	/* if (windows[win].buffer_lines[line].chars[x1].mapped_ch == CH_MAR_B)
	     ignore!  ; */
	/* if (windows[win].buffer_lines[line].chars[x1].mapped_ch == CH_MAR_E)
	     ignore!  ; */

	if (windows[win].buffer_lines[line].chars[x1].mapped_ch == CH_SPC_C &&
	    windows[win].showspc) {
	  pos++;
	}

	if (windows[win].buffer_lines[line].chars[x1].mapped_ch == CH_SPC) {
	  pos++;
	  if (spfill > 0 && x1 < X2P-1) { 
	    spfill--;
	    pos++;
	  }
	}

	if (windows[win].buffer_lines[line].chars[x1].mapped_ch == CH_TAB)
	  if (windows[win].tab_step > 0)
	    pos += windows[win].tab_step-pos%windows[win].tab_step;

	if (windows[win].buffer_lines[line].chars[x1].mapped_ch == CH_SHY) {
	   if (x1 == X2P-1) pos += 1;
	   else extra = 1;
	 }
	if (windows[win].buffer_lines[line].chars[x1].mapped_ch == CH_SHY2) {
	   if (x1 == X2P-1) pos += 1;
	   else extra = 1;
	 }
	if (windows[win].buffer_lines[line].chars[x1].mapped_ch == CH_LSEP) {
	  int n = printable_len(1,&CH_LSEP,NULL,windows[win].showspc);
	   if (x1 < X2P-1 && x1 > X1) pos += n;
	   else extra = n;
	 }
	if (windows[win].buffer_lines[line].chars[x1].mapped_ch == CH_PSEP) {
	  int n = printable_len(1,&CH_PSEP,NULL,windows[win].showspc);
	   if (x1 < X2P-1 && x1 > X1) pos += n;
	   else extra = n;
	 }

	if (windows[win].buffer_lines[line].chars[x1].mapped_ch == CH_NBSP)
	   pos += 1;
	if (windows[win].buffer_lines[line].chars[x1].mapped_ch == CH_NBHY)
	   pos += 1;

	if (lenmap)
	  lenmap[x1] = pos + extra;
      }
  }

  for (x = X2P; x < X2; x++)
    if (lenmap) lenmap[x] = pos;

  if (DEBUG_WRAP) {
#define WRAP_LEN 130
    char P[WRAP_LEN + (WRAP_LEN / 5 * MAX_FALLBACK) ],*c;
    int j;

    print_debug("Wrapping win=%d line=%d X1=%d X2=%d (X2P=%d) indent=%d RESULT=%d",
		win,line,X1,X2,X2P,indent,pos);

    for (j = X1; j < X2; j += WRAP_LEN / 5 - 5) {
      int i,k=j+WRAP_LEN / 5 - 5;
#undef WRAP_LEN

      if (k > X2) k = X2;
      
      c = P;
      
      for (i = j; i < k; i++) {
	char *prev = c;
	strcpy(c,chars[windows[win].buffer_lines[line].chars[i].mapped_ch].fallback);
	c += strlen(c);
	while (c - prev < 5)
	  *c++ = ' ';
      }
      *c++ = '\0';
      print_debug("Line        : %s",P);
      c = P;
      for (i = j; i < k; i++) {
	char *prev = c;
	if (windows[win].buffer_lines[line].chars[i].mapped_ch < 0)
	  sprintf(c,"-%04X",(unsigned int) -windows[win].buffer_lines[line].chars[i].mapped_ch);
	else
	  sprintf(c,"%04X",(unsigned int) windows[win].buffer_lines[line].chars[i].mapped_ch);
	c += strlen(c);
	while (c - prev < 5)
	  *c++ = ' ';
      }
      *c++ = '\0';
      print_debug("Line (hex)  : %s",P);
      c = P;
      if (lenmap) {
	for (i = j; i < k; i++) {
	  char *prev = c;
	  sprintf(c,"%d",lenmap[i]);
	  c += strlen(c);
	  while (c - prev < 5)
	    *c++ = ' ';
	}
	*c++ = '\0';
	print_debug("Line len    : %s",P);
      }
    }
  }

  if (slenlist != lenlist)
    FREE(lenlist);
  if (smapped != mapped)
    FREE(mapped);
  return pos;
}

static void calculate_prompt(void) {
  if (DEBUG_PROMPT) {
      print_debug("prompt_win = %d, prompt_X = %d, prompt_Y = %d",
	   prompt_win, prompt_X,prompt_Y);
  }
  if (-1 == prompt_win || -1 == prompt_X || -1 == prompt_Y)
    set_hard(-1,-1);
  else if (-2 == prompt_win)
    set_hard(prompt_X,prompt_Y);
  else if (!windows[prompt_win].mapped) 
    set_hard(-1,-1);
  else {
    int RY,RX;
    if (prompt_Y < 1) { RY = prompt_Y; RX = prompt_X; }
    else if (prompt_Y > windows[prompt_win].buffer_len) {
      RY = windows[prompt_win].mapping_len + prompt_Y 
	- windows[prompt_win].buffer_len;
      RX = prompt_X; 
    } else {
      int Xup;
      int ml = windows[prompt_win].buffer_lines[prompt_Y-1].wrappos;
      while(ml+1 < windows[prompt_win].mapping_len &&
	    prompt_Y-1 == windows[prompt_win].line_mapping[ml+1].line &&
	    prompt_X >= windows[prompt_win].line_mapping[ml+1].offset+1)
	ml++;
      RY = ml +1;
      Xup = prompt_X-1;
      if (Xup > windows[prompt_win].buffer_lines[prompt_Y-1].len)
	Xup = windows[prompt_win].buffer_lines[prompt_Y-1].len;

      if (windows[prompt_win].pagemode && 
	  windows[prompt_win].line_mapping[ml].pagenum 
	  != windows[prompt_win].curpage)
	RX = 1;
      else 
	RX = 1+shoft_line_len(prompt_win,prompt_Y-1,
			      windows[prompt_win].line_mapping[ml].offset,
			      Xup,NULL,
			      windows[prompt_win].line_mapping[ml].indent,
			      windows[prompt_win].line_mapping[ml].fill);
      if (DEBUG_PROMPT) 
	print_debug("prompt: ml=%d, Xup=%d,RX=%d",ml,Xup,RX);

      RX += prompt_X - Xup -1;
    }
    if (DEBUG_PROMPT) 
	print_debug("prompt: RX=%d, RY=%d",RX,RY);

    if (RY < windows[prompt_win].first_line+1 ||
	RY > windows[prompt_win].first_line + visible_len(prompt_win) +1)
      set_hard(-1,-1);
    else 
      set_hard(RX,
	       windows[prompt_win].first + RY 
	       - windows[prompt_win].first_line -1);
  }
}

void set_win_prompt(int win,int X,int Y) {
  prompt_win = win;
  prompt_X = X;
  prompt_Y = Y;
  calculate_prompt();
}

static void print_win_tail(int win, int from) {
  int Y = windows[win].first;
  int i = windows[win].first_line,j;
  int vlen = visible_len(win);

  for (j = from; j < vlen; j++)
    print_win_line(win,i+j,Y+j);

  if (was_last(win))
    clear_from_line(Y+vlen);
}

static void print_window(int win,int clear_line) {
  int off = windows[win].first_line;
  int adding = 0;
  if (off < 0) adding = -off;
  if (clear_line >= 0 && adding > clear_line) 
    adding = clear_line;
  print_win_tail(win,adding);
}

static int check_indent(int cindent, int margin) {
  if (cindent > margin-5) 
    cindent = margin-5;
  if (cindent < 0) cindent = 0;
  return cindent;
}

static void wrap_line(int win,int Y) {
    int swraplist[STATLEN],swraplens[STATLEN],sindentlist[STATLEN],
        sfilllist[STATLEN],smarginlist[STATLEN],spglist[STATLEN];
    int LEN=windows[win].buffer_lines[Y-1].len;
    int *pglist= LEN <= STATLEN ? spglist : 
      (int *)MALLOC(LEN*sizeof (int));
    int *marginlist= LEN <= STATLEN ? smarginlist :
      (int *)MALLOC(LEN*sizeof (int));
    int *filllist= LEN <= STATLEN ? sfilllist :
      (int *)MALLOC(LEN*sizeof (int));
    int *wraplist= LEN <= STATLEN ? swraplist :
      (int *)MALLOC(LEN*sizeof (int));
    int *wraplens = LEN <= STATLEN ? swraplens :
      (int *)MALLOC(LEN*sizeof (int));
    int *indentlist = LEN <= STATLEN ? sindentlist :
      (int *)MALLOC(LEN*sizeof (int));
    int ry1=windows[win].buffer_lines[Y-1].wrappos;
    int ry2= Y < windows[win].buffer_len ?
	windows[win].buffer_lines[Y].wrappos : windows[win].mapping_len;
    int W1=ry2-ry1,WL=1,YS=ry1+windows[win].first-windows[win].first_line;
    int spos=0,i,pg=1,lastpg=0;
    int cindent = windows[win].buffer_lines[Y-1].indent > 0 ?
      windows[win].buffer_lines[Y-1].indent : 0;
    int rmargin = windows[win].buffer_lines[Y-1].submargin;
    int margin;
    int rindent = windows[win].buffer_lines[Y-1].rindent;

    if (ry1 > 0 && windows[win].mapping_len > ry1-1) 
      pg = windows[win].line_mapping[ry1-1].pagenum;

    if (!windows[win].pagemode) pg = 0;
    if (ry2 > 0 && windows[win].mapping_len > ry2-1) 
      lastpg = windows[win].line_mapping[ry2-1].pagenum;

    margin = columns - windows[win].indent_step * rmargin;
    if (margin < 5) margin = 5;
    if (margin > columns) margin = columns;

    cindent = check_indent(cindent+rindent*windows[win].indent_step,
			   margin);
    wraplist[0]=0;
    filllist[0]=0;
    marginlist[0]=margin;
    indentlist[0] = cindent;
    pglist[0] = pg;
    if (DEBUG_WRAP || DEBUG_MWRAP)
      print_debug("  (win=%d,%d,%d) First line; indent=%d (rindent=%d), "
		  "margin=%d (rmargin=%d) page=%d",
		  win,Y,WL,cindent,rindent,margin,rmargin,pg);

    if (windows[win].enable_wrapping || 
	windows[win].buffer_lines[Y-1].force_wrap ||
	windows[win].pagemode || windows[win].enable_tab) {
      /* enable_tab wraps LINE SEPARATOR (CH_LSEP) 
	 and PARAGRAPH SEPARATOR (CH_PSEP) */
      /* Count wrapping */
      while(spos < LEN) {
	int x, pos1 = spos, pos2 = spos, OLD=spos;
	int THIS=shoft_line_len(win,Y-1,spos,LEN,wraplens,cindent,0);

	for (x=spos; x < LEN; x++) {
	  int clen = wraplens[x];
	  
	  if (CH_IND_B == windows[win].buffer_lines[Y-1].chars[x].mapped_ch)
	    rindent++;
	  if (CH_IND_E == windows[win].buffer_lines[Y-1].chars[x].mapped_ch)
	    rindent--;
	  if (CH_MAR_B == windows[win].buffer_lines[Y-1].chars[x].mapped_ch)
	    rmargin++;
	  if (CH_MAR_E == windows[win].buffer_lines[Y-1].chars[x].mapped_ch)
	    rmargin--;

	  margin = columns - windows[win].indent_step * rmargin;
	  if (margin < 5) margin = 5;
	  if (margin > columns) margin = columns;

	  cindent = windows[win].buffer_lines[Y-1].indent < 0 ?
	    -windows[win].buffer_lines[Y-1].indent : 0;
	  cindent = check_indent(cindent+rindent*windows[win].indent_step,
				 margin);

	  if (is_pagebreak(win,
			   windows[win].buffer_lines[Y-1].chars[x].mapped_ch)){
	    if (WL >0 && wraplist[WL-1] == x) WL--;
	    wraplist[WL]=x;
	    filllist[WL]=0;
	    pglist[WL]=++pg;
	    if (WL > 0) marginlist[WL-1]=margin;
	    marginlist[WL]=margin;
	    indentlist[WL++] = cindent;
	    spos = x;

	    if (DEBUG_WRAP || DEBUG_MWRAP)
	      print_debug("  (win=%d,%d,%d) Pagebreak on position=%d "
			  "len=%d indent=%d "
			  "(rindent=%d) margin=%d (rmargin=%d) "
			  "page=%d",
			  win,Y,WL,x,
			  x ? wraplens[x-1] : 0,
			  cindent,rindent,margin,rmargin,
			  pg);
	    break;
	  }


	  if ((windows[win].enable_wrapping || 
	      windows[win].buffer_lines[Y-1].force_wrap) &&
	      clen > margin -1 && pos1==spos && pos2 > spos) {
	    wraplist[WL]=pos2;
	    filllist[WL]=0;
	    pglist[WL]=pg;
	    if (WL > 0) marginlist[WL-1]=margin;
	    marginlist[WL]=margin;
	    indentlist[WL++] = cindent;
	    spos = pos2;

	    if (DEBUG_WRAP || DEBUG_MWRAP)
	      print_debug("  (win=%d,%d,%d) Hard wrap on position=%d "
			  "len=%d indent=%d "
			  "(rindent=%d) margin=%d (rmargin=%d) "
			  "page=%d",
			  win,Y,WL,pos2,
			  pos2 ? wraplens[pos2-1] : 0,
			  cindent,rindent,margin,rmargin,
			  pg);
	    break;
	  }

	  if (x < LEN-1 &&
	      is_break(win,windows[win].buffer_lines[Y-1].chars[x].mapped_ch)){
	    wraplist[WL]=x+1;
	    filllist[WL]=0;
	    pglist[WL]=pg;
	    if (WL > 0) marginlist[WL-1]=margin;
	    marginlist[WL]=margin;
	    indentlist[WL++] = cindent;
	    spos = x+1;

	    if (DEBUG_WRAP || DEBUG_MWRAP)
	      print_debug("  (win=%d,%d,%d) (Line)break on position=%d "
			  "len=%d indent=%d "
			  "(rindent=%d) margin=%d (rmargin=%d) "
			  "page=%d",
			  win,Y,WL,x+1,
			  x+1 ? wraplens[x] : 0,
			  cindent,rindent,margin,rmargin,
			  pg);
	    break;
	  }
	  
	  if ((windows[win].enable_wrapping || 
	       windows[win].buffer_lines[Y-1].force_wrap) &&
	      clen > margin && pos1 > spos) {
	    wraplist[WL]=pos1;
	    filllist[WL]=0;
	    pglist[WL]=pg;
	    if (WL > 0) marginlist[WL-1]=margin;
	    marginlist[WL]=margin;
	    indentlist[WL++] = cindent;
	    spos = pos1;
	    
	    if (DEBUG_WRAP || DEBUG_MWRAP)
	      print_debug("  (win=%d,%d,%d) Shoft wrap on position=%d "
			  "len=%d indent=%d "
			  "(rindent=%d) margin=%d (rmargin=%d) "
			  "page=%d",
			  win,Y,WL,pos1,
			  pos1 ? wraplens[pos1-1] : 0,
			  cindent,rindent,margin,rmargin,
			  pg);
	    break;
	  }
	  
	  pos2 = x+1;
	  if ((windows[win].enable_wrapping || 
	       windows[win].buffer_lines[Y-1].force_wrap) &&
	      is_wrapping(win,
			  windows[win].buffer_lines[Y-1].chars[x].mapped_ch)) {
	    pos1=x+1;
	  }	 
	}
	
	if (!(windows[win].enable_wrapping || 
	      windows[win].buffer_lines[Y-1].force_wrap)) {
	  if (DEBUG_WRAP || DEBUG_MWRAP)
	    print_debug("  (win=%d,%d,%d) word wrap disabled",
			win,Y,WL);
	  break;
	}
	
	if (THIS < margin) {
	  if (DEBUG_WRAP || DEBUG_MWRAP)
	    print_debug("  (win=%d,%d,%d) wrapping done (len=%d)",
			win,Y,WL,THIS);
	  break;
	}
	if (spos == OLD) {
	  if (DEBUG_WRAP || DEBUG_MWRAP)
	    print_debug("  (win=%d,%d,%d) Can't wrap after position %d",
			win,Y,WL,spos);
	  break;
	}
      }
    }
      
    /* if we have put wrapping to end of line */
    if (WL > 1 && wraplist[WL-1] >= LEN) 
      WL--;

    if (windows[win].buffer_lines[Y-1].justify != JUST_left) {
      /* Justify */
      for (i=0; i < WL; i++) {
	int lastpos = i < WL-1 ? wraplist[i+1] : LEN;
	int clen = shoft_line_len(win,Y-1,wraplist[i],lastpos,
				  NULL,indentlist[i],0);
	int left = marginlist[i] - clen;
	int fill;

	/* Take care of wrap character in end of line !! */
	if (i < WL-1 && lastpos > 0 &&
	    !is_wrapping(win,
			 windows[win].buffer_lines[Y-1].chars[lastpos-1].
			 mapped_ch))
	  left--;

	if (left < 0) left = 0;
	fill = left;
	if (windows[win].buffer_lines[Y-1].justify == JUST_center)
	  fill = fill/2;

	if (windows[win].buffer_lines[Y-1].justify == JUST_both) {
	  fill = -fill;
	  if (i == WL-1) fill = 0;
	}
	
	if (DEBUG_WRAP || DEBUG_MWRAP)
	  print_debug("  (win=%d,%d,%d) justify: margin=%d "
		      "len=%d left=%d fill=%d",
		      win,Y,i+1,marginlist[i],clen,left,fill);
	filllist[i] = fill;	  
      }
    }


   /* Adjust line_mapping */
   if (WL < W1) {
     int vl = visible_len(win);
     int bl = windows[win].first+vl-1;
		
     /* We need visible_len before when mapping_len is updated
	(window is shrinking) */
    
     for (i = ry1 + W1; i < windows[win].mapping_len; i++) {
       windows[win].line_mapping[i-W1+WL] =
	 windows[win].line_mapping[i];
       windows[win].line_mapping[i-W1+WL].pagenum += pg - lastpg;
     }
     windows[win].mapping_len += -W1+WL;
     
     for (i=Y; i < windows[win].buffer_len; i++)
       windows[win].buffer_lines[i].wrappos += -W1+WL;
   	
     if (windows[win].mapped &&
	 ((YS+WL >= windows[win].first && YS+WL <= bl) || 
	  (YS+W1 >= windows[win].first && YS+W1 <= bl))) {
       
       if (YS+WL >= windows[win].first && YS+WL <= bl) {
	 set_margins(windows[win].first,bl);
	 
	 for (i=0; i < W1-WL; i++) delete_line(YS+WL);
       
	 set_margins(-1,-1);
       }
       
       /* print_win_line may need this */
       for (i=0; i < WL; i++) {
	 windows[win].line_mapping[ry1+i].offset=wraplist[i];
	 windows[win].line_mapping[ry1+i].fill=filllist[i];
	 windows[win].line_mapping[ry1+i].indent=indentlist[i];
	 windows[win].line_mapping[ry1+i].pagenum=pglist[i];
	 windows[win].line_mapping[ry1+i].line = Y-1;
       }
       
       for (i=0; i < W1-WL && i < vl; i++)
	 print_win_line(win,
			bl+windows[win].first_line+
			-windows[win].first-i,
			bl-i);
       
     } else if (YS+WL < windows[win].first) {
       windows[win].first_line -= W1-WL;
     }
     
   } else if (WL > W1) {
     windows[win].line_mapping = (struct wrappos *)
       REALLOC((void *)windows[win].line_mapping,
	       (windows[win].mapping_len-W1+WL)*
	       sizeof(struct wrappos));
     for (i = windows[win].mapping_len-1; i >= ry1+W1; i--) {
       windows[win].line_mapping[i-W1+WL] =
	 windows[win].line_mapping[i];
       windows[win].line_mapping[i-W1+WL].pagenum += pg - lastpg;
     }
     windows[win].mapping_len += -W1+WL;
     
     /* We need visible_len after when mapping_len is updated
	(window is growing) */
     
     for (i=Y; i < windows[win].buffer_len; i++)
       windows[win].buffer_lines[i].wrappos += -W1+WL;
     
     if (windows[win].mapped &&
	 YS+W1 >= windows[win].first &&
	 YS+W1 <= windows[win].first+visible_len(win)-1) {
       
       set_margins(windows[win].first,
		   windows[win].first+visible_len(win)-1);
       
       for (i=0; i < WL-W1; i++) insert_line(YS+W1);
       set_margins(-1,-1);
       
     } else if (YS+W1 < windows[win].first) {
       windows[win].first_line -= W1-WL;
     }
     
   }
    
   for (i=0; i < WL; i++) {
     windows[win].line_mapping[ry1+i].offset=wraplist[i];
     windows[win].line_mapping[ry1+i].fill=filllist[i];
     windows[win].line_mapping[ry1+i].indent=indentlist[i];
     windows[win].line_mapping[ry1+i].pagenum=pglist[i];
     windows[win].line_mapping[ry1+i].line = Y-1;
   }

    if (indentlist != sindentlist)
      FREE((void *)indentlist);
    
    if (wraplens != swraplens)
      FREE(wraplens);
    
    if (wraplist != swraplist)
      FREE((void *)wraplist);

    if (filllist != sfilllist)
      FREE((void *)filllist);

    if (marginlist != smarginlist)
      FREE((void *)marginlist);

    if (pglist != spglist) 
      FREE((void *)pglist);
}

static void refresh_line(int win, int Y) {
  int line1 = windows[win].buffer_lines[Y-1].wrappos;
  int line2 = Y < windows[win].buffer_len ?
	windows[win].buffer_lines[Y].wrappos :
        windows[win].mapping_len;
  int rline;

  for (rline = line1; rline < line2; rline++)
  	if (rline >= windows[win].first_line &&
      	    rline < windows[win].first_line + visible_len(win))
    	  print_win_line(win,
		         rline,
		         rline-windows[win].first_line+windows[win].first);
}

void get_line_status(int win,int Y, int *indent, int *rindent, 
		     int *submargin, int *justify,
		     int *excerpt_count, int *excerpt_flag,
		     int *force_wrap) {
  reserve_line(win,Y);
  *indent =  windows[win].buffer_lines[Y-1].indent;
  *rindent =  windows[win].buffer_lines[Y-1].rindent;
  *submargin =  windows[win].buffer_lines[Y-1].submargin;
  *justify =  windows[win].buffer_lines[Y-1].justify;
  *excerpt_count =  windows[win].buffer_lines[Y-1].excerpt_count;
  *excerpt_flag =  windows[win].buffer_lines[Y-1].excerpt_flag;
  *force_wrap = windows[win].buffer_lines[Y-1].force_wrap;
}

static void repage_window(int win);

void set_line_status(int win,int Y, int indent, int rindent,int submargin,
		     int justify, int excerpt_count, int excerpt_flag,
		     int force_wrap) {
  reserve_line(win,Y);
  windows[win].buffer_lines[Y-1].indent = indent;
  windows[win].buffer_lines[Y-1].rindent = rindent;
  windows[win].buffer_lines[Y-1].submargin = submargin;
  windows[win].buffer_lines[Y-1].justify = justify;
  windows[win].buffer_lines[Y-1].excerpt_count = excerpt_count;
  windows[win].buffer_lines[Y-1].excerpt_flag = excerpt_flag;
  windows[win].buffer_lines[Y-1].force_wrap = force_wrap;

  if (DEBUG_WRAP || DEBUG_MWRAP) {
    print_debug("set_line_status(win=%d,Y=%d,indent=%d,rindent=%d"
		",submargin=%d,justify=%d,excerpt_count=%d,excerpt_flag=%d"
		",force_wrap=%d)",
		win,Y,indent,rindent,submargin,justify,
		excerpt_count,excerpt_flag,force_wrap);
  }

  wrap_line(win,Y);
  repage_window(win);

  if (!windows[win].mapped) return;

  refresh_line(win,Y);
}

void set_win_hl(int win,int flags,int X1,int X2,int Y) {
  int old_Y = hl_Y;
  int old_win = hl_win;
  
  hl_win = win;
  hl_flags = flags;
  hl_Y = Y;
  hl_X1 = X1;
  hl_X2 = X2;
  
  if (-1 != old_Y && -1 != old_win &&
      old_Y > 0 && old_Y <= windows[old_win].buffer_len)
    refresh_line(old_win,old_Y);
  
  if (-1 != hl_Y && -1 != hl_win &&
      hl_Y > 0 && hl_Y <= windows[hl_win].buffer_len)
    refresh_line(hl_win,hl_Y);
}

void print_to_window(int win,int X, int Y, int flags,int len,CHAR_IDX *mapped)
{
  int i;

  if (-1 == Y) Y = windows[win].buffer_len +1;
  if (-1 == X) {
    if (Y <= windows[win].buffer_len)
      X = windows[win].buffer_lines[Y-1].len +1;
    else
      X = 1;
  }

  for (i = 0; i < len; i++)
    add_char(win,X+i,Y,flags,mapped[i]);

  wrap_line(win,Y);
  repage_window(win);

  if (!windows[win].mapped) return;

  refresh_line(win,Y);
  calculate_prompt();
}

void clear_window_line(int win,int Y) {
  if (-1 == Y) Y = windows[win].buffer_len +1;
  reserve_line(win,Y);

  if (windows[win].buffer_lines[Y-1].chars) {
    FREE((void *)windows[win].buffer_lines[Y-1].chars);
    windows[win].buffer_lines[Y-1].chars = NULL;
    windows[win].buffer_lines[Y-1].len = 0;
  }

  reset_line_flags(win,Y);

  wrap_line(win,Y);
  repage_window(win);

  if (!windows[win].mapped) return;

  refresh_line(win,Y);
  calculate_prompt();
}

void cut_window_line(int win,int X, int Y) {
  if (1 == X) {
    clear_window_line(win,Y);
    return;
  }
  reserve_line(win,Y);
  
  if (windows[win].buffer_lines[Y-1].len < X) return;
  
  windows[win].buffer_lines[Y-1].len = X-1;
  
  wrap_line(win,Y);
  repage_window(win);

  if (!windows[win].mapped) return;
  
  refresh_line(win,Y);
  calculate_prompt();
}

int win_is_page_top(int win) {
  if (!windows[win].pagemode) return windows[win].first_line < 1;
  if (windows[win].first_line < 1) return 1;
  if (windows[win].first_line > windows[win].mapping_len) return 0;
  return windows[win].curpage != 
    windows[win].line_mapping[windows[win].first_line-1].pagenum;
}

static void repage_window(int win) {
  int lastpage = windows[win].curpage;
  int newpage=0;

  if (windows[win].pagemode && 
      windows[win].mapping_len > windows[win].first_line) {
    if (windows[win].first_line >= 0) 
      newpage = windows[win].line_mapping[windows[win].first_line].pagenum;
    else if (windows[win].mapping_len > 0)
      newpage = windows[win].line_mapping[0].pagenum;
  } else if (windows[win].pagemode && windows[win].mapping_len > 0)
    newpage = windows[win].line_mapping[windows[win].mapping_len-1].pagenum;

  if (newpage != lastpage) {
    int i;
    if (DEBUG_WRAP || DEBUG_MWRAP) 
      print_debug("repage_window: Current page changed - new=%d (old=%d)",
		  newpage,lastpage);
    windows[win].curpage = newpage;

    if (!windows[win].mapped) return;
 
    for (i = 1; i <= windows[win].buffer_len; i++)
      refresh_line(win,i);

    calculate_prompt();
  }
}

static void rewrap_window(int win) {
  int line;
  for (line = 0; line < windows[win].buffer_len; line++) 
    wrap_line(win,line+1);
  /* if lines are refressed before repaging, lines can be
     erased and immediately reprinted !! */
  repage_window(win);
  for (line = 0; line < windows[win].buffer_len; line++) 
    refresh_line(win,line+1);	
}

void get_window_status(int win,int *tab, int *wrap, int *showspc,
		       int *tab_step, int *indent_step, int *priority,
		       int *pagemode) {
  *tab = windows[win].enable_tab;
  *wrap = windows[win].enable_wrapping;
  *showspc = windows[win].showspc;
  *tab_step = windows[win].tab_step;
  *indent_step = windows[win].indent_step;
  *priority = windows[win].priority;
  *pagemode = windows[win].pagemode;
}

#define BOOL(x) (!(!(x)))
void set_window_status(int win,int tab, int wrap, int showspc,
		       int tab_step, int indent_step, int priority,
		       int pagemode) {
  int prev_pri = windows[win].priority;
  /* booleans: */
  windows[win].enable_tab      = BOOL(tab);
  windows[win].enable_wrapping = BOOL(wrap);
  windows[win].showspc         = BOOL(showspc);
  windows[win].pagemode        = BOOL(pagemode);
  /* values: */
  windows[win].tab_step    = tab_step;
  windows[win].indent_step = indent_step;
  windows[win].priority    = priority;
  if (DEBUG_WRAP || DEBUG_MWRAP) 
    print_debug("set_window_status(win=%d,tab=%d,wrap=%d,showspc=%d,"
		"tab_step=%d,indent_step=%d,priority=%d,"
		"pagemode=%d)",
		win,tab,wrap,showspc,
		tab_step,indent_step,priority,
		pagemode);

  rewrap_window(win);
  flush_to_priority(prev_pri < priority ? prev_pri : priority);
  calculate_prompt();
}
#undef BOOL

void rewrap_windows(void) {
  int win;

  for (win = 0; win < WINDOW_COUNT; win++) 
    rewrap_window(win);
  calculate_prompt();
}

void unmap_window(int win)  {
  int previous = -1,i;
  int Y = windows[win].first;
  int off = windows[win].first_line;
  int adding = 0;
  int vlen = visible_len(win);

  if (off < 0) adding = -off;

  for (i = 0; i < win; i++) if (windows[i].mapped)
    previous = i;

  windows[win].mapped = 0;
  if (-1 == previous) {
    if (was_last(win))
      clear_from_line(Y+adding);
    else
      for (i = adding; i < vlen; i++)
	clear_line(Y+i);
  } else {
    int off2 = windows[previous].first_line;
    int adding2 = 0;
    if (off2 < 0) adding2 = -off2;
    if (adding2 < adding) adding  = adding2;
    print_win_tail(previous,Y-windows[previous].first+adding);
  }
  calculate_prompt();
}

void scroll_window(int win) {
  int Y=windows[win].first,vlen=visible_len(win);

  windows[win].first_line++;

  if (!windows[win].mapped || vlen < 1) {
    repage_window(win);
    return;
  }

  scroll_area(Y,Y+vlen-1);
  /*
     set_margins(Y,Y+vlen-1);
     
     delete_line(Y);
     
     set_margins(-1,-1);
     */

  /* Needed if MAX_LINES defined */
  print_win_tail(win,vlen-1);
  repage_window(win);

  calculate_prompt();
}

void scroll_window_R(int win) {
  int Y=windows[win].first,vlen=visible_len(win);

  windows[win].first_line--;

  if (!windows[win].mapped || vlen < 1) {
    repage_window(win);
    return;
  }
  
  if (!was_last(win)) set_margins(Y,Y+vlen-1);
  
  insert_line(Y);
  
  set_margins(-1,-1);

  print_win_line(win,windows[win].first_line,Y);
  repage_window(win);

  calculate_prompt();
}

void map_window(int win,int line,int Y) {
  int clear_area = -1;
  int priority = windows[win].priority;

  windows[win].priority = 0;
  if (windows[win].mapped && windows[win].first != Y) 
    unmap_window(win);
  else if (windows[win].mapped) {
    clear_area = -windows[win].first_line;
    if (clear_area < 0) clear_area = 0;
  }
  
  windows[win].first = Y;
  windows[win].first_line = line;
  windows[win].mapped = 1;
  windows[win].priority = priority;

  repage_window(win);
  print_window(win,clear_area);
  calculate_prompt();
}

int window_lines(int win) {
  return windows[win].buffer_len;
}

int window_rlines(int win) {
  return windows[win].mapping_len;
}

void window_pos(int win,int *Y,int *offset, int *roffset,
		int *pagenum, int *pages) {
  *Y = windows[win].first;
  *roffset = windows[win].first_line;
  if (windows[win].first_line < 0) *offset = windows[win].first_line;
  else if (windows[win].first_line >= windows[win].mapping_len) 
    *offset = windows[win].first_line + windows[win].buffer_len 
      - windows[win].mapping_len;
  else 
    *offset = windows[win].line_mapping[windows[win].first_line].line;
  *pagenum = windows[win].curpage;
  *pages=0;
  if (windows[win].mapping_len > 0)
    *pages=windows[win].line_mapping[windows[win].mapping_len-1].pagenum;
}

void clear_window(int win) {
  int i,DIFF=0;
  int clear_area = -1;
  if (!windows[win].buffer_lines) return;

  if (windows[win].first_line > 0) {
    if (windows[win].first_line < windows[win].mapping_len)
      DIFF = windows[win].first_line - 
	windows[win].line_mapping[windows[win].first_line].line;
    else
      DIFF = windows[win].mapping_len - windows[win].buffer_len;
  }

  clear_area = -windows[win].first_line;
  if (clear_area < 0) clear_area = 0;

  for (i = 0; i < windows[win].buffer_len; i++) {
    if (windows[win].buffer_lines[i].chars) {
      FREE((void *)windows[win].buffer_lines[i].chars);
      windows[win].buffer_lines[i].chars = NULL;
      windows[win].buffer_lines[i].len = 0;
    }
  }  

  FREE((void *)windows[win].buffer_lines);
  FREE((void *)windows[win].line_mapping);
  windows[win].buffer_lines = NULL;
  windows[win].line_mapping = NULL;
  windows[win].buffer_len = 0;
  windows[win].mapping_len = 0;

  windows[win].enable_tab = 1;
  windows[win].enable_wrapping = 1;
  windows[win].showspc = 0;
  windows[win].tab_step = 8;
  windows[win].indent_step = 4;
  windows[win].priority = 0;
  windows[win].pagemode = 0;
  windows[win].first_line -= DIFF;

  if (hl_win == win) hl_win = -1;
  repage_window(win);
  
  if (!windows[win].mapped) return;
  
  print_window(win,clear_area);
  calculate_prompt();
}

static int eq_pair(int win,CHAR_IDX M1, CHAR_IDX M2) {
  if (M1 == M2) return 1;
  if (is_hyphen(win,M1) && is_hyphen(win,M2)) return 1;
  if (is_space(win,M1) && is_space(win,M2)) return 1;
  return 0;
}

int find_buffer_text(int win,int len,CHAR_IDX *mapped,int *X,int *Y,int *rY,
		     int *mlen) {
  int y;
  if (*Y < 1) *Y = 1;
  if (*X < 1) *X = 1;

  for (y = *Y-1; y < windows[win].buffer_len; y++) {
    int x;
    for (x = y == *Y-1 ? *X-1 : 0; x < windows[win].buffer_lines[y].len; x++) {
      int i,j;
      for (i = 0, j=0; 
	   i < len && x + j < windows[win].buffer_lines[y].len; i++,j++) 
	if (!eq_pair(win,windows[win].buffer_lines[y].chars[x+j].mapped_ch,
		    mapped[i])) {
	  if (CH_MAR_B == windows[win].buffer_lines[y].chars[x+j].mapped_ch)
	    i--;
	  else if (CH_MAR_E == 
		   windows[win].buffer_lines[y].chars[x+j].mapped_ch)
	    i--;
	  else if (CH_IND_B == 
		   windows[win].buffer_lines[y].chars[x+j].mapped_ch)
	    i--;
	  else if (CH_IND_E == 
		   windows[win].buffer_lines[y].chars[x+j].mapped_ch)
	    i--;
	  else if (CH_SHY == windows[win].buffer_lines[y].chars[x+j].mapped_ch)
	    i--;
	  else if (CH_SHY == mapped[i])
	    j--;
	  else break;
	}
	  
      if (len == i) {
	int ml;
	*X = x+1;
	*Y = y+1;
	ml = windows[win].buffer_lines[y].wrappos;
	while(ml+1 < windows[win].mapping_len &&
	      y == windows[win].line_mapping[ml+1].line &&
	      x >= windows[win].line_mapping[ml+1].offset)
	  ml++;
	*rY = ml+1;
	*mlen = j;
	return 1;
      }
    }
  }
  return 0;
}

CHAR_IDX read_win_char(int win, int X, int Y) {
  if (Y > windows[win].buffer_len) {
    print_notify(
		 "(%s) Software error/read_win_char: Y=%d > windows[win=%d].buffer_len=%d",
		 prog,Y,win,windows[win].buffer_len);
    close_files(); close_terminal();
    exit(8);
  }

  if (-1 == X) 
    X = windows[win].buffer_lines[Y-1].len;
  
  if (0 == X || X > windows[win].buffer_lines[Y-1].len) 
    return CH_NL;

  return windows[win].buffer_lines[Y-1].chars[X-1].mapped_ch;
}

int win_line_len(int win,int Y) {
  if (Y > windows[win].buffer_len) {
    print_notify(
		 "(%s) Software error/win_line_len: Y=%d > windows[win=%d].buffer_len=%d",
		 prog,Y,win,windows[win].buffer_len);
    close_files(); close_terminal();
    exit(8);
  }
  return windows[win].buffer_lines[Y-1].len;
}

extern void flush_window(int win) {
  if (!windows[win].mapped) return;
  flush_to_priority(windows[win].priority);
}
