/*  file: terminal.c
 *
 *  kehpager, Charset aware pager, Kari E. Hurtta
 *
 *  Copyright (c) 1993, 1994 Kari E. Hurtta
 *
 *  Redistribution and use in source and binary forms are permitted
 *  provided that the above copyright notice and this paragraph are
 *  duplicated in all such forms. This software is provided 'as is'
 *  and without any warranty. 
 */

#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <stdarg.h>

#include <termios.h>
#include <unistd.h>

#include <sys/types.h>
#include <sys/time.h>
#include <sys/wait.h>

#if defined(__386BSD__) || defined(__FreeBSD__) || defined(__NetBSD__)
#include <sys/ioctl.h>
#endif

/* #include <signal.h> terminal.h includes this */
#include "kehpager.h"
#include "charset.h"
#include "terminal.h"
#include "control.h"
#include "memory.h"
#include "rc.h"
#include "keyboard.h"

extern int sys_nerr;
extern char *sys_errlist[];
extern int errno;

#ifndef __sgi
extern int sigaction (int sig, const struct sigaction *act, 
		      struct sigaction *oact);
#endif

#if defined(SUNOS5) || defined(_HPUX_SOURCE) || defined(__386BSD__) || defined(__FreeBSD__) || defined(__NetBSD__)
#define PID pid_t 
#else
#define PID int 
#endif

extern PID getpid(void);
extern int kill(PID pid, int sig);
extern PID fork(void);
extern PID waitpid(PID pid, int *statusp, int options);
extern int pause(void);

int terminal = -1;
int printerror = 1;
static int termios_set = 0;
static volatile int tty_state_set = 0;
static pid_t oldgroup = 0;
static PID mypid = 0;
static int stderr_isterm = 1;
static volatile PID myfriendpid = 0;
static volatile int allow_dead = 0;
static volatile int have_group = 0;
/* We no have terminal longer */
volatile int terminal_have_lost = 0;

static struct termios tty_state, tty_origstate;

extern int tcsetpgrp(int fd, pid_t pgrp_id);
extern pid_t tcgetpgrp(int fd);
extern int setpgid(pid_t pid, pid_t pgrp);

#ifdef SUNOS4
extern int getpgid(int pid);
#else
extern pid_t getpgrp(void);
#endif

volatile int lines=24, columns=80;
int def_lines = 24, def_columns = 80; /* import_vars chages these */

init_item terminal_items[] = {
  { &def_lines,   "use.terminal.lines", V_NUM },
  { &def_columns, "use.terminal.columns", V_NUM },
  { NULL, NULL, V_LAST }
};

void BLOCK  (block_state *s) {
  sigset_t blockmask;
  sigemptyset(&blockmask);

  sigaddset(&blockmask,SIGTTOU);
  sigaddset(&blockmask,SIGTSTP);
  sigaddset(&blockmask,SIGTTIN);
  sigaddset(&blockmask,SIGINT);
  sigaddset(&blockmask,SIGWINCH);

  if (-1 == sigprocmask(SIG_BLOCK,&blockmask,s))
    abort();

}
extern void UNBLOCK(block_state *s) {
  if (-1 == sigprocmask(SIG_SETMASK,s,NULL))
    abort();
}


char *error_text(int code) {
  if (code < sys_nerr) return sys_errlist[code];
  return "Unknow error";
}

static void put_stderr(CHAR *text) {
  CHAR buffer[ERROR_LEN+80],*c=buffer,*c1;
  int res,ptr=0;

  if (stderr_isterm && termios_set) *c++ = '\r';
  for (c1=text; '\0' != *c1; c1++) {
    if (stderr_isterm && termios_set && '\n' == *c1) 
      *c++ = '\r';
    *c++ = *c1;
  }

  /* to stderr */
 again:
  if (-1 == (res = write(2,Cs(buffer+ptr),c-buffer-ptr))) {
    if (EINTR == errno) goto again;
    return;  /* We fail */
  } else if (res < c-buffer-ptr) {
    ptr += res;
    goto again;
  }
}

void feed_queued_error(CHAR *message) {
  /* hopeless ... */
  if (terminal_have_lost) return;
  
  flush_buffer(1); /* print pending output */

  if (stderr_isterm)
    put_stderr(message);
}

void print_error(int code, char *frm, ...) {
  va_list args;
  char buffer [ERROR_LEN];

  if (!printerror) return;

  va_start(args,frm);
  vsprintf(buffer,frm,args);
  va_end(args);
  sprintf(&(buffer[strlen(buffer)]),": %s (errno=%d)\n",error_text(code),code);

  /* hopeless ... */
  if (stderr_isterm && terminal_have_lost) return;

  flush_buffer(1); /* print pending output */

  if ((!terminal_have_lost && !error_callback(rCs(buffer))) || !stderr_isterm) 
    put_stderr(rCs(buffer));
}

void print_notify(char *frm, ...) {
  va_list args;
  char buffer [ERROR_LEN];

  if (!printerror) return;

  va_start(args,frm);
  vsprintf(buffer,frm,args);
  va_end(args);
  sprintf(&(buffer[strlen(buffer)]),"\n");

  /* hopeless ... */
  if (stderr_isterm && terminal_have_lost) return;

  flush_buffer(1); /* print pending output */

  if ((!terminal_have_lost && !error_callback(rCs(buffer))) || !stderr_isterm) 
    put_stderr(rCs(buffer));
}

void print_debug(char *frm, ...) {
  va_list args;
  char buffer [ERROR_LEN];

  if (!printerror) return;

  va_start(args,frm);
  vsprintf(buffer,frm,args);
  va_end(args);
  sprintf(&(buffer[strlen(buffer)]),"\n");

  /* hopeless ... */
  if (stderr_isterm && terminal_have_lost) return;

  put_stderr(rCs(buffer));
}

static void get_winsize(int silent ) {
  struct winsize buffer;

  if (DEBUG_CURSOR) {
    print_debug("Changing terminal size from: lines=%d columns=%d",
		lines,columns);
  }

  if (-1 == terminal) 
    return;

  lines = def_lines;
  columns = def_columns;
  
 again:
  if (-1 == ioctl(terminal,TIOCGWINSZ,&buffer)) {
    int code = errno;
    if (EINTR == code) goto again;

    if (DEBUG_CURSOR) {
      print_debug("  Using default terminal size: lines=%d columns=%d",
		  lines,columns);
    }

    if (!silent) {
      print_error(code,"(%s) get_winsize/Can't get tty size of /dev/tty",prog);
    }
    return;
  }
  if (buffer.ws_row) lines =  buffer.ws_row;
  if (buffer.ws_col) columns = buffer.ws_col;
  if (DEBUG_CURSOR) {
    print_debug("  Using terminal size: lines=%d columns=%d",
		lines,columns);
  }
}

void set_terminal_size(int col,int lin) {
  struct winsize buffer;

  lines=lin; columns=col;

  /* initialize structure (notice that we don't know all possible fields
     of structure) */
 again1:
  if (-1 == ioctl(terminal,TIOCGWINSZ,&buffer)) {
    int code = errno;
    if (EINTR == code) goto again1;
    /* (silent failure)
       print_error(code,"(%s) set_terminal_size/Can't get tty size of /dev/tty",prog);
       */
    return;
  }

  buffer.ws_row = lin;
  buffer.ws_col = col;
  again2:
    if (-1 == ioctl(terminal,TIOCSWINSZ,&buffer)) {
      int code = errno;
      if (EINTR == code) goto again2;
      print_error(code,"(%s) set_terminal_size/Can't set tty size of /dev/tty",prog);
    }
  /* if terminal driver does some fancy */
  lines=lin;
  columns=col;
}

static void QUIT_pager(int sig);

static void check_terminal_group(void) {
  int cgroup;
 again:
  while (oldgroup != (cgroup = tcgetpgrp(terminal))) {
    if (-1 == cgroup) {
      int code = errno;
      if (EINTR == code) goto again;

      terminal_have_lost = 1; /* default */
 
      print_error(code,"(%s) check_terminal_group/Can't get current group",prog);
      QUIT_pager(0);
    }
    if(DEBUG_SIGNAL)
      print_debug("Terminal group = %d, my group = %d. Suspending...",
		  cgroup,oldgroup);
	
    kill(mypid,SIGSTOP); /* wait that I am in foreground */

    if(DEBUG_SIGNAL)
      print_debug("Wake up");
  }
}

void set_pager_ttystate(void) {
  if (!termios_set) return;
  
  check_terminal_group();

  if (have_group) {
    sigset_t oldmask,blockmask;
    
    sigemptyset(&blockmask);
    sigaddset(&blockmask,SIGTTOU);
    if (-1 == sigprocmask(SIG_BLOCK,&blockmask,&oldmask)) {
      int code = errno;
      print_error(code,"(%s) set_pager_ttystate/Can't block SIGTTOU",prog);
    } else {
      
    again0:
      if (-1 == tcsetpgrp(terminal,mypid)) {
	
	int code = errno;
	if (EINTR == code) goto again0;
	
      print_error(code,"(%s) set_pager_ttystate/Can't change terminal group to %d (my pid)",
		  prog,mypid);
      } else if(DEBUG_SIGNAL || DEBUG_TERM)
	print_debug("Terminal group is now changed to %d (my pid)",mypid);
      
      if (-1 == sigprocmask(SIG_SETMASK,&oldmask,NULL)) {
	int code = errno;
	print_error(code,"(%s) set_pager_ttystate/Can't unblock SIGTTOU",prog);
      }
    }
  }

 again1:
  if (-1 == tcsetattr(terminal,TCSADRAIN,&tty_state)) { /* when output is wrote */
    int code = errno;
    if (EINTR == code) goto again1;
    print_error(code,"(%s) set_pager_ttystate/Can't change tty state of /dev/tty",prog);
  } else tty_state_set = 1;
  get_winsize(1);
}

void set_caller_ttystate(void ) {
  if (!termios_set) return;
 again1:
  if (-1 == tcsetattr(terminal,TCSADRAIN,&tty_origstate)) {
    int code = errno;
    if (EINTR == code) goto again1;
    print_error(code,"(%s) set_caller_ttystate/Can't restore tty state for /dev/tty",prog);
  } else tty_state_set = 0;

  if (have_group) {
  again2:
    if (-1 == tcsetpgrp(terminal,oldgroup)) {
      
      int code = errno;
      if (EINTR == code) goto again2;
      
      print_error(code,"(%s) set_caller_ttystate/Can't change terminal group to %d",
		  prog,oldgroup);
    } else if(DEBUG_SIGNAL || DEBUG_TERM)
      print_debug("Terminal group is now restored to %d",oldgroup);
  }
}

static void deliver_signal(int sig)  {
  if (!have_group || mypid == oldgroup) return;
  
  if (-1 == kill(-oldgroup,sig)) {
    int code = errno;
    print_error(code,"(%s) deliver_signal/Can't deliver signal %d to group %d",
		prog,sig,oldgroup);
  } else if (DEBUG_SIGNAL) 
    print_notify("Signal %d delivered to group %d",sig,oldgroup);
}

static void stop_me(void) {

  if (have_group && mypid != oldgroup) {
    if (-1 == setpgid(mypid,oldgroup)) {
      int code = errno;
      print_error(code,"SUSPENDING CANCELED!\n(%s) stop_me/Can't join again my old group = %d",
		  prog, oldgroup);

      /* IF we now stop, we stuck when kehpager is resumed !!! */
      return;
    } else if (DEBUG_SIGNAL || DEBUG_TERM)
      print_debug("Joined to old terminal group %d",oldgroup);
  }

  if (DEBUG_SIGNAL)
    print_debug("Sending SIGSTOP to myself");
  kill(mypid,SIGSTOP);
  if (DEBUG_SIGNAL)
    print_debug("Woked up from SIGSTOP");

  if (have_group && mypid != oldgroup) {
    if (-1 == setpgid(mypid,mypid)) {
      int code = errno;
      print_error(code,"(%s) stop_me/Can't recreate again my new group = %d (my pid)",
		  prog,mypid);
      have_group = 0;
    } else if (DEBUG_SIGNAL || DEBUG_TERM)
      print_debug("Joined to new terminal group %d (my pid)",mypid);
  }
}

static void friend_have_died(int res) {
  if (allow_dead) return;
  print_notify("(%s) Reserver process (%d) died - exiting (%d)...",
	       prog,res,mypid);

  myfriendpid = 0;

  reset_terminal_state();    /* give original terminal characters */
  set_caller_ttystate();

  

  close_files();
  close_terminal();

  if (DEBUG_SIGNAL) 
    print_debug("Exit  ...");
		
  kill(mypid,SIGTERM);          /* Kill me */
  exit(6);

}

static void kill_friend(void) {
  if (!myfriendpid) return;

  allow_dead = 1;  
  if (-1 == kill(myfriendpid,SIGKILL)) {
    int code =errno;
    print_error(code,"(%s) kill_friend/Error when killing company process %d",
		prog,myfriendpid);
  }
  else {
    if (DEBUG_SIGNAL)
      print_debug("Process %d killed",myfriendpid);
    myfriendpid = 0;
  }
}

/* friend have got SIGHUP */
static void friend_exit(int sig) {
  _exit(0);
}

static int friend(void) {
  int res,i;
  struct sigaction T,new_sighup;
  
  T.sa_handler = SIG_IGN;
  sigemptyset(&T.sa_mask);
  T.sa_flags = 0;

  if (-1 == (res = fork())) {
    int code = errno;
    print_error(code,"(%s) friend/Can't fork",prog);
    return 0;
  } else if (res > 0) {
    myfriendpid = res;

    if (DEBUG_SIGNAL)
      print_debug("Reserver process = %d",res);

    return 1;
  }

    for (i = 0; i < NSIG; i++) {
      /* SIGIOT or SIGABRT is used by abort() */
#ifdef SIGIOT
    if (i != SIGIOT)
#endif
#ifdef SIGABRT
      if (i != SIGABRT)
#endif
	sigaction(i,&T,NULL);
  }

  new_sighup.sa_handler = friend_exit;
  sigemptyset(&new_sighup.sa_mask);
  new_sighup.sa_flags = 0;
  if (-1 == sigaction(SIGHUP,&new_sighup,NULL))
    abort();

  /* Close standard input and output so that don't interact
   * when kehpager is in pipe 
   */
  close(0);
  close(1);
  /* fd = terminal and fd = 2  are still open */

  while(1)
    pause();

}

static void QUIT_pager(int sig) {
  if (DEBUG_SIGNAL || DEBUG_TERM) {
    if (sig) 
      print_debug("Going down, signal=%d",sig);
    else 
      print_debug("Going down ...");
  }

  reset_terminal_state();    /* give original terminal characters */
  set_caller_ttystate();
  
  if (sig)
    deliver_signal(sig); /* Now my callers turn */
  
  close_files();
  close_terminal();
  
  if (DEBUG_SIGNAL || DEBUG_TERM) 
    print_debug("Killing myself...");
  
  kill(mypid,SIGTERM);          /* Kill me */
  _exit(6);
  abort();
}

static void handle_sighup(int sig) {
  terminal_have_lost = 1;
  
  if (DEBUG_SIGNAL) 
    print_debug("Got   SIGHUP (sig=%d)", sig);

  QUIT_pager(sig);
}

static void handle_sigchld(int sig) {
  int res;

  if (DEBUG_SIGNAL) 
    print_debug("Got   SIGCHLD (sig=%d)", sig);

  if (myfriendpid) {
  again:
    if (0 < (res=waitpid(myfriendpid,NULL,WNOHANG)))
      friend_have_died(res);
    if (-1 == res) {
      int code=errno;
      if (EINTR==code) goto again;
      print_error(code,"(%s) Error when waiting died of process %d",
		  prog,myfriendpid);
    } 
  }
  if (DEBUG_SIGNAL) 
    print_debug("Exit  SIGCHLD (sig=%d)", sig);

}


static void handle_sigwinch(int sig) { /* window size changed */
  if (DEBUG_SIGNAL) 
    print_debug("Got   SIGWINCH (sig=%d)",sig);
  get_winsize(0);
  need_redraw = 1;
  if (DEBUG_SIGNAL) 
    print_debug("Done  SIGWINCH (sig=%d)",sig);
}

void handle_sigtstp(int sig) {   /* User have pressed Ctrl-Z */
  /* In SUNOS5 we need block SIGTTOU */
  if (DEBUG_SIGNAL) 
    print_debug("Got   SIGTSTP (sig=%d)", sig);

  reset_terminal_state();   /* Give original terminal characters */
  flush_buffer(0);          /* and print it to terminal */
  set_caller_ttystate();   
  
  deliver_signal(sig); /* Now my callers turn */

  stop_me();

  set_pager_ttystate();
  need_redraw = 1;

  if (DEBUG_SIGNAL) 
    print_debug("Done  SIGTSTP (sig=%d)", sig);
}

void handle_sigint(int sig) {   /* User have prseed Ctrl-C */
  if (DEBUG_SIGNAL) 
    print_debug("Got   SIGINT (sig=%d)", sig);

  QUIT_pager(sig);
}

void handle_sigttou(int sig) {    /* Output to terminal when backgroud */
  if (DEBUG_SIGNAL) 
    print_debug("Got   SIGTTOU (sig=%d)", sig);

  set_caller_ttystate();         /* This should succeed, because SIGTTOU is 
				  * blocked
				  */

  deliver_signal(sig); /* Now my callers turn */

  stop_me();

  set_pager_ttystate();
  need_redraw = 1;

  if (DEBUG_SIGNAL) 
    print_debug("Done  SIGTTOU (sig=%d)", sig);

}

static struct sigaction old_sigtstp, old_sigttin, old_sigttou, old_sigint, 
old_sigwinch,old_sigchld,old_sighup;
static struct sigaction new_sigtstp, new_sigttin, new_sigttou, new_sigint, 
new_sigwinch,new_sigchld,new_sighup;

static int isval = 0;
static sigset_t signalset;
static int have_set(int sig) {
  if (!isval) return 0;
  return sigismember(&signalset,sig);
}

int open_terminal(void) {
  CHAR ch;
  mypid = getpid();

  sigemptyset(&signalset);
  isval = 1;

  new_sighup.sa_handler = handle_sighup;
  sigemptyset(&new_sighup.sa_mask);
  new_sighup.sa_flags = 0;
  if (-1 == sigaction(SIGHUP,&new_sighup,&old_sighup))
    print_error(errno,"Error setting of SIGHUP (sigaction)");
  else 
    sigaddset(&signalset,SIGHUP);

  new_sigttou.sa_handler = handle_sigttou;
  sigemptyset(&new_sigttou.sa_mask);
  new_sigttou.sa_flags = 0;
  if (-1 == sigaction(SIGTTOU,&new_sigttou,&old_sigttou))
    print_error(errno,"Error setting of SIGTTOU (sigaction)");
  else 
    sigaddset(&signalset,SIGTTOU);

  new_sigtstp.sa_handler = handle_sigtstp;
  sigemptyset(&new_sigtstp.sa_mask);
  new_sigtstp.sa_flags = 0;
  if (-1 == sigaction(SIGTSTP,&new_sigtstp,&old_sigtstp))
    print_error(errno,"Error setting of SIGTSTP (sigaction)");
  else 
    sigaddset(&signalset,SIGTSTP);

  new_sigttin.sa_handler = handle_sigttou;
  sigemptyset(&new_sigttin.sa_mask);
  sigaddset(&new_sigttin.sa_mask,SIGTTOU); 
  /* lets SIGTTOU handler handle this */
  new_sigttin.sa_flags = 0;
  if (-1 == sigaction(SIGTTIN,&new_sigttin,&old_sigttin))
    print_error(errno,"Error setting of SIGTTIN (sigaction)");
  else 
    sigaddset(&signalset,SIGTTIN);

  new_sigint.sa_handler = handle_sigint;
  sigemptyset(&new_sigint.sa_mask);
  new_sigint.sa_flags = 0;
  if (-1 == sigaction(SIGINT,&new_sigint,&old_sigint))
    print_error(errno,"Error setting of SIGINT (sigaction)");
  else 
    sigaddset(&signalset,SIGINT);

  new_sigwinch.sa_handler = handle_sigwinch;
  sigemptyset(&new_sigwinch.sa_mask);
  new_sigwinch.sa_flags = 0;
  if (-1 == sigaction(SIGWINCH,&new_sigwinch,&old_sigwinch))
    print_error(errno,"Error setting of SIGWINCH (sigaction)");
  else 
    sigaddset(&signalset,SIGWINCH);

  new_sigchld.sa_handler = handle_sigchld;
  sigemptyset(&new_sigchld.sa_mask);
  new_sigchld.sa_flags = 0;
  if (-1 == sigaction(SIGCHLD,&new_sigchld,&old_sigchld))
    print_error(errno,"Error setting of SIGCHLD (sigaction)");
  else 
    sigaddset(&signalset,SIGCHLD);

  stderr_isterm = isatty(2);

 again1:
  terminal = open("/dev/tty",O_RDWR);
  if (-1 == terminal) {
    int code=errno;
    if (EINTR==errno) goto again1;

    print_error(code,"(%s) Can't open /dev/tty",prog);
    return 0;
  }

#ifdef SUNOS4
  oldgroup = getpgrp(mypid);
#else
  oldgroup = getpgrp();
#endif

  if (DEBUG_SIGNAL || DEBUG_TERM)
    print_debug("Old process group is %d",oldgroup);

  check_terminal_group();

  if (oldgroup == mypid) 
    have_group = 0;
  else if (!DEBUG_NOGROUP) {
    if (!friend()) {
      close_terminal();
      return 0;
    }
      
    
  again12:
    if (-1 == setpgid(mypid,mypid)) 
      {
	int code = errno;
	if (EINTR == code) goto again12;
	if (EPERM == code) {
	  if (DEBUG_SIGNAL || DEBUG_TERM)
	    print_notify("Can't do new process group for me (EPERM)");
	  have_group = 0;
	} else {
	  print_error(code,"(%s) Can't set process group",prog);
	  have_group = 0;
	}
    } else {
      have_group = 1;
      if (DEBUG_SIGNAL || DEBUG_TERM)
	print_debug("New process group is now created: %d (my pid)",mypid);
    }
  } else {
    print_debug("I will no create own group!");
  }
  
  if (have_group) {
    sigset_t oldmask,blockmask;

    sigemptyset(&blockmask);
    sigaddset(&blockmask,SIGTTOU);
    if (-1 == sigprocmask(SIG_BLOCK,&blockmask,&oldmask)) {
      int code = errno;
      print_error(code,"(%s) Can't block SIGTTOU",prog);
    } else {
      
    again13:
      if (-1 == tcsetpgrp(terminal,mypid)) {
	
	int code = errno;
	if (EINTR == code) goto again13;
	
	print_error(code,"(%s) Can't change terminal group to %d (my pid)",
		    prog,mypid);
      } else if(DEBUG_SIGNAL || DEBUG_TERM)
	print_debug("Terminal group is now changed to %d (my pid)",mypid);

      if (-1 == sigprocmask(SIG_SETMASK,&oldmask,NULL)) {
	int code = errno;
	print_error(code,"(%s) Can't unblock SIGTTOU",prog);
      }
    }
  }

 again2:
  if (-1 == tcgetattr(terminal,&tty_state)) {
    int code=errno;
    if (EINTR == code) goto again2;
    
    print_error(code,"(%s) Can't read tty state from /dev/tty",prog);
    close(terminal);
    terminal = -1;
    return 0;
  }
  tty_origstate = tty_state;
  termios_set =1;

#ifdef VLNEXT
#ifdef __sgi
  if (tty_state.c_line == LDISC1)
    make_default(&key_compose,tty_state.c_cc[VLNEXT]);
#else
  make_default(&key_compose,tty_state.c_cc[VLNEXT]);
#endif
#endif
  make_default(&key_kill,tty_state.c_cc[VINTR]);
  make_default(&key_suspend,tty_state.c_cc[VSUSP]);
#ifdef VDISCARD
  make_default(&key_flush,tty_state.c_cc[VDISCARD]);
#else
#ifdef VFLUSHO
#ifdef __sgi
  if (tty_state.c_line == LDISC1)
    make_default(&key_flush,tty_state.c_cc[VFLUSHO]);
#else
    make_default(&key_flush,tty_state.c_cc[VFLUSHO]);
#endif
#endif
#endif
  make_default(&key_abort,tty_state.c_cc[VQUIT]);  
#ifdef __sgi
  if (tty_state.c_line == LDISC1) {
    make_default(&key_res1,tty_state.c_cc[VSTART]);  
    make_default(&key_res2,tty_state.c_cc[VSTOP]);
  }
#else
  make_default(&key_res1,tty_state.c_cc[VSTART]);  
  make_default(&key_res2,tty_state.c_cc[VSTOP]);
#endif

#ifdef VREPRINT
  make_default(&key_redraw,tty_state.c_cc[VREPRINT]);
#else
#ifdef VRPRNT
#ifdef __sgi
  if (tty_state.c_line == LDISC1)
    make_default(&key_redraw,tty_state.c_cc[VRPRNT]);
#else
  make_default(&key_redraw,tty_state.c_cc[VRPRNT]);
#endif
#endif
#endif
  make_default(&key_eerase,tty_state.c_cc[VKILL]);
  make_default(&key_edel,tty_state.c_cc[VERASE]);


#ifdef __sgi
  if (tty_state.c_line != LDISC1) {
    tty_state.c_line = LDISC1;      /* BSD line discipline */
    tty_state.c_cc[VLNEXT] = 'V'&31;
    make_default(&key_compose,tty_state.c_cc[VLNEXT]);
    tty_state.c_cc[VWERASE] = 'W'&31;
    tty_state.c_cc[VRPRNT] = 'R'&31;
    make_default(&key_redraw,tty_state.c_cc[VRPRNT]);
    tty_state.c_cc[VFLUSHO] = 'O'&31;
    make_default(&key_flush,tty_state.c_cc[VFLUSHO]);
    tty_state.c_cc[VSTOP] = 'S'&31;
    make_default(&key_res2,tty_state.c_cc[VSTOP]);
    tty_state.c_cc[VSTART] = 'O'%31;
    make_default(&key_res1,tty_state.c_cc[VSTART]);  
  }
#endif


  /* LNEXT Character is SunOS 4 and SunOS 5 works differently */
#if defined(VLNEXT) && defined(VLNEXT_preserved)
  if (control_key(&key_compose,&ch)) {
    tty_state.c_cc[VLNEXT] = ch;
  } else
#ifdef SUNOS4
  /* SunOS 4 not include _POSIX_VDISABLE in <unistd.h> */
    tty_state.c_cc[VLNEXT] = 0;
#else
    tty_state.c_cc[VLNEXT] = _POSIX_VDISABLE;
#endif
#elif defined(VLNEXT) && !defined(VLNEXT_preserved)
  tty_state.c_cc[VLNEXT] = _POSIX_VDISABLE;
#endif
 
  if (control_key(&key_kill,&ch)) 
    tty_state.c_cc[VINTR] = ch;

#ifdef VSUSP
  if (control_key(&key_suspend,&ch)) 
    tty_state.c_cc[VSUSP] = ch;
#else
#ifdef VSWTCH
  if (control_key(&key_suspend,&ch)) 
    tty_state.c_cc[VSWTCH] = ch;
#endif
#endif
#ifdef VDISCARD
  if (control_key(&key_flush,&ch)) 
    tty_state.c_cc[VDISCARD] = ch;
#else
#ifdef VFLUSHO
  if (control_key(&key_flush,&ch)) 
    tty_state.c_cc[VFLUSHO] = ch;
#endif
#endif

  tty_state.c_cc[VMIN] = WANNA_CHARACTERS;    /* want this many chars */
  tty_state.c_cc[VTIME] = timer_per_char;   /* timeout 0.1 seconds per character */
  /* c_lflag */
  tty_state.c_lflag &= ~ICANON;    /* No line editor */
  tty_state.c_lflag &= ~ECHO;      /* No echo */
#ifdef TOSTOP
  tty_state.c_lflag |= TOSTOP;     /* Send SIGTTOU for background output */
#else
#ifdef ITOSTOP
  /* not yet implemented? */
  tty_state.c_lflag |= ITOSTOP;    /* Send SIGTTOU for background output */
#endif
#endif
  /* c_iflag */
  tty_state.c_iflag &= ~ICRNL;     /* No CR -> LF mapping on input */
  tty_state.c_iflag &= ~ISTRIP;    /* Don't strip 8-bit on input */
  tty_state.c_iflag &= ~IGNCR;     /* Don't ignore CR */
  tty_state.c_iflag &= ~INLCR;     /* No LF -> CR mapping on input */
#ifdef IUCLC
  tty_state.c_iflag &= ~IUCLC;     /* No upper-case -> lower-case mapping on 
				      input */
#endif

  /* c_oflag */
#ifdef ONLCR
  tty_state.c_oflag &= ~ONLCR;     /* No LF -> CR LF mapping on output */
#else
  HAZ_DOES(HAZ_ONLCR);
#endif
#ifdef OCRNL
  tty_state.c_oflag &= ~OCRNL;     /* No CR -> LF mapping on output */
#endif
#ifdef OLCUC
  tty_state.c_oflag &= ~OLCUC;     /* No lower-case -> upper-case mappin on 
				      output */
#endif
#if (defined(TABDLY) && (defined(XTABS)) || defined(TAB3)) || defined(OXTABS)
#if defined(TABDLY) && defined(XTABS)
/* SunOS */
  if ((tty_state.c_oflag & TABDLY) == XTABS) {
    tty_state.c_oflag &= ~TABDLY;  /* no TAB -> SPCs expanding */
    tty_state.c_oflag |= TAB0;     
  }
#endif
#if defined(TABDLY) && defined(TAB3) && !defined(XTABS)
/* HPUX && SGI */
  if ((tty_state.c_oflag & TABDLY) == TAB3) {
    tty_state.c_oflag &= ~TABDLY;  /* no TAB -> SPCs expanding */
    tty_state.c_oflag |= TAB0;     
  }
#endif
#ifdef OXTABS
/* netBSD(?) */
  tty_state.c_oflag   &= ~OXTABS;  /* No TAB -> SPCs expanding */
#endif
#else
  HAZ_DOES(HAZ_TAB);
#endif


  /* This kills telnetd and rlogind: !! */
  /* tty_state.c_cflag &= CSIZE;
  tty_state.c_cflag |= CS8; */       /* Character size is 8-bits */

 again3:  
  if (-1 == tcsetattr(terminal,TCSAFLUSH,&tty_state)) {
    int code = errno;
    if (EINTR == code) goto again3;
    print_error(code,"(%s) Can't change tty state of /dev/tty",prog);
    close_terminal();
    return 0;
  }
  tty_state_set = 1;


 again4:  
  if (-1 == tcflush(terminal,TCIFLUSH)) {
    int code = errno;
    if (EINTR == code) goto again4;
    print_error(code,"(%s) Can't flush input for /dev/tty",prog);
  }

  get_winsize(1);
  return 1;
}

void close_terminal(void) {
  if (-1 == terminal) return;

  if (!terminal_have_lost)
    flush_buffer(0);          /* print buffer to terminal */

  if (termios_set) {
  again1:
    if (-1 == tcsetattr(terminal,TCSADRAIN,&tty_origstate)) {
      int code = errno;
      if (EINTR == code) goto again1;
      print_error(code,"(%s) Can't restore tty state for /dev/tty",prog);
    }
    termios_set = 0;
  }
  if (have_group) {
  again2:
    if (-1 == tcsetpgrp(terminal,oldgroup)) {
      
      int code = errno;
      if (EINTR == code) goto again2;
      
      print_error(code,"(%s) Can't change terminal group to %d",
		  prog,oldgroup);
    } else if(DEBUG_SIGNAL || DEBUG_TERM)
      print_debug("Terminal group is now restored to %d",oldgroup);
  }

  /* Terminal group must restore out keeper process is killed */
  kill_friend();

  if (have_set(SIGCHLD) && 
      -1 == sigaction(SIGCHLD,&old_sigchld,NULL))
    print_error(errno,"Error restoring of SIGCHLD (sigaction)");
  
  if (have_set(SIGTTOU) && 
      -1 == sigaction(SIGTTOU,&old_sigttou,NULL))
    print_error(errno,"Error restoring of SIGTTOU (sigaction)");

  if (have_set(SIGTSTP) && 
      -1 == sigaction(SIGTSTP,&old_sigtstp,NULL))
    print_error(errno,"Error restoring of SIGTSTP (sigaction)");

  if (have_set(SIGTTIN) && 
      -1 == sigaction(SIGTTIN,&old_sigttin,NULL))
    print_error(errno,"Error restoring of SIGTTIN (sigaction)");
  
  if (have_set(SIGWINCH) && 
      -1 == sigaction(SIGWINCH,&old_sigwinch,NULL))
    print_error(errno,"Error restoring of SIGWINCH (sigaction)");

  if (have_set(SIGHUP) && 
      -1 == sigaction(SIGHUP,&old_sighup,NULL))
    print_error(errno,"Error restoring of SIGHUP (sigaction)");

  if (-1 == close(terminal))
#ifdef __sgi
    if (ENODEV != errno)   /* I don't know from where this error comes ! */
#endif
      print_error(errno,"Error when closing terminal");
  terminal = -1;
}

#define BUFFER_LEN 1000

static volatile CHAR buffer[BUFFER_LEN], *lptr = buffer;

static int add_to_buffer(int len, CHAR *text) {
  block_state S;

  if (lptr + len >= buffer + BUFFER_LEN) return 0;
  BLOCK(&S);
  if (lptr + len >= buffer + BUFFER_LEN) {
    UNBLOCK(&S);
    return 0;
  }

  memcpy((char *)lptr,(void *)text,len);
  lptr += len;
  UNBLOCK(&S);

  if (DEBUG_BUFFER) {
    print_debug("BUFFER: added %d bytes to buffer",len);
  }



  return 1;
}

int need_flush_buffer(void) {
  int res = lptr > buffer;
  if (DEBUG_BUFFER) {
    print_debug("BUFFER: need_flush_buffer = %d",res);
  }
  return res;
}

void flush_buffer(int noerror) {
  block_state S;
  volatile CHAR *last,*ptr=buffer;
  int len,res;

  /* We have problem: BLOCKing isn't allowed during write (bacause 
     we wanna get signal when kehpager is in background). So
     this routine will be interpret by self !!
     */


 again:
  /* hopeless ... */

  last = lptr;
  len = last - ptr;

  if (len <= 0) {
    if (DEBUG_BUFFER) {
      print_debug("BUFFER: flushing - already done");
    }

    return;  
  }

  if (DEBUG_BUFFER) {
    print_debug("BUFFER: flushing %d bytes",len);
  }

  /* If some other calling of flush_buffer is already writed this !! */
  if (terminal_have_lost) return;
  if (!tty_state_set) set_pager_ttystate();

  res = write(terminal,(char *)buffer,len);
  if (-1 == res) {
    int code = errno;

    if (DEBUG_BUFFER) {
      print_debug("BUFFER: ... flushing fail");
    }
    
    if (EINTR == code) goto again;
    if (noerror) return;
    print_error(code,"(%s) Failed to write to /dev/tty",
		prog);
    return;
  }

  if (DEBUG_BUFFER) {
    print_debug("BUFFER: ... flushed %d bytes",res);
  }

  ptr += res;
  if (ptr < lptr)
    goto again;

  BLOCK(&S);
  if (ptr < lptr) {
    UNBLOCK(&S);
    goto again;
  }
  lptr = buffer;
  UNBLOCK(&S);
}

void print_to_terminal(CHAR *text) {
  int len = strlen(Cs(text));

  if (add_to_buffer(len,text)) return;
  flush_buffer(0);
  if (!add_to_buffer(len,text)) {
    print_notify("(%s) print_to_error: Can't add to buffer (too long line?), "
		"len=%d, MAX=%d", prog,len,BUFFER_LEN);
  }
}



int read_from_terminal(CHAR *buffer) {   
  int res;

  /* hopeless */
  if (terminal_have_lost) {
    /* We should not be here ... */
    print_notify("(%s) read_from_terminal/terminal_have_lost = 1, aborting...",
		 prog);
    
    reset_terminal_state();    /* give original terminal characters */
    set_caller_ttystate();
    
    close_files();
    close_terminal();
    
    abort();
  }
  if (!tty_state_set) set_pager_ttystate();
  res = read(terminal,Cs(buffer),INPUT_BUFFER_SIZE);
  if (res < 0) {
    int code = errno;
    if (EINTR == code) {
      buffer[0] = '\0';
      return 0;
    }
    print_error(code,"(%s) Failed to read from /dev/tty",
		prog);
    quitflag = 1;
    buffer[0] = '\0';
    return 0;
  }

  if (0 == res) {
    print_notify("(%s) Unexpected eof when reading /dev/tty\n",prog);
    buffer[0] = '\0';
    quitflag = 1;
    return 0;
  }

  buffer[res] = '\0';
  return res;
}
